/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME:         blddisplay.c
 *
 * SCCSINFO:            @(#)blddisplay.c	1.7 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Tim Teitenberg, 23-March-1990
 *
 * MODIFICATIONS:
 *	<list mods here with name and date>
 *
 * DESCRIPTION:
 * This module contains functions which traverse through the selection
 * structure and displays it on the chosen device-type. The module builds
 * a widget structure for the X-window display if no other device-type is
 * selected.
 *
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 * void blddisplay(reference_structure *rsp)
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "aimtypes.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_blddisplay.h"

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimpropagate.h"
#include "f_xstuff.h"
#include "f_uim_propagate.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* used for logging messages */
extern int LL_uid;			/* liblincks.a */
extern char *sys_errlist[];		/* liblincks.a */
extern int sys_nerr;			/* liblincks.a */
extern int errno;			/* liblincks.a */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* number of formats one can have associated with one widget */
#define NUM_FORMATS 100		

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static formentry *create_format_list P_(( void ));
static void display_node P_(( reference_structure *ref_struct, infonode
                               *np ));
static void free_format_strings P_(( int level ));
static void stack_formats P_(( infonode *np ));

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
static Widget previouswidp;
static formentry format_stack[NUM_FORMATS];	/* wild guess */
static int format_stack_ptr;

/*  */
/*
 * Description of formatting attribute handling:
 *	At the database level, formatting information is stored as
 *	name/value text strings which are to be used by the user
 *	interface layer (UIM) in the manner appropriate to it.  For each
 *	infonode encountered, the template reference structure and the 
 *	target reference structure are checked for formatting
 *	information attached to an infonode.  This is done in the
 *	function stack_formats(), which puts all of the formatting
 *	attributes onto a stack local to this file.  As the function
 *	display_node() descends, new attributes are added in order to
 *	facilitate inheritance through the reference structure.
 *	However, when a call to display_node returns, the stack is
 *	restored to the level it was when that call to display_node was
 *	made.  This is the function of free_format_strings().  Just
 *	before each call to propagation_queue_add(), a copy is created
 *	of the current state of the global stack.  This list is then
 *	attached to the propagation queue element to be used when
 *	creating the widget.  The function which creates this list is
 *	create_format_list().
 */

/*  */
/**********************************************************************
 *
 * Function: void blddisplay(reference_structure *rsp)
 * 
 * This function builds a display for a reference structure, and maps it
 * to the chosen device, default is text (on screen).
 * 
 * Modified:
 * 	1992-02-07 David Partain:  
 *		(1) rsp->flags is incremented on the call to mapwidgets.
 *		    mapwidgets() is only called once.
 *		(2) propagation_queue_process is called to process 
 *		    the propagation queue.
 *	1992-09-17, Ralph: added call to refreshlayout().
 *	1992-09-21, David and Ralph:  changed to make it do changes to 
 *		the window only if it's REALLY changed.
 *	1992-10-11, Ralph: replaced using rsp->flags with checking
 *		rsp->contw assignment.
 *	1993-03-01, Ralph: removed the FocusOnDisplay call.
 *	1993-03-23, David: removed the FocusOnDisplay parameter, removed
 *		speech stuff.
 *
 *	<list mods here with name and date>
 */
void blddisplay(rsp)
  reference_structure *rsp;
{
  int newref = 0;

  if (rsp == (reference_structure *) NULL)
    return;

  format_stack_ptr = 0;	/* initialize pointer into stack of formats */

  previouswidp = NULL;
  if ((newref = (rsp->contw == (Widget)NULL)))
    create_container(rsp);

  setupdefaultargs(rsp);	/* once per reference structure */

  if (rsp->flags.bound_changed) {
    markbound(rsp, (int)rsp->flags.bound);
    rsp->flags.bound_changed = 0;
  }

  display_node(rsp, rsp->target->value);

  if (rsp->pqueue.tail == (propagation_q_element *) NULL)
    return;

  if ((propagation_queue_process(rsp->pqueue.head)) == FAIL)
    fatal_error("propagation_queue_process failed:blddisplay\n");

  rsp -> pqueue.tail = rsp -> pqueue.head = (propagation_q_element *) NULL;

  if (newref)
    mapwidgets();

  refreshlayout(rsp,0);
}

/*  */
/**********************************************************************
 * Function: static void display_node(reference_structure ref_struct, infonode *np)
 *
 * This function builds a display for an infonode.
 *
 * Modified:
 *	1992-02-07 David Partain:  adopt the new propagation strategy.
 *
 * 	<list mods here with name and date>
 */
static void display_node(ref_struct, np)
  reference_structure *ref_struct;
  infonode *np;
{
  formentry *sent_formats = (formentry *) NULL;
  identry *idp;
  int current_stack_level;

/* 
 * add the formatting attributes to a stack of formatting attributes,
 * which will then be attached to a propagation queue element and 
 * handled in propagation_queue_process() in uim_propagate.c
 */
  current_stack_level = format_stack_ptr;	/* used to restore */
  stack_formats(np->head->tmpl->value);		/* Generic formatting */
  stack_formats(np);				/* Specific formatting */

  if ((idp = np->subnodes) == (identry *) NULL) {
    if (np->widp == NULL) {	/* Only create widgets for new infonodes. */
      /* 
       * copy current stack of formatting attributes and get a
       * pointer to the first one so that it can be sent to the 
       * propagation queue processing function.
       */
      sent_formats = create_format_list();
      /* deal with the propagation queue */
      if ((propagation_queue_add(ref_struct,
				   np,
				   previouswidp, 
				   sent_formats,
                                   PQE_OP_MAKEWIDGET)) == FAIL) {
	  fatal_error("propagation_queue_add in display_node\n");
      }
    }
    previouswidp = np->widp;
  } 
  else {
    while (idp != (identry *) NULL) {
      for (np = idp->value; np != (infonode *) NULL; np=np->next) {
	display_node(ref_struct,np);
      }
      idp = idp->next;
    }
  }
  /* 
   * reset the format stack to level when entered this function
   * and free up the strings that are in the structures above that level
   */
  free_format_strings(current_stack_level);
}

/*  */
/**********************************************************************
 * Function: static formentry *create_format_list()
 *
 * This function creates a list of formatting attributes which will
 * be used in creation of the widget (in X).  The global structure
 * format_stack is copied in order to accomplish this.  This strategy
 * is used in order to facilitate proper formatting inheritance within
 * the reference structure.  The list will then be associated with a 
 * propagation queue element and used in the function 
 * propagation_queue_process() or whatever function processes the 
 * propagation queue.
 *
 * Modified: 
 *	<list mods here with name and date>
 *
 * NOTE:  memory is supposed to be free'ed by propagation_queue_process
 *
 */
static formentry *create_format_list()
{
  int current = 0;
  formentry *tmp_format = (formentry *) NULL;
  formentry *return_stuff = (formentry *) NULL;
  formentry *last_format = (formentry *) NULL;

  while ((format_stack[current].formid) && (format_stack[current].formval)) {
    /* 
     * This memory must be free'ed by propagation_queue_process 
     */
    tmp_format = (formentry *) malloc(sizeof(formentry));
    if(tmp_format == (formentry *)NULL) {
      LogMess(LL_uid,
        "aim: malloc error: pqe_element propagation_queue_add_format, %s",
        sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    }
    tmp_format->next = (formentry *) NULL;
    tmp_format->formid = (char *)strdup(format_stack[current].formid);
    tmp_format->formval = (char *)strdup(format_stack[current].formval);
 
    if (return_stuff == (formentry *)NULL) {
      return_stuff = tmp_format;
      last_format = tmp_format;
    }
    else {
      last_format -> next = tmp_format;
      last_format = tmp_format;
    }
    current++;
  } /* while */
  return(return_stuff);
}

/*  */
/*********************************************************************
 * Function: static void stack_formats(infonode *np)
 *
 * This function "stacks" formatting attributes for a particular 
 * infonode onto a global stack which will then be copied and sent
 * to the propagation queue processing routine.
 *
 * Modifications:
 *	<list mods here with name and date>
 * 
 */
static void stack_formats(np)
  infonode *np;
{
  formentry *fp;

  for (fp = np->formats; fp != (formentry *) NULL; fp = fp->next) {
    format_stack[format_stack_ptr].formid = (char *)strdup(fp->formid);
    format_stack[format_stack_ptr].formval = (char *)strdup(fp->formval);
    format_stack_ptr++;
    if (format_stack_ptr >= NUM_FORMATS) {
      fatal_error("stack_formats: overflow in num of formatting elements\n");
    }
  }
}

/*  */
/*********************************************************************
 * Function: static void free_format_strings(int level)
 *
 * This function frees up strings allotted to an element of the 
 * format stack and sets them to null pointers.
 * 
 * Modifications:
 *	<list mods here with name and date>
 */
static void free_format_strings(level)
  int level;		/* level from which to begin free'ing */
{
  format_stack_ptr = level;
  while ((format_stack[level].formid) && (format_stack[level].formval)) {
    free((FREEPTR *) format_stack[level].formid);
    free((FREEPTR *) format_stack[level].formval);
    format_stack[level].formid = (char *) NULL;
    format_stack[level].formval = (char *) NULL;
    level++;
  }
}
