/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	autolinks.c
 *
 * SCCSINFO:		@(#)autolinks.c	1.8 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Ralph R|nnquist and Patrick Oln{s, 1991-03-13
 *
 * MODIFICATIONS:
 *      <list mods with name and date>
 *
 * DESCRIPTION:
 *
 * This file contains the autolink functionality.
 *
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 * int autolink(reference_structure *ref, infonode *inode)
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "parser.h"
#include "aimtypes.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_autolinks.h"

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimsubr.h"
#include "f_aimcmdstream.h"

/* libshared */
extern char *strdup( /* char *incoming */ );

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static ACTION(link1);
static ACTION(link2);
static ACTION(link3);
static ACTION(newplace0);
static ACTION(newplace1);
static ACTION(newplace2);
static ACTION(putlink);
static ACTION(steplink);
static void initautolinks P_(( void ));
static int peepsemicolon P_(( char **str ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/*
 * The following is set up at first initialisation prior to processing
 * the autolink expression.
 */
static label rootlbl;		/* Root of orginiating view (%SELF%) */
static label datalbl;		/* Object to carry autolinks */
static char *gpdtag;		/* Tag to find autolink expression for */

/*
 * The following constitute an "access record" which is updated
 * according to each autolink phrase.
 */
static label thelbl;		/* The object to link to. */
static char *thegrp = 0;	/* The last link group tag used. */
static char *thefld = 0;	/* The last link field tag used. */
static int thepos;		/* The last link item position used. */

/*
 * The AUTOLINKS language. Assumes <id> <integer> and <empty> to be
 * defined non-terminals, and "drop1" and "noop" to be predefined
 * actions.
 */
static char *AUTOLINK_SYNTAX[] =
{

  "<AUTOLINKS> ::= [ <autolink> while ; ] => drop1",

  "<autolink> ::= <newplace> = <oldplace> => putlink",

  "<newplace> ::= <id> <id> => newplace2 !\
	<id> => newplace1 !\
	<empty> => newplace0",

  "<oldplace> ::= %SELF% => noop !\
	[ <link> while + ] => drop1",

  "<link> ::= <id> <peepsemicolon> => link1 !\
	<id> <id> <integer> => link3 !\
	<id> <id> => link2 !\
	<id> => link1 !\
	<empty> => steplink",
  0
};

static actionentry AUTOLINK_ACTIONS[] =
{
  {"putlink", putlink},
  {"newplace2", newplace2},
  {"newplace1", newplace1},
  {"newplace0", newplace0},
  {"link3", link3},
  {"link2", link2},
  {"link1", link1},
  {"steplink", steplink},
  {0, 0}
};

/*  */
/**********************************************************************
 * Function: static PRIMARY(peepsemicolon)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static PRIMARY(peepsemicolon) /* Begins the function body!! */
  p = NULL;
  Ensuretoken;
  Failif(Peep != ';');
  return(ACCEPT);
}

/*  */
/**********************************************************************
 * Function: static void initautolinks()
 *
 * This function (re)initialises the access record.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void initautolinks()
{
  if (thegrp != (char *)NULL)
    free(thegrp);
  if (thefld != (char *)NULL)
    free(thefld);

  (void)copylbl(&thelbl, &rootlbl);	/* Assume the root object (%SELF%) */
  /*
   * The default access for a %SELF% link is BACKLINKS:<gpdtag>:1
   */
  thegrp = strdup("BACKLINKS");
  thefld = strdup(gpdtag);
  thepos = 1;
}

/*  */
/**********************************************************************
 * Function: int autolink(label *gpd,label *root,label *new,char *tag)
 *
 * This function is the entry point which processes the autolinks.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int autolink(gpd, root, new, tag)
  label *gpd, *root, *new;
  char *tag;
{
  static int autolink_inited = 0;
  int rtn;
  label rpd;

  if (!autolink_inited) {
    /*
     * This fork is followed only once in order to define the
     * AUTOLINKS language syntax and actions.
     */
    defineactions(AUTOLINK_ACTIONS);
    bindprimary("peepsemicolon",peepsemicolon);
    definegrammar(AUTOLINK_SYNTAX);
    autolink_inited = 1;
  }
  gpdtag = strdup(tag);	/* Grab private copy. */
  (void)copylbl(&rpd, gpd);
  (void)copylbl(&rootlbl, root);
  (void)copylbl(&datalbl, new);

  initautolinks();

  /* Process the autolinks expression (if any) */
  rtn = DoAttribute(&rpd, "AUTOLINKS", gpdtag, "AUTOLINKS");
  if (gpdtag)
    free(gpdtag);		/* Release private copy. */
  return rtn;
}


/*  */
/**********************************************************************
 * Function: static ACTION(steplink)
 *
 * Step to next link using the access record.
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(steplink)
{
  label lbl;

  if (thelbl.vs == -1)
    END_ACTION;

  if (GLI_GETLINKITEM(&thelbl, thegrp, thefld, thepos, &lbl) != SUCCESS)
    thelbl.vs = thelbl.inst = -1;
  else
    (void)copylbl(&thelbl, &lbl);
  END_ACTION;
}


/*  */
/**********************************************************************
 *
 * Following are the actions associated to the AUTOLINKS language.
 *
 */

static ACTION(putlink)
{
  char *s;

  if ((s = popdata()) != (char *)NULL) {
    free(thefld);
    thefld = s;
  }
  if ((s = popdata()) != (char *)NULL) {
    free(thegrp);
    thegrp = s;
  }
  if ((thelbl.vs != -1) && (thegrp != (char *)NULL) && (thefld != (char *)NULL))
    newlastcmd(LINKOP, &datalbl, LINKITEM,
	       thegrp, thefld, 1, (attrval *) NULL, &thelbl);

  /*
   * Reset access record for next (if any) autolink
   * expression.
   */
  initautolinks();
  END_ACTION;
}


static ACTION(newplace2)
{
  /* Do nothing */
  if (thefld)
    free(thefld);
  if (thegrp)
    free(thegrp);
  thefld = popdata();
  thegrp = popdata();
  pushdata(strdup(thegrp));
  pushdata(strdup(thefld));
  END_ACTION;
}

static ACTION(newplace1)
{
  if (thegrp)
    free(thegrp);
  thegrp = popdata();
  pushdata(strdup(thegrp));
  pushdata((char *)NULL);
  END_ACTION;
}

static ACTION(newplace0)
{
  pushdata((char *)NULL);
  pushdata((char *)NULL);
  END_ACTION;
}

static ACTION(link3)
{
  thepos = popint();
  free(thefld);
  thefld = popdata();
  free(thegrp);
  thegrp = popdata();
  (void)steplink();
  END_ACTION;
}

static ACTION(link2)
{
  thepos = 1;
  free(thefld);
  thefld = popdata();
  free(thegrp);
  thegrp = popdata();
  (void)steplink();
  END_ACTION;
}

static ACTION(link1)
{
  thepos = 1;
  free(thegrp);
  thegrp = popdata();
  (void)steplink();
  END_ACTION;
}
