/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	aimsubr.c
 *
 * SCCSINFO:		@(#)aimsubr.c	1.12 5/19/94
 *
 * ORIGINAL AUTHOR(S):  Ralph R|nnquist, 1990-03-13
 *
 * MODIFICATIONS:
 *      1993-11-02 Martin Sjlin - moved whichref from aimcommand
 *                 into here, since it is of general interests.
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 * This file contains a number of useful subroutines for the Aim module.
 *
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 * Copy object label
 * label *copylbl(label *tolbl, label *fromlbl)
 *
 * Find value of inheritance attribute
 * char *InheritAttribute(label *obj, char *grp, char *fld)
 *
 * Parse string to string goal
 * int DoParseExec(char *str,char *goal);
 *
 * Parse and execute inheritance attribute value
 * int DoAttribute(label *lbl,char *grp,char *fld,char *goal)
 *
 * Get real group tag
 * char *grouptag(identry *idp)
 *
 * Get real field tag
 * char *fieldtag(identry *idp)
 *
 * Find the tag to expand an item by
 *  char *findexpandtag(infonode *np)
 *
 * int clean_and_store_bt(reference_structure *ref)
 *
 * Find reference structure of infonode or identry, depending on "which"
 * which == 0, check in, which == 1, check id.
 *  reference_structure *whichref(infonode *in,identry *id,int which)
 *
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "parser.h"
#include "aimtypes.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_aimsubr.h"

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_xstuff.h"

#ifdef NEED_STRDUP
extern char *strdup( /* char *incoming */ );
#endif	/* NEED_STRDUP */

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* aimstart.c */
extern reference_structure *refstructs;	/* List of reference structures.   */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
enum errors { e_group, e_value };

typedef struct _gpderrs {
  struct _gpderrs *next;
  enum errors error;
  infonode *parent;
} gpderrs;

#define GROUP_WARN \
"The current GPD uses the %%GROUP%% directive to access the\n\
group tag of the parent.  However, the parent is an IMAGE\n\
and has a NULL group tag.  The GPD is wrong and should be fixed.\n\
You will only get this message once."

#define VALUE_WARN \
"The current GPD uses the %%VALUE%% directive to access the\n\
value of the parent.  However, the parent is an IMAGE\n\
and has a NULL value.  The GPD is wrong and should be fixed.\n\
You will only get this message once."

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* Parse and execute string */
static char *idtag P_(( char *tag, infonode *parent ));
static int add_from_identry P_(( label *bt, identry *idp ));
static int add_from_infonode P_(( label *bt, infonode *inp ));
static int expandtagmapfn P_(( char *p, char *field, int c, int n ));
static void gpd_error P_(( infonode *parent, enum errors error ));

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
static struct {
  label obj;
  char *group;
  char *gpdname;
  int agree;
} expandmapinfo;


/*  */
/**********************************************************************
 * Function: label *copylbl(label *tolbl, label *fromlbl)
 *
 * Copies the label fromlbl into tolbl. Returns tolbl
 *
 * Modifications:
 *      <list mods with name and date>
 */
label *copylbl(tolbl,fromlbl)
     label *tolbl, *fromlbl;
{
  tolbl->vs = fromlbl->vs;
  tolbl->inst = fromlbl->inst;
  return(tolbl);
}

/*  */
/**********************************************************************
 * Function: char *InheritAttribute(label *obj, char *grp, char *fld)
 *
 * Follow chain in links grp:fld:1 until an attribute grp:fld is
 * found. Return the value of this attribute, and set obj to the
 * instance containing the attribute.
 * Errors are reported by NULL returns.
 *
 * Modifications:
 *      <list mods with name and date>
 */
char *InheritAttribute(obj,grp,fld)
     label *obj;
     char *grp, *fld;
{
  label lbl;
  attrval v;

  if (!BOUND(obj) && GC_GETCURRENT(obj) != SUCCESS) 
    return NULL;
  while (GA_GETATTR(obj,grp,fld,&v) != SUCCESS) {
    if (GLI_GETLINKITEM(obj,grp,fld,1,&lbl) != SUCCESS) 
      return NULL;
    (void)copylbl(obj,&lbl);
    if (!BOUND(obj) && GC_GETCURRENT(obj) != SUCCESS) 
      return NULL;
  }
  if (v.attsize == 0) return NULL;
  *(v.attvalue+v.attsize-1) = '\0';
  return v.attvalue;
}

  
/*  */
/**********************************************************************
 * Function: int DoParseExec(char *str, char *goal)
 *
 * This function parses and executes the string str towards goal.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int DoParseExec(str,goal)
     char *str, *goal;
{
  funcell *fxpr;
  int code;

  code = parse(findsymbol(goal),&str);
  if (code != ACCEPT && code != ACCEPTX)
    return FAIL;
  fxpr = funexpr();
  execute(fxpr);
  freefuncell(fxpr);
  return SUCCESS;
}


/*  */
/**********************************************************************
 * Function: int DoAttribute(label *lbl,char *grp,char *fld,char *goal)
 *
 * This function uses InheritAttribute to find a string which
 * is then parsed towards the given goal. If parsing is successful,
 * the associated actions are also executed.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int DoAttribute(lbl,grp,fld,goal)
     label *lbl;
     char *grp, *fld, *goal;
{
  label xlbl;
  char *str;

  (void)copylbl(&xlbl,lbl);
  str = InheritAttribute(&xlbl,grp,fld);
  if (!str)
    return FAIL;
  if (DoParseExec(str,goal) != SUCCESS) {
    free(str);
    return FAIL;
  }
  free(str);
  (void)copylbl(lbl,&xlbl);
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static char *idtag(char *tag, infonode *parent)
 *
 * The following functions are used for determining the access
 * group and field tags for target identries, implementing the
 * ability to re-use the parent group tag or value as an access tag.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static char *idtag(tag,parent)
     char *tag;
     infonode *parent;
{ /* Translates tag */
  if (tag == (char *) NULL) return tag;
  if (strcmp(tag,"%GROUP%") == 0) {
    if (!parent->group)
      gpd_error(parent, e_group);
    return parent->group;
  }
  if (strcmp(tag,"%VALUE%") == 0) {
    if (!parent->val.attvalue) {
      /* next condition tests to see if it has automagic initialization */
      if (!((parent->head) &&	
          (parent->head->tmpl) && 
          (parent->head->tmpl->owner) && 
          (parent->head->tmpl->owner->val.attvalue)))
      gpd_error(parent, e_value);
    }
    return parent->val.attvalue;
  }
  return tag;
}

/*  */
/**********************************************************************
 * Function: static void gpd_error( infonode *parent, enum errors error)
 *
 * pop up an error message about bad usage of %VALUE% or %GROUP%
 * (if it hasn't been popped up about this parent before)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void gpd_error(parent,error)
  infonode *parent;
  enum errors error;
{
  static gpderrs *head;
  gpderrs *tmp;

  for (tmp = head; tmp; tmp = tmp->next)
    if ((tmp->parent == parent) && (tmp->error == error))
      return;	/* just get the message once */
  
  tmp = (gpderrs *)malloc(sizeof(gpderrs));
  if (!tmp)
    fatal_error("%s\n", "malloc failed in gpd_error");
  tmp->parent = parent;
  tmp->error = error;
  tmp->next = head;
  head = tmp;

  if (error == e_value) {
    popup_message(POPUP_MESS_ERROR, VALUE_WARN);
    return;
  }
  popup_message(POPUP_MESS_ERROR, GROUP_WARN);
}

/*  */
/**********************************************************************
 * Function: char *grouptag(identry *idp)
 *
 * Determines access group tag for target identry.
 *
 * Modifications:
 *      <list mods with name and date>
 */
char *grouptag(idp)
     identry *idp;
{
  return idtag(idp->tmpl->value->group,idp->owner);
}

/*  */
/**********************************************************************
 * Function: char *fieldtag(identry *idp)
 *
 * Determines access field tag for target identry.
 *
 * Modifications:
 *      <list mods with name and date>
 */
char *fieldtag(idp)
     identry *idp;
{
  return idtag(idp->tmpl->value->field,idp->owner);
}

/* ^L */
/* ARGSUSED */
/**********************************************************************
 * Function: static int expandtagmapfn(char *p, *field,int c, n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int expandtagmapfn(p,field,c,n)
  char *p;
  char *field;
  int c;
  int n;
{
  attrval v;

  if (c == 0)
    return GVS_CONTINUE;
  v.attvalue = 0;
  v.attsize = 0;
  if (GA_GETATTR(&expandmapinfo.obj,expandmapinfo.group,field,&v) ||
      !v.attvalue) {
    expandmapinfo.agree = 0;
    return GVS_STOP;
  }
  if (c == 1) {
    expandmapinfo.gpdname = v.attvalue;
    return GVS_CONTINUE;
  }
  if (strcmp(v.attvalue,expandmapinfo.gpdname)) {
    expandmapinfo.agree = 0;
    free(v.attvalue);
    return GVS_STOP;
  }
  free(v.attvalue);
  return GVS_CONTINUE;
}

/*  */
/**********************************************************************
 * Function: char *findexpandtag(infonode *np)
 *
 * This function determines how to normally view an infonode; it returns
 * a name for a GPD to use for expansion.
 * The initial tag is obtained as (np->head->idname), and it is resolved
 * into a GPD name as follows (in priority order):
 * 1. Use "EXPAND AS":<tag> for expansion object
 * 2. Use "EXPAND PART":<tag> for root object
 * 3. Use EXPAND:<tag> for GPD
 * 4. Use any "EXPAND AS":xxx value if all such values agrees
 * 4. Use copy of tag.
 *
 * Modifications:
 *      <list mods with name and date>
 */
char *findexpandtag(np)
     infonode *np;
{
  attrval v;
  label lbl;
  char *tag = np->head->idname;
  char *s;
  infonode *saveinp = np;

  (void)copylbl(&lbl,(np->index == LINKITEM)?(&(np->lbl)):(&(np->obj)));
  if ((lbl.vs != -1) && !GA_GETATTR(&lbl,"EXPAND AS",tag,&v))
    return v.attvalue;
  (void)copylbl(&expandmapinfo.obj,&lbl);
  expandmapinfo.group = "EXPAND AS";
  expandmapinfo.agree = 1;
  expandmapinfo.gpdname = (char *)NULL;

  while (np->parent)
    np = np->parent;

  (void)copylbl(&lbl,&(np->obj));
  if (!GA_GETATTR(&lbl,"EXPAND PART",tag,&v))
    return v.attvalue;

  (void)copylbl(&lbl,&(np->selobj));
  if ((s = InheritAttribute(&lbl,"EXPAND",tag)))
    return s;

  (void)copylbl(&lbl,&(saveinp->selobj));
  if ((s = InheritAttribute(&lbl,"EXPAND",tag)))
    return s;

  if (expandmapinfo.obj.vs != -1) {
    (void)GAN_GETATTRNAMES(&expandmapinfo.obj, expandmapinfo.group,
			   expandtagmapfn, (char *)NULL);
    if (expandmapinfo.gpdname) {
      if (expandmapinfo.agree)
	return expandmapinfo.gpdname;
      free(expandmapinfo.gpdname);
    }
  }
  return strdup(tag);
}

/*  */
/**********************************************************************
 * Function: int clean_and_store_bt(reference_structure *ref)
 *
 * go through the binding table, remove all the components and rebuild
 * the list.  this is to get rid of unneeded links in the binding table.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int clean_and_store_bt(ref)
  reference_structure *ref;
{
  label bt;
  
  if (!ISTRANSIENT(copylbl(&bt, &(ref->bt))))
    return BT_NOT_TRANSIENT;
  
  switch (CH_RLV_REMOVELINKVAL(&bt, "COMPOSITION","Components")) {
    case SUCCESS:
    case ERR_FIELD:
      break;
    default:
      return FAIL;
  }
  if (add_from_identry(&bt, ref->target))
    return BT_SOME_TRANSIENT;
  return CH_SO_STOREOBJ(&bt);
}

/*  */
/**********************************************************************
 * Function: static int add_from_identry(label *bt, identry *idp)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int add_from_identry(bt, idp)
  label *bt;
  identry *idp;
{
  int rtn = 0;

  while (idp) {
    rtn |= add_from_infonode(bt, idp->value);
    idp = idp->next;
  }
  return rtn;
}

/*  */
/**********************************************************************
 * Function: static int add_from_infonode(label *bt, infonode *inp)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int add_from_infonode(bt, inp)
  label *bt;
  infonode *inp;
{
  int rtn = 0;
  label lbl;

  while (inp) {
    if ((inp->index == LINKITEM) && !(inp->iflg & NONEXIST_MSK)) {
      if (GBTC_GETBTCOMP(bt, copylbl(&lbl, &(inp->lbl)), 1, 0))
	return 1;
      rtn |= ISTRANSIENT(&lbl);
    }
    rtn |= add_from_identry(bt, inp->subnodes);
    inp = inp->next;
  }
  return rtn;
}

/*  */
/**********************************************************************
 * Function: reference_structure *whichref(infonode *in,identry *id,int which)
 *
 * Find reference structure of infonode or identry, depending on "which"
 *	which == 0, check in,
 *	which == 1, check id.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
reference_structure *whichref(in, id, which)
  infonode *in;
  identry *id;
  int which;
{
  reference_structure *ref;

  if (which == 0) {
    if (in == (infonode *) NULL)
      return (reference_structure *) NULL;
    id = in -> head;
  }

  while (id->owner != (infonode *) NULL)
    id = id->owner->head;

  if (id->tmpl != (identry *) NULL)
    id = id->tmpl;

  /* That's the root of the template */

  for (ref = refstructs; ref != (reference_structure *) NULL; ref = ref->next)
    if (ref->template == id)
      return ref;

  return (reference_structure *) NULL;
}
