/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME:		aimexport.c
 *
 * SCCSINFO:		@(#)aimexport.c	1.17 5/19/94
 *
 * ORIGINAL AUTHOR:	David Partain, 1990-06-06
 *
 * MODIFICATIONS:
 *	1993-11-26 Martin Sjlin and David Partain.  Reworked several
 *			things, but the gist of it is that there are now
 *			two additional configuration options.  (1) you
 *			set whether to escape characters for LaTeX.  (2)
 *			you say whether you want placeholders included.
 *      1993-12-27 Martin Sjlin. Moved font references into 
 *                      application resource file as well fall back resource
 *                      (to ease running small screens)
 *      1994-05-14 Martin Sjlin. Get defaults values for choices from 
 *                      application defaults.
 *      1994-05-17 Martin Sjlin. Moved all related stuff to its on file.
 *
 *      <list mods here with name and date>
 *
 * DESCRIPTION:
 *	This module will contain the functions that will enable
 *	the user to export a window as text (either LaTeX or straight)
 *	text.
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include "aimtypes.h"
#include "xconfig.h"			/* for DBDIR  */

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_aimexport.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 *********************************************************************/
char *default_style      = "";
int   default_usedeep    = TRUE;
int   default_fontsize   = 11;
int   default_useph      = TRUE;
int   default_doesc      = TRUE;
int   default_latex      = TRUE;

EscType global_escape = doit;		/* whether to escape chars for LaTeX */
DumpType global_type = LaTeX;		/* whether to dump as straight text */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#ifdef NEED_STRDUP
extern char *strdup( /* char *incoming */ );
#endif	/* NEED_STRDUP */

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/

/* file which holds font translations from their X to LaTeX forms */
#define FONT_TRANSLATION_FILE ".fonttrans"

/*
 * In escape_sp_chars, these two are used to signify a left quotation
 * mark and to signify a right quotation mark.
 */
#define LEFT 0
#define RIGHT 1

/* these indicate the depth of the LaTeX file to be created.  */
#define FIRST_LEVEL 1
#define SECOND_LEVEL 2
#define THIRD_LEVEL 3
#define FOURTH_LEVEL 4
#define FIFTH_LEVEL 5

/* print_part prints all of these to the file */
#define PART 1
#define PARAGRAPH 2
#define SECTION 3
#define SUBSECTION 4
#define SUBSUBSECTION 5
#define TRAILER 6
#define LETTERHEAD 7
#define LETTERTAIL 8
#define SQUIGGLES 9

/* when creating title, these are used to choose parameters.  */
#define TITLE 1
#define DATE 2
#define AUTHOR 3

#define BUF_SIZE 100		/* for strings */
#define MAX_STACK 10		/* number of LaTeX fonts on stack */

/*
 * This structure holds a pair of information.  When an X font is found,
 * its corresponding LaTeX font is loaded into "fontname", and the
 * number of {'s to close is put into "completion".
 */
typedef struct x_LaTeXfonts {
  char *fontname;
  int completion;
} LaTeXfonts;

typedef struct _filefontinfo {
  struct _filefontinfo *next;
  char *Xfontname;
  char *LaTeXname;
  int squigglies;
} filefontinfo;

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int add_formats P_(( infonode *infoptr, char *leftmargin ));
static char *attval_plus P_(( infonode *infoptr, char *localfont ));
static int check_identry P_(( identry *idp, char **title, char **author,
                               char **date));
static int check_infonode P_(( infonode *inp, char **title, char
                                **author, char **date));
static void clear_printed P_(( infonode *inodeptr ));
static int compare_strings P_(( char *string, int to_find, int *mark ));
static void delete_formats P_(( int where ));
static char *escape_sp_chars P_(( char *text ));
static void find_sect_title P_(( _textdump *textdump, infonode *infoptr, int
                                  check_level, FILE *outfile ));
static char *get_file_name P_(( void ));
static filefontinfo *get_format P_(( char *lookfor ));
static void get_header P_(( _textdump *textdump, FILE *outfile ));
static void get_title P_(( FILE *outfile, identry *idptr));
static void LaTeX_node P_((_textdump *textdump, infonode *infoptr, FILE *outfile ));
static int load_file_fonts P_(( void ));
static void make_art_title P_(( FILE *outfile, char *title, char *author, 
			       char *date ));
static int my_level P_(( identry *idp ));
static filefontinfo *new_filefontinfo P_(( void ));
static void num_levels P_(( infonode *inodeptr, int level ));
static void print_indent P_(( FILE *outfile, int debug ));
static void print_node P_(( _textdump *textdump, infonode *infoptr, FILE *outfile ));
static void print_part P_((FILE *outfile, char *part, int which ));
static char *substitute_title P_(( identry *idptr ));


/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static int prevlevel = 0;	/* level of previous leaf node */
static int thislevel = 0;	/* level of this leaf node */
static int depth_signal = 0;	/* signals that depth > 4 */
static int indent = 0;		/* number spaces to indent == indent * 3 */
static LaTeXfonts fontstack[MAX_STACK];/* LaTeX fonts */
static int stkptr = -1;		/* top of stack pointer in LaTeXfonts */
static char *prevfont = NULL;	/* last LaTeX font written to file */
static int lefttoprint = 0;	/* number of open squigglies */
static int global_level = 0;	/* global for number of levels */
static filefontinfo *fontinfo = NULL;

/*  */
/*******************************************************************
 * Function: static void num_levels(infonode *inodeptr, int level)
 *
 * This function finds the number of levels in a given reference
 * structure.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void num_levels(inodeptr, level)
  infonode *inodeptr;
  int level;
{
  identry *identptr = NULL;
  infonode *infoptr = NULL;

  if ((identptr = inodeptr->subnodes) == (identry *)NULL) {
    return;
  } else {
    level++;
    if (level > global_level) {
      global_level = level;
    }
    while (identptr != (identry *)NULL) {
      for (infoptr = identptr->value;
	   infoptr != (infonode *)NULL; infoptr = infoptr->next) {
	num_levels(infoptr, level);
      }
      identptr = identptr->next;
    }
  }

  if (global_level > 4)
    depth_signal = TRUE;
  return;
}

/*  */
/*******************************************************************
 * Function: static void get_title(FILE *outfile, identry *idptr);
 *
 * This function will find the appropriate title, author, and date
 * for the document and make the title for the LaTeX document.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void get_title(outfile, idptr)
  FILE *outfile;		/* pointer to LaTeX file */
  identry *idptr;		/* pointer to identry where search starts */
{
  char *author = NULL;
  char *title = NULL;
  char *date = NULL;
  identry *saveidptr = idptr;

  (void)check_identry(idptr, &title, &author, &date);
  if (!title)
    if (!(title = substitute_title(saveidptr)))
      title = strdup("This should be some title");;
  if (!author)
    author = strdup("This should be your name");
  if (!date)
    date = strdup("This should be the date");

  make_art_title(outfile, title, author, date);

  if (title)
    free((FREEPTR *)title);
  if (date)
    free((FREEPTR *)date);
  if (author)
    free((FREEPTR *)author);
  return;
}

/*  */
/*******************************************************************
 * Function: static int check_identry(FOUR PARAMETERS)
 * Parameters:
 *	identry *idp
 *	char **title
 *	char **author
 *	char **date
 *
 *  go through the identries in a reference structure, calling
 *  check_infonode for each set of infonodes hanging off of the identry.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static int check_identry(idp, title, author, date)
  identry *idp;
  char **title;
  char **author;
  char **date;
{
  int rtn = 0;

  while (idp) {
    rtn |= check_infonode(idp->value, title, author, date);			  
    if (rtn)
      return rtn;
    idp = idp->next;
  }
  return rtn;
}

/*  */
/*******************************************************************
 * Function: static int check_infonode(FOUR PARAMETERS)
 * Parameters:
 *	infonode *inp
 *	char **title
 *	char **author
 *	char **date
 *  
 *  at each leaf infonode, check to see if it is appropriate as a
 *  title, author or date.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static int check_infonode(inp, title, author, date)
  infonode *inp;
  char **title;
  char **author;
  char **date;
{
  int rtn = 0;
  int found = 0;
  int i;
  int mark = 0;

  while (inp) {
    if ((inp->index == ATTRVALUE) || (inp->index == LINKITEM)) {
      for (i = TITLE; i <= AUTHOR; i++) {
        switch(i) {
        case TITLE:
          if (*title)
            continue;
          break;
        case AUTHOR:
          if (*author)
            continue;
          break;
        case DATE:
          if (*date)
            continue;
          break;
        default:
          break;
        }
	found = compare_strings(inp->field, i, &mark);
	if (found == TRUE) {
	  switch (i) {
	  case TITLE:
	    if (!*title) {
	      *title = strdup(inp->val.attvalue);
	      *title = escape_sp_chars(*title);
	    }
	    break;
	  case AUTHOR:
	    if (!*author) {
	      *author = strdup(inp->val.attvalue);
	      *author = escape_sp_chars(*author);
	    }
	    break;
	  case DATE:
	    if (!*date) {
	      *date = strdup(inp->val.attvalue);
	      *date = escape_sp_chars(*date);
	    }
	    break;
	  default:
	    break;
	  }
	  if (mark)
	    inp->printed = 1;
	}
      }
    }
    rtn |= check_identry(inp->subnodes,title,author,date);
    inp = inp->next;
  }
  return rtn;
}


/*  */
/*******************************************************************
 * Function: static char *substitute_title(identry *idptr)
 *
 *  This function will provide a title when none has been found.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static char *substitute_title(idptr)
  identry *idptr;
{
  char *title = NULL;

  if (!(idptr && idptr->value && idptr->value->val.attsize)) 
    return strdup(idptr->idname);

  if (idptr->value->val.attsize < 30) 
    return strdup(idptr->value->val.attvalue);

  if (!(title = (char *)malloc((ALLOC_T)30)))
    return NULL;
  (void)strncpy(title, idptr->value->val.attvalue, 25);
  (void)strcat(title, " ...");

  return title;
}

/*  */
/*******************************************************************
 * Function: static int compare_strings(char *string, char *to_find, int *mark)
 *
 * This function does just that.  Only twist is that it checks
 * them in all lower case.  Actually, it looks for a match within
 * the substring of the first argument.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static int compare_strings(string, to_find, mark)
  char *string;
  int to_find;
  int *mark;			/* whether it should be marked as printed */
{
  char buf[BUF_SIZE];
  int i;
  int index = 0;
  int in_len = 0;
  int iteration = 0;

  static char *authors[] =
  {
    "author",
    "owner"
  };
  static char *dates[] =
  {
    "date",
    "created"
  };

  *mark = 0;
  if (to_find == TITLE)
    in_len = strlen("title");

  for (iteration = 0; iteration < 2; iteration++) {
    index = 0;
    if (to_find == DATE)
      in_len = strlen(dates[iteration]);
    else if (to_find == AUTHOR)
      in_len = strlen(authors[iteration]);

    (void)memset((char *)buf, 0, (int)BUF_SIZE);
    while (index <= (int) ((strlen(string)) - in_len)) {
      for (i = 0; i < in_len; i++) {	/* string of first in_len length */
	if (islower(string[i + index]))
	  buf[i] = string[i + index];
	else
	  buf[i] = tolower(string[i + index]);
      }
      buf[i] = '\0';
      switch (to_find) {
      case TITLE:
	if (!strcmp("title", buf)) {
	  *mark = 1;
	  return (TRUE);
	}
	break;
      case DATE:
	if (iteration == 0) {
	  if (!strcmp("date", buf)) {
	    *mark = 1;
	    return (TRUE);
	  }
	} else {
	  if (!strcmp("created", buf))
	    return (TRUE);
	}
	break;
      case AUTHOR:
	if (iteration == 0) {
	  if (!strcmp("author", buf)) {
	    *mark = 1;
	    return (TRUE);
	  }
        } else {
	  if (!strcmp("owner", buf))
	    return (TRUE);
	}
	break;
      default:
	break;
      }
      index++;
    }
  }
  return (FALSE);
}

/*  */
/*******************************************************************
 * Function: static void LaTeX_node(_textdump *textdump, infonode *infoptr, FILE *outfile)
 *
 * This function performs a traversal of the reference structure.
 * When a leaf node is encountered, it writes the infonode to the
 * LaTeX file.  It should also maintain a certain sense of
 * hierarchy in the file.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void LaTeX_node(textdump, infoptr, outfile)
  _textdump *textdump;		/* index into open_win */
  infonode *infoptr;		/* pointer to the infonode */
  FILE *outfile;		/* pointer to the LaTeX file */
{
  identry *idptr = NULL;	/* pointer to an identry */
  int where = 0;		/* current position of stackpointer */

  /* get formats whether it's LaTeX or not - for indentation */
  where = add_formats(infoptr->head->tmpl->value, textdump->leftmargin);
  (void)add_formats(infoptr, textdump->leftmargin);

  if ((idptr = infoptr->subnodes) == (identry *) NULL) {
    print_node(textdump, infoptr, outfile);
  } else {
    while (idptr != (identry *) NULL) {
      for (infoptr = idptr->value; infoptr; infoptr = infoptr->next) {
	LaTeX_node(textdump, infoptr, outfile);
      }
      if (idptr->value->index != ATTRVALUE)
	prevlevel = 0;
      idptr = idptr->next;
    }
  }

  delete_formats(where);
  return;
}

/*  */
/*******************************************************************
 * Function: static void print_node(_textdump *textdump, infonode *info, FILE *outfile)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void print_node(textdump, infoptr, outfile)
  _textdump *textdump;
  infonode *infoptr;
  FILE *outfile;
{
  char *text = NULL;
  char *localfont = NULL;

  if (!infoptr || infoptr->printed)
    return;

  /* print placeholder only if they want it */
  if (infoptr->iflg & NONEXIST_MSK) {
    if(textdump->placeholders == yes) {
      char *text = strdup(infoptr->val.attvalue);
      text = escape_sp_chars(text);
      print_part(outfile, text, PARAGRAPH);
      free(text);
    }
    infoptr->printed = 1;
    return;
  }
  
  /* if we're dumping straight text, do it now */
  if (global_type == straight) {
    print_part(outfile, infoptr->val.attvalue, PARAGRAPH);
    infoptr->printed = 1;
    return;
  }
  
  if (thislevel != prevlevel) {
    print_part(outfile, (char *)NULL, SQUIGGLES);
  }
  if (stkptr > -1) {
    if ((thislevel != prevlevel) ||
      (prevfont && (strcmp(prevfont, fontstack[stkptr - 1].fontname) != 0))) {
      if (prevfont)
	free((FREEPTR *)prevfont);
      prevfont = strdup(fontstack[stkptr - 1].fontname);
      localfont = strdup(fontstack[stkptr - 1].fontname);
    }
  }

  if (!(text = attval_plus(infoptr, localfont))) {
    (void)fprintf(stderr, "print_node: attval_plus failed\n");
    return;
  }

  if (textdump->structure == flat) {
    print_part(outfile, text, PARAGRAPH);
    infoptr->printed = 1;
  } else {
    thislevel = my_level(infoptr->head);
    if (thislevel != prevlevel) {
      prevlevel = thislevel;
      switch (thislevel) {
      case SECOND_LEVEL:
      case THIRD_LEVEL:
      case FOURTH_LEVEL:
      case FIFTH_LEVEL:
	find_sect_title(textdump, infoptr, thislevel, outfile);
	if (!infoptr->printed) {
	  print_part(outfile, text, PARAGRAPH);
	  infoptr->printed = 1;
	}
	break;
      case FIRST_LEVEL:
      default:
	print_part(outfile, text, PARAGRAPH);
	infoptr->printed = 1;
	break;
      }
    } else {
      print_part(outfile, text, PARAGRAPH);
      infoptr->printed = 1;
    }
  }

  if (text)
    free((FREEPTR *)text);
  if (localfont)
    free((FREEPTR *)localfont);
}

/*  */
/*******************************************************************
 * Function: static char *attval_plus(infonode *infoptr,char *localfont)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static char *attval_plus(infoptr, localfont)
  infonode *infoptr;
  char *localfont;
{
  char *text = NULL;
  char *newtext = NULL;

  if (!infoptr || !infoptr->val.attvalue)
    return NULL;

  text = strdup(infoptr->val.attvalue);
  text = escape_sp_chars(text);

  if (!(newtext = (char *)malloc((ALLOC_T)strlen(text) +
				 ((localfont) ? strlen(localfont) + 10 : 10)))){
    if (text) 
      free(text);
    return NULL;
  }

  if (localfont) {
    (void)sprintf(newtext, "%s%s", localfont, text);
    lefttoprint += fontstack[stkptr - 1].completion;
  } else {
    (void)strcpy(newtext, text);
  }
  if (text)
    free(text);
  return newtext;
}

/*  */
/*******************************************************************
 * Function: static int my_level(identry *identptr)
 *
 * This function returns the level of the current infonode.  It makes
 * use of the function stepup() from aimeditops.c by Tim Teitenberg.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static int my_level(idp)
  identry *idp;
{
  int level;

  for (level=0; idp; idp = idp->owner->head, level++)
    if (!idp->owner)
      break;
  return (level);
}

/*  */
/*******************************************************************
 * Function: static char *escape_sp_chars(char *text)
 *
 *  The purpose of this function is to escape special LaTeX characters
 *  before sending the string to be printed to the output file.  At this
 *  point, I will brute force escape them.  This is the solution I
 *  choose simply because there are not terribly many of them and
 *  mapping to functions seems overly complicated.
 *
 *  Arguments:
 *    pointer to the string to be checked.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static char *escape_sp_chars(text)
  char *text;			/* pointer to string */
{
  char *tmptext = NULL;		/* string that will be sent back eventually */
  char searchchar;		/* character to be checked */
  int index = 0;		/* index into the text string */
  int quotes = LEFT;		/* first quote is a left quote */

  if (!text) {
    (void)fprintf(stderr, "escape_sp_chars: NULL input\n");
    return NULL;
  }

  if (global_escape == dont || global_type == straight)
    return text;

  if (!(tmptext = (char *)malloc((ALLOC_T)(strlen(text) * 3))))
    return text;
  (void)memset((char *)tmptext, 0, (int)strlen(text) * 3);
  for (index = 0; index < (int)strlen(text); index++) {
    searchchar = text[index];
    switch ((int)searchchar) {
    case '"':
      if (quotes == LEFT) {
	(void)strcat(tmptext, "``");
	quotes = RIGHT;
      } else {
	(void)strcat(tmptext, "''");
	quotes = LEFT;
      }
      break;
    case '\\':
      (void)strcat(tmptext, "$\\backslash{}$");
      break;
    case '~':
      (void)strcat(tmptext, "\\~{}");
      break;
    case '^':
      (void)strcat(tmptext, "\\^{}");
      break;
    case '<':
      (void)strcat(tmptext, "$<$");
      break;
    case '>':
      (void)strcat(tmptext, "$>$");
      break;
    case '|':
      (void)strcat(tmptext, "$\\mid$");
      break;
    case '{':
      (void)strcat(tmptext, "\\{");
      break;
    case '}':
      (void)strcat(tmptext, "\\}");
      break;
    case '$':
    case '%':
    case '&':
    case '#':
    case '_':			/* perform the same on all */
      (void)strcat(tmptext, "\\");
    default:
      (void)strncat(tmptext, &searchchar, 1);
      break;
    }
  }
  free((FREEPTR *)text);
  return tmptext;
}

/*  */
/*******************************************************************
 * Function: static void find_sect_title(FOUR PARAMETERS)
 * Parameters:
 *	_textdump *textdump
 *	infonode *infoptr
 *	int check_level
 *	FILE *outfile
 *
 * This function will find the appropriate title to give to the
 * heading of the current level section (section, subsection, etc.).
 * If there is an ATTRVALUE at this level, it will check to see if
 * the infonode->field has the string "title" in any part of it.
 * I will essentially to a "window idiom" search and compare of the
 * string.  If I find something at this level, it will be used.
 * Otherwise, I will give the section the title "Temporary Section
 * Title" and let the user change it in the file as s/he sees fit.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void find_sect_title(textdump, infoptr, check_level, outfile)
  _textdump *textdump;
  infonode *infoptr;		/* ptr to infonode where you begin */
  int check_level;		/* current level. only check this level */
  FILE *outfile;		/* output file */
{
  infonode *tmpinfoptr = NULL;
  identry *tmpidptr = NULL;
  int found = FALSE;
  char *title = NULL;
  int mark = 0;			/* passed to compare_strings(), used
				 * to signal whether an infonode should
				 * be marked as printed or not
				 */

/*
 * look at the first infonode at this level of the reference
 * structure.  If it has the string "title" in the "field" field of the
 * infonode, then you have found your section title.
 */
  tmpidptr = infoptr->head;
  if (tmpidptr) {
    tmpinfoptr = tmpidptr->value;
    if (tmpinfoptr->index == ATTRVALUE)
      found = compare_strings(tmpinfoptr->field, TITLE, &mark);
  }
  if (found) {
    title = strdup(tmpinfoptr->val.attvalue);
    title = escape_sp_chars(title);
    lefttoprint = 0;
  }
  switch (check_level) {
  case SECOND_LEVEL:
    if (found && depth_signal)
      print_part(outfile, title, PART);
    else if (found)
      print_part(outfile, title, SECTION);
    else
      print_part(outfile, "Temporary Section Title", SECTION);
    break;
  case THIRD_LEVEL:
    if (found && depth_signal)
      print_part(outfile, title, SECTION);
    else if (found == TRUE)
      print_part(outfile, title, SUBSECTION);
    else
      print_part(outfile, "Temporary Subsection Title", SUBSECTION);
    break;
  case FOURTH_LEVEL:
    if (found && depth_signal)
      print_part(outfile, title, SUBSECTION);
    else if (found == TRUE)
      print_part(outfile, title, SUBSUBSECTION);
    else
      print_part(outfile, "Temporary Subsubection Title", SUBSUBSECTION);
    break;
  case FIFTH_LEVEL:
    if (found && depth_signal)
      print_part(outfile, title, SUBSUBSECTION);
    break;
  case FIRST_LEVEL:
  default:
    (void)fprintf(stderr, "find_sect_title:  no such case\n");
    break;
  }
  if (mark)
    tmpinfoptr->printed = 1;

  if (title)
    free((FREEPTR *)title);
  return;
}

/*  */
/*******************************************************************
 * Function: static void clear_printed(infonode *inodeptr)
 *
 * This function does a traversal of the reference structure to clear
 * all of the "printed" fields in the infonodes.  This is done so that
 * the next time the file is LaTeX'ed, the information will be valid.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void clear_printed(inodeptr)
  infonode *inodeptr;
{
  identry *identptr = NULL;
  infonode *infoptr = NULL;

  if ((identptr = inodeptr->subnodes) == (identry *)NULL) {
    inodeptr->printed = FALSE;
    return;
  } else {
    while (identptr != (identry *)NULL) {
      for (infoptr = identptr->value;
	   infoptr != (infonode *)NULL;
	   infoptr = infoptr->next) {
	clear_printed(infoptr);
      }
      identptr = identptr->next;
    }
  }
  return;
}

/*  */
/**********************************************************************
 * Function: static int add_formats(infonode *infoptr, char *leftmargin)
 *
 * This function adds fonts from generic and specific fonting
 * information to a stack of fonts.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static int add_formats(infoptr, leftmargin)
  infonode *infoptr;
  char *leftmargin;
{
  formentry *tmp = NULL;
  filefontinfo *f = NULL;
  int level = 0;

  if (stkptr == -1)
    level = 0;
  else
    level = stkptr;

  for (tmp = infoptr->formats; tmp; tmp = tmp->next) {
    if (strcmp(tmp->formid, "font") == 0) {
      f = get_format(tmp->formval);
      if (f) {
	if (stkptr == -1)
	  stkptr = 0;
	if (fontstack[stkptr].fontname)
	  free((FREEPTR *)fontstack[stkptr].fontname);
	fontstack[stkptr].fontname = strdup(f->LaTeXname);
	fontstack[stkptr].completion = f->squigglies;
	stkptr++;
      }
    }
    if (strcmp(tmp->formid, leftmargin) == 0) {
      int tmpint = atoi(tmp->formval);
      indent = (int)(tmpint / 10);
    }
  }
  return (level);
}

/*  */
/**********************************************************************
 * Function: static void delete_formats(int where)
 *
 * This function puts the stack of fonts back to the state it was
 * in at the invocation of the function.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static void delete_formats(where)
  int where;
{
  int i = 0;

  for (i = where; i < stkptr; i++) {
    if (fontstack[i].fontname)
      free((FREEPTR *)fontstack[i].fontname);
    fontstack[i].fontname = NULL;
    fontstack[i].completion = 0;
  }
  if (where == 0)
    stkptr = -1;
  else
    stkptr = where;
}

/*  */
/**********************************************************************
 * Function: static int load_file_fonts()
 *
 * load all of the font translations from file.  this is done
 * once per session the first time that the text dumping
 * operation is carried out.
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static int load_file_fonts()
{
  FILE *fontfile = NULL;
  filefontinfo *tmp = NULL;
  int i = 0, j = 0;
  char f1[BUF_SIZE];
  char f2[BUF_SIZE];
  static int done = 0;
  char filename[MAXPATHLEN];

  if (done)
    return 1;
  done = 1;

  (void)sprintf(filename, "%s/%s", DBDIR, FONT_TRANSLATION_FILE);

  if (!(fontfile = fopen(filename, "r"))) {
    (void)fprintf(stderr, 
		  "Cannot open %s\nfor X to LaTeX translations.\n", filename);
    (void)fprintf(stderr, 
		  "Contact your systems administrator to get it installed.\n");
    return (0);
  }

  (void)memset((char *)f1, 0, (int)sizeof(f1));
  (void)memset((char *)f2, 0, (int)sizeof(f2));

  while ((i = fscanf(fontfile, "%s %s %d", f1, f2, &j)) != EOF) {
    if (i < 3)
      break;
    tmp = new_filefontinfo();
    tmp->Xfontname = strdup(f1);
    tmp->LaTeXname = strdup(f2);
    tmp->squigglies = j;
    if (!fontinfo)
      fontinfo = tmp;
    else {
      tmp->next = fontinfo;
      fontinfo = tmp;
    }
    (void)memset((char *)f1, 0, (int)sizeof(f1));
    (void)memset((char *)f2, 0, (int)sizeof(f2));
  }
  if ((fclose(fontfile)) != 0) {
    (void)fprintf(stderr, "fclose failed in load_file_fonts\n");
  }
  return 1;
}

/*  */
/**********************************************************************
 * Function: static filefontinfo *new_filefontinfo()
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static filefontinfo *new_filefontinfo()
{
  filefontinfo *tmp = NULL;

  tmp = (filefontinfo *) malloc(sizeof(filefontinfo));
  tmp->next = NULL;
  tmp->Xfontname = NULL;
  tmp->LaTeXname = NULL;
  tmp->squigglies = 0;
  return tmp;
}

/*  */
/**********************************************************************
 * Function: static filefontinfo *get_format(char *lookfor)
 *
 *  Modifications:
 *	<list mods with name and date>
 */
static filefontinfo *get_format(lookfor)
  char *lookfor;
{
  filefontinfo *f = NULL;

  for (f = fontinfo; f; f = f->next) {
    if (!(strcmp(lookfor, f->Xfontname)))
      return f;
  }
  return NULL;
}


/*  */
/**********************************************************************
 * Function: _textdump *initialize_textdump()
 *
 *  Modifications:
 *      1994-05-14 Martin Sjlin. Uses application defaults ....
 *	<list mods with name and date>
 */
_textdump *initialize_textdump()
{
  _textdump *textdump = NULL;

  textdump = (_textdump *) malloc(sizeof(_textdump));
  textdump->outfile = get_file_name();
  textdump->docstyle = strcmp(default_style,"letter") == 0 ? letter : article;
  textdump->structure = default_usedeep ? deep : flat;
  textdump->fontsize = default_fontsize <= 10 ? small :
                         default_fontsize == 11 ? medium : large;
  textdump->placeholders = default_useph ? yes : no;
  textdump->leftmargin = NULL;
  global_escape = default_doesc ? doit : dont;
  global_type = default_latex ? LaTeX : straight;

  if (!load_file_fonts())
    (void)fprintf(stderr, "load_file_fonts: cannot open fontfile.\n");

  return textdump;
}

/*  */
/**********************************************************************
 * Function: static char *get_file_name()
 *
 * choose a suitable file for the default output
 *
 * Modifications:
 *      <list mods with name and date>
 */
static char *get_file_name()
{
  struct stat buf;		/* stat information returned here */
  char *home = NULL;
  char *base = NULL;
  char *try = NULL;
  int done = 0;
  int count = 0;
  extern char *getenv();

  if ((home = (char *)getenv("HOME")) != NULL) {
    base = (char *)malloc((ALLOC_T)strlen(home) + 20);
    (void)sprintf(base, "%s%s", home, "/");
  } else {
    base = (char *)malloc((ALLOC_T)25);	/* /tmp/ + some */
    (void)sprintf(base, "%s", "/tmp/");
  }

  try = (char *)malloc((ALLOC_T)strlen(base) + 15);
  (void)sprintf(try, "%s%s%s", base, "tmpfile", ".tex");
  while (!done) {
    if (stat(try, &buf) != -1) {
#ifdef SPRINTF_INT
      if (sprintf(try, "%s%s%d%s", base, "tmpfile", ++count, ".tex") == EOF)
#else
      if (sprintf(try, "%s%s%d%s", base, "tmpfile", ++count, ".tex") == NULL)
#endif
	(void)fprintf(stderr, "get_file_name: sprintf failed.\n");
    } else {
      free((FREEPTR *)base);
      return try;
    }
  }

  free((FREEPTR *)base);
  return try;
}

/*  */
/**********************************************************************
 * Function: static void get_header(_textdump textdump, FILE *outfile)
 *
 * This function will include the header for the LaTeX file.
 * This header is partly included in the global formating attributes
 * in formats for each document.  The type size and
 * document style are taken from that structure.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void get_header(textdump, outfile)
  _textdump *textdump;
  FILE *outfile;
{
  static char *dockinds[] = {
    "article",
    "letter"
  };

  char *doc = (char *)dockinds[textdump->docstyle];
  int font = textdump->fontsize;

  if (textdump->docstyle == article)
    (void)fprintf(outfile, "%s%d%s%s%s\n", "\\documentstyle[",
		  font, "pt,a4,par]{", doc, "}");
  else
    (void)fprintf(outfile, "%s%d%s%s%s\n",
		  "\\documentstyle[", font, "pt]{", doc, "}");

  (void)fprintf(outfile, "%s\n%s\n%s\n",
		"\\marginparwidth 0pt",
		"\\oddsidemargin  35pt",
		"\\evensidemargin  35pt");
  (void)fprintf(outfile, "%s\n%s\n",
		"\\marginparsep 0pt",
		"\\topmargin   0pt");
  (void)fprintf(outfile, "%s\n%s\n%s\n%s\n",
		"\\textwidth=422pt",
		"\\textheight=25cm",
		"\\parindent=0cm  % if you want indention, change this",
		"\\headsep=1cm");
  (void)fprintf(outfile, "%s\n\n",
		"\\begin{document}");
#ifdef DEBUG
  if (textdump->docstyle == article)
    (void)printf("%s%d%s%s%s\n", "\\documentstyle[", font, "pt,a4,par]{", doc, "}");
  else
    (void)printf("%s%d%s%s%s\n", "\\documentstyle[", font, "pt]{", doc, "}");
  (void)printf("%s\n%s\n%s\n",
	       "\\marginparwidth 0pt",
	       "\\oddsidemargin  35pt",
	       "\\evensidemargin  35pt");
  (void)printf("%s\n%s\n",
	       "\\marginparsep 0pt",
	       "\\topmargin   0pt");
  (void)printf("%s\n%s\n%s\n%s\n",
	       "\\textwidth=422pt",
	       "\\textheight=25cm",
	       "\\parindent=0cm  % if you want indention, change this",
	       "\\headsep=1cm");
  (void)printf("%s\n\n",
	       "\\begin{document}");
#endif
}

/*  */
/**********************************************************************
 * Function: static void make_art_title(FOUR PARAMETERS)
 * Parameters:
 *	FILE *outfile
 *	char *title
 *	char *author
 *	char *date
 *
 *  This function inserts text at the paragraph level.  It will grab
 *  the appropriate text from the appropriate place and put it into the
 *  file where it should be.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void make_art_title(outfile, title, author, date)
  FILE *outfile;		/* the output LaTeX file */
  char *title;			/* the title of the document */
  char *author;			/* the author of the document */
  char *date;			/* the date of the document */
{
  (void)fprintf(outfile, "%s%s%s\n%s%s%s\n%s%s%s\n\n%s",
		"\\title{", title, "}",
		"\\author{", author, "}",
		"\\date{", date, "}",
		"\\maketitle");
#ifdef DEBUG
  (void)printf("%s%s%s\n%s%s%s\n%s%s%s\n\n%s\n",
	       "\\title{", title, "}",
	       "\\author{", author, "}",
	       "\\date{", date, "}",
	       "\\maketitle");
#endif

}

/*  */
/**********************************************************************
 * Function: static void print_indent(FILE *outfile, int debug)
 *
 * Print out some spaces (there must have been a leftMargin (X11) 
 * formatting directive).
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void print_indent(outfile, debug)
  FILE *outfile;
  int debug;
{
  int i;

  for (i = 0; i < indent; i++) {
    if (debug)
      (void)printf("  ");
    else
      (void)fprintf(outfile, "  ");
  }
}

/*  */
/**********************************************************************
 * Function: static void print_part(THREE PARAMETERS)
 * Parameters:
 *	FILE *outfile
 *	char *part		-string to print
 *	int which		- which part
 *
 *  This function prints some part of the LaTeX to the output
 *  file.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void print_part(outfile, part, which)
  FILE *outfile;		/* the output LaTeX file */
  char *part;			/* the title of the part */
  int which;
{
  int j;

  (void)fprintf(outfile, "\n\n");
#ifdef DEBUG
  (void)printf("\n\n");
#endif

  switch (which) {
  case PART:
    (void)fprintf(outfile, "%s", "\\part{");
#ifdef DEBUG
    (void)printf("%s", "\\part{");
#endif
    break;
  case SECTION:
    (void)fprintf(outfile, "%s", "\\section{");
#ifdef DEBUG
    (void)printf("%s", "\\section{");
#endif
    break;
  case SUBSECTION:
    (void)fprintf(outfile, "%s", "\\subsection{");
#ifdef DEBUG
    (void)printf("%s", "\\subsection{");
#endif
    break;
  case SUBSUBSECTION:
    (void)fprintf(outfile, "%s", "\\subsubsection{");
#ifdef DEBUG
    (void)printf("%s", "\\subsubsection{");
#endif
    break;
  default:
    break;
  }

  switch (which) {
  case PART:
  case SECTION:
  case SUBSECTION:
  case SUBSUBSECTION:
  case PARAGRAPH:
    print_indent(outfile, 0);
    (void)fprintf(outfile, "%s%s", part, (which == PARAGRAPH) ? " " : "}");
#ifdef DEBUG
    print_indent(outfile, 1);
    (void)printf("%s%s", part, (which == PARAGRAPH) ? " " : "}");
#endif
    break;
  case TRAILER:
    if (global_type == LaTeX) {
      (void)fprintf(outfile, "\\end{document}\n");
#ifdef DEBUG
      (void)printf("\\end{document}\n");
#endif
    } else {
      (void)fprintf(outfile, "\n");
#ifdef DEBUG
      (void)printf("\n");
#endif
    }
    break;
  case LETTERHEAD:
    (void)fprintf(outfile, "%s\n\n%s\n\n%s\n\n%s",
		  "\\signature{Your name goes here}",
		  "\\address{Your address\\\\ should go here}",
		  "\\begin{letter}{The recipients address\\\\ should go here}",
		  "\\opening{Dear Somebody}   % the salutation");
#ifdef DEBUG
    (void)printf("%s\n\n%s\n\n%s\n\n%s\n\n",
		 "\\signature{Your name goes here}",
		 "\\address{Your address \\\\ should go here}",
		 "\\begin{letter}{The recipients address \\\\ should go here}",
		 "\\opening{Dear Somebody}  % the salutation");
#endif
    break;
  case LETTERTAIL:
    (void)fprintf(outfile, "%s\n\n%s\n",
		  "\\closing{Sincerely}  % You can change this here",
		  "\\end{letter}");
#ifdef DEBUG
    (void)printf("%s\n\n%s\n",
		 "\\closing{Sincerely}  % You can change this here",
		 "\\end{letter}");
#endif
    break;
  case SQUIGGLES:
    for (j = 0; j < lefttoprint; j++) {
      (void)fprintf(outfile, "%s", "}");
#ifdef DEBUG
      (void)printf("%s", "}");
#endif
    }
    lefttoprint = 0;
    break;
  default:
    (void)fprintf(stderr, "print_part: invalid case\n");
    break;
  }
  indent = 0;	/* no. of times to indent */
}

/*  */
/**********************************************************************
 * Function: void export_file(reference_structure *ref, _textdump * textdump,
 *                           FILE *file);
 *
 * Modifications:
 *
 *      <list mods with name and date>
 *
 */
void  export_file (ref, textdump, file)
  reference_structure *ref;
 _textdump * textdump;
  FILE *file;
{
  int level = 0;			/* dummy for num_levels */

  num_levels(ref->target->value, level);

  if (global_type == LaTeX) {
    get_header(textdump, file);
    if (textdump->docstyle == article)
      get_title(file, ref->target);
    else
      print_part(file, (char *)NULL, LETTERHEAD);
  }

  LaTeX_node(textdump, ref->target->value, file);

  if (global_type == LaTeX)
    print_part(file, (char *)NULL, SQUIGGLES);
  if ((global_type == LaTeX) && 
      (textdump->docstyle == letter))
    print_part(file, (char *)NULL, LETTERTAIL);
  print_part(file, (char *)NULL, TRAILER);

  /* clear "printed" in infonodes */
  clear_printed(ref->target->value);

  if (prevfont)
    free((FREEPTR *)prevfont);
  prevfont = NULL;

}
