/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME:         aimeditops.c
 *
 * SCCSINFO:            @(#)aimeditops.c	1.7 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Tim Teitenberg, 1990-04-23
 *
 * MODIFICATIONS:
 *      1993-11-02 Martin Sjlin - whichref is moved to aimsubr.c and 
 *                 changed external section accordingly.
 *	<list mods here with name and date>
 *
 * DESCRIPTION:
 *
 * This file contains the specific editing functions performed against
 * the reference structure.  Except for an operation on formatting
 * information, the reference structure itself is not
 * altered here; only the database contents are altered by calling
 * routines within the aimdbupdate module.  The reference structure is
 * updated later when the changes to the database contents are propagated
 * into the reference structure.
 *
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "aimtypes.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_aimeditops.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimbuildref.h"
#include "f_aimcmdstream.h"
#include "f_aimcommand.h"
#include "f_aimstruct.h"
#include "f_aimsubr.h"
#include "f_autolinks.h"
#include "f_blddisplay.h"
#include "f_xstuff.h"

/* libshared */
extern char *strdup( /* char *incoming */ );

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* liblincks.a */
extern int LL_uid;
extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;

/* comhist.c */
extern int global_comhistoff;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define oldtag	where->val.attvalue
#define newtag	v->attvalue

struct expandstruct {
    char *s;
    label lbl;
    reference_structure *r;
    int mode;
};

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void expandpart2 P_(( Widget widget, caddr_t client_data, caddr_t
                              call_data ));
static void expandstorecmd P_(( Widget widget, caddr_t client_data,
                                 caddr_t call_data ));
static popd_list *make_pdl_entry P_(( reference_structure *ref ));

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: int create(identry *idp, attrval *v, int pos)
 * 
 * Creates a new target infonode to carry a given text.
 *
 *	Input:	idp = creation identry
 *		v = new text
 *		pos = identry value list position.
 *	Result:	Error code.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int create(idp, v, pos)
  identry *idp;
  attrval *v;
  int pos;
{
  label *obj;
  label lbl;
  int index;
  infonode *top;

  if (idp->owner == (infonode *)NULL) {
    /* Creation attempted at reference structure top; cannot do that */
    return FAIL;
  }

  if (idp->owner->index == LINKITEM)
    obj = &(idp->owner->lbl);
  else
    obj = &(idp->owner->obj);
  
  switch (index = idp->tmpl->value->index) {
  case ATTRGROUPTAG:
  case LINKGROUPTAG:
    newlastcmd(CREATEOP,obj,index,(char *)NULL,(char *)NULL,0,v,(label *)NULL);
    break;

  case ATTRFIELDTAG:
  case LINKFIELDTAG:
    newlastcmd(CREATEOP,obj,index,grouptag(idp),(char *)NULL,0,v,(label *)NULL);
    break;

  case LINKITEM:
    if (CO_CREATEOBJ(&lbl) || UBL_UNBINDLABEL(&lbl))
      return FAIL;
    newlastcmd(CHANGEOP,&lbl,IMAGE,(char *)NULL,(char *)NULL,0,v,(label*)NULL);
    newlastcmd(LINKOP,obj,LINKITEM,grouptag(idp),fieldtag(idp),pos,
	       (attrval *)NULL, &lbl);
    /* AUTOLINK-ing */
    for (top=idp->owner; (top) && (top->parent); top=top->parent)
      ; /* Finds the top of the target ref. struct. */

    /* this is autolinking based upon AUTOLINK:idp->idname resolved gpd */
    (void) autolink(&(idp->value->selobj),&(idp->value->obj),&lbl,idp->idname);

    /* this is autolinking based upon AUTOLINK:idp->idname in the root gpd */
    (void) autolink(&(top->selobj),&(top->obj),&lbl,idp->idname);
    break;

  case ATTRVALUE:
    if (!(idp->value->iflg & NONEXIST_MSK))
      return FAIL;
    newlastcmd(CHANGEOP,obj,index,grouptag(idp),fieldtag(idp),0,v,(label *)NULL);
    break;

  case IMAGE:
    if (!(idp->value->iflg & NONEXIST_MSK))
      return FAIL;
    newlastcmd(CHANGEOP,obj,index,(char *)NULL,(char *)NULL,0,v,(label *)NULL);
    break;

  case SAMEASPARENT:
    return create(idp->owner->head,v,pos);

  case INTERNAL:
  default:
    /* These cannot be created */
    return FAIL;
  }

  return SUCCESS;
}


/*  */
/**********************************************************************
 * Function: int change(infonode *where, attrval *v)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
int change(where, v)
  infonode *where;
  attrval *v;
{
  label *obj;
  int index;

  if (where->iflg & NONEXIST_MSK)
    return create(where->head,v,1);

  if (where->parent == (infonode *)NULL)
    obj = &(where->obj);
  else if (where->head->owner->index == LINKITEM)
    obj = &(where->head->owner->lbl);
  else
    obj = &(where->head->owner->obj);
  
  switch (index = where->head->tmpl->value->index) {
  case ATTRGROUPTAG:
  case LINKGROUPTAG:
    newlastcmd(CHANGEOP,obj,index,where->val.attvalue,
	       (char *)NULL, 0, v, (label *)NULL);
    break;
  case ATTRFIELDTAG:
  case LINKFIELDTAG:
    newlastcmd(CHANGEOP,obj,index,grouptag(where->head),where->val.attvalue,
	       0,v,(label *)NULL);
    break;
  case IMAGE:
    newlastcmd(CHANGEOP,obj,index,(char *)NULL,(char *)NULL,0,v,(label *)NULL);
    break;
  case LINKITEM:
    newlastcmd(CHANGEOP,&(where->lbl),IMAGE,(char *)NULL,
	       (char *)NULL,0,v,(label *)NULL);
    break;
  case ATTRVALUE:
    newlastcmd(CHANGEOP,obj,index,grouptag(where->head),
	       fieldtag(where->head),0,v,(label *)NULL);
    break;

  case SAMEASPARENT:
    if (where->parent == (infonode *)NULL)
      return FAIL;
    return change(where->parent,v);
  case INTERNAL:
  default:
    return FAIL;
  }

  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: int expand(infonode *where, int inplace)
 * 
 * If the given infonode is a LINKITEM, a reference structure is created
 * for the object linked to, using the GPD tag findexpandtag(infonode).
 * 
 *	 Input:	where = The infonode to expand upon.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int expand(where, inplace)
  infonode *where;
  int inplace;
{
  static struct expandstruct expand_data;

  if ((where == (infonode *)NULL) || (where->iflg & NONEXIST_MSK))
    return FAIL;

  determinenode(where,&(expand_data.lbl));

  expand_data.s = findexpandtag(where);

  if (expand_data.s == NULL) {
    short_error("Couldn't find expansion type\n");
    return FAIL;
  }

  expand_data.r = whichref(where, (identry *)NULL, 0);
  expand_data.mode = inplace;
  if (expand_data.r->flags.bound) {
    (void)GBTC_GETBTCOMP(&(expand_data.r->bt),&(expand_data.lbl),FALSE,0);
  }
  if (expand_data.mode) {	/* mode 1 is "expand in same window" */
    /* 
     * first check if the window needs saving, issue expandstorecmd
     * and expandpart2 if it needed saving and they choose the "yes"
     * button in yesnomenu.  if they choose "no", just do expandpart2
     */
    checksave(expand_data.r,expandstorecmd, expandpart2, (caddr_t)&expand_data);
  } else {
    expandpart2((Widget)NULL, (caddr_t)&expand_data, (caddr_t)NULL);
  }
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static void expandstorecmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget widget;
 *	caddr_t client_data;
 *	caddr_t call_data;
 *
 * Callback to checksave()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void expandstorecmd(widget, client_data, call_data)
  Widget widget;
  caddr_t client_data;
  caddr_t call_data;
{
  struct expandstruct *expand_data = (struct expandstruct *)client_data;

  storecmd(widget, (caddr_t) expand_data->r, call_data);
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void expandpart2(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget widget;
 *	caddr_t client_data;
 *	caddr_t call_data;
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void expandpart2(widget, client_data, call_data)
  Widget widget;
  caddr_t client_data;
  caddr_t call_data;
{
  struct expandstruct *expand_data = (struct expandstruct *)client_data;
  popd_list *pdl;

  pdl = make_pdl_entry(expand_data->r);

  expand_data->r = buildref(&(expand_data->lbl),
			    expand_data->s,
			    ((expand_data->mode) ? expand_data ->r : NULL));

  if (!expand_data->r) {
    short_error("Couldn't create view using `%s'.\nNo such GPD.\n",
		expand_data -> s);
    free(expand_data->s);
    free((FREEPTR *)pdl->tag);
    free((FREEPTR *)pdl);
    return;
  }
  
  expand_data->r->pdl = pdl;
  if (!expand_data->mode) {	/* in place */
    copy_pdls(&(pdl->next));
  }
  blddisplay(expand_data->r);
  /* FocusOnDisplay moved to blddisplay to decrease X-dependency */
  free(expand_data->s);
}

/*  */
/**********************************************************************
 * Function: int remove_infonode(infonode *where)
 * 
 * This function removes an item corresponding to an infonode.
 *
 * Input:	The infonode for what should be removed.
 * Result:	Error code.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int remove_infonode(where)
  infonode *where;
{
  label *obj;
  int index;
  char *group = (char *)NULL;
  char *field = (char *)NULL;
  int pos = 0;

  if (where->iflg & NONEXIST_MSK)
    return FAIL;
  
  obj = &(where->obj);

  switch(index = where->index) {
  case ATTRGROUPTAG:
  case LINKGROUPTAG:
    group = where->val.attvalue;
    break;
  case ATTRFIELDTAG:
  case LINKFIELDTAG:
    group = grouptag(where->head);
    field = where->val.attvalue;
    break;
  case LINKITEM:
    group = grouptag(where->head);
    field = fieldtag(where->head);
    pos = where->pos;
    break;
  case ATTRVALUE:
    group = grouptag(where->head);
    field = fieldtag(where->head);
    break;
  case IMAGE:
    break;
  case INTERNAL:
    return FAIL;
  case SAMEASPARENT:
    if (where->parent == (infonode *)NULL)
      return FAIL;
    return remove_infonode(where->parent);
  }
  
  newlastcmd(REMOVEOP,obj,index,group,field,pos, (attrval *)NULL, (label *)NULL);
  return SUCCESS;
}


/*  */
/**********************************************************************
 * Function: int linkcopy(infonode *inp, identry *idp, int pos)
 * 
 * This function copies a linkitem by setting a new link to
 * the source linkitem from the given destination identry
 * and position.
 *
 *  Input:        inp = The infonode of the source linkitem.
 *                idp = The destination identry.
 *                pos = The destination position within the identry.
 *  Result:	Error code.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int linkcopy(inp, idp, pos)
     infonode *inp;
     identry *idp;
     int pos;
{
  char *field;
  label obj;
  label lbl;

  if ((inp->iflg&NONEXIST_MSK) || (inp->index != LINKITEM))
    (void)copylbl(&lbl,&(inp->obj));
  else
    (void)copylbl(&lbl,&(inp->lbl));

  while (idp->tmpl->value->index != LINKITEM) {
    if (idp->owner == (infonode *)NULL) return FAIL;
    if (idp->tmpl->value->index == LINKFIELDTAG) break;
    for (inp=idp->owner->head->value, pos=1; inp; inp = inp->next, pos++)
      if (inp == idp->owner)
	break;
    idp = idp->owner->head;
  }

  determinenode(idp->owner,&obj);

  if (idp->tmpl->value->index == LINKFIELDTAG) {
    field = idp->value->val.attvalue;
    pos = 1;
  } else {
    field = fieldtag(idp);
    if (idp->value->iflg & NONEXIST_MSK)
      pos = 0;
    pos += 1;
  }
  
  newlastcmd(LINKOP, &obj, LINKITEM, grouptag(idp), field, /* fieldtag(idp), */
	     pos,(attrval *)NULL,&lbl);

  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: int isstructurechanged(infonode *top)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
int isstructurechanged(top)
  infonode *top;
{
  identry *idp;
  infonode *inp;
  int edited = FALSE;
  int editedtmp = FALSE;
  reference_structure *ref = (!top->parent) ? whichref(top,(identry *)NULL,0)
					    : NULL;
  if (ref && EDITED(&(ref->bt))) {
    trace("bt <%d:%d> changed.\n", ref->bt.vs, ref->bt.inst);
    if (edited != BT_SOME_TRANSIENT)
      edited = BT_TRANSIENT;
  }

  for (idp=top->subnodes; idp!=(identry *)NULL; idp=idp->next) {
    for (inp=idp->value; inp!=(infonode *)NULL; inp=inp->next) {
      if (inp->subnodes != (identry *)NULL) {
	editedtmp = isstructurechanged(inp);
	if (edited == FALSE)
	  edited = editedtmp;
	if (edited == BT_TRANSIENT && editedtmp == BT_SOME_TRANSIENT)
	  edited = BT_SOME_TRANSIENT;
      } else {
	if (!(inp->iflg & NONEXIST_MSK)) {
	  if (EDITED(&inp->obj)) {
            trace("obj <%d:%d> changed.\n", inp->obj.vs, inp->obj.inst);
            edited = BT_SOME_TRANSIENT;
	  }
	  if (inp->index == LINKITEM && EDITED(&inp->lbl)) {
            trace("lbl <%d:%d> changed.\n", inp->lbl.vs, inp->lbl.inst);
            edited = BT_SOME_TRANSIENT;
	  }
	}
      }
    }
  }
  return edited;
}

/*  */
/**********************************************************************
 * Function: void storesubnodes(infonode *top)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
void storesubnodes(top)
     infonode *top;
{
  identry *idp;
  infonode *inp;
  label lbl;
  int comhistoff = global_comhistoff;

  global_comhistoff = 1;
  for (idp=top->subnodes; idp!=(identry *)NULL; idp=idp->next) {
    for (inp=idp->value; inp!=(infonode *)NULL; inp=inp->next) {
      if (inp->subnodes != (identry *)NULL)
	(void) storesubnodes(inp);
      else {
	if (!(inp->iflg & NONEXIST_MSK)) {
	  if (inp->index == LINKITEM &&
	      EDITED(&(inp->lbl))) {
	    (void)GC_GETCURRENT(copylbl(&lbl,&(inp->lbl)));
	    trace("Storing lbl <%d:%d>", lbl.vs, lbl.inst);
	    trace(" (%d)\n",SO_STOREOBJ(&lbl));
	  }
	  if (EDITED(&inp->obj)) {
	    (void)GC_GETCURRENT(copylbl(&lbl,&(inp->obj)));
	    trace("Storing obj <%d:%d>", lbl.vs, lbl.inst);
	    trace(" (%d)\n",SO_STOREOBJ(&lbl));
	  }
	}
      }
    }
  }
  global_comhistoff = comhistoff;
}

/*  */
/**********************************************************************
 * Function: static popd_list *make_pdl_entry(reference_structure *ref)
 * 
 * Creates and returns a popdown list entry to be attached to those
 * associated with a reference structure.  The user can go backwards
 * through the popdown list to previous compositions (windows).
 * Consider it like a stack of windows of which you pop off the topmost.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static popd_list *make_pdl_entry(ref)
  reference_structure *ref;
{
  popd_list *pdl;

  pdl = (popd_list *)malloc(sizeof(popd_list));
  pdl -> next = ref -> pdl;
  pdl -> tag = strdup(ref->template->idname);
  pdl -> bound = ref->flags.bound;
  (void) copylbl(&(pdl -> bt), &(ref->bt));
  return pdl;
}

/*  */
/**********************************************************************
 * Function: void copy_pdls(popd_list **pdl_ptr)
 * 
 * copies the pdl list of one reference structure into another.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void copy_pdls(pdl_ptr)
  popd_list **pdl_ptr;
{
  popd_list *pdl;

  while (*pdl_ptr) {
    pdl = (popd_list *)malloc(sizeof(popd_list));
    pdl -> next = (*pdl_ptr) -> next;
    pdl -> tag = strdup((*pdl_ptr)->tag);
    pdl -> bound = (*pdl_ptr) -> bound;
    (void) copylbl(&(pdl -> bt), &((*pdl_ptr)->bt));
    *pdl_ptr = pdl;
    pdl_ptr = &(pdl->next);
  }
}
