/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	aimcutbuf.c
 *
 * SCCSINFO:		@(#)aimcutbuf.c	1.9 5/3/94
 *
 * ORIGINAL AUTHOR(S):  1992-09-01, Ralph R\"onnquist
 *
 * DESCRIPTION:
 *	 Handles cut buffers.
 *
 * MODIFICATIONS:
 *      1993-11-02 Martin Sjlin - whichref is moved to aimsubr.c and 
 *                 changed external section accordingly.
 *	<list mods with name and date>
 *
 *********************************************************************/
/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE
 *********************************************************************
 * void make_cut_of_string(char *s)
 * void pop_cut_buffers()
 * void copy_region(infonode *fst,infonode *lst,int remove)
 * int paste(infonode *inp,int after)
 */ 

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "aimtypes.h"
#include "aimstredit.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE
 *********************************************************************/
#include "f_aimcutbuf.h"

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimsubr.h"
#include "f_aimeditops.h"
#include "f_aimcmdstream.h"
#include "f_xstuff.h"

/* libshared */
extern char *strdup( /* char *incoming */ );

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern char *indextbl[];

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define PASTE_AFTER		1
#define PASTE_COPY_LEAF		2
#define PASTE_COPY_WHOLE	4
#define PASTE_SHARE		8

#define ADAPT_FIRST		1
#define ADAPT_DROP_END  	2
#define ADAPT_DROP_INNER	4
#define ADAPT_ALL       (ADAPT_FIRST | ADAPT_DROP_END | ADAPT_DROP_INNER)
/*
 * Type (def_rec) records hold the "fragment" definitions. Each record
 * uniquely identifies a fragment type.
 */
typedef struct _def_rec {
  struct _def_rec *next;
  identry *idp;
} def_rec;

/*
 * Type (edge_pnt) holds a suggested edge point with a flag marking
 * if it is reducable.
 */
typedef struct _edge_pnt {
  struct _edge_pnt *next;
  identry *idp; /* in target */
  int pos;
  int reducable;
} edge_pnt;

/*
 * Type (edge_list) is essentially a list of edge points that actualises
 * all edge points indicated by an edge_pnt.
 */
typedef struct _edge_list {
  struct _edge_list *next;
  def_rec *fragment;
  int  pos;			/* Infonode position concerned. */
  int reducable;		/* End item (in tail edge). */
  identry *idp;			/* In target or null */
  label obj;			/* object supposed to carry the item */
  attrval val;			/* Presumptive value (when !idp) */
  struct _edge_list *owner;	/* Point of expansion */
} edge_list;

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* these are debugging functions that are currently unused */
static int EDGE P_(( edge_list *e ));
static int CUT P_(( cut_item c ));

static int do_paste_edge P_(( int mode ));
static void free_cut_buffer P_(( cut_item ci ));
static char *get_field P_(( edge_list *edge ));
static char *get_group P_(( edge_list *edge ));
static void get_real_obj P_(( label *obj, edge_list *edge ));
static int isplural P_(( identry *idp ));
static void new_cut_buffer P_(( void ));
static cut_item new_cut_item P_(( void ));
static cut_item put_cut_item P_(( infonode *in ));
static int traverse_infonode P_(( infonode *ip ));
static void value_of_cut P_(( edge_list *edge, cut_item from ));
static void value_of_edge P_(( edge_list *edge, edge_list *from ));


/* Managing fragment definitions... */
static def_rec *finddef P_(( identry *s ));
static int addtodefs P_(( identry *idp ));
static void free_defs P_(( void ));
static void define_fragments P_(( identry *idp ));

/* Managing cut edge alternatives.. */
static void catch_edge P_(( identry *idp, int pos, int flag ));
static void clear_edge P_(( void ));

/* Adapting edges.. */
static void into_edge_list P_(( edge_list **list, identry *idp, int pos,
                                 int flag ));
static void free_edge_list P_(( edge_list *list ));
static int length P_(( edge_list *e, int all ));
static void make_edge_pnt P_(( identry *idp, int pos, int flag,
                                edge_list **list ));

/* Edge matching.. */
static int moveattrfield P_(( char *p, char *field, int c, int n ));
static int movelinkfield P_(( char *p, char *field, int c, int n ));
static int movelinkitem P_(( char *p, label *item, int c, int n ));
static void edge_match P_(( edge_list *left, edge_list *right ));
static int expand_by_fragment P_(( edge_list **list, edge_list *owner ));
static int adapt_by_expanding P_(( edge_list *left, edge_list *right ));
static int adapt_edge P_(( edge_list *left, edge_list *right, int exp ));
static void erase_subtree P_(( identry *idp, int pos ));

/* Determining the fringes.. */
static void init_cuts P_(( void ));
static void make_edge P_(( int mode ));
static int reduce_to_fragments P_(( infonode *inp, int mode ));
static void cut_out_region P_(( infonode *inp, infonode *inq ));
static cut_item get_cut_buffer P_(( void ));

/* Pasting .. */
static edge_list *reverse_edge P_(( edge_list *edge ));
static void cut_match P_(( edge_list *edge, cut_item cut ));
static int paste_by_expanding P_(( cut_item cut, edge_list *edge,
                                    edge_list *right, int mode ));
static int adapt_paste P_(( cut_item cut, edge_list *edge, edge_list
                             *right, int mode ));
static void paste_reduce P_(( void ));
static int paste_edge P_(( void ));
static int find_paste_fragments P_(( infonode *inp, int mode ));

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
static cut_buffer CBPDL;	/* Cut Buffer Push Down List */
static infonode *begincut, *endcut, *insert_point;
static int started = 0;
static int paste_option;
static int pasting = 0;

static def_rec *defs = (def_rec *)NULL;	/* Fragment definitions */
static edge_pnt *edge = (edge_pnt *)NULL;	/* Tentative edge points */
static int ocount = -2;		/* "new-object counter & flag" */
static struct {
  int assigned;
  int assumed;
}  newobj[1000];	/* NOTE: at most 1000 new objects... */
static struct {
  int count;
  edge_list *edge[50];
} left, right;	/* The cut fringes; each at most 50 alternative edges */
static struct {
  label source;
  label obj;
  char *group;
  char *field;
} mapinfo;

/*  */     
/*********************************************************************
 * Function: static void new_cut_buffer()
 *
 * Make new cut buffer by pushing down previous.
 *
 * Modification:
 *	<list mod's with name and date>
 */
static void new_cut_buffer()
{
  int i;
  cut_buffer cb, cbx;

  for (cb = CBPDL, i=1; cb; i++, cb=cb->next)
    if (i == 7)
      break;
  if (cb) {
    cbx = cb->next;
    cb->next = (cut_buffer)NULL;
    for (cb = cbx; cb; cb = cbx) {
      cbx = cb->next;
      free_cut_buffer(cb->head);
      free((FREEPTR *)cb);
    }
  }

  cb = (cut_buffer)malloc(sizeof(struct aim_cut_buffer));
  cb->next = CBPDL;
  cb->head = (cut_item)NULL;
  cb->tail = (cut_item)NULL;
  cb->removed = 0;
  CBPDL = cb;
}

/*  */
/*********************************************************************
 * Function: static void free_cut_buffer(cut_item ci)
 *
 * Reclaim the memory of cut item list beginning with ci.
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void free_cut_buffer(ci)
  cut_item ci;
{
  cut_item cix;

  for (cix = ci; cix; cix = ci) {
    ci = cix->next;
    if (cix->group)
      (void)free(cix->group);
    if (cix->field)
      (void)free(cix->field);
    if ((cix->val.attsize > 0) && (cix->val.attvalue))
      (void)free(cix->val.attvalue);
    free((FREEPTR *)cix);
  }
}

/*  */
/*********************************************************************
 * Function: static cut_item new_cut_item()
 *
 * Create a new "empty" cut item and append it to current cut buffer.
 *
 * Modification:
 *	<list mod's with name and date>
 */
static cut_item new_cut_item()
{
  cut_item ci;

  if (!started)
    return (cut_item)NULL;

  ci = (cut_item)malloc(sizeof(struct aim_cut_item));
  ci->next = (cut_item)NULL;
  if (!(ci->prev = CBPDL->tail))
    CBPDL->head = ci;
  else
    CBPDL->tail->next = ci;
  CBPDL->tail = ci;

  ci->prev = (cut_item)NULL;
  ci->removed = 0;
  ci->ancestor = (cut_item)NULL;
  ci->visible = 0;
  ci->index = 0;
  ci->obj.vs = -1;
  ci->obj.inst = -1;
  ci->val.attsize = 0;
  ci->val.attvalue = (char *)NULL;
  ci->pos = 0;
  ci->lbl.vs = -1;
  ci->lbl.inst = -1;
  ci->group = (char *)NULL;
  ci->field = (char *)NULL;
  return ci;
}


/*  */
/*********************************************************************
 * Function: static void put_cut_item(infonode *in,cut_item anc)
 *
 * Append new item to current cut buffer.
 * Resolves SAMEASPARENT chain to the "real" infonode, but visibility
 * is derived from the original infonode.
 * All data are copied.
 *
 * Modification:
 *	<list mod's with name and date>
 */
static cut_item put_cut_item(in)
     infonode *in;
{
  cut_item ci;

  if (!started)
    return (cut_item)NULL;

  ci = new_cut_item();

  ci->ancestor = (cut_item)NULL;
  ci->visible = (in->widp != 0);

  while (in->index == SAMEASPARENT)
    in = in->parent;

  ci->index = in->index;
  (void)copylbl(&(ci->obj),&(in->obj));
  if ((ci->val.attsize = in->val.attsize) > 0) {
    ci->val.attvalue = malloc((ALLOC_T)in->val.attsize);
    (void)strncpy(ci->val.attvalue,in->val.attvalue,in->val.attsize);
  }

  switch (in->index) {
  case LINKITEM:
    ci->pos = in->pos;
    (void)copylbl(&(ci->lbl),&(in->lbl));
  case ATTRVALUE:
    ci->field = strdup(fieldtag(in->head));
  case ATTRFIELDTAG:
  case LINKFIELDTAG:
    ci->group = strdup(grouptag(in->head));
    break;
  case INTERNAL:
  case ATTRGROUPTAG:
  case LINKGROUPTAG:
  case IMAGE:
  default:
    break;
  }
  return ci;
}


/*  */
/*********************************************************************
 * Function: static int traverse_infonode(infonode *ip)
 *
 * Put visible items and end items into cut buffer.
 *
 * Modification:
 *	<list mod's with name and date>
 */
static int traverse_infonode(ip)
     infonode *ip;
{
  identry *idp;

  for ( ; ip; ip = ip->next) {
    if (!started) {
      if (ip == begincut) {
	started = 1;
      } else {
	if (ip == endcut) {
	  endcut = begincut;
	  begincut = ip;
	  started = 1;
	}
      }
    }
    if (!(ip->iflg & NONEXIST_MSK)) {
      (void)put_cut_item(ip);
      for (idp = ip->subnodes; idp; idp = idp->next) {
	if (traverse_infonode(idp->value))
	  return 1;
      }
    }
    if (ip == endcut)
      return 1;
  }
  (void)new_cut_item(); /* Puts an "end" item here */
  return 0;
}

/*  */
/*********************************************************************
 * Function: static int isplural(identry *idp)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int isplural(idp)
     identry *idp;
{
  if (!idp)
    return FALSE;
  if (idp->tmpl)
    idp = idp->tmpl;
  switch (idp->value->index) {
  case INTERNAL:
  case ATTRVALUE:
  case IMAGE:
  case SAMEASPARENT:
    return FALSE;
  case ATTRGROUPTAG:
  case ATTRFIELDTAG:
  case LINKGROUPTAG:
  case LINKFIELDTAG:
    return TRUE;
  case LINKITEM:
    return (idp->value->pos == -1);
  default:
    return TRUE;
  }
}

/*  */
/*********************************************************************
 * Function: static char *get_group(edge_list *edge)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static char *get_group(edge)
  edge_list *edge;
{
  static char *tmpbuf = (char *)NULL;
  infonode *inp;
  int pos;
  char *s;

  if (tmpbuf) {
    free(tmpbuf);
    tmpbuf = (char *)NULL;
  }
  if (edge->idp)
    return grouptag(edge->idp);
  s = edge->fragment->idp->value->group;
  if (s && !strcmp(s,"%GROUP%"))
    return (edge->owner)? get_group(edge->owner): "??";
  if (s && strcmp(s,"%VALUE%"))
    return s;
  if (!edge->owner)
    return edge->fragment->idp->idname;
  if (edge->owner->val.attvalue)
    return edge->owner->val.attvalue;
  if (!edge->owner->idp)
    return edge->fragment->idp->idname;
  for (inp = edge->owner->idp->value, pos=1; inp; inp=inp->next, pos++) {
    if (pos == edge->owner->pos) {
      tmpbuf = (char *)malloc((ALLOC_T)(inp->val.attsize + 2));
      for (pos=0,s=inp->val.attvalue; pos<inp->val.attsize; s++,pos++) {
        tmpbuf[pos] = ' ';
        if (*s)
          tmpbuf[pos] = *s;
      }
      tmpbuf[pos++] = '+';
      tmpbuf[pos++] = '\0';
      popup_message(POPUP_MESS_INFO, "%s%s%s\n%s\n%s%s%s\n%s\n%s",
		    "Cutting the item with the value \"",
		    inp->val.attvalue, 
		    "\"",
		    "will result in losing structure below.",
		    "Creating an item \"",
		    tmpbuf,
		    "\" to hold substructure.",
		    "If you wish to delete the item and its",
		    "substructure, use Meta-l-r.");
      return tmpbuf;
    }
  }
  return edge->fragment->idp->idname;
}

/*  */
/*********************************************************************
 * Function: static char *get_field(edge_list *edge)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static char *get_field(edge)
  edge_list *edge;
{
  static char *tmpbuf = (char *)NULL;
  infonode *inp;
  int pos;
  char *s;

  if (tmpbuf) {
    free(tmpbuf);
    tmpbuf = (char *)NULL;
  }
  if (edge->idp)
    return fieldtag(edge->idp);
  s = edge->fragment->idp->value->field;
  if (s && !strcmp(s,"%GROUP%"))
    return (edge->owner)? get_field(edge->owner): "??";
  if (s && strcmp(s,"%VALUE%"))
    return s;
  if (!edge->owner)
    return edge->fragment->idp->idname;
  if (edge->owner->val.attvalue)
    return edge->owner->val.attvalue;
  if (!edge->owner->idp)
    return edge->fragment->idp->idname;
  for (inp = edge->owner->idp->value, pos=1; inp; inp=inp->next, pos++) {
    if (pos == edge->owner->pos) {
      tmpbuf = (char *)malloc((ALLOC_T)(inp->val.attsize + 2));
      for (pos=0,s=inp->val.attvalue; pos<inp->val.attsize; s++,pos++) {
	tmpbuf[pos] = ' ';
	if (*s)
	  tmpbuf[pos] = *s;
      }
      tmpbuf[pos++] = '+';
      tmpbuf[pos++] = '\0';
      popup_message(POPUP_MESS_INFO, "%s%s%s\n%s\n%s%s%s\n%s\n%s",
		    "Cutting the item with the value \"",
		    inp->val.attvalue, 
		    "\"",
		    "will result in losing structure below.",
		    "Creating an item \"",
		    tmpbuf,
		    "\" to hold substructure.",
		    "If you wish to delete the item and its",
		    "substructure, use Meta-l-r.");
      return tmpbuf;
    }
  }
  return edge->fragment->idp->idname;
}


/*  */
/*********************************************************************
 * Function: static def_rec *finddef(identry *s)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static def_rec *finddef(s)
  identry *s;
{
  def_rec *p;
  identry *tmpidp = NULL;

  /*
   * get to the template identry: this must be the same identry as 
   * was put in by addtodefs
   */
  if (!s->tmpl)
    tmpidp = s;			/* template identries don't set this */
  else
    tmpidp = s->tmpl;		/* get to the template */

  tmpidp = tmpidp->value->head;	/* is this safe? is it always set */

  for (p = defs; p; p = p->next)
    if (tmpidp == p->idp)
      return p;
  return (def_rec *)NULL;
}


/*  */
/*********************************************************************
 * Function: static int addtodefs(identry *idp)
 *
 * Add an identry to the list of identries used later for lookup in 
 * edges.  We add the template identry, while also making sure that it's
 * the head identry if the template is recursive.
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int addtodefs(idp)
     identry *idp;
{
  def_rec *p;
  identry *tmpidp = NULL;

  /* get to the template identry */
  if (!idp->tmpl)
    tmpidp = idp;		/* template identries don't set this */
  else
    tmpidp = idp->tmpl;		/* get to the template */

  tmpidp = tmpidp->value->head;	/* is this safe? is it always set */

  if (finddef(tmpidp))
    return 0;
  p = (def_rec *)malloc(sizeof(def_rec));
  p->next = defs;
  defs = p;
  p->idp = tmpidp;
  return 1;
}

/*  */
/*********************************************************************
 * Function: static void free_defs()
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void free_defs()
{
  def_rec *p;

  for ( p = defs; defs; p = defs) {
    defs = defs->next;
    free((FREEPTR *)p);
  }
}

/*  */
/*********************************************************************
 * Function: static void define_fragments(identry *idp)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void define_fragments(idp)
     identry *idp;
{
  identry *idx;

  if (idp != idp->value->head) {
    /* This identry is a recursive thing defined elsewhere */
    return;
  }
  (void)addtodefs(idp);
  for (idx = idp->value->subnodes; idx; idx = idx->next)
    define_fragments(idx);
}

/*  */
/*********************************************************************
 * Function: static void catch_edge(identry *idp, int pos, int flag)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void catch_edge(idp,pos,flag)
     identry *idp; /* in target */
     int pos;
     int flag;
{
  edge_pnt *p;

  p = (edge_pnt *)malloc(sizeof(edge_pnt));
  p->next = edge;
  p->idp = idp;
  p->pos = pos;
  p->reducable = flag;
  edge = p;
}

/*  */
/*********************************************************************
 * Function: static void clear_edge()
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void clear_edge()
{
  edge_pnt *p;

  for (p = edge; p; p = edge) {
    edge = p->next;
    free((FREEPTR *)p);
  }
}

/*  */
/*********************************************************************
 * Function: static void into_edge_list(list,idp,pos,flag)
 * Parameters:
 *	edge_list **list
 *	identry *idp	 in target or template
 *	int pos
 *	int flag
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void into_edge_list(list,idp,pos,flag)
     edge_list **list;
     identry *idp; /* in target or template */
     int pos;
     int flag;
{
  edge_list *p;

  p = (edge_list *)malloc(sizeof(edge_list));
  p->fragment = finddef(idp);
  p->idp =  (idp->tmpl)? idp: (identry *)NULL; /* Discard template ptr */
  p->obj.vs = -1;
  p->obj.inst = -1;
  p->pos = pos;
  p->reducable = flag;
  p->val.attsize = 0;
  p->val.attvalue = (char *)NULL;
  p->owner = (edge_list *)NULL;
  p->next = (*list);
  (*list) = p;
}

/*  */
/*********************************************************************
 * Function: static void free_edge_list(edge_list *list)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void free_edge_list(list)
     edge_list *list;
{
  edge_list *p;

  for (p = list; p; p = list) {
    list = p->next;
    free((FREEPTR *)p);
  }
}

/*  */
/*********************************************************************
 * Function: static int length(edge_list *e, int all)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int length(e,all)
     edge_list *e;
     int all;
{
  int i;

  for (i=0; e; e = e->next)
    if (all || !e->reducable)
      i++;
  return i;
}

/*  */
/*********************************************************************
 * Function: static void make_edge_pnt(idp,pos,flag,list)
 * Parameters:
 *	identry *idp	in target
 *	int pos
 *	int flag
 *	edge_list **list
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void make_edge_pnt(idp,pos,flag,list)
     identry *idp; /* in target */
     int pos;
     int flag;
     edge_list **list;
{
  infonode *owner;
  infonode *inp;
  label obj;

  if (!idp)
    return;
  (void)copylbl(&obj,&(idp->value->obj));
  if (idp->value->iflg & NONEXIST_MSK)
    pos = 1;
  into_edge_list(list,idp,pos,flag);
  (void)copylbl(&((*list)->obj),&obj);
  owner = idp->owner;
  while ((idp = idp->next)) {
    flag = ((idp->value->iflg & NONEXIST_MSK) != 0);
    into_edge_list(list,idp,1,flag);
    (void)copylbl(&((*list)->obj),&obj);
  }
  while (owner) {
    idp = owner->head;
    (void)copylbl(&obj,&(idp->value->obj));
    for (inp = idp->value, pos = 1; inp; inp = inp->next, pos++)
      if (inp == owner)
        break;
    flag = (owner->next == (infonode *)NULL);
    if (isplural(idp)) {
      into_edge_list(list,idp,pos+1,flag);
      (void)copylbl(&((*list)->obj),&obj);
    }
    while ((idp = idp->next)) {
      flag = ((idp->value->iflg & NONEXIST_MSK) != 0);
      into_edge_list(list,idp,1,flag);
      (void)copylbl(&((*list)->obj),&obj);
    }
    owner = owner->parent;
  }
}

/*  */
/*ARGSUSED*/
/*********************************************************************
 * Function: static int moveattrfield(char *p, *field,int c, n)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int moveattrfield(p,field,c,n)
     char *p;
     char *field;
     int c;
     int n;
{
  attrval v;

  if (n == 0) {
    v.attsize = strlen(mapinfo.group)+1;
    v.attvalue = mapinfo.group;
      newlastcmd(CREATEOP, &mapinfo.obj, ATTRGROUPTAG,
		 (char *)NULL,(char *)NULL,0,&v,(label *)NULL);
    return GVS_STOP;
  }
  if (c == 0)
    return GVS_CONTINUE;
  if (GA_GETATTR(&mapinfo.source, mapinfo.group, field, &v)) {
    newlastcmd(CHANGEOP, &mapinfo.obj, ATTRVALUE,
	       mapinfo.group, field,0,&v,(label *)NULL);
    free(v.attvalue);
  }
  return GVS_CONTINUE;
}  
    
/*  */
/*ARGSUSED*/
/*********************************************************************
 * Function: static int movelinkitem(char *p,label *item,int c, n)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int movelinkitem(p,item,c,n)
     char *p;
     label *item;
     int c;
     int n;
{
  attrval v;

  if (n == 0) {
    v.attsize = strlen(mapinfo.field)+1;
    v.attvalue = mapinfo.group;
    newlastcmd(CREATEOP, &mapinfo.obj, LINKFIELDTAG,
	       mapinfo.group,(char *)NULL,0,&v,(label *)NULL);
    return GVS_STOP;
  }
  if (c == 0)
    return GVS_CONTINUE;
  newlastcmd(LINKOP, &mapinfo.obj, LINKITEM, mapinfo.group,
	     mapinfo.field, c, (attrval *)NULL, item);
  return GVS_CONTINUE;
}

/*  */
/*ARGSUSED*/
/*********************************************************************
 * Function: static int movelinkfield(char *p, *field,int c, n)

 *
 * Modification:
 *      <list mod's with name and date>
 */
static int movelinkfield(p,field,c,n)
     char *p;
     char *field;
     int c;
     int n;
{
  attrval v;

  if (n == 0) {
    v.attsize = strlen(mapinfo.group)+1;
    v.attvalue = mapinfo.group;
      newlastcmd(CREATEOP, &mapinfo.obj, LINKGROUPTAG,
		 (char *)NULL,(char *)NULL,0,&v,(label *)NULL);
    return GVS_STOP;
  }
  if (c == 0)
    return GVS_CONTINUE;
  mapinfo.field = field;
  (void)GLV_GETLINKVAL(&mapinfo.source, mapinfo.group, field,
		       movelinkitem, (char *)NULL);
  return GVS_CONTINUE;
}

/*  */
/*********************************************************************
 * Function: static void get_real_obj(label *obj,edge_list *edge)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void get_real_obj(obj,edge)
  label *obj;
  edge_list *edge;
{
  int ix;

  (void)copylbl(obj,&(edge->obj));
  if ((ix = obj->vs) < 0) {
    obj->vs = newobj[-2 - ix].assigned;
    obj->inst = -1;
    if (obj->vs == -1) {
      (void)CO_CREATEOBJ(obj);
      (void)UBL_UNBINDLABEL(obj);
      newobj[-2 - ix].assigned = obj->vs;
    }
  }
}

/*  */
/*********************************************************************
 * Function: static void erase_subtree(identry *idp,int pos)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void erase_subtree(idp,pos)
     identry *idp;
     int pos;
{
  identry *idx;
  infonode *inp;
  int i;

  if (!isplural(idp)) {
    for (idx=idp->value->subnodes; idx; idx=idx->next)
      erase_subtree(idx,1);
  }
  if (idp->tmpl->value->index == SAMEASPARENT)
    return;
  for (inp=idp->value, i=1; inp; inp=inp->next, i++) {
    if (i < pos)
      continue;
    inp->pos = pos;
    remove_infonode(inp);
    inp->pos = i;
  }
}

/*  */
/*********************************************************************
 * Function: static void edge_match(edge_list *left, *right, int i, j)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void edge_match(left,right)
     edge_list *left;
     edge_list *right;
{
  edge_list *vedge;
  int section = 0;
  char *group;
  char *field;
  infonode *inp;
  int pos;
  label obj;
  int i;

  if (!right) {
    /* Ensure that the left edge ends at left->pos */
    if (left && left->idp)
      erase_subtree(left->idp,left->pos);
    return;
  }
  if (!left) {
    /* Ensure that the right edge is empty */
    if (!right || !right->idp)
      return;
    if (right->idp->tmpl->value->index == SAMEASPARENT)
      return;
    for (inp = right->idp->value, pos = 1; inp; inp = inp->next, pos++) {
      if (pos >= right->pos)
	break;
      inp->pos = 1; /* Fake the position!! */
      (void)remove_infonode(inp);
      inp->pos = pos; /* Restore position!! */
      /*
       * Note: by faking pos=1 for all infonodes, a list of link items
       * becomes cleared through the successive removes of the first
       * link item.
       */
    }
    return;
  }
  get_real_obj(&obj,left);

  for (vedge = left; vedge; vedge = vedge->owner) {
    vedge->val.attsize = right->idp->value->val.attsize;
    vedge->val.attvalue = right->idp->value->val.attvalue;
    if (vedge->fragment->idp->value->index != SAMEASPARENT)
      break;
  }

  if (!vedge)
    return;
  switch (vedge->fragment->idp->value->index) {
  case ATTRGROUPTAG:
    section = 1;
  case LINKGROUPTAG:
    if (obj.vs == right->obj.vs) {
      for (inp = right->idp->value, pos = 1; inp; inp = inp->next, pos++) {
        if (pos >= right->pos)
          break;
        if (pos >= left->pos) {
	  (void)remove_infonode(inp);
        }
      }
      return;
    }
    if (left->idp) {
      for (inp = left->idp->value, pos = 1; inp; inp = inp->next, pos++) {
	if (pos >= left->pos) {
          (void)remove_infonode(inp);
	}
      }
    }
    for (inp = right->idp->value, pos = 1; inp; inp = inp->next, pos++) {
      if ((pos >= right->pos) && !(inp->iflg & NONEXIST_MSK)) {
	(void)copylbl(&mapinfo.source,&(inp->obj));
	(void)copylbl(&mapinfo.obj,&obj);
	mapinfo.group = inp->val.attvalue;
	if (section == 1) {
	  (void)GAN_GETATTRNAMES(&(inp->obj),inp->val.attvalue, moveattrfield,
			    (char *)NULL);
	} else {
	  (void)GLN_GETLINKNAMES(&(inp->obj),inp->val.attvalue, movelinkfield,
			   (char *)NULL);
	}
      }
    }
    break;
  case ATTRFIELDTAG:
    section = 1;
  case LINKFIELDTAG:
    group = get_group(vedge);
    if ((obj.vs == right->obj.vs) &&
	!strcmp(group,right->idp->value->group)) {
      for (inp = right->idp->value, pos = 1; inp; inp = inp->next, pos++) {
        if (pos >= right->pos)
          break;
        if (pos >= left->pos) {
	  (void)remove_infonode(inp);
	}
      }
      return;
    }
    if (left->idp) {
      for (inp = left->idp->value, pos = 1; inp; inp = inp->next, pos++) {
	if (pos >= left->pos) {
          (void)remove_infonode(inp);
	}
      }
    }
    for (inp = right->idp->value, pos = 1; inp; inp = inp->next, pos++) {
      if ((pos >= right->pos) && !(inp->iflg & NONEXIST_MSK)) {
        (void)copylbl(&mapinfo.source,&(inp->obj));
        (void)copylbl(&mapinfo.obj,&obj);
        mapinfo.group = grouptag(inp->head);
	mapinfo.field = inp->val.attvalue;
	(void)GLV_GETLINKVAL(&mapinfo.source, mapinfo.group, mapinfo.field,
			     movelinkitem, (char *)NULL);
      }
    }
    break;
  case ATTRVALUE:
    group = get_group(vedge);
    field = get_field(vedge);
    /* sanity check */
    if (!group || !field) {
      return;
    }
    if ((obj.vs == right->obj.vs) &&
	(right->idp->value->index == ATTRVALUE) &&
	!strcmp(group,right->idp->value->group) &&
	!strcmp(field,right->idp->value->field))
      return;
    newlastcmd(CHANGEOP, &obj, ATTRVALUE, group, field, 0,
	       &(right->idp->value->val),(label *)NULL);
    break;
  case LINKITEM:
    group = get_group(vedge);
    field = get_field(vedge);
    /* sanity check */
    if (!group || !field) {
      return;
    }
    if ((obj.vs == right->obj.vs) &&
        !strcmp(group,right->idp->value->group) &&
        !strcmp(field,right->idp->value->field)) {
      for (inp = right->idp->value, pos = 1; inp; inp = inp->next, pos++) {
	if (pos >= right->pos)
	  break;
	if (pos >= left->pos) {
	  i = inp->pos;
	  inp->pos = left->pos;
	  (void)remove_infonode(inp);
	  inp->pos = i;
	}
      }
      return;
    }
    if (left->idp && !(left->idp->value->iflg & NONEXIST_MSK)) {
      for (inp=left->idp->value, pos=1; inp; inp=inp->next, pos++) {
	if (pos >= left->pos) {
	  i = inp->pos;
	  inp->pos = left->pos;
          (void)remove_infonode(inp);
	  inp->pos = i;
	}
      }
    }
    for (inp=right->idp->value, pos=1; inp; inp=inp->next, pos++) {
      if ((pos >= right->pos) && !(inp->iflg & NONEXIST_MSK))  {
	if (pasting) {
	  i = inp->pos;
	  inp->pos = right->pos;
          (void)remove_infonode(inp);
	  inp->pos = i;
	}
	newlastcmd(LINKOP, &obj, LINKITEM, group,
		   field, left->pos + (pos - right->pos), (attrval *)NULL,
		   &(inp->lbl));
#ifdef DEBUG
	(void)printf("Set link %s:%s:%d of <%d:-1> to <%d:%d>\n",
	       group, field, left->pos + (pos - right->pos),
	       obj.vs,inp->lbl.vs,inp->lbl.inst);
#endif	/* DEBUG */
      }
    }
    break;
  case IMAGE:
    if (obj.vs == right->obj.vs)
      return;
    newlastcmd(CHANGEOP, &obj, IMAGE, (char *)NULL, (char *)NULL, 0,
	       &(right->idp->value->val),(label *)NULL);
    break;
  case SAMEASPARENT:
  case INTERNAL:
  default:
    /* Don't bother with these ... */
    break;
  }
}

/*  */
/*********************************************************************
 * Function: static int expand_by_fragment(edge_list **list, *owner)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int expand_by_fragment(list,owner)
     edge_list **list;
     edge_list *owner;
{
  identry *tmpl = owner->fragment->idp;
  identry *idp;
  label obj;

  if (!tmpl->value->subnodes)
    return FALSE;
  if (tmpl->value->index == LINKITEM) {
    obj.vs = ocount--;
    obj.inst = -1;
    newobj[-2 - obj.vs].assigned = -1;
    newobj[-2 - obj.vs].assumed = -1;
  } else {
    (void)copylbl(&obj,&(owner->obj));
  }
  for (idp = tmpl->value->subnodes; idp; idp=idp->next) {
    into_edge_list(list,idp,1,1);
    (void)copylbl(&((*list)->obj),&obj);
    (*list)->owner = owner;
  }
  return TRUE;
}

/*  */
/*********************************************************************
 * Function: static int adapt_by_expanding(edge_list *left, *right, int i, j)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int adapt_by_expanding(left,right)
     edge_list *left;
     edge_list *right;
{
  identry *tmpl = left->fragment->idp;
  int ocount_saved = ocount;
  int rtn = 0;
  edge_list *eol = left->next;
  edge_list *list = left->next;
  edge_list *p;
  char *group;
  char *field;
  label obj;
  label lbl;

  if (!expand_by_fragment(&list,left))
    return FALSE;
  rtn = (adapt_edge(list,right,0) || adapt_edge(list,right,1));
  if (rtn && (tmpl->value->index == LINKITEM)) {
    group = get_group(left);
    field = get_field(left);
    get_real_obj(&lbl,list);
    get_real_obj(&obj,left);
    newlastcmd(LINKOP, &obj, LINKITEM,group,field,left->pos,
	       (attrval *)NULL, &lbl);
#ifdef DEBUG
    (void)printf("Set expanded link %s:%s:%d of <%d:-1> to <%d:-1>\n",
	   group, field, left->pos,obj.vs,lbl.vs);
#endif	/* DEBUG */
  }
  for (p = list; p != eol; p = list) {
    list = p->next;
    free((FREEPTR *)p);
  }
  if (!rtn)
    ocount = ocount_saved;
  return rtn;
}

/*  */
/*********************************************************************
 * Function: static void value_of_edge(edge_list *edge,*from)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void value_of_edge(edge,from)
  edge_list *edge;
  edge_list *from;
{
  edge->val.attsize = 0;
  edge->val.attvalue = (char *)NULL;
  if (!from || !from->idp || !from->idp->value)
    return;
  if (from->idp->value->iflg & NONEXIST_MSK)
    return;
  edge->val.attsize = from->idp->value->val.attsize;
  edge->val.attvalue = from->idp->value->val.attvalue;
}

/*  */
/*********************************************************************
 * Function: static int adapt_edge(edge_list *left, *right,int exp)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int adapt_edge(left,right,exp)
     edge_list *left;
     edge_list *right;
     int exp;
{
  edge_list *e;

  if (!left) {
    for (e = right; e; e = e->next)
      if (!e->reducable)
	return FALSE;

    for (e = right; e; e = e->next)
      edge_match((edge_list *)NULL,e);

    return TRUE;
  }
  value_of_edge(left,(edge_list *)NULL);
  if (!right) {
    for (e = left; e; e = e->next)
      edge_match(e,(edge_list *)NULL);
    return TRUE;
  }
  if (left->fragment == right->fragment) {
    value_of_edge(left,right);
    if (adapt_edge(left->next,right->next,exp)) {
      edge_match(left,right);
      return TRUE;
    }
    value_of_edge(left,(edge_list *)NULL);
  }
  if (length(left,1) > length(right,0)) {
    if (adapt_edge(left->next,right,exp)) {
      edge_match(left,(edge_list *)NULL);
      return TRUE;
    }
  }
  if (right->reducable) {
    if (adapt_edge(left,right->next,exp)) {
      return TRUE;
    }
  }
  if (exp) {
    if (isplural(left->fragment->idp)) {
      if (left->fragment == right->fragment) {
	if (adapt_by_expanding(left,right->next)) {
          if (left->fragment->idp->value->index == LINKITEM) {
            left->pos++;
            if (left->obj.vs == right->obj.vs)
              right->pos++;
          }
	  edge_match(left,right);
	  return TRUE;
	}
      }
    } else {
      if (adapt_by_expanding(left,right)) {
	return TRUE;
      }
    }
  }
  return FALSE;
}

/*  */
/*********************************************************************
 * Function: static void init_cuts()
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void init_cuts()
{
  static int first = 1;
  int i;

  if (first) {
    left.count = 50;
    right.count = 50;
  }
  for (i = 0; i<left.count; i++) {
    if (!first)
      free_edge_list(left.edge[i]);
    left.edge[i] = (edge_list *)NULL;
  }
  left.count = 0;
  for (i = 0; i<right.count; i++) {
    if (!first)
      free_edge_list(right.edge[i]);
    right.edge[i] = (edge_list *)NULL;
  }
  right.count = 0;
  first = 0;
}

/*  */
/*********************************************************************
 * Function: static void make_edge(int mode)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void make_edge(mode)
     int mode;
{
  edge_list **list;
  edge_pnt *p;
  int i, j;

  for (i = 0, p = edge; p; p = p->next, i++) {
    list = (mode)? &right.edge[i]: &left.edge[i];
    make_edge_pnt(p->idp,p->pos,p->reducable,list);
  }

  if (!mode) {
    left.count = i;
    return;
  }
  right.count = i;

  ocount = -2;
  for (j = left.count-1; j>=0; j--) {
    for (i = right.count-1; i>=0; i--) {
      if (adapt_edge(left.edge[j],right.edge[i],1))
        return;
    }
  }
}

/*  */
/*********************************************************************
 * Function: static int reduce_to_fragments(infonode *inp,int mode)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int reduce_to_fragments(inp,mode)
     infonode *inp;
     int mode;
{
  identry *idp;
  infonode *xnp;
  int pos;

  for (idp = inp->subnodes; idp; idp = idp->next) {
    for (xnp = idp->value, pos = 1; xnp; xnp = xnp->next, pos++) {
      catch_edge(idp, pos, ((xnp->iflg & NONEXIST_MSK) != 0));
      if (xnp == begincut) {
        make_edge(0);
        mode = 1;
      }
      if ((mode == 0) && (xnp == endcut)) {
        endcut = begincut;
        begincut = xnp;
        make_edge(0);
        mode = 1;
      }
      if (!(xnp->iflg & NONEXIST_MSK)) {
        if (!xnp->subnodes) {
          if (mode == 2) {
            make_edge(1);
            clear_edge();
            return 3;
          }
          clear_edge();
        }
        if ((mode = reduce_to_fragments(xnp,mode)) == 3)
          return 3;
      }
      if (xnp == endcut) {
        mode = 2;
      }
    }
    if (isplural(idp)) {
      catch_edge(idp, pos,1);
    }
  }
  return mode;
}

/*  */
/*********************************************************************
 * Function: static void cut_out_region(infonode *inp, *inq)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void cut_out_region(inp,inq)
     infonode *inp;
     infonode *inq;
{
  reference_structure *ref = whichref(inp, (identry *)NULL, 0);

  if (!ref)
    return;
  defs = (def_rec *)NULL;
  begincut = inp;
  endcut = inq;

  pasting = 0;
  define_fragments(ref->template);
  init_cuts();
  clear_edge();
  if (reduce_to_fragments(ref->target->value,0) < 3) {
    make_edge(1);
  }
  clear_edge();
  free_defs();
}

/*  */
/*********************************************************************
 * Function: static cut_item get_cut_buffer()
 *
 * Returns pointer to current cut buffer.
 *
 * Modification:
 *	<list mod's with name and date>
 */
static cut_item get_cut_buffer()
{
  if (CBPDL)
    return CBPDL->head;
  return (cut_item)NULL;
}

/*  */
/*********************************************************************
 * Function: void make_cut_of_string(char *s)
 *
 * Modification:
 *      <list mod's with name and date>
 */
void make_cut_of_string(s)
  char *s;
{
  cut_item ci;

  new_cut_buffer();

  started = 1;
  ci = new_cut_item();

  ci->ancestor = (cut_item)NULL;
  ci->visible = 1;
  ci->obj.vs = -77; /* An arbitrary neg value except -1 */
  ci->index = INTERNAL;
  if (s) {
    ci->val.attsize = strlen(s)+1;
    ci->val.attvalue = strdup(s);
  }
  started = 0;
}

/*  */
/*********************************************************************
 * Function: void pop_cut_buffers()
 *
 * Pops the cut buffer push down list.
 *
 * Modification:
 *	<list mod's with name and date>
 */
void pop_cut_buffers(dofree)
     int dofree;
{
  cut_buffer cb;

  if (!CBPDL)
    return;
  if (dofree)
    free_cut_buffer(CBPDL->head);
  cb = CBPDL;
  CBPDL = cb->next;
  free((FREEPTR *)cb);
}


/*  */
/*********************************************************************
 * Function: void copy_region(infonode *fst,infonode *lst,int cut)
 *
 * Copy region to new cut buffer. Remove if cut non-zero.
 * Region is marked by two infonodes within the same reference
 * structure.
 * The cut buffer contains visible items and end items reached during
 * depth-first traversal, beginning after reaching the first of the given
 * infonodes, and ending when having traversed the last of the given
 * infonodes.
 *
 * Modification:
 *	<list mod's with name and date>
 */
void copy_region(fst,lst,cut)
     infonode *fst, *lst;
     int cut;
{
  infonode *root;

  if (!fst)
    return;

  new_cut_buffer();

  for (root = fst; root->parent; root = root->parent)
    ;

  begincut = fst; /* These may be given in wrong order... */
  endcut = lst; 
  started = 0;
  (void)traverse_infonode(root);
  if (cut)
    cut_out_region(begincut,endcut);    
}

/*  */
/*********************************************************************
 * Function: static edge_list *reverse_edge(edge_list *edge)
 *
 * Reverses the edge point list destructively.
 *
 * Modification:
 *      <list mod's with name and date>
 */
static edge_list *reverse_edge(edge)
     edge_list *edge;
{
  edge_list *p, *q;

  if (!edge || !edge->next)
    return edge;

  p = edge->next;
  q = reverse_edge(p);
  p->next = edge;
  edge->next = (edge_list *)NULL;
  return q;  
}

/*  */
/*********************************************************************
 * Function: static void cut_match(edge_list *edge,cut_item cut)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void cut_match(edge,cut)
  edge_list *edge;
  cut_item cut;
{
  infonode *inp;
  edge_list *vedge = (edge_list *)NULL;
  char *group = (char *)NULL;
  char *field = (char *)NULL;
  label obj;
  label lbl;
  int i;
  int pos;
  int same_item;

  if (!cut) {
    /* The edge point is matched with an inserted end item */
    if (!edge || !edge->idp /* || edge->fragment->idp->value->subnodes */)
      return;
    for (inp = edge->idp->value, pos=1; inp; inp=inp->next) {
      if (pos >= edge->pos) {
        i = inp->pos;
        inp->pos = edge->pos;
	(void)remove_infonode(inp);
        inp->pos = i;
      }
    }
    return;
  }
  if (!edge) {
    /* The cut item is discarded */
    return;
  }
  get_real_obj(&obj,edge);

  for (vedge = edge; vedge; vedge = vedge->owner) {
    vedge->val.attsize = cut->val.attsize;
    vedge->val.attvalue = cut->val.attvalue;
    if (vedge->fragment->idp->value->index != SAMEASPARENT)
      break;
  }

  same_item = (vedge->fragment->idp->value->index == cut->index);
        
  switch (vedge->fragment->idp->value->index) {
  case ATTRFIELDTAG:
  case LINKFIELDTAG:
    group = get_group(vedge);
    if (same_item)
      same_item = (strcmp(group,cut->group) == 0);
  case ATTRGROUPTAG:
  case LINKGROUPTAG:
    newlastcmd(CREATEOP, &obj, vedge->fragment->idp->value->index,
	       group, (char *)NULL, 0, &(cut->val),(label *)NULL);
    break;
  case ATTRVALUE:
    group = get_group(vedge);
    field = get_field(vedge);
    if (same_item)
      same_item = ((strcmp(group,cut->group) == 0) &&
		   (strcmp(field,cut->field) == 0));
    newlastcmd(CHANGEOP, &obj, ATTRVALUE, group, field, 0,
	       &(cut->val),(label *)NULL);
    break;
  case LINKITEM:
    group = get_group(vedge);
    field = get_field(vedge);
    if ((cut->index == LINKITEM) &&!(paste_option & PASTE_COPY_LEAF)) {
      if (same_item)
	same_item = ((strcmp(group,cut->group) == 0) &&
		     (strcmp(field,cut->field) == 0));
      newlastcmd(LINKOP, &obj, LINKITEM, group, field, edge->pos,
		 (attrval *)NULL, &(cut->lbl));
#ifdef DEBUG
      (void)printf("Set link %s:%s:%d in object <%d:-1> to ",
	     group,field,edge->pos,obj.vs);
      (void)printf("<%d:%d>\n",cut->lbl.vs,cut->lbl.inst);
#endif	/* DEBUG */
      break;
    }
    same_item = FALSE;
    (void)CO_CREATEOBJ(&lbl);
    (void)UBL_UNBINDLABEL(&lbl);
    if (paste_option & PASTE_COPY_WHOLE) {
      if (cut->index == LINKITEM)
	newlastcmd(COPYOP, &lbl, LINKITEM, (char *)NULL, (char *)NULL, 0,
		   (attrval *)NULL,&(cut->lbl));
      else
	newlastcmd(COPYOP, &lbl, LINKITEM, (char *)NULL, (char *)NULL, 0,
		   (attrval *)NULL,&(cut->obj));
    }
    newlastcmd(CHANGEOP, &lbl, IMAGE, (char *)NULL, (char *)NULL, 0,
	       &(cut->val),(label*)NULL);
    newlastcmd(LINKOP, &obj, LINKITEM, group, field, edge->pos,
               (attrval *)NULL, &lbl);
#ifdef DEBUG
    (void)printf("Set link %s:%s:%d in object <%d:-1> to new\n",
	   group,field,edge->pos,obj.vs);
#endif	/* DEBUG */
    break;
  case IMAGE:
    newlastcmd(CHANGEOP, &obj, IMAGE, (char *)NULL, (char *)NULL, 0,
               &(cut->val),(label *)NULL);
    break;
  case SAMEASPARENT:
  case INTERNAL:
  default:
    /* Don't bother with these ... */
    return;
  }
  if (edge->obj.vs >= 0)
      return;
  if (same_item && (cut->obj.vs == newobj[-2 - edge->obj.vs].assumed))
    return;
  newobj[-2 - edge->obj.vs].assumed = -1;
}

/*  */
/*********************************************************************
 * Function: static void value_of_cut(edge_list *edge,cut_item from)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void value_of_cut(edge,from)
  edge_list *edge;
  cut_item from;
{
  edge->val.attsize = 0;
  edge->val.attvalue = (char *)NULL;
  if (!from || (from->obj.vs == -1))
    return;
  edge->val.attsize = from->val.attsize;
  edge->val.attvalue = from->val.attvalue;
  if (edge->fragment->idp->value->index != SAMEASPARENT)
    return;
  if (edge->owner)
    value_of_cut(edge->owner,from);
}

/*  */
/*********************************************************************
 * Function: static int paste_by_expanding(cut_item cut,edge_list *edge, *right,int mode)
 *
 * Tries to adapt the edge to the cut by expanding the edge point for
 * a new item.
 * Modification:
 *      <list mod's with name and date>
 */
static int paste_by_expanding(cut,edge,right,mode)
  cut_item cut;
  edge_list *edge;
  edge_list *right;
  int mode;
{
  edge_list *list = (edge_list *)NULL;
  edge_list *eol = (edge_list *)NULL;
  int ocount_saved = ocount;
  int rtn = FALSE;

  if (!expand_by_fragment(&list,edge))
    return FALSE;
  if (isplural(edge->fragment->idp)) {
    into_edge_list(&list,edge->fragment->idp,edge->pos,1);
    (void)copylbl(&(list->obj),&(edge->obj));
    list->owner = edge->owner;
    list->idp = edge->idp;
  }

  eol = list;
  list = reverse_edge(list);
  eol->next = edge->next;
  if (edge->fragment->idp->value->index == LINKITEM) {
    if (cut->index == LINKITEM) {
      newobj[-2 - ocount_saved].assumed = cut->lbl.vs;
    } else {
      newobj[-2 - ocount_saved].assumed = cut->obj.vs;
    }
  }
  if (cut->visible) {
    value_of_cut(edge,(cut_item)NULL);
    rtn = adapt_paste(cut,list,right,mode);
  } else {
    value_of_cut(edge,cut);
    rtn = adapt_paste(cut->next,list,right,mode);
  }
  if (rtn) {
    if (edge->fragment->idp->value->index == LINKITEM) {
      char *group;
      char *field;
      label lbl;
      label obj;

      group = get_group(edge);
      field = get_field(edge);
      get_real_obj(&obj,edge);
      get_real_obj(&lbl,list);
      if (!cut->visible) {
	if (paste_option & PASTE_COPY_WHOLE) {
	  label old, new;
	  int ix = list->obj.vs;
	  if (ix<0) {
	    new.vs = newobj[-2-ix].assigned;
	    new.inst = -1;
	    old.vs = newobj[-2-ix].assumed;
	    old.inst = -1;
	    if ((new.vs != -1) && (old.vs != -1))
	      newlastcmd(COPYOP, &new, LINKITEM, (char *)NULL, (char *)NULL, 0,
			 (attrval *)NULL,&old);
	  }
	} else {
	  newlastcmd(CHANGEOP, &lbl, IMAGE, (char *)NULL, (char *)NULL, 0,
		     &(cut->val),(label*)NULL);
	}
      }
      newlastcmd(LINKOP, &obj, LINKITEM, group, field, edge->pos,
		 (attrval *)NULL, &lbl);
#ifdef DEBUG
      (void)printf("Set link %s:%s:%d in object <%d:-1> to <%d:-1>\n",
	     group,field,edge->pos,obj.vs,lbl.vs);
#endif	/* DEBUG */
    } else {
      if (!cut->visible) {
	cut_match(edge,cut);
      }
    }
  }
  eol->next = (edge_list *)NULL;
  free_edge_list(list);
  if (!rtn)
    ocount = ocount_saved;
  return rtn;
}


static int CUT(c)
     cut_item c;
{
  while (c) {
#ifdef DEBUG
    (void)printf("%c ",((c->visible)?'v':((c->obj.vs != -1)?'i':'e')));
#endif	/* DEBUG */
    c = c->next;
  }
#ifdef DEBUG
  (void)printf("\n");
#endif	/* DEBUG */
  return 0;
}

static int EDGE(e)
     edge_list *e;
{
  while (e) {
#ifdef DEBUG
    (void)printf("\"%s\"" ,e->fragment->idp->idname);
#endif	/* DEBUG */
    e = e->next;
  }
#ifdef DEBUG
  (void)printf("\n");
#endif	/* DEBUG */
  return 0;
}

/*  */
/*********************************************************************
 * Function: static int adapt_paste(cut_item cut,edge_list *edge, *right, int mode)
 *
 * NOTE: the 'edge' is in reverse order to 'right', i.e. we fill in cut
 * items from top to bottom in the reference structure.
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int adapt_paste(cut,edge,right,mode)
  cut_item cut;
  edge_list *edge;
  edge_list *right;
  int mode;
{
  int rtn;

  if (!cut) {
    edge = reverse_edge(edge);
    rtn = adapt_edge(edge,right,0);
    edge = reverse_edge(edge);
    return rtn;
  }
  if (!edge) {
    for ( ; cut; cut = cut->next)
      if (!(cut->obj.vs == -1))
	return FALSE;
    return adapt_edge((edge_list *)NULL,right,0);
  }
  if (cut->obj.vs == -1) {
    value_of_cut(edge,(cut_item)NULL);
    if (adapt_paste(cut->next,edge->next,right,mode)) {
      cut_match(edge,(cut_item)NULL);
      return TRUE;
    }
    value_of_cut(edge,(cut_item)NULL);
    if (!edge->fragment->idp->value->subnodes) {
      if (adapt_paste(cut,edge->next,right,mode)) {
	cut_match(edge,(cut_item)NULL);
	return TRUE;
      }
    }
    if ((mode & ADAPT_DROP_END) &&
	adapt_paste(cut->next,edge,right,mode)) {
      return TRUE;
    }
    return FALSE;
  }
  if (cut->visible) {
    if (!edge->fragment->idp->value->subnodes) {
      value_of_cut(edge,cut);
      if (!isplural(edge->fragment->idp)) {
	if (cut->next && (cut->next->obj.vs == -1)) {
	  if (adapt_paste(cut->next->next,edge->next,right, mode)) {
	    cut_match(edge,cut);
	    return TRUE;
	  }
	}
      } else { /* isplural(edge->fragment->idp) */
	if (adapt_paste(cut->next,edge,right,mode)) {
	  cut_match(edge,cut);
	  return TRUE;
	}
      }
      value_of_cut(edge,cut);
      if (adapt_paste(cut->next,edge->next,right,mode)) {
	cut_match(edge,cut);
	return TRUE;
      }
      return FALSE;
    }
    /* edge->fragment->idp->value->subnodes */
    value_of_cut(edge,(cut_item)NULL);
    if (mode & ADAPT_FIRST) {
      if (paste_by_expanding(cut,edge,right,mode)) {
	return TRUE;
      }
    }
    value_of_cut(edge,(cut_item)NULL);
    if (adapt_paste(cut,edge->next,right,mode)) {
      cut_match(edge,(cut_item)NULL);
      return TRUE;
    }
    return FALSE;
  }
  /* !cut->visible */
  if (edge->fragment->idp->value->subnodes) {
    if (paste_by_expanding(cut,edge,right,mode)) {
      return TRUE;
    }
    value_of_cut(edge,cut);
    if (adapt_paste(cut->next,edge->next,right,(mode&~ADAPT_FIRST))) {
      cut_match(edge,cut);
      return TRUE;
    }
  }
  value_of_cut(edge,(cut_item)NULL);
#if 0
  if (adapt_paste(cut,edge->next,right,mode)) {
    cut_match(edge,(cut_item)NULL);
    return TRUE;
  }
#endif
  if (!(mode & ADAPT_DROP_INNER))
    return FALSE;
  if (adapt_paste(cut->next,edge,right,mode)) {
    return TRUE;
  }
  return FALSE;
}

/*  */
/*********************************************************************
 * Function: static void paste_reduce()
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void paste_reduce()
{
  int i;

  for (i=0; i<(-2-ocount); i++)
    if (newobj[i].assumed >= 0)
      revise_command_stream(newobj[i].assigned,newobj[i].assumed);
}



/*  */
/*********************************************************************
 * Function: static int do_paste_edge()
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int do_paste_edge(mode)
     int mode;
{
  int i, ix, incr;

  ocount = -2;
  if (paste_option & PASTE_AFTER) {
    ix = left.count-1;
    incr = -1;
  } else {
    ix = 0;
    incr = 1;
  }
  for (i = 0; i < left.count; i++, ix += incr) {
    if (adapt_paste(get_cut_buffer(),left.edge[ix],right.edge[ix],mode)) {
      if (paste_option & PASTE_SHARE)
        paste_reduce();
      return 1;
    }
  }
  return 0;
}


/*  */
/*********************************************************************
 * Function: static int paste_edge()
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int paste_edge()
{
  edge_list **list;
  edge_pnt *p;
  int i;

  for (i = 0, p = edge; p; p = p->next, i++) {
    list = &left.edge[i];
    make_edge_pnt(p->idp,p->pos,p->reducable,list);
    left.edge[i] = reverse_edge(left.edge[i]);
    list = &right.edge[i];
    make_edge_pnt(p->idp,p->pos,p->reducable,list);
  }

  left.count = i;
  right.count = i;

  if (do_paste_edge(ADAPT_FIRST)) {
#ifdef DEBUG
    (void)printf("Done.\n");
#endif	/* DEBUG */
    return 1;
  }
  (void)printf("Hmm.. Maybe needing to relax structure..\n");
  if (do_paste_edge(ADAPT_ALL)) {
#ifdef DEBUG
    (void)printf("Done.\n");
#endif	/* DEBUG */
    return 1;
  }
  (void)printf("Nope.. Couldn't paste it there..\n");
  return 0;
}

/*  */
/*********************************************************************
 * Function: static int find_paste_fragments(infonode *inp,int mode)
 *
 * Modification:
 *      <list mod's with name and date>
 */
static int find_paste_fragments(inp,mode)
     infonode *inp;
     int mode;
{
  identry *idp;
  infonode *xnp;
  int pos;

  for (idp = inp->subnodes; idp; idp = idp->next) {
    for (xnp = idp->value, pos = 1; xnp; xnp = xnp->next, pos++) {
      if (!(xnp->iflg & NONEXIST_MSK))
	catch_edge(idp,pos,0);
      if (!xnp->subnodes) {
	if (mode == 1)
	  return 3;
	if (xnp == insert_point) {
	  if (!(paste_option & PASTE_AFTER))
	    return 3;
	  mode = 1;
	}
	clear_edge();
      }
      if ((mode = find_paste_fragments(xnp,mode)) == 3)
	return 3;
    }
    if (isplural(idp))
      catch_edge(idp, pos,1);
  }
  return mode;
}

/*  */
/*********************************************************************
 * Function: int paste(infonode *inp,int option)
 *
 *
 * PASTE_AFTER		1
 * PASTE_COPY_LEAF	2
 * PASTE_COPY_WHOLE	4
 * PASTE_SHARE		8
 *
 * Modification:
 *	David, 1993-05-28: make it return a value.
 *      <list mod's with name and date>
 */
int paste(inp,option)
  infonode *inp;
  int option;
{
  reference_structure *ref = whichref(inp, (identry *)NULL, 0);
  int ret_val = 0;

  if (!ref)
    return ret_val;
  defs = (def_rec *)NULL;
  insert_point = inp;
  paste_option = option;

  pasting = 1;
  define_fragments(ref->template);
  init_cuts();
  clear_edge();
  if (find_paste_fragments(ref->target->value,0)) {
    ret_val = paste_edge();
  }
  clear_edge();
  free_defs();

  return ret_val;
}
