/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	aimcursor.c
 *
 * SCCSINFO:		@(#)aimcursor.c	1.6 5/3/94
 *
 * ORIGINAL AUTHOR(S):  1992-08-29, Ralph R\"onnquist
 *
 * DESCRIPTION:
 *	A cursor handling module.
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 *********************************************************************/
/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 * cursor clipcursor(cursor c1, cursor c2)
 * cursor freecursor(cursor c)	- Reclaim memory and return (cursor)0
 * int cursordepth(cursor c)	- Tell number of identry levels
 * int cursorat(cursor c)	- Tell index for cursored infonode
 * identry *cursorptr(cursor c)	- Tell cursored identry
 * int ispluralpoint(cursor c)	- Tell if cursored point is plural
 * cursor copycursor(cursor c)	- Make new cursor as copy
 * int movecursor(Cursorstep step, cursor c) - Move cursor forward
 * int applycursor(identry *root, cursor c) - Make target cursor
 * int cursorcmp(cursor c1, cursor c2) - Compare cursors
 * infonode *findinfonode(infonode *root,cursor c) -
 * cursor targetcursor(infonode *inp) - Make cursor for target point
 * cursor templatecursor(cursor c) - Redirect cursor into template
 * int sprintfcursor(char *buf,cursor c,int len) - Make text cursor
 * void free_cursor_list(cursor_list cl) - Reclaim memory.
 * cursor targetcursor(infonode *inp) - Build cursor from target
 *********************************************************************/

/*********************************************************************
 * INCLUDES
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "aimtypes.h"
#include "aimstredit.h"
#include "parser.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_aimcursor.h"

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static cursor createcursorrec P_(( identry *ptr, int pos, cursor next ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/* ^L */
/*********************************************************************
 * Function: static cursor createcursorrec(identry *ptr,int pos,cursor next)
 *
 * Create a cursor record with argument fields.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
static cursor createcursorrec(ptr,pos,next)
     identry *ptr;
     int pos;
     cursor next;
{
  cursor p = (cursor) malloc(sizeof(struct aim_cursor));

  p->next = next;
  p->ptr = ptr;
  p->pos = pos;
  return p;
}

/* ^L */
/*
 * Public cursor handling functions.
 */
/*********************************************************************
 * Function: int cursordepth(cursor c)
 *
 * Determine the number of identries to pass down to the cursored
 * point.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
int cursordepth(c)
     cursor c;
{
  int n = 0;

  while (c != (cursor)NULL) {
    n += 1;
    c = c->next;
  }
  return n;
}

/* ^L */
/*********************************************************************
 * Function: cursor clipcursor(cursor c1,cursor c2)
 *
 * Point at the record in c1 that is the largest prefix to both
 * c1 and c2 (the last counts may differ).
 *
 * Modifications:
 *	<list mod's with name and date>
 */
cursor clipcursor(c1,c2)
     cursor c1,c2;
{
  int i;
  cursor p;

  if (c1 == (cursor)NULL) return (cursor)NULL;
  if (c2 == (cursor)NULL) return (cursor)NULL;

  p = clipcursor(c1->next,c2);

  if (p != c1->next) return p;

  if ((i = (cursordepth(c2) - cursordepth(p))) <= 0) return p;
  while (i-- > 1) c2 = c2->next;

  if (p == (cursor)NULL) {
    if (c2->next != (cursor)NULL) return p;
  } else {
    if (p->pos != c2->next->pos) return p;
  }
  if (c1->ptr != c2->ptr) return p;

  return c1;
}


/* ^L */
/*********************************************************************
 * Function: cursor freecursor(cursor c)
 *
 * Reclaim the records that make up a cursor.
 * Returns (cursor)NULL.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
cursor freecursor(c)
     cursor c;
{
  cursor p;

  while ((p=c) != (cursor)NULL) {
    c = c->next;
    (void)free((FREEPTR *)p);
  }
  return (cursor)NULL;
}



/* ^L */
/*********************************************************************
 * Function: int cursorat(cursor c)
 *
 * Returns the index innermost for the cursored point.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
int cursorat(c)
     cursor c;
{
  return (c->pos + 1);
}


/* ^L */
/*********************************************************************
 * Function: identry *cursorptr(cursor c)
 *
 * Returns ptr to innermost identry of cursored point.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
identry *cursorptr(c)
     cursor c;
{
  return c->ptr;
}


/* ^L */
/*********************************************************************
 * Function: int ispluralpoint(cursor c)
 *
 * Tells if identry access is of plural kind.
 *
 * Modification:
 *	Ralph, 1992-09-18: exported it.
 *	<list mod's with name and date>
 */
int ispluralpoint(c)
     cursor c;
{
  identry *idp = c->ptr;

  if (idp->tmpl != (identry*)NULL) idp = idp->tmpl;

  switch (idp->value->index) {
  case LINKITEM:
    if (idp->value->pos != -1) return 0;
  case ATTRGROUPTAG:
  case ATTRFIELDTAG:
  case LINKGROUPTAG:
  case LINKFIELDTAG:
    return 1;
  case INTERNAL:
  case ATTRVALUE:
  case IMAGE:
  case SAMEASPARENT:
  default:
    return 0;
  }
}


/* ^L */
/*********************************************************************
 * Function: identry *cursorptr(cursor c)
 *
 * Create a cursor as a copy of c.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
cursor copycursor(c)
     cursor c;
{
  if (c != (cursor)NULL) {
    c = createcursorrec(c->ptr,c->pos,copycursor(c->next));
  }
  return c;
}


/* ^L */
/*********************************************************************
 * Function: int movecursor(Cursorstep step, cursor c)
 *
 * Moves the cursor c to new point as specified by step.
 *	GODOWN	  = to first of subnodes of present
 *	GONEXT	  = to sibling of present
 *	GOFORWARD = to subsequent identry at level of present
 *	GOUP	  = to parent of present.
 * Returns 1 if the movement is successful, otherwise 0.
 * NOTE: the cursor is modified also at 0 return.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
int movecursor(step,c)
     Cursorstep step;
     cursor c;
{
  cursor p;

  if (c == (cursor)NULL) return 0;

  switch (step) {
  case GODOWN:
    if ((c->ptr == (identry*)NULL) ||
	(c->ptr->value == (infonode*)NULL) ||
	(c->ptr->value->subnodes == (identry*)NULL))
      return 0;
    c->next = createcursorrec(c->ptr,c->pos,c->next);
    c->ptr = c->ptr->value->subnodes;
    c->pos = 0;
    return 1;
  case GONEXT:
    if (!ispluralpoint(c)) return 0;
    c->pos += 1;
    return 1;
  case GOFORWARD:
    if ((c->ptr == (identry*)NULL) ||
	(c->ptr->next == (identry*)NULL))
      return 0;
    c->ptr = c->ptr->next;
    c->pos = 0;
    return 1;
  case GOUP:
    if (c->next == (cursor)NULL) {
      (void)freecursor(c);
      return 0;
    }
    p = c->next;
    c->next = p->next;
    c->ptr = p->ptr;
    c->pos = p->pos;
    (void)free((FREEPTR *)p);
    return 1;
  default:
    return 0;
    break;
  }
  /*NOTREACHED*/
}
  

/* ^L */
/*********************************************************************
 * Function: int applycursor(identry *root,cursor c)
 *
 * Redirects template cursor c to be cursor for target from root.
 * Returns 1 if the movement is successful, otherwise 0.
 * NOTE: The cursor is modified also at 0 return.
 * NOTE: All intermediate points (i.e. all ancestors of the cursored
 *	 point) must be present in the target for successful return.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
int applycursor(root,c)
     identry *root;
     cursor c;
{
  identry *idp;
  infonode *inp;
  int i;

  if (c == (cursor)NULL) return 0;

  if (c->next == (cursor)NULL) {
    if (root->tmpl != c->ptr) return 0;
    c->ptr = root;
    return 1;
  }

  if (!applycursor(root,c->next)) return 0;
  if ((idp = c->next->ptr) == (identry*)NULL) return 0;
  if ((inp = idp->value) == (infonode*)NULL) return 0;

  for (i = c->next->pos; i > 0; i--) 
    if ((inp = inp->next) == (infonode*)NULL) return 0;

  for (idp = inp->subnodes; idp != (identry*)NULL; idp = idp->next) 
    if (idp->tmpl == c->ptr) {
      c->ptr = idp;
      return 1;
    }

  return 0;
}


/* ^L */
/*********************************************************************
 * Function: int cursorcmp(cursor c1,cursor c2)
 *
 * Compares two cursors and returns
 *	-1 if c1 is a prefix for c2,
 *	 0 if c1 and c2 are equal,
 *	 1 otherwise.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
int cursorcmp(c1,c2)
     cursor c1, c2;
{
  cursor p;

  if ((p = clipcursor(c1,c2)) == (cursor)NULL) return 1;
  if (p != c1) return 1;
  if (cursordepth(c1) == cursordepth(c2)) return 0;
  return -1;

}

/* ^L */
/*********************************************************************
 * Function: infonode *findinfonode(infonode *root,cursor c)
 *
 * Point to the infonode of root that matches the cursor.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
infonode *findinfonode(root,c)
     infonode *root;
     cursor c;
{
  identry *idp;
  int i;

  if (c == (cursor)NULL) return (infonode*)NULL;
  if (c->next == (cursor)NULL) return root;
  if ((root = findinfonode(root,c->next)) == (infonode*)NULL)
    return (infonode*)NULL;

  for (idp = root->subnodes; idp != (identry*)NULL; idp = idp->next)
    if (strcmp(idp->idname,c->ptr->idname) == 0)
      break;

  if (idp == (identry*)NULL) return (infonode*)NULL;
  root = idp->value;

  for (i = c->pos; i > 0; i--)
    if ((root = root->next) == (infonode*)NULL)
      return (infonode*)NULL;

  return root;
}


/* ^L */
/*********************************************************************
 * Function: cursor targetcursor(infonode *inp)
 *
 * Make a cursor for a given infonode.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
cursor targetcursor(inp)
     infonode *inp;
{
  infonode *inx;
  int n = 0;

  if (inp == (infonode*)NULL) return (cursor)NULL;

  for (inx = inp->head->value; inx != inp; inx = inx->next)
    n += 1;

  return createcursorrec(inp->head,n,targetcursor(inp->parent));
}


/* ^L */
/*********************************************************************
 * Function: cursor templatecursor(cursor c)
 *
 * Make a template cursor of a target cursor.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
cursor templatecursor(c)
     cursor c;
{
  cursor p;

  for (p = c; p != (cursor)NULL; p = p->next)
    if ((p->ptr != (identry*)NULL) && (p->ptr->tmpl != (identry*)NULL))
      p->ptr = p->ptr->tmpl;
  
  return c;
}


/* ^L */
/*********************************************************************
 * Function: int sprintfcursor(char *buf,cursor c,int len)
 *
 * Print the cursor in buffer of length len. Returns length of text.
 * Printing is interrupted at "even cursor points" whenever the
 * buffer is filled. Then only the outermost part of the cursor is
 * put in the buffer.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
int sprintfcursor(buf,c,len)
     char *buf;
     cursor c;
     int len;
{
  int x,y,z;

  if (c == (cursor)NULL) return 0;

  x = sprintfcursor(buf,c->next,len);

  y = strlen(c->ptr->idname)+4;
  for (z = c->pos; z > 10; z = z/10)
    y += 1;

  if (len < x+y) return x;

  if (c->next == (cursor)NULL) {
    (void)sprintf(buf,"%s(%d)",c->ptr->idname,c->pos);
    return y-1;
  }

  (void)sprintf(buf+x,".%s(%d)",c->ptr->idname,c->pos);
  return x+y;
}
  
  
/* ^L */
/*********************************************************************
 * Function: void free_cursor_list(cursor_list cl)
 *
 * Reclaim memory.
 *
 * Modifications:
 *	<list mod's with name and date>
 */
void free_cursor_list(cl)
     cursor_list cl;
{
  cursor_list nx;

  if (cl == (cursor_list)NULL) return;

  for (nx = cl; nx != (cursor_list)NULL; nx = cl) {
    cl = nx->next;
    (void)freecursor(nx->c);
    (void)free((FREEPTR *)nx);
  }
}
