/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME:         aimcommand.c
 *
 * SCCSINFO:		@(#)aimcommand.c	1.13 5/20/94
 *
 * ORIGINAL AUTHOR(S):  Ralph R|nnquist, 1990-03-27
 *
 * MODIFICATIONS:
 * 	<list mods with name and date>
 *
 * DESCRIPTION:
 *  This file contains the AIM command menu functions. In the final version,
 *  this file should not be needed.
 *  Note: all support functions that are useful also for the final program
 *  are collected in the file aimsubr.c.
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Paned.h>	/* for quitcmd */
#include <X11/Xaw/Box.h>	/* for quitcmd */
#include <X11/Xaw/Label.h>	/* for quitcmd */
#include <X11/Xaw/Command.h>	/* for quitcmd */

#include "aimtypes.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_aimcommand.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 *********************************************************************/
Widget command_menu;

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern reference_structure *refstructs;	/* List of reference structures.  */
extern infonode *currentinfonode;	/* The currently selected infonode */
extern label userroot;		/* label to access the user root obj */
extern label sysroot;		/* label to access the systerm root obj */
extern int suppress;		/* whether certain output is supressed */
extern Widget gpdmenu;		/* gpdmenu.c */
extern Widget toplevel;         /* xstuff.c */
extern infonode *previousinfonode;      

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimbuildref.h"
#include "f_aimcmdstream.h"
#include "f_aimdbdump.h"
#include "f_aimeditops.h"
#include "f_aimpropagate.h"
#include "f_aimstredit.h"
#include "f_aimstruct.h"
#include "f_aimsubr.h"
#include "f_blddisplay.h"
#include "f_comhist.h"
#include "f_pen_notify.h"
#include "f_xdumpref.h"
#include "f_xstuff.h"
#include "f_xaimexport.h"
#include "f_chng_prop.h"

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
typedef struct menu_struct {
  char *name;
  void (*func) ();
  enum {
    callback, dialog
  } type;
} Menu;

/* Strings for the yes/no menu */
#define QUIT_QUESTION	"You have made changes.\nDo you wish to save them?"
#define SOME_CHANGE	"Window(s) have changes.\nDo you wish to save them?"
#define SAVE_CHOICE	"Save changes"
#define NOSAVE_CHOICE	"Don't save changes"
#define CANCEL_CHOICE	"Cancel"

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void add_to_widl P_(( reference_structure *ref, Widget widget ));
static void closeanyway P_(( Widget w, caddr_t client_data, caddr_t
                              call_data ));
static void closewindow P_(( Widget w, caddr_t client_data, caddr_t
                              call_data ));
static void dbdumpcmd P_(( Widget w, caddr_t client_data, caddr_t
                            call_data ));
static void Dorefetch P_(( char *p, label *lbl ));
static void followcmd P_(( Widget w, caddr_t client_data, caddr_t
                            call_data ));
static void helpwin_cmd P_(( Widget w, caddr_t client_data, caddr_t
                              call_data ));
static void linkcmd P_(( Widget w, caddr_t client_data, caddr_t
                          call_data ));
static void logout P_(( Widget w, caddr_t client_data, caddr_t call_data ));
static int obj_changed P_(( reference_structure *ref ));
static int promotefn P_(( char *p, label *item, int c, int n ));
static int refetchfn P_(( char *p, label *lbl, int c, int n ));
static void userroot_cmd P_(( Widget w, caddr_t client_data, caddr_t
                               call_data ));
static void yesnomenu P_(( String question, caddr_t data, String
                            yesstring, void (*yesfn) (), String nostring,
                            void (*bothfn) (), String cancelstring ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static Menu cmdmenu[] =
{
  { WEXPAND, followcmd, callback},
  { WCUTANDPASTE, streditcmd, callback},
  { WITEMHISTORY, nodehistcmd, callback},
  { WUSERROOT, userroot_cmd, callback},
  { WHELPWINDOW, helpwin_cmd, callback},
  { WADDLINK, linkcmd, callback},
  { WDBDUMP, dbdumpcmd, callback},
  { WSTOREALL, store_all_cmd, callback},
  { WCLOSEALL, closeallcmd, callback},
  { WLOGOUT, quitcmd, callback},
  { WCLOSEAUX, closewidlcmd, callback},
  { WCOMPARE, objdiffcmd, callback}
#ifdef EXPERIMENT
  , { WPROPAGATE, propagatecmd, callback}
  , { WCOMHIST, cmdhistcmd, callback}
#endif	/* EXPERIMENT */
};

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void buildcommandmenu(Widget parent)
 *
 * This function builds the widget structure for the Aim command menu.
 * The command menu is a box widget as a parent widget child.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void buildcommandmenu(parent)
  Widget parent;
{
  Widget button;
  int i;

  command_menu =
    XtVaCreateManagedWidget("commandmenu", boxWidgetClass, parent, NULL);
  XtAddCallback(command_menu, XtNdestroyCallback, (XtCallbackProc)quitcmd,
		(char *)NULL);

  for (i = 0; i < XtNumber(cmdmenu); i++) {
    button = XtVaCreateManagedWidget(cmdmenu[i].name,
				     commandWidgetClass,
				     command_menu,
				     XtNlabel, cmdmenu[i].name,
				     NULL);
    XtAddCallback(button, XtNcallback, cmdmenu[i].func, cmdmenu[i].name);
  }
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void dumpcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * This function calls the function in xdumpref.c that creates a
 * window that presents the reference structure graphically.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void dumpcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *ref = (reference_structure *) client_data;
  Widget widget;

  widget = xdumpref(ref);
  add_to_widl(ref, widget);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void histcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * This function calls the function in xdumpref.c that creates a
 * window that presents the history of the current reference structure.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void histcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  Widget widget;
  reference_structure *ref = (reference_structure *) client_data;
  label lbl;

  (void)copylbl(&lbl,&ref->bt);
  widget = dump_temporal_history(&lbl, ref, 0);
  add_to_widl(ref, widget);
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void Dorefetch(label *lbl)
 *
 * Synchronise the object with database and remove its "penned" status.
 * (Not propagated?)
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void Dorefetch(p,lbl)
  char *p;
  label *lbl;
{
  if (clear_penned(lbl->vs))
    *p = 'Y';
  (void)OS_OBJECTSYNC(lbl);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int refetchfn(char *p,label *lbl,int c,int n)
 *
 * Map function: Call Dorefetch.
 * (Not propagated?)
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static int refetchfn(p,lbl,c,n)
  char *p;
  label *lbl;
  int c;
  int n;
{
  if (c)
    Dorefetch(p,lbl);
  return 0;
}

#ifdef EXPERIMENT
/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void cmdhistcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void cmdhistcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *r;
  label lbl;

  lbl.vs = -1;
  
  if (CH_get_user_process_node(&lbl) || lbl.vs == -1) {
    trace("No session history.\n");
    return;
  }
  lbl.inst = -1;
  if (!(r = buildref(&lbl, "Session History", (reference_structure *)NULL)) &&
      !(r = buildref(&lbl, "node", (reference_structure *)NULL))) {
    (void)printf("In 'cmdhistcmd' -- Shouldn't happen!\n");
    return;
  }
  blddisplay(r);
}
#endif	/* EXPERIMENT */

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void refetchcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * Release all objects concerned by the view and rebuild it using the
 * same root and the same gpd name.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void refetchcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *ref = (reference_structure *) client_data;
  label *bt = (ref) ? &(ref->bt) : NULL;
  label lbl;
  char flag = (char)NULL;

  if (!ref) {
    short_error("Peculiar call to refetchcmd\n");
    return;
  }

  if (GLI_GETLINKITEM(bt, BTBL_ROOT, 1, &lbl)) {
    short_error("No root");
    return;
  }
  (void)Dorefetch(&flag,bt);
  (void)GC_GETCURRENT(bt);
  (void)Dorefetch(&flag,&lbl);
  (void)GLV_GETLINKVAL(bt, BTBL_PARTS, refetchfn, &flag);
  if (flag)
    mark_reference(refstructs, uim_highlight);
  (void)GBTC_GETBTCOMP(bt,&lbl,TRUE,1);
  switch (clean_and_store_bt(buildrefbt(ref->template->idname,bt,FALSE,ref))) {
  case BT_NOT_TRANSIENT:
    break;
  case BT_SOME_TRANSIENT:
    if (!suppress)
      popup_message(POPUP_MESS_INFO, "%s\n%s\n%s",
                    "The latest binding table version for the object",
                    "you just refetched refers to transient component",
                    "version(s) and was not stored automatically.");
    break;
  case FAIL:
    (void)fprintf(stderr, "refetch: clean_and_store_bt failed\n");
    break;
  case SUCCESS:
    (void)fprintf(stderr, "Storing binding table <%d:%d> (0)\n",
                  ref->bt.vs, ref->bt.inst);
    break;
  default:
    break;
  }
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void dupwindowcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * Pick up the binding table and rebuild the view in a new window.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void dupwindowcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *ref = (reference_structure *) client_data;
  reference_structure *newref = NULL;
  label *bt = (ref) ? &(ref->bt) : NULL;
  char *tag = (ref) ? ref->template->idname : NULL;

  if (!ref) {
    short_error("Pecuilar dupwindowcmd call\n");
    return;
  }

  if ((newref = buildrefbt(tag,
                           bt,
                           (int)ref->flags.bound,
                           (reference_structure *)NULL))) {
    newref->pdl = ref->pdl;
    copy_pdls(&(newref->pdl));
  }
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int promotefn(char *p,label *item,int c,int n)
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static int promotefn(p,item,c,n)
  char *p;
  label *item;
  int c;
  int n;
{
  label lbl;

  if (c != 0) {
    (void)GC_GETCURRENT(copylbl(&lbl,item));
    if (lbl.inst != item->inst) {
#ifdef DEBUG
      (void)printf("Promoting component <%d:%d>\n",item->vs,item->inst);
#endif
      newlastcmd(COPYOP,&lbl,0,(char *)NULL,(char *)NULL,0,(attrval *)NULL, item);
    }
  }
  return 0;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void promotecmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * Promotes the present version of a composition to be the latest.
 * The promotion means to map through the given binding table version
 * and copy each component whose version in the binding table is not
 * the latest version.
 * If the binding table is not edited at entry, then the indicated
 * binding table version is also copied (with CPO_COPYOBJ) to be the
 * latest binding table version.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void promotecmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *ref = (reference_structure *) client_data;
  label *bt = (ref) ? &(ref->bt) : NULL;

  if (!bt) {
    short_error("Peculiar Promotecmd call\n");
    return;
  }
    
  if (!EDITED(bt)) {
#ifdef DEBUG
      (void)printf("Promoting binding table <%d:%d>\n",bt->vs,bt->inst);
#endif
    (void)CPO_COPYOBJ(bt,bt);
  }

  (void)GLV_GETLINKVAL(bt, BTBL_ROOT, promotefn, (char *)NULL);
  (void)GLV_GETLINKVAL(bt, BTBL_PARTS, promotefn, (char *)NULL);
  ref->flags.bound_changed = (ref->flags.bound != False);
  ref->flags.bound = False;
  set_global_propagate_called();  /* so changes do propagate */
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void dbdumpcmd(CALLBACK PROTOCOL)
 * Parameters:
 *    Widget w,
 *    caddr_t client_data,
 *    caddr_t call_data
 *
 * Processes a dump file description and generates a dump file
 *
 * Modifications:
 *    <list mods with name and date>
 */
static void dbdumpcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  label lbl;
  reference_structure *ref;

  if (currentinfonode == (infonode *)NULL) {
    (void)printf("Please indicate dump file description first.\n");
    return;
  }

  if ((ref = whichref(currentinfonode, (identry *)NULL, 0))) {
    (void)printf("Dumping takes some time, so please be patient.\n");
    XFlush(XtDisplay(w));
    (void)copylbl(&lbl,&(ref->target->value->obj));
    (void)printf("dumping returned %d.\n", make_t2l_file(&lbl));
  }
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void nodehistcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * Presents the history of the selected node.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void nodehistcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *ref;
  label lbl;
  Widget widget;

  if (currentinfonode == (infonode *)NULL) {
    (void)printf("Please select node first.\n");
    return;
  }
  if (currentinfonode->iflg & NONEXIST_MSK) {
    (void)printf("You have selected a place holder.\n");
    (void)copylbl(&lbl,&(currentinfonode->obj));
  } else {
    if (currentinfonode->index == LINKITEM) {
      (void)printf("You have selected a link item.\n");
      (void)copylbl(&lbl,&(currentinfonode->lbl));
    } else {
      (void)copylbl(&lbl,&(currentinfonode->obj));

    }
  }
      
  ref = whichref(currentinfonode, (identry *)NULL, 0);
  (void)printf("Making history of <%d:%d>\n",lbl.vs,lbl.inst);
  widget = dump_temporal_history(&lbl, ref, 1);
  add_to_widl(ref, widget);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void showgpdcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void showgpdcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  label lbl;
  reference_structure *r;

  /* get the reference structure for this window */
  r = (reference_structure *) client_data;
  (void)UBL_UNBINDLABEL(copylbl(&lbl,&(r->gpd)));

  if (!(r = buildref(&lbl, "gpd", (reference_structure *)NULL)) &&
      !(r = buildref(&lbl, "node", (reference_structure *)NULL))) {
    (void)printf("In 'showgpdcmd' -- Shouldn't happen!\n");
    return;
  }
  blddisplay(r);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void showbtcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void showbtcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  label lbl;
  reference_structure *r;

  /* get the reference structure for this window */
  r = (reference_structure *) client_data;

  (void)UBL_UNBINDLABEL(copylbl(&lbl,&(r->bt)));

  if (!(r = buildref(&lbl, "binding table", (reference_structure *)NULL)) &&
      !(r = buildref(&lbl, "node", (reference_structure *)NULL))) {
    (void)printf("In 'showbtcmd' -- Shouldn't happen!\n");
    return;
  }
  blddisplay(r);

}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void storecmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void storecmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *rsp;

  /* get reference structure and store stuff */
  rsp = (reference_structure *) client_data;
  storesubnodes(rsp->target->value);
  switch (clean_and_store_bt(rsp)) {
    case FAIL:
      trace("Storing bt <%d:%d> (-1)\n", rsp->bt.vs, rsp->bt.inst);
      break;
    case SUCCESS:
      trace("Storing bt <%d:%d> (0)\n", rsp->bt.vs, rsp->bt.inst);
      (void) CH_update("Composite store", 2, (char *)NULL, (char *)NULL, (char *)NULL,
		       (char *)NULL, 0, &(rsp->bt),(label *)NULL,(label *)NULL);
    break;
    case BT_SOME_TRANSIENT:
    case BT_NOT_TRANSIENT:
    default:
      break;
  }
}

/*  */
/**********************************************************************
 * Function: static void yesnomenu(MANY PARAMETERS (7))
 * Parameters:
 *	String question,
 *	caddr_t data,
 *	String yesstring,
 *	void (*yesfn)()		Uses callback protocol,
 *	String nostring,
 *	void (*bothfn)()	Uses callback protocol.
 *	String cancelstring
 *
 * Make a three-choice exclusive popup menu.
 *
 * both the yesfn and the bothfn are added as callbacks to the "yes"
 * button, and the bothfn is added to the "no" button.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void yesnomenu(question, data,
		      yesstring, yesfn, nostring, bothfn, cancelstring)
  String question;
  caddr_t data;
  String yesstring;
  void (*yesfn) ();		/* Uses callback protocol */
  String nostring;
  void (*bothfn) ();		/* Uses callback protocol */
  String cancelstring;
{
  Widget query, pane, menu, yes, no, cancel, lbl;

  /* Setup the popup shell */
  query = XtVaCreatePopupShell("query",
			       transientShellWidgetClass,
			       toplevel,
			       XtNtitle, "Yes/No/Cancel menu",
			       XtNallowShellResize, TRUE,
			       XtNsaveUnder, TRUE,
			       NULL);

  /* Insert a pane list */
  pane = XtVaCreateManagedWidget("pane", panedWidgetClass, query, NULL);

  /* First pane is the question */
  lbl = XtVaCreateManagedWidget("question", labelWidgetClass, pane,
				  XtNlabel, question,
				  NULL);
  /* Add menu */
  menu = XtVaCreateManagedWidget("menu", boxWidgetClass, pane, NULL);

  /* Create "yes" command widget */
  yes = XtVaCreateManagedWidget("yes",
				commandWidgetClass,
				menu,
				XtNlabel, yesstring,
				NULL);

  /* Create "no" command widget */
  no = XtVaCreateManagedWidget("no",
			       commandWidgetClass,
			       menu,
			       XtNlabel, nostring,
			       NULL);

  /* Create "no" command widget */
  cancel = XtVaCreateManagedWidget("cancel",
				   commandWidgetClass,
				   menu,
				   XtNlabel, cancelstring,
				   NULL);

  /* Setup all callbacks */
  XtAddCallback(yes, XtNcallback, yesfn, data);
  XtAddCallback(yes, XtNcallback, bothfn, data);
  XtAddCallback(no, XtNcallback, bothfn, data);
  {				/* Used by XtCallbackPopdown() */
    static XtPopdownIDRec pdrec;/* MUST be static !! */
    pdrec.shell_widget = query;
    pdrec.enable_widget = menu;
    XtAddCallback(yes, XtNcallback, XtCallbackPopdown, (XtPointer)&pdrec);
    XtAddCallback(no, XtNcallback, XtCallbackPopdown, (XtPointer)&pdrec);
    XtAddCallback(cancel, XtNcallback, XtCallbackPopdown, (XtPointer)&pdrec);
  }

  centerPopupOverCursor(query);
  XtPopup(query, XtGrabExclusive);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void closecmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void closecmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *tmp_ptr = (reference_structure *) client_data;
  checksave(tmp_ptr, storecmd, closewindow, client_data);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void exportcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * This function pops off to export a file given the
 * reference structure passed in as client_data.
 * It returns either success (1) or failure (-1)
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void exportcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  Widget widget = NULL;
  reference_structure *ref = (reference_structure *) client_data;

  if(!(widget = build_export(ref))) {
    (void)fprintf(stderr, "export failed.\n");
  } else {
    add_to_widl(ref, widget);
  }
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void quitcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w
 *	caddr_t client_data
 * 	caddr_t calldata
 *
 *   This function implements the Quit button in the Aim command menu.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void quitcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  checksave((reference_structure *)NULL, store_all_cmd, logout, client_data);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void logout(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w
 *	caddr_t client_data
 * 	caddr_t calldata
 *
 * This function actually calls functions to terminate xlincks
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void logout(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  switch (LOGOUT()) {
  case ERR_RCBAD:
    popup_message(POPUP_MESS_INFO,"%s\n%s\n%s\n%s\n",
		  "Cannot log out at this time because the",
		  "database server has not yet processed",
		  "all previous store requests.",
		  "Please try again in a moment.");
    return;
  case SUCCESS:
  default:
    exit(0);
    return;
  }
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void closewindow(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w
 *	caddr_t client_data	- pointer to right ref. structure
 * 	caddr_t calldata
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void closewindow(w, client_data, call_data)
  Widget w;
  caddr_t client_data;	/* the reference structure associated with window */
  caddr_t call_data;
{
  Widget wp;
  reference_structure *tmp_ptr = (reference_structure *) client_data;

  if (tmp_ptr == whichref(currentinfonode, (identry *)NULL, 0))
    currentinfonode = (infonode *)NULL;

  if (tmp_ptr == whichref(previousinfonode, (identry *)NULL, 0))
    previousinfonode = (infonode *)NULL;

  if (tmp_ptr->contw != (Widget) NULL) {
    /* whack the window and free stuff */
    for (wp = tmp_ptr->contw;
	 (wp != (Widget) NULL) && (!XtIsSubclass(wp, shellWidgetClass));
	 wp = XtParent(wp))
      /*empty body*/ ;
    XtDestroyWidget(wp);
    free_ref(tmp_ptr);
  }
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void store_all_cmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w
 *	caddr_t client_data
 * 	caddr_t calldata
 *
 * Go through all the current reference structures and
 * store them if there have been changes.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void store_all_cmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *rsp;
  int err;

  /* go through all the reference structures */
  for (rsp = refstructs; rsp != NULL; rsp = rsp->next) {
    if (isstructurechanged(rsp->target->value) != FALSE) {
      storesubnodes(rsp->target->value);
      trace("Storing bt <%d:%d>", rsp->bt.vs, rsp->bt.inst);
      if (!(err = SO_STOREOBJ(&(rsp->bt))))
	(void) CH_update("Composite store", 2, (char *)NULL, (char *)NULL, (char *)NULL,
			 (char *)NULL, 0, &(rsp->bt),(label *)NULL,(label *)NULL);
      trace(" (%d)\n", err);
    }
  }
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void translinkcmd(TRANSLATION PROTOCOL)
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void translinkcmd(w, event, params, num_params)
  Widget w;
  XEvent *event;
  String *params;
  Cardinal *num_params;
{
  int mode = 1;		/* after */
  int position = 0;

  if (currentinfonode == (infonode *)NULL) {
    (void)printf("Please set the destination.\n");
    return;
  }

  if (previousinfonode == (infonode *)NULL) {
    (void)printf("Please set the source infonode first.\n");
    return;
  }

  if (whichref(currentinfonode, (identry *)NULL, 0)->flags.bound) {
    XBell(XtDisplay(w), 50);
    return;
  }

  /* Determine mode */
  if (num_params && (*num_params) == 1) {
    static char *mode_table[] = {"before" ,"after"};
    int i;

    for (i = 0; i < XtNumber(mode_table); i++)
      if (!strcmp(params[0],mode_table[i])) {
        mode = i;
        break;
      }
  }

  if (mode == 1) {	/* after */
    position = (currentinfonode->iflg & NONEXIST_MSK) 
	         ? 1 
	         : currentinfonode->pos;
  } else {
    position = (currentinfonode->iflg & NONEXIST_MSK) 
	         ? 1 
	         : (currentinfonode->pos - 1);
    if (position < 0)
      position = 0;
  }

  (void)fprintf(stderr, "Linking in object %s destination.\n", 
		(mode == 0) ? "before" : "after");

  if ((linkcopy(previousinfonode, currentinfonode->head, position)) == FAIL)
    short_error("Linking failed.\nCannot link into that position.\n");
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void linkcmd(CALLBACK PROTOCOL)
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void linkcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  translinkcmd(w, (XEvent *)NULL, (String *)NULL, (Cardinal *)NULL);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void followcmd(CALLBACK PROTOCOL)
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void followcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  XtPopup(gpdmenu, XtGrabNone);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void helpwin_cmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w		- unused
 *	caddr_t client_data	- unused
 * 	caddr_t calldata	- unused
 *
 * rebuild the help window under HELP:help window in the system root.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void helpwin_cmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  label lbl;
  label helpwinlbl;
  reference_structure *r;

  if (GLI_GETLINKITEM(&sysroot, "HELP", "help window", 1, &helpwinlbl)) {
    (void)fprintf(stderr, "Cannot build help window.\n");
    return;
  }

  (void)UBL_UNBINDLABEL(copylbl(&lbl,&helpwinlbl));
  if (!(r = buildref(&lbl, "text", (reference_structure *)NULL))) {
    (void)printf("helpwin_cmd: shouldn't happen.\n");
    return;
  }
  blddisplay(r);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void userroot_cmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w		- unused
 *	caddr_t client_data	- unused
 * 	caddr_t calldata	- unused
 *
 * Should the user lose their user root object (login window), this
 * function will build a new one.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void userroot_cmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  label lbl;
  reference_structure *r;
  /*
   * build a reference structure using the "user" gpd of the object
   * accessed by the label userroot.
   */
  (void)UBL_UNBINDLABEL(copylbl(&lbl,&userroot));
  if (!(r = buildref(&lbl, "user", (reference_structure *)NULL))) {
    (void)printf("userroot_cmd: shouldn't happen.\n");
    return;
  }
  blddisplay(r);
}

/*  */
/**********************************************************************
 * Function: void checksave(FOUR PARAMETERS)
 * Parameters:
 *	reference_structure *ref
 *	void (*yesfn)()		called first by the "yes" button
 *	void (*bothfn)()	called second by the "yes" button and
 *				the only call by the "no" button
 *	caddr_t client_data	passed as client_data to both functions
 *	
 * Modifications:
 *      <list mods with name and date>
 */
void checksave(ref, yesfn, bothfn, client_data)
  reference_structure *ref;
  void (*yesfn)();
  void (*bothfn)();
  caddr_t client_data;
{
  int edited = FALSE;

  if (!ref)
    for (ref = refstructs; ref; ref = ref->next)
      edited |= obj_changed(ref);
  else
    edited = obj_changed(ref);

  if (edited != TRUE) {
    (* bothfn)(NULL, client_data, NULL);
    return;
  }
  yesnomenu(QUIT_QUESTION, client_data, SAVE_CHOICE,
	    yesfn, NOSAVE_CHOICE, bothfn, CANCEL_CHOICE);
}
/*  */
/**********************************************************************
 * Function: static int obj_changed(reference_structure *ref)
 *
 * Check to see if some object, as opposed to just a binding table,
 * has changed.  If it's only a binding table, save it automagically.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int obj_changed(ref)
  reference_structure *ref;
{
  int err = 0;

  switch (isstructurechanged(ref->target->value)) {
  case BT_TRANSIENT:
    trace("Storing bt <%d:%d>", ref->bt.vs, ref->bt.inst);
    if (!(err = CH_SO_STOREOBJ(&(ref->bt))))
      (void)CH_update("Composite store", 2,(char *)NULL,(char *)NULL,(char *)NULL,
                      (char *)NULL, 0, &(ref->bt),(label *)NULL,(label *)NULL);
    trace(" (%d)\n", err);
    return FALSE;
  case BT_SOME_TRANSIENT:
    return TRUE;
  case FALSE:
  default:
    return FALSE;
  } 
}

/*  */
/**********************************************************************
 * Function: static void add_to_widl(reference_structure *ref, Widget widget)
 *	
 * This function adds a widget to the list of widgets that need to be 
 * destroyed when a window is closed.  This might be a history graph,
 * the graphical reference structure, etc.  Note that the destroying and
 * free'ing is done in free_ref(), NOT in free_ref_cont().  
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void add_to_widl(ref, widget)
  reference_structure *ref;
  Widget widget;
{
  destroy_list *dl;

  if (!widget || !ref)
    return;

  dl = (destroy_list *)malloc(sizeof(destroy_list));
  dl -> next = ref->widl;
  dl -> widget = widget;
  ref->widl = dl;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void closeanyway(CALLBACK PROTOCOL)
 * Parameters:
 *      Widget w
 *      caddr_t client_data
 *      caddr_t calldata
 *
 * close all the windows, irregardless of whether they have changed.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void closeanyway(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *ref = NULL;

  for (ref = refstructs; ref; ref = ref->next)
    closewindow(w, (caddr_t)ref, (caddr_t)NULL);
} 

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void closeallcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * Close all open windows, checking to see if there are any changes,
 * etc.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void closeallcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *ref = NULL;
  reference_structure *next = NULL;
  int edited = 0;

  for (ref = refstructs; ref; ref = next) {
    next = ref->next;
    if (!isstructurechanged(ref->target->value))
      closewindow(w, (caddr_t)ref, (caddr_t)NULL);
    else
      edited = 1;
  }
  if (edited)
    yesnomenu(SOME_CHANGE, (caddr_t)NULL, SAVE_CHOICE,
	      store_all_cmd, NOSAVE_CHOICE, closeanyway, CANCEL_CHOICE);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: void closewidlcmd(CALLBACK PROTOCOL)
 * Parameters:
 *	Widget w,
 *	caddr_t client_data,
 *	caddr_t call_data
 *
 * Close all auxiliary windows attached to the open windows.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
void closewidlcmd(w, client_data, call_data)
  Widget w;
  caddr_t client_data;
  caddr_t call_data;
{
  reference_structure *ref = NULL;

  for (ref = refstructs; ref; ref = ref->next)
    free_widls(ref);
}

/*  */
/*ARGSUSED*/
/***********************************************************************
 * Function: void popdown_yesnomenu(FOUR PARAMETERS)
 * Parameters:
 *	Widget w
 *	XEvent *ev
 *	String *parms
 *	Cardinal *num_parms
 *
 * Modifications:
 *	<list mods here with name and date>
 */
void popdown_yesnomenu(w, ev, parms, num_parms)
  Widget w;
  XEvent *ev;
  String *parms;
  Cardinal *num_parms;
{
  WidgetList worf;

  while (!XtIsSubclass(w, panedWidgetClass))
    if(!(w = XtParent(w)))
      return;
  XtVaGetValues(w, XtNchildren, &worf, NULL);
  w = worf[1];
  XtVaGetValues(w, XtNchildren, &worf, NULL);

  XtCallActionProc(worf[0], "set", ev, (String *)NULL, (Cardinal)0);
  XSync(XtDisplay(w), False);

#ifdef HAVE_USLEEP 
  usleep(500000);
#else
  {
    struct timeval tval;
    tval.tv_sec  = 0;
    tval.tv_usec =  500000;
    select(0, NULL, NULL, NULL, &tval);
  }
#endif /* n HAVE_USLEEP */
  XtCallActionProc(worf[0], "notify", ev, (String *)NULL, (Cardinal)0);
  XtCallActionProc(worf[0], "unset", ev, (String *)NULL, (Cardinal)0);
}


