/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME:         aimbuildref.c
 *
 * SCCSINFO:            @(#)aimbuildref.c	1.10 5/19/94
 *
 * ORIGINAL AUTHOR(S):  Ralph R|nnquist, 1990-03-14
 *
 * MODIFICATIONS:
 *      1993-11-02 Martin Sjlin - whichref is moved to aimsubr.c and 
 *                 changed external section accordingly.
 *      1994-05-17 Martin Sjlin - added DoParseExecInfoNode
 *                 to be able parse/exec GPD grammar from
 *                 other places .... 
 *	<list mods here with name and date>
 *
 * DESCRIPTION:
 *
 * This file contains procedures to build a reference structure. It includes
 * the ACTIONs for the grammar rules STRUCTURE, ACCESS, and FORMAT.
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "parser.h"
#include "aimtypes.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_aimbuildref.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 *********************************************************************/
int global_unbind = 1;

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimcmdstream.h"
#include "f_aimpropagate.h"
#include "f_aimstruct.h"
#include "f_aimsubr.h"
#include "f_xstuff.h"
#include "f_comhist.h"

/* libshared */
extern char *strdup( /* char *incoming */ );

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* aimstart.c */
extern label userroot, sysroot;
extern reference_structure *refstructs;
extern int suppress;

/* comhist.c */
extern int global_comhistoff;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define ACCBUFSIZE 1000

#define LINK_ERROR \
"LINK returned ERR_PARAMS.\n\
The object was not linked in.\n\
This is most likely a GPD error.\n"

#define Labelcmp(a,b) \
	(((a)->vs == (b)->vs && (a)->inst == (b)->inst))

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* Local status used by grammar rule ACTIONs */
static infonode *curnode;

/*
 * Below is the context used for an "instantiation" of a template infonode
 * towards a given target. The variables listp and lastp are used for
 * constructing the "instance" value list.
 */
static infonode *listp, *lastp;
static identry *head;
static label *obj;
static reference_structure *ref = NULL;

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void addformatspec P_(( infonode *inp, label *spd, char *acc ));
static void appendtolist P_(( void ));
static void doexpandtarget P_(( infonode *np, reference_structure *ref,
                                 char *acc ));
static void expandtemplate P_(( identry *expidp, label *gpd ));
static int findexpandGPD P_(( char *tag, label *gpd ));
static char *makeacc P_(( char *acc, char *tag, int ix ));
static char *makeacchead P_(( infonode *np, char *acc ));
static int mklinknode P_(( void *p, label *item, int c, int n ));
static int mktaginfonode P_(( void *p, char *tag, int c, int n ));
static void set_bt_image P_(( reference_structure *ref ));

static ACTION(attrgrptagaccess);
static ACTION(attrtagaccess);
static ACTION(attrvalaccess);
static ACTION(defineformat);
static ACTION(emptyinitstr);
static ACTION(gpdstrinit);
static ACTION(imageaccess);
static ACTION(linkgrpaccess);
static ACTION(linkitemaccess);
static ACTION(linktagaccess);
static ACTION(linkvalaccess);
static ACTION(makesubitem);
static ACTION(makesubtags);
static ACTION(outervalueaccess);

/*  */
/**********************************************************************
 * The following functions implements the semantics of structure formation
 * statements, i.e. the <STRUCTURE>, <ACCESS>, and <FORMAT> parts of the
 * grammar.
 */

/* STRUCTURE ACTIONs */

static ACTION(makesubitem)
{
  curnode->iflg = (curnode->iflg & (~LEVEL_MSK)) | ITEMLEVEL;
  END_ACTION;
}

static ACTION(makesubtags)
{
  int n;
  identry *p;

  n = popint();
  while (n-- > 0) {
    p = newidentry();
    p->idname = popdata();
    p->tmpl = (identry *) NULL;
    p->owner = curnode;
    p->next = curnode->subnodes;
    curnode->subnodes = p;
  }
  END_ACTION;
}

/*  */
/* ACCESS ACTIONs */

static ACTION(gpdstrinit)
{
  char *string;

  if ((string = popdata()) != (char *)NULL) {
    curnode->val.attvalue = string;
    curnode->val.attsize = strlen(string) + 1;
  }
  END_ACTION;
}

static ACTION(emptyinitstr)
{
  pushdata((char *)NULL);
  END_ACTION;
}

static ACTION(imageaccess)
{
  curnode->index = IMAGE;
  END_ACTION;
}

static ACTION(attrvalaccess)
{
  curnode->index = ATTRVALUE;
  curnode->field = popdata();
  curnode->group = popdata();
  END_ACTION;
}

static ACTION(attrtagaccess)
{
  curnode->index = ATTRFIELDTAG;
  curnode->group = popdata();
  END_ACTION;
}

static ACTION(attrgrptagaccess)
{
  curnode->index = ATTRGROUPTAG;
  END_ACTION;
}

static ACTION(linkitemaccess)
{
  curnode->index = LINKITEM;
  curnode->pos = 1;		/* = Singular LINKITEM */
  curnode->field = popdata();
  curnode->group = popdata();
  END_ACTION;
}

static ACTION(linkvalaccess)
{
  curnode->index = LINKITEM;
  curnode->pos = -1;		/* = ALL */
  curnode->field = popdata();
  curnode->group = popdata();
  END_ACTION;
}

static ACTION(linktagaccess)
{
  curnode->index = LINKFIELDTAG;
  curnode->group = popdata();
  END_ACTION;
}

static ACTION(linkgrpaccess)
{
  curnode->index = LINKGROUPTAG;
  END_ACTION;
}

static ACTION(outervalueaccess)
{
  curnode->index = SAMEASPARENT;
  END_ACTION;
}

/*  */
/* FORMAT ACTIONs */

static ACTION(defineformat)
{
  int n;
  formentry *fp;

  n = popint();
  while (n-- > 0) {
    fp = (formentry *) malloc(sizeof(formentry));
    if (fp == (formentry *) NULL) {
      fatal_error("%s\n", "malloc failed in defineformat");
    }
    fp->formval = popdata();
    fp->formid = popdata();
    fp->next = curnode->formats;
    curnode->formats = fp;
  }
  END_ACTION;
}

/*  */
/*
 * The following table is used at initialisation to define text names
 * for actions.
 */

actionentry EXPANSION_ACTIONS[] =
{				/* !!! Initialisation table !!! */
/* Actions for <STRUCTURE> */
  {"makesubitem", makesubitem},
  {"makesubtags", makesubtags},

/* Actions for <ACCESS> */
  {"gpdstrinit", gpdstrinit},
  {"emptyinitstr", emptyinitstr},
  {"imageaccess", imageaccess},
  {"attrvalaccess", attrvalaccess},
  {"attrtagaccess", attrtagaccess},
  {"attrgrptagaccess", attrgrptagaccess},
  {"linkitemaccess", linkitemaccess},
  {"linkvalaccess", linkvalaccess},
  {"linktagaccess", linktagaccess},
  {"linkgrpaccess", linkgrpaccess},
  {"outervalueaccess", outervalueaccess},

/* Actions for <FORMAT> */
  {"defineformat", defineformat},

  {0, 0}
};

/*  */
/* The following table defines the syntax for GPDs */
char *EXPANSION_SYNTAX[] =
{
  "<id> ::= <identifier> => noop ! <string>  => noop ! <blword> => noop",

  "<STRUCTURE> ::= ITEM => makesubitem ! [ <id> until <eoln> ] => makesubtags",

  "<ACCESS> ::= <access> <initstring> => gpdstrinit",

  "<initstring> ::= = <string> => noop ! <empty> => emptyinitstr",

  "<access> ::= IMAGE => imageaccess !\
		ATTR <id> <id> => attrvalaccess !\
		ATTR <id> => attrtagaccess !\
		ATTR => attrgrptagaccess !\
		LINK <id> <id> 1 => linkitemaccess !\
		LINK <id> <id> => linkvalaccess !\
		LINK <id> => linktagaccess !\
		LINK => linkgrpaccess !\
		$ => outervalueaccess",

  "<FORMAT> ::= [ <id> = <id> while ; ] => defineformat",

  "<EXPAND> ::= <id> => notused",

  (char *)NULL
};

/*  */
/**********************************************************************
 * Function: static void expandtemplate(identry *expidp, label *gpd)
 *
 * Expand the given template tag expidp using the root presentation
 * desctription object gpd.
 *
 * Note: the result of expandtemplate is an expansion of the 'value' field
 * of the given identry expidp.
 *
 * Modifications:
 * 	<list modifications with name and date>
 */
static void expandtemplate(expidp, gpd)
  identry *expidp;
  label *gpd;
{
  label rgpd;		/* Resolved GPD */
  infonode *inp = NULL;
  identry *idp = NULL;
  char *str = NULL;

/* 
 * The first step is to determine whether the given tag (expidp->idname) is
 * recursive or not. A recursive identry gets a value ptr to the
 * previously built value for the same tag.
 */
  (void)copylbl(&rgpd, gpd);
  if ((str = InheritAttribute(&rgpd, "STRUCTURE", expidp->idname)) != NULL)
    free(str);

  for (inp = expidp->owner; inp && inp->parent; inp = inp->parent) {
    if (Labelcmp(&(inp->selobj), &rgpd) &&
	strcmp(inp->head->idname, expidp->idname) == 0) {
      expidp->value = inp;
      expidp->flags.is_recursive = 1;
      return;			/* returns for recursive concepts */
    }
  }

  /* The following is only for non-recursive concepts */

  expidp->value = inp = newinfonode();
  inp->head = expidp;
  inp->parent = expidp->owner;
  (void)copylbl(&(inp->selobj), &rgpd);

  curnode = inp;

  /* Default structure definition is ITEM */
  if (DoAttribute(&rgpd, "STRUCTURE", expidp->idname, "STRUCTURE") != SUCCESS)
    (void)makesubitem();

  /* 
   * Default access definition is "$", except for the top-most
   * identry which is defined as "IMAGE".
   */
  if (DoAttribute(&rgpd, "ACCESS", expidp->idname, "ACCESS") != SUCCESS) {
    if (!(expidp->owner))	/* top most */
      (void)imageaccess();
    else
      (void)outervalueaccess();	/* the rest */
  }

  /* Default format definition is "no format" */
  if (DoAttribute(copylbl(&rgpd, &(inp->selobj)),
		  "FORMAT", expidp->idname, "FORMAT") != SUCCESS) {
    /* do nothing */
  }

  for (idp = inp->subnodes; idp != (identry *)NULL; idp = idp->next)
    expandtemplate(idp, &(inp->selobj));

}

/*  */
/**********************************************************************
 * Function: int DoParseExecInfoNode(infonode *inp, char *str, char *goal)
 *
 * This function parses and executes the string str towards goal,
 * but more important, we have possible data hanging on the infonode
 * inp ....
 *
 * Modifications:
 * 	<list mods with name and date>
 */
int DoParseExecInfoNode(inp, str, goal)
  infonode *inp;
  char *str;		  
  char *goal;
{
  curnode = inp;
  return(DoParseExec(str,goal));
}

/*  */
/**********************************************************************
 * Function: static void addformatspec(infonode *inp, label *spd, char *acc)
 *
 * Add the specific format information to the infonode *inp as
 * defined by the Specific Presentation Description spd using the
 * logical access for acc.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void addformatspec(inp, spd, acc)
  infonode *inp;
  label *spd;
  char *acc;
{
  label lbl;

  if (spd == (label *) NULL || spd->vs == -1)
    return;
  curnode = inp;
  /* should returned be checked here ??? */
  (void)DoAttribute(copylbl(&lbl, spd), "FORMATS", acc, "FORMAT");
}

/*  */
/**********************************************************************
 * Function: static void appendtolist()
 *
 * Create an infonode for present context and append it to the listp
 * list.
 *
 * Modifications:
 * 	<list mods with name and date>
 */
static void appendtolist()
{
  if (lastp == (infonode *) NULL) {
    listp = lastp = createinfonode(head, obj);
  } else {
    lastp->next = createinfonode(head, obj);
    lastp = lastp->next;
  }
}

/*  */
/**********************************************************************
 * The below functions are used as map functions for the Lincks Application
 * Program Interface, to catch tag lists and link item lists.
 */

/*ARGSUSED*/
/**********************************************************************
 * Function: static int mktaginfonode(void *p, char *tag, int c, int n)
 *
 * Modifications:
 *	<list mods with date and name>
 */
static int mktaginfonode(p, tag, c, n)
  void *p;
  char *tag;
  int c, n;
{
  if (c != 0) {
    appendtolist();
    lastp->val.attsize = strlen(tag) + 1;
    lastp->val.attvalue = strdup(tag);
  }
  return GVS_CONTINUE;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int mklinknode(void *p,label *item,int c,int n)
 *
 * Modifications:
 *	<list mods with date and name>
 */
static int mklinknode(p, item, c, n)
  void *p;
  label *item;
  int c, n;
{
  if (c != 0) {
    appendtolist();
    lastp->pos = c;
    (void)copylbl(&(lastp->lbl), item);
    if (!BOUND(item)) {
      (void)GBTC_GETBTCOMP(&(ref->bt), item, !(ref->flags.bound),0);
    }
    (void)GI_GETIMAGE(item, &(lastp->val));
  }
  return GVS_CONTINUE;
}

/*  */
/**********************************************************************
 * Function: infonode *makeinfonodes(identry *hd, label *target)
 *
 * Performs the NODE access according to the specification in the
 * template, and builds the corresponding value list.
 *
 * Modifications:
 *	<list mods with date and name>
 */
infonode *makeinfonodes(hd, target)
  identry *hd;
  label *target;
{
  label tmplbl;

  head = hd;
  obj = target;
  ref = whichref((infonode *)NULL, hd, 1);
  listp = lastp = (infonode *) NULL;

  switch (hd->tmpl->value->index) {
  case INTERNAL:		/* The value is a constant */
    listp = createinfonode(hd, obj);
    listp->val.attsize = hd->tmpl->value->val.attsize;
    listp->val.attvalue = hd->tmpl->value->val.attvalue;
    break;
  case ATTRGROUPTAG:
    (void)GAGN_GETATTRGROUPNAMES(target, mktaginfonode, (void *)NULL);
    break;
  case ATTRFIELDTAG:
    (void)GAN_GETATTRNAMES(target, grouptag(hd), mktaginfonode, (void *)NULL);
    break;
  case ATTRVALUE:
    listp = createinfonode(hd, obj);
    (void)GA_GETATTR(target, grouptag(hd), fieldtag(hd), &(listp->val));
    break;
  case LINKGROUPTAG:
    (void)GLGN_GETLINKGROUPNAMES(target, mktaginfonode, (void *)NULL);
    break;
  case LINKFIELDTAG:
    (void)GLN_GETLINKNAMES(target, grouptag(hd), mktaginfonode, (void *)NULL);
    break;
  case LINKITEM:
    if (hd->tmpl->value->pos == -1) {
      (void)GLV_GETLINKVAL(target,
			   grouptag(hd),
			   fieldtag(hd),
			   mklinknode,
			   (void *)NULL);
    } else {
      /* This branch is for "singular" LINKITEMs, with position 1 given.
         The GPD language allows only position 1 for singularity. */
      switch (GLI_GETLINKITEM(target,
			      grouptag(hd),
			      fieldtag(hd),
			      1, &tmplbl)) {
      case 2:			/* Error code 2 is for "no such group" */
      case 4:			/* Error code 4 is for "no such field" */
      case 5:			/* Error code 5 is for "no such position" */
	/* The requested link does not exist, so it must be created. */
	(void)CO_CREATEOBJ(&tmplbl);
	(void)SO_STOREOBJ(&tmplbl);
	(void)UBL_UNBINDLABEL(&tmplbl);
	if ((LINK(target, grouptag(hd), fieldtag(hd), &tmplbl, 1))==ERR_PARAMS)
	  short_error(LINK_ERROR);
	(void)SO_STOREOBJ(target);
	(void)UBL_UNBINDLABEL(target);
	/* The case continues into SUCCESS case below */
      case SUCCESS:
	listp = createinfonode(hd, obj);
	(void)copylbl(&(listp->lbl), &tmplbl);
	(void)GI_GETIMAGE(&tmplbl, &(listp->val));
	break;
      default:
	/* Funny error codes, return will be (infonode *)NULL */
	break;
      }
    }
    break;
  case IMAGE:
    listp = createinfonode(hd, obj);
    (void)GI_GETIMAGE(target, &(listp->val));
    break;
  case SAMEASPARENT:
    listp = createinfonode(hd, obj);
    listp->val.attsize = hd->owner->val.attsize;
    listp->val.attvalue = hd->owner->val.attvalue;
    break;
  default:
    break;
  }
  if (!listp || listp->val.attsize == 0)
    add_def_values(hd, &(listp->obj), ref);

  return listp;
}

/*  */
/**********************************************************************
 * Function: static char *makeacc(char *acc,char *tag,int ix)
 *
 * Extend the access form acc with tag and index ix. The assumption
 * is that acc points to a large enough buffer. The current
 * implementation is not the fastest possible.
 *
 * Modifications:
 *	<list mods with date and name>
 */
static char *makeacc(acc, tag, ix)
  char *acc, *tag;
  int ix;
{
  static char buffer[100];
  (void)sprintf(buffer, ".%s(%d)", tag, ix);
  if ((int)strlen(acc)+(int)strlen(buffer) >= ACCBUFSIZE - 1) {
    (void)fprintf(stderr,"makeacc: %s\n%s\n",
	    "The access tag is too large.",
	    "(Probably due to a recursion over a cycle)");
    return NULL;
  }
  (void)strcat(acc, buffer);
  return acc;
}

/*  */
/**********************************************************************
 * Function: static void doexpandtarget(infonode *np,reference_structure *ref,char *acc)
 *
 * Expand a reference structure. The infonode np is the infonode to
 * expand, the label spd refers to the Specific Presentation
 * Description, and acc is a large enough character buffer to hold
 * the deepest logical access form.
 *
 * Modifications:
 *	<list mods with date and name>
 */
static void doexpandtarget(np, ref, acc)
  infonode *np;
  reference_structure *ref;
  char *acc;
{
  label target;
  identry *npx, **at;
  char *endp;
  infonode *vp;
  int i;
  int saveflag = global_unbind;

  if (!acc)    
    return;
  if (!np)
    return;
  if (np->subnodes)
    return;
  if (!np->head->tmpl->value)
    return;

  addformatspec(np, &(ref->spd), acc);

  if (((np->iflg = np->head->tmpl->value->iflg) & LEVEL_MSK) == ITEMLEVEL)
    return;

  if (np->index != LINKITEM) {
    (void)copylbl(&target, &(np->obj));
  } else {
    global_unbind = 0;

    if (!BOUND(copylbl(&target, &(np->lbl)))) {
      if (ref->flags.bound) {
	(void)GBTC_GETBTCOMP(&(ref->bt), &target, FALSE, 0);
      } else {
	(void)GBTC_GETBTCOMP(&(ref->bt), &target, TRUE, 0);
	global_unbind = 1;
      }
    }
  }

  endp = acc + strlen(acc);	/* Where the '\0' character is */
  at = &(np->subnodes);
  for (npx = np->head->tmpl->value->subnodes; npx; npx = npx->next) {
    (*at) = newidentry();
    (*at)->idname = npx->idname;
    (*at)->tmpl = npx;
    (*at)->owner = np;
    (*at)->value = makeinfonodes((*at), &target);
    if ((*at)->value == (infonode *)NULL) {
      /* Create a place holder */
      (*at)->value = createinfonode((*at), &target);
      setplaceholder((*at)->value,(int)NONEXIST_MSK);
    } else {
      for (i = 0, vp = (*at)->value; vp != (infonode *)NULL; vp = vp->next) {
	doexpandtarget(vp, ref, makeacc(acc, npx->idname, i++));
	*endp = '\0';
      }
    }
    at = &((*at)->next);
  }
  global_unbind = saveflag;

}

/*  */
/**********************************************************************
 * Function: static char *makeacchead(infonode *np,char *acc)
 *
 * Compute the initial access form for the infonode np.
 *
 * Modifications:
 *	<list mods with date and name>
 */
static char *makeacchead(np, acc)
  infonode *np;
  char *acc;
{
  infonode *xp;
  int i;

  if (np->parent == (infonode *)NULL) {
    (void)strcpy(acc, np->head->idname);
    return (acc + strlen(acc));
  }
  acc = makeacchead(np->parent, acc);
  *(acc++) = '.';
  (void)strcpy(acc, np->head->idname);
  for (xp = np->head->value, i = 0; xp != np; xp = xp->next, i++);
  (void)sprintf(acc, "(%d)", i);
  return (acc + strlen(acc));
}

/*  */
/**********************************************************************
 * Function: void expandtarget(reference_structure *ref,infonode *np)
 *
 * Expand the infonode `np' in teh reference structure `ref'.
 *
 * Modifications:
 *	<list mods with date and name>
 */
void expandtarget(ref, np)
  reference_structure *ref;
  infonode *np;
{
  static char acc[ACCBUFSIZE];

  (void)makeacchead(np, acc);
  doexpandtarget(np, ref, acc);
}

/*  */
/**********************************************************************
 * Function: static int findexpandGPD(char *tag,label *gpd)
 *
 * This function determines a GPD object for a given tag using the
 * present GPDmap chain.
 *
 * Modifications:
 *	<list mods with date and name>
 */
static int findexpandGPD(tag, gpd)
  char *tag;
  label *gpd;
{
  int i;
  label map;
  static char grp[] = "SYSTEM";
  static char fld[] = "GPDmaplist";
  static char mapgrp[] = "GPDmap";

  /* Search in maps from user root */
  for (i = 1; GLI_GETLINKITEM(&userroot, grp, fld, i, &map) == SUCCESS; i++) {
    if (GLI_GETLINKITEM(&map, mapgrp, tag, 1, gpd) == SUCCESS)
      return SUCCESS;
  }

  /* Search in maps from system root */
  for (i = 1; GLI_GETLINKITEM(&sysroot, grp, fld, i, &map) == SUCCESS; i++) {
    if (GLI_GETLINKITEM(&map, mapgrp, tag, 1, gpd) == SUCCESS)
      return SUCCESS;
  }
  return FAIL;
}

/*  */
/**********************************************************************
 * Function: reference_structure *buildref(label *tgobj, char *tag,
 *				           reference_structure *oldref)
 *
 * if oldref is not null, then we're reusing that reference structure
 * in this expansion.
 *
 * Modifications:
 *	<list mods with date and name>
 */
reference_structure *buildref(tgobj, tag, oldref)
  label *tgobj;
  char *tag;
  reference_structure *oldref;
{
  label gpd;

  if (oldref && oldref->template && !strcmp(tag,oldref->template->idname)) {
    (void)copylbl(&gpd,&(oldref->gpd));
  } else {
    if (findexpandGPD(tag, &gpd) != SUCCESS)
      return (reference_structure *) NULL;
  }
  return buildrefgpd(tgobj, &gpd, tag, oldref);
}

/*  */
/**********************************************************************
 * Function: reference_structure *buildrefgpd(FOUR PARAMETERS)
 * Parameters:
 *	label *rootobj
 *	label *gpd
 *	char *tag
 *	reference_structure *oldref
 *
 * Build a currency structure for target object obj using the generic
 * presentation description root tag tag in the generic presentation
 * description object gpd.
 *
 * Modifications:
 *	<list mods with date and name>
 */
reference_structure *buildrefgpd(rootobj, gpd, tag, oldref)
  label *rootobj, *gpd;
  char *tag;
  reference_structure *oldref;
{
  reference_structure *ref;
  label boundroot;
  label bt;
  label boundgpd;
  int bound;
  int edited = 0;
  int created = 0;

  bound = BOUND(rootobj);
  (void)copylbl(&boundroot,rootobj);
  if (!bound && GC_GETCURRENT(&boundroot))
    return (reference_structure *)NULL;

  if (GC_GETCURRENT(copylbl(&boundgpd,gpd)))
    return (reference_structure *)NULL;

  switch (GBT_GETBINDINGTABLE(&boundgpd,&boundroot,&bt,&created)) {
  case ERR_VERSION:
    switch (GBTC_GETBTCOMP(&bt, &boundroot, FALSE, 0)) {
    case ERR_VERSION:
      if (bound)
        return (reference_structure *)NULL;
    case SUCCESS:
      if (bound && (rootobj->inst != boundroot.inst)) {
	short_error("buildrefgpd: no binding table version for <%d:%d>",
		    rootobj->vs, rootobj->inst);
	return (reference_structure *)NULL;
      }
      edited = EDITED(&bt);
      if (GBTC_GETBTCOMP(&bt, &boundroot, TRUE, 1))
        return (reference_structure *)NULL;
      if (GBTC_GETBTCOMP(&bt, &boundgpd, TRUE, 2))
        return (reference_structure *)NULL;
      break;
    default:
      return (reference_structure *)NULL;
    }
    break;
  case SUCCESS:
    edited = EDITED(&bt);
    break;
  default:
    short_error("buildrefgpd: unable to get binding table version for <%d:%d>",
		rootobj->vs, rootobj->inst);
    return (reference_structure *)NULL;
    break;
  }

  if ((ref = buildrefbt(tag,&bt,bound,oldref)) == (reference_structure *)NULL)
    return ref;
  if (created || (!edited && EDITED(&(ref->bt)))) {
    switch (clean_and_store_bt(ref)) {
    case BT_NOT_TRANSIENT:
      (void)fprintf(stderr, "buildrefgpd: how peculiar!\n");
      break;
    case BT_SOME_TRANSIENT:
#ifdef DEBUG /* these messages are annoying :-) */
      if (!suppress)
        popup_message(POPUP_MESS_INFO, "%s\n%s\n%s",
		    "The latest binding table version for the object",
		    "you just expanded refers to transient component",
		    "version(s) and was not stored automatically.");
#endif /* DEBUG */
      break;
    case FAIL:
      (void)fprintf(stderr, "buildrefgpd: clean_and_store_bt failed\n");
      break;
    case SUCCESS:
      (void)fprintf(stderr, "Storing expanded binding table <%d:%d> (0)\n",
    		  ref->bt.vs, ref->bt.inst);
      break;
    default:
      break;
    }
  }
  if (created)
    set_bt_image(ref);
  return ref;
}

/*  */
/**********************************************************************
 * Function: reference_structure *buildrefbt(FOUR PARAMETERS)
 * Parameters:
 * 	char *tag
 * 	label *bt
 * 	int bound
 * 	reference_structure *oldref
 *
 * Build a reference structure for target object obj using the generic
 * presentation description root tag "tag" in the generic presentation
 * description object gpd and through the binding table bt.
 *
 * if (bound), the reference structure is created as a
 * bound structure, which in particular means that the exact component
 * versions indicated by the given `bt' version are used without
 * considering component development. For an unbound reference structure,
 * component development is taken into account, and made to update the
 * binding table to a version holding the most recent versions of all
 * components.
 *
 * If ref is null, a new reference structure is built, otherwise the old
 * one is reused.  
 *
 * Modifications:
 *	<list mods with date and name>
 */
reference_structure *buildrefbt(tag, bt, bound, oldref)
  char *tag;
  label *bt;
  int bound;
  reference_structure *oldref;
{
  reference_structure *rsp;
  infonode *p;
  int comhistoff;

  rsp = (reference_structure *) malloc(sizeof(reference_structure));
  if (rsp == (reference_structure *) NULL) {
    fatal_error("fatal error: %s\n", "malloc failed in buildrefbt.");
  }
  
  /* must put this in because subsequent functions depend on having
   * the reference structure available.  it will be removed at the
   * end of this function
   */
  rsp->next = refstructs;
  refstructs = rsp;

  rsp->contw = NULL;
  rsp->widl = NULL;
  rsp->pdl = NULL;
  rsp->pqueue.head = rsp->pqueue.tail = (propagation_q_element *) NULL;

  /*
   * Setup the bt field; the binding table.  if the binding table we
   * have has been edited, but it isn't the current transient version,
   * force it to be bound.
   */
  if (!bound && EDITED(bt) && !ISTRANSIENT(bt)) {
    bound = TRUE;
  }
  (void)copylbl(&(rsp->bt), bt);
  rsp->flags.bound_changed = (bound == TRUE);
  rsp->flags.bound = bound;
  (void)copylbl(&(rsp->bt), bt);
  /*
   * Setup the root and gpd fields using the binding table.
   */
  if (GLI_GETLINKITEM(bt, "COMPOSITION", "Root", 1, &(rsp->root)) ||
      GLI_GETLINKITEM(bt, "COMPOSITION", "Gpd", 1, &(rsp->gpd))) {
    trace("Incomplete binding table version <%d:%d>\n", bt->vs, bt->inst);
    refstructs = rsp->next;
    free((FREEPTR *)rsp);
    return NULL;
  }
  /*
   * Block command history temporarily.
   */
  comhistoff = global_comhistoff;
  global_comhistoff = 1;
  /*
   * Setup the spd field; the specific presentation decription.
   * (spd.vs == -1) marks non-existent spd.
   */
  if (GLI_GETLINKITEM(&(rsp->root), "FORMAT", tag, 1, &(rsp->spd))) {
    rsp->spd.vs = -1;
    rsp->spd.inst = -1;
  }
  /*
   * Build the template structure.
   * The root is forced to be of IMAGE kind.
   */
  rsp->template = newidentry();
  rsp->template->idname = strdup(tag);
  expandtemplate(rsp->template, &(rsp->gpd));
  p = rsp->template->value;
  p->iflg = (p->iflg & (!LEVEL_MSK)) | SUBLEVEL;
  p->index = IMAGE;
  /*
   * Build the target structure.
   */
  rsp->target = newidentry();
  rsp->target->idname = rsp->template->idname;
  rsp->target->tmpl = rsp->template;
  global_unbind = !bound;
  rsp->target->value = makeinfonodes(rsp->target, &(rsp->root));
  expandtarget(rsp, rsp->target->value);

  set_global_propagate_called();	/* so changes do propagate */

  /*
   * Re-enable command history. Note: the operation type 2 marks that it
   * is unspecified whether it involves database communication or not.
   */
  global_comhistoff = comhistoff;
  (void) CH_update("Composite expand", 2, (char *)NULL, (char *)NULL, (char *)NULL,
		   (char *)NULL, 0, &(rsp->bt),(label *)NULL,(label *)NULL);
  if (oldref == NULL) {
    return (rsp);
  }

  /* from here on is for when we're reusing a reference structure */
  /* Free the old reference structure contents; may build pqueue */
  free_ref_cont(oldref);

  oldref->template = rsp->template;
  oldref->target = rsp->target;
  (void)copylbl(&(oldref->root),&(rsp->root));
  (void)copylbl(&(oldref->gpd),&(rsp->gpd));
  (void)copylbl(&(oldref->spd),&(rsp->spd));
  (void)copylbl(&(oldref->bt),&(rsp->bt));
  oldref->flags.bound_changed = (oldref->flags.bound != bound);
  oldref->flags.bound = bound;

  refstructs = rsp->next;
  free((FREEPTR *)rsp);
  return (oldref);
}

/*  */
/**********************************************************************
 * Function: void add_def_values(THREE PARAMETERS)
 * Parameters:
 *	identry *hd
 *	label *target
 *	reference_structure *ref
 *
 * If the identry's value in the template has a default value for a
 * group, field, or attrval, add a command to carry this out into the
 * command queue.  The command queue will be flushed at the main loop.
 *
 * Modifications:
 *	<list mods with date and name>
 */
void add_def_values(hd, target, ref)
  identry *hd;
  label *target;
  reference_structure *ref;
{
  if ((hd->tmpl->value->val.attsize == 0) || 
      (ref->flags.bound) || 
      (BOUND(target)))
    return;

  switch (hd->tmpl->value->index) {
  case ATTRGROUPTAG:
    newlastcmd(CREATEOP, target, ATTRGROUPTAG, (char *)NULL, (char *)NULL,
	       0, &(hd->tmpl->value->val), (label *) NULL);
    break;
  case ATTRFIELDTAG:
    newlastcmd(CREATEOP, target, ATTRFIELDTAG, grouptag(hd), (char *)NULL,
	       0, &(hd->tmpl->value->val), (label *) NULL);
    break;
  case ATTRVALUE:
    newlastcmd(CHANGEOP, target, ATTRVALUE, grouptag(hd), fieldtag(hd),
	       0, &(hd->tmpl->value->val), (label *) NULL);
    break;
  case LINKGROUPTAG:
    newlastcmd(CREATEOP, target, LINKGROUPTAG, (char *)NULL, (char *)NULL,
	       0, &(hd->tmpl->value->val), (label *) NULL);
    break;
  case LINKFIELDTAG:
    newlastcmd(CREATEOP, target, LINKFIELDTAG, grouptag(hd), (char *)NULL,
	       0, &(hd->tmpl->value->val), (label *) NULL);
    break;
  case SAMEASPARENT:
  case INTERNAL:
  case IMAGE:
  case LINKITEM:
  default:
    break;
  }
}

/*  */
/**********************************************************************
 * Function: static void set_bt_image(reference_structure *ref)
 *
 * Modifications:
 *	<list mods with date and name>
 */
static void set_bt_image(ref)
  reference_structure *ref;
{
  attrval v;
  char *title = makeshelltitle(ref->target);

  v.attsize = strlen("Binding table for ") + strlen(title) + 
	      strlen(" in view ") + strlen(ref->target->idname) + 1;
  v.attvalue = (char *)malloc((ALLOC_T)(v.attsize));
  (void)sprintf(v.attvalue, "%s%s%s%s", "Binding table for ",
	  title, " in view ", ref->target->idname);
  (void)SI_SETIMAGE(&(ref->bt), &v);
  free(v.attvalue);
}
