/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME:		parser.h
 *
 * SCCSINFO:            @(#)parser.h	1.6 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Ralph Ronnquist
 *
 * MODIFICATIONS:
 *	< add modifications with name and date >
 *
 * DESCRIPTION:
 *	This file is included by the parser module user.
 */

#ifndef PARSER_H
#define PARSER_H

#ifndef FILE
#include <stdio.h>
#endif

/* useful #defines */
#ifdef __STDC__
#define ACTION(fn)	int fn P_(( void ))
#else	/* __STDC__ */
#define ACTION(fn)	int fn()
#endif	/* __STDC__ */
#define END_ACTION	return 0

char *popdata P_(( void ));
/*
 * PARSERTYPES
 *  This part contains internal type declarations for the parser module,
 *  and forward declarations for structure operating functions.
 */

typedef struct x_bnftoken {	/* rule element record */
  struct x_bnftoken  *nextp;
  struct x_bnftoken  *extra;
  int		     termtoken;
  struct x_head      *token;
} bnftoken;

typedef struct x_bnfrule {	/* alternative rule record */
  struct x_bnfrule   *nextp;
  struct x_head      *semantic;
  bnftoken 	     *elements;
} bnfrule;

typedef struct x_head {		/* vocabulary entry */
  struct x_head      *nextp;
  struct x_head      *primary;	
  int		     (*actionfn)();
  char		     *pname;
  bnfrule	     *syntax;
} word;

/*  */
/* Definitions for function expressions */

typedef struct {
  word *fn;
  struct x_argcell *args;
} funcell;

typedef struct x_argcell {
  struct x_argcell *nextp;
  funcell *fnexpr;
  char *data;
} argcell;

typedef struct {		/* For defining action tables */
  char *fnname;
  int (*fn)();
} actionentry;


/*  */
/* Function prototypes */

word *makesymbol P_(( char *pn ));
/*
 * Installs a new word record with pname pn at the top of the
 * current vocabulary.
 */

word *findsymbol P_(( char *s ));
/*
 * Returns ptr to first word record in vocabulary with pname s, or a
 * zero ptr if no such record is found.
 */

word *symbol P_(( char *s, int flg ));
/*
 * Locates first word record in vocabulary with pname s, or creates
 * such. Returns pointer to it. The argument flg marks (if non-zero)
 * that the argument string should be free-ed before return.
 */

word *setvocabulary P_(( word *w ));
/*
 * Reassigns the vocabulary to w (if not zero). Returns previous
 * vocabulary. Note: setvocabulary((word *) NULL) retrieves current
 * vocabulary.
 */

word *hidevocabulary P_(( void ));
/* Reassigns the vocabulary to the predfinedwords vocabulary. */

word *setbnf P_(( char *s ));
/* Sets grammar target for bnfdef parsing. */

void freeargcells P_(( argcell *ap ));
/* Free all memory for givel argcell list, including sub-lists */

void freefuncell P_(( funcell *fp ));
/* Free all memory for given funcell, including its argcell list */

argcell *appendargs P_(( argcell *a1, argcell *a2 ));
/*
 * Destructively append the argcell list a2 to the end of argcell
 * list a1.
 */

void printfunexpr P_(( funcell *fexpr, int n ));
/* Print function expression in LISP format */

funcell *funexpr P_(( void ));
/* Returns the function expression after a successful parsing */

/*  */
/* PARSER ACTIONS
 * This part contains useful definitions for semantic action functions.
 */

/*
 * Pops and returns the top element from the stack. Returns zero ptr
 * if the stack is empty.
 */

int popint P_(( void ));
/*
 * Pops the top of stack element and interprets it as a number,
 * which is returned after that the stacked element is free-ed.
 * Returns zero if the stack is empty.
 */

void pushdata P_(( char *s ));
/* Pushes string s (without copying) onto the data stack. */

void cleardata P_(( int i, int flg ));
/*
 * Clears the data stack down to the i:th level. If flg is non-zero,
 * all stack elements are free'ed.
 */

int datadepth P_(( void ));
/* Returns number of used entries in the data stack. */

void abortexec P_(( void ));
/* Aborts current (on-going) execution. */

void traceinput P_(( int flg ));
/* Enables/disables tracing of input */

FILE *usefile P_(( FILE *f ));
/* Redirects input to file f. Previous input file is returned. */

FILE *promptfile P_(( FILE *f ));
/* Redirects prompt output */

void droplastline P_(( void ));
/* Drops the last input line from the buffer. */

word *setmeta P_(( char *s ));
/* Sets the secondary parsing goal. Returns previous meta goal. */

word *setgoal P_(( char *s ));
/* Sets the main parsing goal. Returns the new goal. */

void interact P_(( char *g ));
/*
 * Repeatedly calls readline() and parses towards goal g. The
 * function is exited by a semantic action that calls
 * quitinteract()
 */

int quitinteract P_(( void ));
/*
 * Forces an exit from interact() after completed treatment of
 * current input.
 */

void takefile P_(( char *fn, char *g ));
/* Calls interact() with goal g, taking input from file fn. */

/*  */
/* PARSERPRIMARIES 
 * This part contains some macros that clean up the code of primary
 * non-terminal functions. 
 *
 * PRIMARY(myfunction)
 *   The macro PRIMARY contains the initial '{' for the function
 *   body and that should therefore be ommitted. The function
 *   'myfunction' is defined as 'static int', taking as argument a
 *   pointer to a string ptr. Further, the local variable
 *   'char *p' is defined in the function body.
 *
 * Ensuretoken;
 *   This macro moves the input line ptr to point to the first
 *   up-coming non-space character. If this exhausts the input,
 *   the primary function will return signalling EXHAUST.
 *
 * Consume(mybuffer,test);
 *    This macro consumes input characters into the buffer while
 *    the test condition is satisfied. A null character is appended
 *    after the last consumed character.
 *
 *  Acceptif(buffer,test)
 *    This macro REJECTs unless the test is satisfied, in which
 *    case the buffer is given as data from the primary function
 *    and the function ACCEPTs
 *
 *  Pick
 *    Removes and returns next input character from the input line.
 *
 *  Peep
 *    Returns (without removing) the next input character from
 *    the input line.
 *
 *  Failif(test)
 *    Calls 'return(REJECT)' if the test is satisfied.
 *
 *  Exhaustif(test)
 *    Calls 'return(EXHAUST)' if the test is satisfied.
 *
 */

#define EXHAUST	(-1)
#define ACCEPT	(1)
#define REJECT	(0)
#define ACCEPTX (2)

#define PRIMARY(name)	int name(str) char **str; { char *p;

#define Pick	(*((*str)++))
#define Peep	(**str)

#define Consume(buffer,test)\
p = buffer; while (Peep!='\0' && (test)) *(p++) = Pick; *p = '\0'

#define Acceptif(buffer,test)\
if (!(test)) return(REJECT); delaypushdata(buffer); return(ACCEPT)
#define Failif(test)	if (test) return(REJECT)
#define Exhaustif(test) if (test) return(EXHAUST)
#define Ensuretoken	Exhaustif(!skipbl(str))

int skipbl P_(( char **str ));
/*
 * Moves the string ptr *str over "space" characters (as defined by
 * the isspace function
 */

void delaypushdata P_(( char *s ));
/*
 * Pushes an action, which when executed pushes a copy of the string
 * s onto the data stack, onto the execution stack.
 */

/*  */
/* PARSER SYNTAX
 * This part defines forwards for the syntax definition
 */

void bindaction P_(( char *s, int (*fn) () ));
/*
 * Define an action fn to an action name s, and install in the
 * current vocabulary. Note: every bindaction() makes a new
 * vocabulary entry when necessary (i.e. existent actions are not
 * redefined)
 */

void defineactions P_(( actionentry *tbl ));
/*
 * The actionentry array is a list of action bindings, terminated by a
 * {0,0} slot. This function installs such a table into the current
 * vocabulary.
 */

void bindprimary P_(( char *s, int (*fn) () ));
/*
 * Defines a primary function fn for the name s and installs in the
 * current vocabulary. The primary field of word s is set to word s,
 * and the action of word s is set to fn. That is, there is a double
 * in-direction.
 */

void defineprimaries P_(( actionentry *tbl ));
/* Like defineactions() but for primary functions. */

void bnfdef P_(( char *s ));
/*
 * Defines a grammar rule in extended bnf format to be included in
 * the current vocabulary. Note: actions and primaries should be
 * bound prior to usage in grammar rules.
 */

void definegrammar P_(( char **syntax ));
/*
 * Processes the array of strings by calling bnfdef() for each string
 * in order. The array should be terminated by a 0 string pointer.
 */

/*  */
/* PARSER CONTROL
 * This part defines forwards for the main control of the parser.
 */

void execute P_(( funcell *fexpr ));
/*
 * Executes actions from the execution stack.  This function takes 1
 * argument: `funcell *fexpr', where fexpr is the function expression
 * constructed by a previous call to `parse(..)' and picked up by the
 * application through the function `funexpr()'.
 */

int parse P_(( word *goal, char **str ));
/*
 * Parses the string *str towards the given goal. If successful,
 * return value is 1, str is moved to after last consumed character,
 * and the semantics execution environment is setup accordingly (see
 * execute). If parsing is unsuccessful, return value is 0, str is
 * unchanged, and the semantics execution environment is empty.
 *
 * Return value from the parse function is one of the following:
 *
 * ACCEPT  meaning that the parsing of the string str towards the
 * goal is accepted, and there is no potential longer acceptable
 * parse. When parsing is accepted (ACCEPT or ACCEPTX), the internal
 * execution stack has been set up with action calls in accordance
 * to the accepted parse (see the execute function). Further, the
 * argument str is moved to the character immediately succeding the
 * portion of the string that was "consumed" by the parse.
 *
 * ACCEPTX meaning that the parsing is accepted, but there is a
 * potential longer acceptable parse.
 *
 * EXHAUST meaning that the string was exhausted before a complete
 * acceptable parse was obtained. This is a reject condition.
 *
 * REJECT meaning that the parsing failed.
 */

char *readline P_(( char *first, char *prompt, int *count ));
/*
 * readline() "appends" next line from infile to current input. You
 * give as argument a string pointer to the first character of the
 * input buffer that should be kept. A NULL pointer indicates that
 * the whole buffer should be thrown, so that the new input line
 * will be the first text fragment. Otherwise, the portion from the
 * given pointer to the end of the current buffer is kept (but moved
 * to the buffer beginning), and the new input line is appended
 * after it.  Note: the newline characters, '\n', appear in the
 * text. A NULL character is added after the buffer and overwritten
 * at next readline().
 *
 * The second argument is a prompt string that is display on stdout
 * in the format "prompt>" when the first textline for the buffer is
 * asked for. Successive lines are prompted with "...>". The
 * prompting occurs when infile is stdin or when the input tracing
 * is enabled. You enable tracing by a call to traceinput() with a
 * non-zero argument, and disable it with a zero argument to
 * traceinput().
 *
 * When tracing is enabled and infile is not stdin, then the prompts
 * occur as above, and in addition, the input line read is also
 * displayed on stdout.
 *
 * Input is taken from an infile, which defaults to stdin. You
 * redirect input by a call to usefile(), which takes an opened
 * input file ptr as argument and returns the previous file pointer
 * (without closing the file). usefile() sets tracing mode so that
 * tracing is disabled for stdin and enabled otherwise.
 */

void init_parser P_(( void ));
/*
 * Initialises the parser module. This function must be called once
 * from the main, prior to any other parser module calls.
 */

void metagrammar P_(( void ));
void printall P_(( word *rh ));

#endif /* PARSER_H */
