/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	rpccalls.c
 *
 * SCCSINFO:		@(#)rpccalls.c	1.11 5/30/94
 *
 * ORIGINAL AUTHOR(S):  ???, 1987-01-18
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *	This file contains the definition of the RPC command functions.
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include <sys/wait.h>

#include "lincks.h"
#include "monitor.h"
#include "libshared.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_rpccalls.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_dbsqueue.h"
#include "f_flags.h"
#include "f_getentry.h"
#include "f_locks.h"
#include "f_labelinfo.h"
#include "f_monerrors.h"
#include "f_putentry.h"
#include "f_misc.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern UID uidcom;			/* monitormain.c */
extern pid_t childpid;			/* monitormain.c */
extern int indexfd;			/* monitormain.c */
extern char *molnames[];		/* monitormain.c */
extern INDEX indextable[];		/* monitormain.c */
extern LabelInfo *editinfop;		/* monitormain.c */
extern DBSQueue *penqueuep;		/* monitormain.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define MAXPENBUFSIZE 512

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: A_ENTRY *Access(L_ENTRY *args)
 * 
 * The RPC function that finds the requested molecule
 * reference data in the index file. Returns a structure
 * of type A_ENTRY.
 *
 * Modifications:
 *      <list mods with name and date>
 */
A_ENTRY *Access(args)
  L_ENTRY *args;
{
    static A_ENTRY info;
    static char fname[MAXPATHLEN];
    INDEX *entry;
    int pos;

    /* Set up default values */
    info.a_filename = "";
    info.a_filepos = (off_t)0;
    info.a_checkno = 0;

    /* Check for lock timeouts */
    (void)TimeLocks(indextable);

    /* Check args */
    if (!args) {
	Error(ER_CALL, "Access: no arguments");
	info.a_status = NOENTRY;
	return(&info);
    }
	
    /* Set communications uid */
    uidcom = args->l_uid;

    /* Get index entry from either file or table */
    if ((pos = GetEntry(args->l_label, indextable, indexfd)) < 0) {
	info.a_status = NOENTRY;
	return(&info);
    }
    entry = &indextable[pos];

    /* Check that the user has read permission on this molecule */
    if (CheckPerm(args->l_uid, entry, READ) < 0) {
	info.a_status = NOREADPERM;
	return(&info);
    }

    /* Check that any locks, if set, belong to the user */
    if (CheckLock(args->l_uid, entry, MAY) < 0) {
	info.a_status = LOCKED;
	return(&info);
    }

    /* Check that the molecule hasn't been deleted */
    if (CheckDel(entry) < 0) {
	info.a_status = DELETED;
	return(&info);
    }
    
    /* Check info */
    if (entry->filenum >= MAXNOOFFILES || molnames[entry->filenum] == NULL) {
	info.a_status = NOENTRY;
	return(&info);
    }
	
    /* Copy information to the info return structure */
    info.a_status = SUCCESS;
    (void)strcpy(fname, molnames[entry->filenum]);
    info.a_filename = fname;
    info.a_filepos = entry->filepos;
    info.a_checkno = entry->checkno;

    /* Return info */
    return(&info);
}

/*  */
/**********************************************************************
 * Function: A_ENTRY *Lock(L_ENTRY *args)
 * 
 * Functions in the same way as access but assumes write access
 * and sets a lock on the molecule
 *
 * Modifications:
 *      <list mods with name and date>
 */
A_ENTRY *Lock(args)
  L_ENTRY *args;
{
    static A_ENTRY info;
    static char fname[MAXPATHLEN];
    INDEX *entry;
    int pos;

    /* Set up default values */
    info.a_filename = "";
    info.a_filepos = (off_t)0;
    info.a_checkno = 0;

    /* Check for lock timeouts */
    (void)TimeLocks(indextable);

    /* Check args */
    if (!args) {
	Error(ER_CALL, "Lock: no arguments");
	info.a_status = NOENTRY;
	return(&info);
    }
	
    /* Set communications uid */
    uidcom = args->l_uid;

    /* Get index entry from either file or table */
    if ((pos = GetEntry(args->l_label, indextable, indexfd)) < 0) {
	info.a_status = NOENTRY;
	return(&info);
    }
    entry = &indextable[pos];

    /* Check that the user has write permission on this molecule */
    if (CheckPerm(args->l_uid, entry, WRITE) < 0) {
	info.a_status = NOWRITEPERM;
	return(&info);
    }

    /* Check that any locks, if set, belong to the user */
    if (CheckLock(args->l_uid, entry, MAY) < 0) {
	info.a_status = LOCKED;
	return(&info);
    }

    /* Sets a lock if necessary */
    if (SetLock(args->l_uid, entry) < 0) {
	info.a_status = LOCKFAIL;
	return(&info);
    }

    /* Check info */
    if (entry->filenum >= MAXNOOFFILES ||
	!molnames[entry->filenum]) {
	info.a_status = NOENTRY;
	return(&info);
    }
	
    /* Copy information to the info return structure */
    info.a_status = SUCCESS;
    (void)strcpy(fname, molnames[entry->filenum]);
    info.a_filename = fname;
    info.a_filepos = entry->filepos;
    info.a_checkno = entry->checkno;

    /* Return info */
    return(&info);
}

/*  */
/**********************************************************************
 * Function: N_ENTRY *Create(L_ENTRY *args)
 * 
 * The RPC function that creates and initializes a new
 * molecule entry - both index and molecule files are
 * updated
 *
 * Modifications:
 *      <list mods with name and date>
 */
N_ENTRY *Create(args)
  L_ENTRY *args;
{
    static N_ENTRY info;
    static char fname[MAXPATHLEN];
    INDEX *entry;
    int pos;

    /* Set default values */
    info.n_label = 0;
    info.n_filename = "";
    info.n_filepos = (off_t)0;
    info.n_checkno = 0;

    /* Check args */
    if (!args) {
	Error(ER_CALL, "Create: no arguments");
	info.n_status = NOENTRY;
	return(&info);
    }
	
    /* Set communications uid */
    uidcom = args->l_uid;

    /* Call NewEntry to create new index and molecule entry */
    if ((pos = NewEntry(indextable, args->l_uid, indexfd)) < 0) {
	info.n_status = CREATEFAIL;
	return(&info);
    }
    entry = &indextable[pos];

    /* Copy to return info structure */
    info.n_status = SUCCESS;
    info.n_label = entry->label;
    (void)strcpy(fname, molnames[entry->filenum]);
    info.n_filename = fname;
    info.n_filepos = entry->filepos;
    info.n_checkno = entry->checkno;

    /* Return pointer to info */
    return(&info);
}

/*  */
/**********************************************************************
 * Function: int *Unlock(L_ENTRY *args)
 * 
 * Removes a lock on the molecule referenced. Checks that
 * user is owner of lock
 *
 * Modifications:
 *      <list mods with name and date>
 */
int *Unlock(args)
  L_ENTRY *args;
{
    static int status;
    INDEX *entry;
    int pos;

    /* Set default value */
    status = SUCCESS;

    /* Check lock timeouts */
    (void)TimeLocks(indextable);

    /* Check args */
    if (!args) {
	Error(ER_CALL, "Unlock: no arguments");
	status = NOENTRY;
	return(&status);
    }
	
    /* Get entry from index table or file */
    if ((pos = GetEntry(args->l_label, indextable, indexfd)) < 0) {
	status = NOENTRY;
	return(&status);
    }
    /* Set pointer to indicate table position */
    entry = &indextable[pos];

    /* Check that any lock on molecule belongs to uid */
    if (CheckLock(args->l_uid, entry, MAY) < 0) {
	status = LOCKED;
	return(&status);
    }

    /* Remove lock */
    (void)ClearLock(entry);

    /* Return pointer to status */
    return(&status);
}

/*  */
/**********************************************************************
 * Function: int *SetProt(S_ENTRY *args)
 * 
 * Changes the read and write protection modes for the
 * molecule to those specified. It must be locked by uid
 *
 * Modifications:
 *      <list mods with name and date>
 */
int *SetProt(args)
  S_ENTRY *args;
{
    static int status;
    INDEX *entry;
    int pos;

    /* Set default value */
    status = SUCCESS;

    /* Check args */
    if (!args) {
	Error(ER_CALL, "SetProt: no arguments");
	status = NOENTRY;
	return(&status);
    }
	
    /* Get entry from index table or file */
    if ((pos = GetEntry(args->s_label, indextable, indexfd)) < 0) {
	status = NOENTRY;
	return(&status);
    }
    /* Set pointer to indicate table position */
    entry = &indextable[pos];

    /* Check that lock on molecule belongs to uid */
    if (CheckLock(args->s_uid, entry, MUST) < 0) {
	status = NOTLOCKED;
	return(&status);
    }

    /* Check that molecule has not been deleted */
    if (CheckDel(entry) < 0) {
	status = DELETED;
	return(&status);
    }

    /* Set new protection modes and update index file */
    if (ChangeProt(args->s_wrperm, args->s_rdperm, entry, indexfd) < 0) {
	status = SYSERR;
	return(&status);
    }

    /* Return pointer to status */
    return(&status);
}

/*  */
/**********************************************************************
 * Function: P_ENTRY *GetProt(L_ENTRY *args)
 * 
 * Returns the read and write protections for the molecule
 *
 * Modifications:
 *      <list mods with name and date>
 */
P_ENTRY *GetProt(args)
  L_ENTRY *args;
{
    static P_ENTRY info;
    INDEX *entry;
    int pos;

    /* Set default values */
    info.p_status = SUCCESS;
    info.p_wrperm = 0;
    info.p_rdperm = 0;

    /* Check lock timeouts */
    (void)TimeLocks(indextable);

    /* Check args */
    if (!args) {
	Error(ER_CALL, "GetProt: no arguments");
	info.p_status = NOENTRY;
	return(&info);
    }
	
    /* Get entry from index table or file */
    if ((pos = GetEntry(args->l_label, indextable, indexfd)) < 0) {
	info.p_status = NOENTRY;
	return(&info);
    }
    /* Set pointer to indicate table position */
    entry = &indextable[pos];

    /* Check that user has read permission on molecule */
    if (CheckPerm(args->l_uid, entry, READ) < 0) {
	info.p_status = NOREADPERM;
	return(&info);
    }

    /* Check that any lock on molecule belongs to uid */
    if (CheckLock(args->l_uid, entry, MAY) < 0) {
	info.p_status = LOCKED;
	return(&info);
    }

    /* Copy information */
    info.p_wrperm = entry->wrperm;
    info.p_rdperm = entry->rdperm;

    /* Return pointer to structure */
    return(&info);
}

/*  */
/**********************************************************************
 * Function: int *Delete(L_ENTRY *args)
 * 
 * Marks the molecule as deleted. It is not actually removed
 * The molecule must be locked by uid
 *
 * Modifications:
 *      <list mods with name and date>
 */
int *Delete(args)
  L_ENTRY *args;
{
    static int status;
    INDEX *entry;
    int pos;

    /* Set default value */
    status = 0;

    /* Check args */
    if (!args) {
	Error(ER_CALL, "Delete: no arguments");
	status = NOENTRY;
	return(&status);
    }
	
    /* Get entry from index table or file */
    if ((pos = GetEntry(args->l_label, indextable, indexfd)) < 0) {
	status = NOENTRY;
	return(&status);
    }
    /* Set pointer to indicate table position */
    entry = &indextable[pos];

    /* Check that lock on molecule belongs to uid */
    if (CheckLock(args->l_uid, entry, MUST) < 0) {
	status = NOTLOCKED;
	return(&status);
    }

    /* Set DELETED marker on molecule and update index file */
    if (DelMark(entry, SET, indexfd) < 0) {
	status = SYSERR;
	return(&status);
    }

    /* Return pointer to status */
    return(&status);
}

/*  */
/**********************************************************************
 * Function: int *Undelete(L_ENTRY *args)
 * 
 * Removes the delete mark from the molecule. The molecule
 * must be locked
 *
 * Modifications:
 *      <list mods with name and date>
 */
int *Undelete(args)
  L_ENTRY *args;
{
    static int status;
    INDEX *entry;
    int pos;

    /* Set default value */
    status = 0;

    /* Check args */
    if (!args) {
	Error(ER_CALL, "Undelete: no arguments");
	status = NOENTRY;
	return(&status);
    }
	
    /* Get entry from index table or file */
    if ((pos = GetEntry(args->l_label, indextable, indexfd)) < 0) {
	status = NOENTRY;
	return(&status);
    }
    /* Set pointer to indicate table position */
    entry = &indextable[pos];

    /* Check that any lock on molecule belongs to uid */
    if (CheckLock(args->l_uid, entry, MUST) < 0) {
	status = NOTLOCKED;
	return(&status);
    }

    /* Clear DELETED mark on molecule and update index file */
    if (DelMark(entry, CLEAR, indexfd) < 0) {
	status = SYSERR;
	return(&status);
    }

    /* Return pointer to status */
    return(&status);
}

/*  */
/**********************************************************************
 * Function: C_ENTRY *GetCheckNo(L_ENTRY *args)
 * 
 * Returns the check number of the molecule
 *
 * Modifications:
 *      <list mods with name and date>
 */
C_ENTRY *GetCheckNo(args)
  L_ENTRY *args;
{
    static C_ENTRY info;
    INDEX *entry;
    int pos;

    /* Set default values */
    info.c_status = 0;
    info.c_checkno = 0;

    /* Check lock timeouts */
    (void)TimeLocks(indextable);

    /* Check args */
    if (!args) {
	Error(ER_CALL, "GetCheckNo: no arguments");
	info.c_status = NOENTRY;
	return(&info);
    }
	
    /* Get entry from index table or file */
    if ((pos = GetEntry(args->l_label, indextable, indexfd)) < 0) {
	info.c_status = NOENTRY;
	return(&info);
    }
    /* Set pointer to indicate table position */
    entry = &indextable[pos];

    /* Check that the uid has read permission on molecule */
    if (CheckPerm(args->l_uid, entry, READ) < 0) {
	info.c_status = NOREADPERM;
	return(&info);
    }

    /* Check that any lock on molecule belongs to uid */
    if (CheckLock(args->l_uid, entry, MAY) < 0) {
	info.c_status = LOCKED;
	return(&info);
    }

    /* Copy information */
    info.c_checkno = entry->checkno;

    /* Return pointer to structure */
    return(&info);
}

/*  */
/**********************************************************************
 * Function: C_ENTRY *UpdateCheckNo(L_ENTRY *args)
 *
 * Increments the molecule check number. Molecule must
 * be locked by uid
 * 
 * Modifications:
 *      <list mods with name and date>
 */
C_ENTRY *UpdateCheckNo(args)
  L_ENTRY *args;
{
    static C_ENTRY info;
    INDEX *entry;
    int pos;

    /* Set default values */
    info.c_status = 0;
    info.c_checkno = 0;

    /* Check args */
    if (!args) {
	Error(ER_CALL, "UpdateCheckNo: no arguments");
	info.c_status = NOENTRY;
	return(&info);
    }
	
    /* Get entry from index table or file */
    if ((pos = GetEntry(args->l_label, indextable, indexfd)) < 0) {
	info.c_status = NOENTRY;
	return(&info);
    }
    /* Set pointer to indicate table position */
    entry = &indextable[pos];

    /* Check that lock on molecule belongs to uid */
    if (CheckLock(args->l_uid, entry, MUST) < 0) {
	info.c_status = NOTLOCKED;
	return(&info);
    }

    /* Increment check number and update index file */
    if (IncrChkNo(entry, indexfd) < 0) {
	info.c_status = SYSERR;
	return(&info);
    }

    /* Copy information */
    info.c_checkno = entry->checkno;

    /* Return pointer to structure */
    return(&info);
}

/*  */
/**********************************************************************
 * Function: int *UpdateEntry(U_ENTRY *args)
 * 
 * Changes the file number and file position of the
 * molecule implying that the actual molecule has
 * been moved. Use with caution. Molecule must be locked
 *
 * Modifications:
 *      <list mods with name and date>
 */
int *UpdateEntry(args)
  U_ENTRY *args;
{
    static int status;
    INDEX *entry;
    int pos, index, filenum;

    /* Set default value */
    status = 0;

    /* Check args */
    if (!args || !args->u_filename) {
	Error(ER_CALL, "UpdateEntry: no arguments");
	status = NOENTRY;
	return(&status);
    }
	
    /* Get entry from index table or file */
    if ((pos = GetEntry(args->u_label, indextable, indexfd)) < 0) {
	status = NOENTRY;
	return(&status);
    }
    /* Set pointer to indicate table position */
    entry = &indextable[pos];

    /* Check that lock on molecule belongs to uid */
    if (CheckLock(args->u_uid, entry, MUST) < 0) {
	status = NOTLOCKED;
	return(&status);
    }

    /* Find file number corresponding to file name given */
    /* It is an error if it does not exist - assume it doesn't */
    filenum = -1;
    /* Search through molecule file names vector */
    index = 0;
    while(*molnames[index]) {
	if (strcmp(molnames[index], args->u_filename))
	  ++index;
	else {
	    filenum = index;
	    break;
	}
    }
    /* If no match was found, return an error */
    if (filenum < 0) {
	status = NOENTRY;
	return(&status);
    }

    /* Otherwise copy information into entry */
    entry->filenum = filenum;
    entry->filepos = args->u_filepos;

    /* Update index file too */
    if (WriteEntry(entry, indexfd) < 0) {
	status = SYSERR;
	return(&status);
    }

    /* Return pointer to status */
    return(&status);
}

/*  */
/**********************************************************************
 * Function: int *Kill(L_ENTRY *args)
 * 
 * Terminates execution of the net server and monitor
 *
 * Modifications:
 *      <list mods with name and date>
 */
int *Kill(args)
  L_ENTRY *args;
{
    static int dummy;

    /* Set common uid */
    if (args)
      uidcom = args->l_uid;

    /* Check user is SUPERUSER */
    if (uidcom != SUPERUSER) {
	Error(ER_SUPER, "Kill: must be super user (%d)", (int) uidcom);
	return(&dummy);
    }

    /* Ignore child's death */
    (void)Signal(SIGCHLD, SIG_DFL);

    /* Send kill signal to net server */
    (void)kill(childpid, SIGHUP);

    /* Wait for it to terminate */
    (void)wait((WAIT_T *)NULL);

    /* Set suicide alarm */
    (void)alarm(2);
  
    /* Parent return */
    return(&dummy);
}

/*  */
/**********************************************************************
 * Function: int *Restart(L_ENTRY *args)
 * 
 * Starts a new net server process - uid must be superuser
 *
 * Modifications:
 *      <list mods with name and date>
 */
int *Restart(args)
  L_ENTRY *args;
{
    int dummy;

    /* Check args */
    if (!args) {
      Error(ER_CALL, "Restart: no arguments");
      return(&dummy);
    }

    /* Check uid is super user */
    if (args->l_uid != SUPERUSER) {
	Error(ER_SUPER, "Restart: must be super user (%d)", 
	      (int) args->l_uid);
	return(&dummy);
    }

    /* Send server restart signal */
    (void)kill(getpid(), SIGUSR1);

    return(&dummy);
}

/*  */
/**********************************************************************
 * Function: int *Remove(R_ENTRY *args)
 * 
 * Removes information about a label for a process
 * identity.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int *Remove(args)
    R_ENTRY *args;
{
    LabelInfo *labelinfop;
    PENInfo *peninfop, *oldp;
    int dummy;

    if (args->r_flag) {
        /* remove all information about a process identity */
        labelinfop = editinfop->nextp; /* first label */
        while (labelinfop != NULL) {
            peninfop = (PENInfo *) labelinfop->infop; /* first info */

            if (peninfop->pid == args->r_pid) {
                oldp = peninfop;
                SetInfo(editinfop, labelinfop->label, (void *) peninfop->nextp);
                free((FREEPTR *)oldp);
            }
            else
                while (peninfop->nextp != NULL) {
                    if (peninfop->nextp->pid == args->r_pid) {
                        /* remove and free info */
                        oldp = peninfop->nextp;
                        peninfop->nextp = peninfop->nextp->nextp;
                        free(oldp->username);
                        free((FREEPTR *)oldp);
                        break;
                    };
                    peninfop = peninfop->nextp;
                };

            labelinfop = labelinfop->nextp; /* next label */
        };

        /* Remove all messages from DBS queue */
        MakeEmpty(penqueuep, args->r_pid);
    }
    else {
        /* remove information about a process identity for a label */
        if ((peninfop = (PENInfo *) RetrieveInfo(editinfop, args->r_label)) == NULL) {
            return(&dummy);
        }

        if (peninfop->pid == args->r_pid) {
            oldp = peninfop;
            SetInfo(editinfop, args->r_label, (void *) peninfop->nextp);
	    free((FREEPTR *)oldp->username);
            free((FREEPTR *)oldp);
        }
        else
            while (peninfop->nextp != NULL) {
                if (peninfop->nextp->pid == args->r_pid) {
                    /* remove and free info */
                    oldp = peninfop->nextp;
                    peninfop->nextp = peninfop->nextp->nextp;
                    free(oldp->username);
                    free((FREEPTR *)oldp);
                    return(&dummy); /* Only one info needs to be removed */
                };
                peninfop = peninfop->nextp;
            }
    }
    return(&dummy);
}

/*  */
/**********************************************************************
 * Function: E_ENTRY *Check(R_ENTRY *args)
 * 
 * Checks and inserts edit information in order to detect
 * parallel editing
 *
 * Modifications:
 *      <list mods with name and date>
 */
E_ENTRY *Check(args)
    R_ENTRY *args;
{
    static E_ENTRY info = { 0, NULL }; 
    static E_ENTRY *infop = NULL;
    static char namebuf[MAXPENBUFSIZE];
    PENInfo *peninfop, *peninfolistp, *newp;
    int len, updateflg;
    char *p;

    /* Free memory used last time */
    if (info.e_usernames != NULL) {
      if ( info.e_usernames != namebuf )
	free(info.e_usernames);
      info.e_usernames = NULL;
    }

    /* Update flag */
    updateflg = args->r_flag;

    /* Default values */
    info.e_label = args->r_label;

    /* Get pointer to edit info list */
    peninfolistp = (PENInfo *) RetrieveInfo(editinfop, args->r_label);

    /* Collect usernames associated with label */
    /* Send signals if parallel editing occurs */
    
    /* Check required space for usernames */
    len = 1;
    for( peninfop = peninfolistp; peninfop != NULL; peninfop=peninfop->nextp) 
      if (peninfop->pid != args->r_pid)
	len += strlen(peninfop->username) + 1; 

    /* Allocate memory if more than our local buffer */
    if ( len < sizeof(namebuf) )
      info.e_usernames = namebuf;
    else if ((info.e_usernames = (char *) malloc((ALLOC_T)len)) == NULL) {
      Error(ER_MALLOC, "Check:");
      return(&info);
    }

    /* Make string empty */
    info.e_usernames[0] = '\0';
    p = &info.e_usernames[0];

    for( peninfop = peninfolistp; peninfop != NULL;) {

        /* Skip user's own name in message */
        if (peninfop->pid == args->r_pid)
	  /* don't create new info, user already has one */
	  updateflg = 0;     
        else {
  	  char *q;

	  /* Build a string of user names and insert line break to seperate names */
	  for( q = peninfop->username; *q != 0; *p++ = *q++) ;

	  *p++ = '\n'; *p = '\0';

	  /* Send notification to other users */
	  if (args->r_flag) {

	    /* Create and build E_ENTRY */
	    if ((infop = (E_ENTRY *) malloc(sizeof(E_ENTRY))) == NULL) {
	      Error(ER_MALLOC, "Check:");
	      return(&info);
	    }
	    infop->e_label = args->r_label;
	    if ((infop->e_usernames = strdup(args->r_username)) == NULL) {
	      Error(ER_MALLOC, "Check:");
	      return(&info);
	    }

	    /* Put it in DBS queue */
	    Enqueue(penqueuep, peninfop->pid, (void *)infop);
#ifdef PENDEBUG
	    Error(0, "Sending pen signal to dbs %d", (int) peninfop->pid);
#endif
	    /* Send signal to DBS */
	    (void)kill(peninfop->pid, SIGUSR2);
	  }
        }
        peninfop = peninfop->nextp;
    }

    /* Insert new edit information */

    if (updateflg) {
        /* insert new edit info */

        if ((newp = (PENInfo *) malloc(sizeof(PENInfo))) == NULL) {
            Error(0, "Check: could not allocate memory");
            return(&info);
        }
        newp->pid = args->r_pid;
        if ((newp->username = strdup(args->r_username)) == NULL) {
            Error(0, "Check: could not allocate memory");
            return(&info);
        }
        newp->nextp = peninfolistp;

        SetInfo(editinfop, args->r_label, (void *) newp);
    };

    /* Complete info */
    info.e_label = args->r_label;

    /* Remove line break in last position */
    if ( p > &info.e_usernames[0] )
      p[-1] = '\0';

    /* Return pointer to info */
    return(&info);
}

/*  */
/**********************************************************************
 * Function: E_ENTRY *Poll(pid_t *pidp)
 * 
 * Dequeues and returns the first message in the DBS queue
 * Returns an empty string if nothing was found
 *
 * Modifications:
 *      <list mods with name and date>
 */
E_ENTRY *Poll(pidp)
    pid_t *pidp;
{
    static E_ENTRY info = { 0, "" }; /* Error result */
    static E_ENTRY *infop = NULL; 

    /* Free memory used last time */
    if (infop != NULL) {
	free(infop->e_usernames);
        free((FREEPTR *)infop);
    }

    /* Get E_ENTRY from queue associated with pid */
    if ((infop = (E_ENTRY *) Dequeue(penqueuep, *pidp)) == NULL) {
        Error(0, "Poll: Didn't find anything for pid=%d", (int) *pidp);
	infop = NULL; /* Didn't use any memory this time */
        return(&info);
    }

    /* Return pointer to E_ENTRY */
    return(infop);
}
