/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	putentry.c
 *
 * SCCSINFO:		@(#)putentry.c	1.7 5/3/94
 *
 * ORIGINAL AUTHOR(S):  ???, 6 Jan, 1987
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *	This file contains NewEntry, MakeEntry and WriteEntry
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "lincks.h"
#include "monitor.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_putentry.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_monerrors.h"
#include "f_newfiles.h"
#include "f_getentry.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int currentmol;		/* monitormain.c */
extern int currentmolfd;	/* monitormain.c */
extern int currentdatafd;	/* monitormain.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: int NewEntry(INDEX table[], UID uid, int fd)
 * 
 * Creates and initializes a new index entry. Also creates
 * dummy molecule file entry.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int NewEntry(table, uid, fd)
  INDEX table[];
  UID uid;
  int fd;
{
    char *MakeName(), *fnamep;
    LABEL newlabel;
    LABEL MakeEntry();
    off_t moladr, datadr;
    INDEX fileentry;
    int newfd, pos;

    /* Check size of current molecule file */
    if ((moladr = lseek(currentmolfd, (off_t)0, L_XTND)) < 0) {
	Error(ER_LSEEK,
	      "NewEntry: Cannot seek to end of molecule file");
	return(FAIL);
    }
    /* If it is too large, build a new one. Also start a new data file */
    if (moladr >= MAXMOLFILESIZE) {
	fnamep = MakeName();
	if ((newfd = NewMolFile(fnamep)) < 0)
	  return(FAIL);
	currentmolfd = newfd;
	if ((newfd = NewDataFile(fnamep)) < 0)
	  return(FAIL);
	currentdatafd = newfd;
	datadr = 0;
	moladr = 0;
    }

    /* Check size of current data file */
    if ((datadr = lseek(currentdatafd, (off_t)0, L_XTND)) < 0) {
	Error(ER_LSEEK, "NewEntry: Cannot seek to end of data file");
	return(FAIL);
    }
    /* If it is too large, start a new one. Also start a new molfile */
    if (datadr > MAXDATAFILESIZE) {
	fnamep = MakeName();
	if ((newfd = NewDataFile(fnamep)) < 0)
	  return(FAIL);
	currentdatafd = newfd;
	if ((newfd = NewMolFile(fnamep)) < 0)
	  return(FAIL);
	currentmolfd = newfd;
	moladr = 0;
	datadr = 0;
    }

    /* Initialize table data */
    fileentry.label = 0;
    fileentry.filenum = currentmol;
    fileentry.filepos = UNDEFINED_ENTRY;
    fileentry.wrperm = uid;
    fileentry.rdperm = uid;
    fileentry.checkno = 0;
    fileentry.flags = 0;

    /* Create index file entry. Copy to index table also */
    if ((newlabel = MakeEntry(&fileentry, fd)) < 0)
      return(FAIL);
    if ((pos = GetEntry(newlabel, table, fd)) < 0)
      return(FAIL);

    /* Save label on file too */
    table[pos].label = newlabel;
    if (WriteEntry(&table[pos], fd) < 0)
      return(FAIL);

    /* Return table pos */
    return(pos);
}

/*  */
/**********************************************************************
 * Function: LABEL MakeEntry(INDEX *entry, int fd)
 * 
 * Creates a new index entry in the file fd
 * Returns the new label for this position
 *
 * Modifications:
 *      <list mods with name and date>
 */
LABEL MakeEntry(entry, fd)
  INDEX *entry;
  int fd;
{
    off_t adr;

    /* Seeks to end of file */
    if ((adr = lseek(fd, (off_t)0, L_XTND)) < 0) {
	Error(ER_LSEEK, "MakeEntry: Cannot seek to end of file");
	return(FAIL);
    }

    /* Write buffer to file */
    if (write(fd, (char *)entry, (IOSIZE_T)INDEXSIZE) != INDEXSIZE) {
	Error(ER_WRITE, "MakeEntry: File write error");
	return(FAIL);
    }

    /* Return label */
    return(ConvToLabel(adr));
}

/*  */
/**********************************************************************
 * Function: int WriteEntry(INDEX *entry, int fd)
 * 
 * Writes an index entry to the file fd. Uses index
 * entry label to seek to correct position in file
 *
 * Modifications:
 *      <list mods with name and date>
 */
int WriteEntry(entry, fd)
  INDEX *entry;
  int fd;
{
    off_t seekadr, adr;
    FLAGS saveflags;

    /* Convert label to seek address */
    if ((adr = ConvToSeek(entry->label)) < 0) {
	Error(ER_SEEKADR, "WriteEntry: Illegal seek address");
	return(FAIL);
    }

    /* Seek to position in file */
    seekadr = lseek(fd, adr, L_SET);
    if (seekadr != adr) {
	Error(ER_LSEEK, "WriteEntry: Unsuccessful seek attempt");
	return(FAIL);
    }

    /* Save flags then mask away all but DELETED */
    saveflags = entry->flags;
    entry->flags &= DELETED;
	
    /* Write buffer to file */
    if (write(fd, (char *)entry, (IOSIZE_T)INDEXSIZE) != INDEXSIZE) {
	Error(ER_WRITE, "WriteEntry: File write error");
	return(FAIL);
    }

    /* Restore flags */
    entry->flags = saveflags;

    /* Return success status */
    return(SUCCESS);
}
