/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	monitormain.c
 *
 * SCCSINFO:		@(#)monitormain.c	1.15 6/6/94
 *
 * ORIGINAL AUTHOR(S):  Peter {\AA}berg, 1987-01-20
 *
 * MODIFICATIONS:
 *       1994-05-11 Martin Sjlin. Added an extra protection file,
 *              if present contains write protected group.
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *
 *	This file contains the definition of main and the functions
 *	MakeName, Error, NewMolFile and NewDataFile
 *
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <sys/wait.h>
#include <netdb.h>			/* for MAXHOSTNAMELEN */
#ifndef MAXHOSTNAMELEN			/* seems like SCO have here */
#include <sys/socket.h>
#endif /* n MAXHOSTNAMELEN */
#include <rpc/types.h>			/* for broken rpc header files */
#include <rpc/xdr.h>			/* for xdr types */

#include "lincks.h"
#include "monitor.h"
#include "xconfig.h"
#include "libshared.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* Global variables */
UID uidcom = SUPERUSER;		/* Currently processed user id */
int debug = 0;			/* Debugging flag */
pid_t childpid;			/* PID of child i.e. net server */
char host[MAXHOSTNAMELEN];		/* Host name */
char lockfile[MAXPATHLEN];      /* lockfile name and path */

/* File descriptors and pointers */
int currentmolfd;		/* Currently open molecule file descriptor */
int currentdatafd;		/* Currently open data file descriptor */
int indexfd;			/* Index file descriptor */

/* File designators */
int currentdata;		/* Current data file number in table */
int currentmol;			/* Current molecule file number in table */

/* Data structures */
char *molnames[MAXNOOFFILES];	   /* Table of existing molecule file names */
char *datanames[MAXNOOFFILES];	   /* Table of existing data file names */
PROT *rdprotlists[PROTLISTSIZE];   /* Table of protection lists */
PROT *wrprotlists[PROTLISTSIZE];   /* Table of protection lists (write) */
LOCK *locklist = NULL;		   /* Lock list tag */
INDEX indextable[INDEXTABLESIZE];  /* Table of index entries */
DBSQueue *penqueuep = NULL;        /* DBS queue */
LabelInfo *editinfop = NULL;       /* Edit information list */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_labelinfo.h"
#include "f_monerrors.h"
#include "f_rpccalls.h"
#include "f_monhandler.h"
#include "f_rpc.h"
#include "f_tcp.h"
#include "f_flags.h"
#include "f_misc.h"
#include "f_xdr.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif /* MAXHOSTNAMELEN */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int InitSignals P_(( void ));

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
/* Database port number */
static u_long lincksport;		/* Database identity number */
static char *monnam;
static FILE *molnamesfp;	/* Molecule names file pointer */
static FILE *datanamesfp;	/* Data names file pointer */

/*  */
/**********************************************************************
 * Function: main(int argc, char **argv)
 *
 * Handles initialization of data structures and opening of
 * all system files. It also sets up the RPC communications
 * Arguments: [-d] debug mode
 *	dbid  database identity number
 * 
 * Modifications:
 *      <list mods with name and date>
 */
int main(argc, argv)
    int argc;
    char **argv;
{
    int c, errflag;
    int lockfd;
    int index;
    char fname[MAXPATHLEN];
    char filename[MAXPATHLEN];
    char netservfn[MAXPATHLEN]; /* Net server name */
    FILE *fopen();
    extern int optind;
    extern DBSQueue *CreateDBSQueue();
    extern LabelInfo *CreateLabelInfo();
          
    /* RPC function definitions */
    A_ENTRY *Access(), *Lock();
    N_ENTRY *Create();
    P_ENTRY *GetProt();
    C_ENTRY *GetCheckNo(), *UpdateCheckNo();
    int *Unlock(), *SetProt(), *Delete(), *Undelete(), *UpdateEntry();
    int *Kill(), *Restart();
    E_ENTRY *Check(), *Poll();
    int *Remove();

    /* Get arguments from argv using getopt function */
    monnam = argv[0];
    errflag = 0;
    while ((c = getopt(argc, argv, "di:")) != EOF)
      switch(c) {
	case 'd':
	  debug = !debug;
	  break;
        default:
	  ++errflag;
	  break;
      }

    if (argc - optind == 1) {
      if (!configuration(argv[optind])) {
        /*
         * we have nowhere to "write" an error,
         * so this will have to do
         */
        exit(-1);
      }
      lincksport = TCPIPNO;
    }
    else 
      ++errflag;

    /* If an illegal argument was given, signal error and exit */
    if (errflag) {
        (void)fprintf(stderr, "usage: monitor [-d] dbid\n");
	exit(1);
    }

    if (setup_daemon() != SUCCESS) {
      Error(ER_INIT,"Failed to change process group, errno %d", errno);
      exit(1);
    }

    /* Register program start */
    (void)gethostname(host, MAXHOSTNAMELEN);
    errno = 0;
    Error(0, "monitor %ld started at %s", lincksport, host ? host : "");

    (void)sprintf(filename, "%s/%s", DBDIR, INDEXFILE);

    /* Open index file for read/write */
    if ((indexfd = open(filename, O_RDWR)) < 0) {
	Error(ER_INIT, 
	      "%s: could not open %s", monnam, filename );
	exit(1);
    }

    (void) sprintf(lockfile, "%s/%s", DBDIR, INDEXLOCK);
 
    /* this can fail under nfs with a loaded server, but ... */
    if ((lockfd = open(lockfile, O_WRONLY|O_CREAT|O_EXCL, 0644)) < 0) {
      Error(ER_INIT, "%s: could not create lockfile %s\n\
 Check if monitor or cutoff is already running.\n", monnam, lockfile);
      exit(1);
    }

    (void) sprintf( filename, "Monitor@%s:%d\n", host, (int) getpid());
    if ( write( lockfd, filename, (IOSIZE_T)strlen(filename)) <= 0) {
      Error(ER_INIT, "%: failed to write to lockfile %s", monnam, lockfile);
      exit(1);
    }

    if ( close(lockfd) < 0 ) {
      Error(ER_INIT, "%s: failed to close lockfile %s", monnam, lockfile);
      exit(1);
    }

    /* Set up signals */
    if (InitSignals() < 0)
      goto quit;

    /* Create path to Net server */
    (void)sprintf(netservfn, "%s/%s", BINARIES, NETSERVER);

    /* Start Net Server process */
    errno = 0;
    Error(0, "netserv starting...");
    if ((childpid = fork()) == 0) {
        (void)Signal(SIGHUP, SIG_DFL);
	if (debug) {
	    if((execl(netservfn, NETSERVER, "-d", 
		      "-h", host, DBDIR, (char *)NULL)) == -1) {
	      Error(ER_INIT, "%s: execl failed to start %s",
		    monnam, netservfn);
	    }
	} else {
	    if((execl(netservfn, NETSERVER, "-h", host, 
		      DBDIR, (char *)NULL)) == -1) {
	      Error(ER_INIT, "%s: execl failed to start %s",
		    monnam, netservfn);
	    }
        }
	exit(1);
    }

    (void)sprintf(filename, "%s/%s", DBDIR, MOLNAMES);

    /* Open molecule names file for reading */
    if ((molnamesfp = fopen(filename, "r")) == NULL) {
	Error(ER_INIT, "%s: could not open %s", monnam, filename);
	goto quit;
    }
    index = 0;
    /* Read file contents into molecule names array */
    while (fscanf(molnamesfp, "%s\n", fname) != EOF) {
        if ((molnames[index] = malloc((ALLOC_T)strlen(fname) + 1)) == NULL) {
	    Error(ER_INIT, "%s: no more memory for molecule names", monnam);
	    goto quit;
	}
        (void)strcpy(molnames[index++], fname);
    }
    currentmol = index - 1;
    (void)fclose(molnamesfp);

    (void)sprintf(filename, "%s/%s", DBDIR, molnames[currentmol]);

    /* Open current molecule file (the one added to) */
    if ((currentmolfd = open(filename, O_RDWR)) < 0) {
	Error(ER_INIT, "%s: could not open %s", monnam, filename);
	goto quit;
    }

    (void)sprintf(filename, "%s/%s", DBDIR, DATANAMES);

    /* Open data names file for reading */
    if ((datanamesfp = fopen(filename, "r")) == NULL) {
	Error(ER_INIT, "%s: could not open %s", monnam, filename);
	goto quit;
    }
    index = 0;
    /* Read file contents into data names array */
    while (fscanf(datanamesfp, "%s\n", fname) != EOF) {
        if ((datanames[index] = malloc((ALLOC_T)strlen(fname) + 1)) == NULL) {
	    Error(ER_INIT, "%s: no more memory for data names", monnam);
	    goto quit;
	}
        (void)strcpy(datanames[index++], fname);
    }
    currentdata = index - 1;
    (void)fclose(datanamesfp);

    (void)sprintf(filename, "%s/%s", DBDIR, datanames[currentdata]);

    /* Open current data file (the one added to) */
    if ((currentdatafd = open(filename, O_RDWR)) < 0) {
	Error(ER_INIT, "%s: could not open %s", monnam, filename);
	goto quit;
    }

    /* Open protection files for reading */
    (void)sprintf(filename, "%s/%s", DBDIR, PROTECTFILE);
    if (ReadProtFile(filename, rdprotlists, monnam, SUCCESS) != SUCCESS) {
      Error(ER_INIT, "%s: failed to setup basic protection", monnam);
      goto quit;
    }
  
    /* and Try to open write protection file */
    (void)sprintf(filename, "%s/%s", DBDIR, WRPROTECTFILE);
    if (ReadProtFile(filename, wrprotlists, monnam, FAIL) != SUCCESS) {
      int grp;
      for( grp = 0; grp < PROTLISTSIZE; grp++)
	wrprotlists[grp] = rdprotlists[grp];
    }

    /* Init data structures */
    editinfop = (LabelInfo *) CreateLabelInfo();
    penqueuep = (DBSQueue *) CreateLabelInfo();
    
    /* Register the available RPC functions */
    if (
	registerrpc((u_long)MONITORPROGNO, lincksport, ACCESS, 
		    (char *(*)())Access, xdr_entry, xdr_access) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, LOCKACCESS,
		    (char *(*)())Lock, xdr_entry, xdr_access) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, UNLOCK,
		    (char *(*)())Unlock, xdr_entry, xdr_int) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, SETPROT,
		    (char *(*)())SetProt, xdr_setprot, xdr_int) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, GETPROT,
		    (char *(*)())GetProt, xdr_entry, xdr_prot) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, DELETE,
		    (char *(*)())Delete, xdr_entry, xdr_int) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, UNDELETE,
		    (char *(*)())Undelete, xdr_entry, xdr_int) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, GETCHECKNO,
		    (char *(*)())GetCheckNo, xdr_entry, xdr_checkno) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, UPDATECHECKNO,
		    (char *(*)())UpdateCheckNo, xdr_entry, xdr_checkno) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, UPDATEENTRY,
		    (char *(*)())UpdateEntry, xdr_update, xdr_int) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, CREATE,
		    (char *(*)())Create, xdr_entry, xdr_new) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, KILL,
		    (char *(*)())Kill, xdr_entry, xdr_void) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, RESTART,
		    (char *(*)())Restart, xdr_entry, xdr_void) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, CHECK,
		    (char *(*)())Check, xdr_remove, xdr_edit) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, REMOVE,
		    (char *(*)())Remove, xdr_remove, xdr_void) ||
	registerrpc((u_long)MONITORPROGNO, lincksport, POLL,
                    (char *(*)())Poll, xdr_pid, xdr_edit)
	) {
	/* registerrpc returned error status */
	Error(ER_INIT, "%s: registerrpc failure", monnam);
	goto quit;
    }

    /* Disconnect standard IO */
    (void)fflush(stderr);
    (void)close(fileno(stderr));
    (void)open("/dev/null", O_WRONLY);
    (void)fflush(stdout);
    (void)close(fileno(stdout));
    (void)open("/dev/null", O_WRONLY);
    (void)close(fileno(stdin));
    (void)open("/dev/null", O_WRONLY);

    /* Start RPC service - never returns */
    svc_run();

    /* If it does: note error and terminate system */
    Error(ER_INIT, "%s: svc_run returned", monnam);

 quit:
    /* Ignore child and kill (them) */
    (void)Signal(SIGCHLD, SIG_IGN);
    (void)kill(childpid, SIGHUP); 
    (void)wait((WAIT_T *)NULL);
    exit(1);
#ifdef lint
/* lint is SOOOOOOOOOOOOO stupid sometimes */
    return 0;
#endif
}

/*  */
/**********************************************************************
 * Function: static int InitSignals()
 * 
 * Sets up monitor signals
 * SIGUSR1 starts a net server process
 *
 * Modifications:
 *      Sat Oct  9 19:28:05 1993 Martin Sjlin. Added check for SIGURG
 *                   and if not defined try SIGPOLL instead (SCO system).
 *      Thu Nov 11 10:25:19 1993 Martin Sjlin - SCO uses SIGUSR1 for
 *                   SIGURG (OOB) and SIGPOLL for async I/O.
 *      <list mods with name and date>
 */
static int InitSignals()
{
    /* Register signals to take care of */
    if (Signal(SIGALRM, Handler) == BADSIG) {
	Error(ER_INIT, "%s: could not register signal SIGALRM.", monnam);
	return(FAIL);
    }

    if (Signal(SIGCHLD, Handler) == BADSIG) {
	Error(ER_INIT, "%s: could not register signal SIGCHLD.", monnam);
	return(FAIL);
    }

#ifdef SIGXFSZ
    if (Signal(SIGXFSZ, Handler) == BADSIG) {
	Error(ER_INIT, "%s: could not register signal SIGXFSZ.", monnam);
	return(FAIL);
    }
#endif /* SIGXFSZ */
#ifndef SIGUSR1_FOR_OOB
    if (Signal(SIGUSR1, Handler) == BADSIG) {
	Error(ER_INIT, "%s: could not register signal SIGUSR1.", monnam);
	return(FAIL);
    }
#endif /* n SIGUSR1_FOR_OOB */
	
    /* Register "clean-up" signals */
    if (Signal(SIGTERM, Handler) == BADSIG ) {
      Error(ER_INIT, "%s: could not register signal SIGTERM.", monnam);
      return(FAIL);
    }
    if (Signal(SIGHUP, Handler) == BADSIG ) {
      Error(ER_INIT, "%s: could not register signal SIGHUP.", monnam);
      return(FAIL);
    }
    if (Signal(SIGQUIT, Handler) == BADSIG ) {
      Error(ER_INIT, "%s: could not register signal SIGQUIT.", monnam);
      return(FAIL);
    }
#ifdef SIGLOST
    if (Signal(SIGLOST, Handler) == BADSIG ) {
      Error(ER_INIT, "%s: could not register signal SIGLOST.", monnam);
      return(FAIL);
    }
#endif

    /* Register signals to ignore */
    (void)Signal(SIGINT, SIG_IGN);
    (void)Signal(SIGPIPE, SIG_IGN);
#ifdef SIGURG
    (void)Signal(SIGURG, SIG_IGN);
#else
#ifdef SIGUSR1_FOR_OOB
    (void)Signal(SIGUSR1, SIG_IGN);
#else
    (void)Signal(SIGPOLL, SIG_IGN);
#endif /* n SIGUSR1_FOR_OOB */
#endif /* n SIGURG */
    return(SUCCESS);
}
