/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	dbserfs.c
 *
 * SCCSINFO:		@(#)dbserfs.c	1.11 6/1/94
 *
 * ORIGINAL AUTHOR(S):  ???, 1987-02-26
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *
 *  This is the part of database server that communicates with the
 *  Monitor; transferring data (molecules) between the database and
 *  the workspace.
 *
 *   "So twisted the cords, & so knotted
 *    The meshes: twisted like to the human brain"
 *                         William Blake
 *
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include <rpc/types.h>			/* broken xdr headers */
#include <rpc/xdr.h>			/* for xdr types  */

#include "lincks.h"
#include "dbserver.h"
#include "libshared.h"
#include "dbcodes.h"
#include "xconfig.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_dbserfs.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_dbserrors.h"
#include "f_dbserv.h"
#include "f_dbrdmol.h"
#include "f_makelinks.h"
#include "f_dbwrmol.h"
#include "f_molfile.h"
#include "f_rpc.h"
#include "f_xdr.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern UID uid;
extern char host[];
extern TABENTRY *filetable;
extern int filetablesize;
extern IT_ENTRY *InterfaceTable;
extern int ITblsize;
extern u_long lincksport;
extern char username[MAXPATHLEN];
extern int synchronous;

extern MOLECULE the_molecule;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define ITABLEINC 256
#define SIZE_VSNET 20		/* Split fragment size */
#define SIZE_VSNET_SPLIT 30	/* Minimum size before splitting */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int clip_vsnet P_(( int chunk, LINKS **vsp, LABEL *cur ));
static LABEL create_new_label P_(( void ));
static int DoStore P_(( WS_LABEL index, LABEL label, MOLECULE *molecule,
                         int moltype, int checkno ));
static int findfield P_(( LINKGROUP *lg, char *tag, int add, LINKS **p ));
static int findgroup P_(( MOLECULE *mol, char *name, int add, LINKGROUP
                           **p ));
static int NewITix P_(( WS_LABEL wanted ));
static int store_fragment P_(( LINKS *vsp, LABEL this, LABEL chain ));
static int vsnet_split P_(( MOLECULE *first_mol, int *check ));
static int WriteAndUpdate P_(( MOLECULE *mol, LABEL label ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static WS_LABEL ITtblfree = 0;	/* start of free items */
 
/*  */
/**********************************************************************
 * Function: void TableInit()
 *
 * Initates the Interface Table and the filetable.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void TableInit()
{
  int i;
#ifdef HAVE_GETDTABLESIZE
  filetablesize = (getdtablesize() - 4) / 2;
#else
#ifdef HAVE_SC_OPEN_MAX
#ifndef NOFILE
#define NOFILE	(sysconf(_SC_OPEN_MAX))
#endif  /* n NOFILE */
	filetablesize = ( NOFILE - 4 ) / 2;
#else /* HAVE_SC_OPEN_MAX */
  /* DAVID:
   * this should work on a sun, too - see man page about getdtablesize
   * concerning caching limit value. Since getdtablesize returns the
   * soft limit, we can use rlim_cur below, but much more efficient
   * would be to set the soft limit to the maximum allowed by setrlimit
   * and use that for filetablesize.
   */
  struct rlimit rlp;
  if ( getrlimit( RLIMIT_NOFILE, &rlp) == -1 )
     /* DAVID: what should we do ... this shouldn't happend? */
     ;
  else
    /* we have a problem here - since rlp.rlim_cur is the current
     * soft limit which we would like to raise with setrlimit, or
     * should we use rlp.rlim_max which is the system imposed limit
     * which only root can increase ... 
     */
    /* could possible use - 2, since we are not using stdin etc */
    filetablesize = (rlp.rlim_cur - 4) / 2;
#endif /* n HAVE_SC_OPEN_MAX */
#endif /* n HAVE_GETDTABLESIZE */
  filetable = (TABENTRY *) calloc((ALLOC_T)filetablesize, sizeof(TABENTRY));

  for (i = 0; i < filetablesize; ++i) {
    filetable[i].molfile.fname = "";
    filetable[i].datfile.fname = "";
  }

  /* Initiate the Interface Table: System molecule and others as deleted */
  ITblsize = 0;
  InterfaceTable = NULL;

  (void)NewITix(IT_SYSTEM_MOLECULE_POSITION);

  InterfaceTable[IT_SYSTEM_MOLECULE_POSITION].label = IT_SYSTEM_MOLECULE_LABEL;
  InterfaceTable[IT_SYSTEM_MOLECULE_POSITION].checknumber = IT_NOT_SENT;
}

/*  */
/**********************************************************************
 * Function: int Release(WS_LABEL index)
 *
 * Marks the indexed label in the Interface Table to be "not sent".
 *
 * Modifications:
 *      <list mods with name and date>
 */
int Release(index)
  WS_LABEL index;
{
  if (!Allowed(index)) {
    Error(ER_ILLEGAL_INDEX, "Illegal index in Release.");
    return (FAIL);
  }
  InterfaceTable[index].checknumber = IT_NOT_SENT;

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: int Retrieve(WS_LABEL index)
 *
 * Sends the current database version of the indexed molecule.
 * If, though, its checknumber is the same in the Interface Table
 * as in the database the molecule is not sent.
 * A new molecule is created if necessary.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int Retrieve(index)
  WS_LABEL index;
{
  A_ENTRY *access;
  int checknumber;
  LABEL label;

  if (!Allowed(index)) {
    Error(ER_ILLEGAL_INDEX, "Illegal index in Retrieve.");
    return (FAIL);
  }
  if ((label = TranslateIndexToLabel(index)) == IT_NO_LABEL)
    return (FAIL);

  if (!(access = MolAccess(label, ACCESS)))
    return FAIL;

  if ((checknumber = access->a_checkno) == IT_NOT_SENT)
    return (RCBUFFER);

  if (InterfaceTable[index].checknumber == checknumber)
    return (RCOK);

  if (ReadMolecule(access))
    return FAIL;

  if (MakeLinksIndices(the_molecule.flinks) == IT_NO_SPACE)
    return (FAIL);

  InterfaceTable[index].checknumber = checknumber;

  return (RCMOL);		/* Another return further up. */
}

/*  */
/**********************************************************************
 * Function: int Store(WS_LABEL index, MOLECULE *molecule, int moltype)
 *
 * see DoStore.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int Store(index, molecule, moltype)
  WS_LABEL index;
  MOLECULE *molecule;
  int moltype;
{
  A_ENTRY *access;
  int errcode;
  LABEL label;

  if (!Allowed(index)) {
    Error(ER_ILLEGAL_INDEX, "Illegal index in Store.");
    return (FAIL);
  }
  if ((label = TranslateIndexToLabel(index)) == IT_NO_LABEL)
    return (FAIL);

  if (!(access = MolAccess(label,  LOCKACCESS)))
    return (FAIL);

  errcode = DoStore(index, label, molecule, moltype,  access->a_checkno);

  if (Unlock(label))
    return (FAIL);

  return errcode;
}

/*  */
/**********************************************************************
 * Function: static int DoStore(WS_LABEL index, LABEL label,  
                                MOLECULE *molecule, int moltype)
 *
 * Receives a molecule from the workspace and stores it in the
 * database. The molecule gets/has an entry in the Interface Table
 * and its checknumbers are updated. Referred molecules' backlinks
 * are updated.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int DoStore(index, label, molecule, moltype,  checkno)
  WS_LABEL index;
  LABEL label;
  MOLECULE *molecule;
  int moltype;
  int checkno;
{
  int split;

  if (checkno != InterfaceTable[index].checknumber) {
    SendOOB(PEN_RESOLVE, index,(char *)NULL);
    return (FAIL);
  }
  if (MakeLinksLabels(molecule->flinks) == IT_NO_LABEL) {
    return (FAIL);
  }
 
  split = 0; /* split is set to 1 if a split is done (in vsnet_split). */

  if ((moltype == VS) && vsnet_split(molecule, &split)) {
    Error(ER_WRITE,  "VSNET_SPLIT failed.");
    return (FAIL);
  } 

  if (WriteAndUpdate(molecule, label))
    return (FAIL);

  if (split) {
    SendOOB(PEN_RESOLVE, index,(char *)NULL);
    return FAIL;
  }

  InterfaceTable[index].checknumber += 1;

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int WriteAndUpdate(MOLECULE *mol, LABEL label)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int WriteAndUpdate(mol, label)
  MOLECULE *mol;
  LABEL label;
{
  static L_ENTRY largs = {(LABEL)0, (UID)0};
  static C_ENTRY checkno = {0, 0};
  int st;

  if (WriteMolecule(mol, label)) {
    Error(ER_WRITE, "WriteMolecule failed.");
    return FAIL;
  }

  largs.l_uid = uid;
  largs.l_label = label;
  if ((st=callrpc(host, (u_long)MONITORPROGNO, lincksport, UPDATECHECKNO,
		  xdr_entry, (char *)&largs, xdr_checkno, (char *)&checkno))) {
    RPCTrouble(UPDATECHECKNO,st);
    return (FAIL);
  }
  if (checkno.c_status != SUCCESS) {
    ErrorRPC(UPDATECHECKNO, checkno.c_status);
    return (FAIL);
  }
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static int clip_vsnet(int chunk, LINKS **vsp, LABEL *cur)
 *
 * Clips vsnet so that a head of 'chunk' slots at (*vsp) are clipped off
 * and (*vsp) is moved to the remaining vsnet. When there is a remaining
 * vsnet, the function also grabs a new molecule label for the vsnet tail
 * and returns this in 'cur', which otherwise is unchanged.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int clip_vsnet(chunk, vsp, cur)
  int chunk;
  LINKS **vsp;
  LABEL *cur;
{
  LINKS *p = NULL;

  if ((chunk <= 0) || (!(*vsp)))
    return FAIL;

  for ( ; (chunk-- > 0) && (*vsp); (*vsp) = p->nextlinks)
    p = (*vsp);

  if (!(*vsp))
    return SUCCESS;

  p->nextlinks = NULL;

  if (((*cur) = create_new_label()) == IT_NO_LABEL)
    return (FAIL);

  return SUCCESS;
}



/*  */
/**********************************************************************
 * Function: static int store_fragment(LINKS *vsp, LABEL this, chain)
 *
 * Makes a molecule to hold the 'vsp' vsnet fragment and stores it under
 * label 'this' with a SYSTEM:Chain link to 'chain'.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int store_fragment(vsp, this, chain)
  LINKS *vsp;
  LABEL this;
  LABEL chain;
{
  static LINKITEM chainlinkrec = {(LABEL)0, (LABEL)0, NULL};
  static LINKS chainrec = {"Chain", &chainlinkrec,  NULL};
  static LINKGROUP systemrec = { "SYSTEM", &chainrec, NULL};
  static LINKGROUP vsnetrec = { "VSNET", NULL, &systemrec};
  static MOLECULE molecule = {&vsnetrec,  NULL, NULL, NULL};
  static int errcode = 0;
  
  chainlinkrec.vs = chain;
  vsnetrec.links = vsp;

  if (!MolAccess(this,  LOCKACCESS))
    return FAIL;

  if (chain == (LABEL)0)
    vsnetrec.nextgroup = NULL;

  errcode = WriteAndUpdate(&molecule, this);

  vsnetrec.nextgroup = &systemrec;

  if (Unlock(this))
    return FAIL;

  return errcode;
}


/*  */
/**********************************************************************
 * Function: static int findgroup(MOLECULE *mol,char *name,int add,LINKGROUP **p)
 *
 * Moves (*p) to the link group record from mol->flinks with matching name. If
 * 'add' is non-zero, then a new link group record is added to the top of
 * the mol->flinks list when no match was found.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int findgroup(mol,name,add,p)
  MOLECULE *mol;
  char *name;
  int add;
  LINKGROUP **p;
{
  for ((*p) = mol->flinks; (*p); (*p) = (*p)->nextgroup)
    if (!strcmp((*p)->name,name))
      return SUCCESS;
  if (!add)
    return FAIL;
  if (!((*p) = (LINKGROUP *)malloc(sizeof(LINKGROUP)))) {
    Error(ER_MALLOC, "findgroup: no more memory");
    return FAIL;
  }
  (*p)->nextgroup = mol->flinks;
  (*p)->links = NULL;
  (*p)->name = strdup(name);
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static int findfield(LINKGROUP *lg,char *tag,int add,LINKS **p)
 *
 * Moves (*p) to the link field record from lg->links with matching tag. If
 * 'add' is non-zero, then a new link field record is added to the top of
 * the lg->links list when no match was found.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int findfield(lg,tag,add,p)
  LINKGROUP *lg;
  char *tag;
  int add;
  LINKS **p;
{
  for ((*p) = lg->links; (*p); (*p) = (*p)->nextlinks)
    if (!strcmp((*p)->tag,tag))
      return SUCCESS;
  if (!add)
    return FAIL;
  if (!((*p) = (LINKS *)malloc(sizeof(LINKS)))) {
    Error(ER_MALLOC, "findfield: no more memory");
    return FAIL;
  }
  (*p)->nextlinks = lg->links;
  (*p)->linkitem = NULL;
  (*p)->tag = strdup(tag);
  lg->links = (*p);
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static int vsnet_split(MOLECULE *first_mol, int *check)
 *
 * Splits the molecule if it is a vs molecule and has a VSNET-group
 * that are > SIZE_VSNET. Do NOT store the molecule, it just adjusts
 * VSNET of the molecule and moves the rest (of VSNET) to new molecules.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int vsnet_split(first_mol, check)
  MOLECULE *first_mol;
  int      *check;

{
  LINKGROUP *lg = NULL;
  LINKS *vstail = NULL;
  LINKS *vshead = NULL;
  LINKS *nx = NULL;
  LABEL last_label = (LABEL)0;
  LABEL chain_label = (LABEL)0;
  LABEL cur_label = (LABEL)0;
  int vsnet_count = 0;
  int chunk = 0;

  /* Find the VSNET group. Must exist for a VS-molecule. */
  if (findgroup(first_mol,"VSNET",0,&lg)) {
    Error(0, "Server: wrong moltype/ no VSNET-group");
    return FAIL;
  }

  /* Compute number of slots in the vsnet group */
  for (vstail = lg->links; vstail; vstail = vstail->nextlinks)
    vsnet_count++;

  /* Hold on to the group in situ (for the while-loop below) */
  vstail = lg->links;

  /* Return when there are fewer slots than the split threshold. */
  if (vsnet_count <= SIZE_VSNET_SPLIT) 
    return (SUCCESS);

  /* Set (*check) to signal that split took place (1). */
  (*check) = 1;

  /* Point 'nx' to the SYSTEM:Chain link field record */
  if (findgroup(first_mol,"SYSTEM",1,&lg) || findfield(lg,"Chain",1,&nx))
    return FAIL;

  /* Ensure a link item record for the vsnet chain */
  if (!nx->linkitem) {
    if (!(nx->linkitem = (LINKITEM *)malloc(sizeof(LINKITEM)))) {
      Error(ER_MALLOC, "vsnet_split: no more memory");
      return FAIL;
    }
    nx->linkitem->nextitem = NULL;
    nx->linkitem->vs = (LABEL)0;
    nx->linkitem->label = (LABEL)0;
  }

  /* Grab the current tail of vsnet chain */
  last_label = nx->linkitem->vs;

  /* Compute size of initial chunk */
  chunk = vsnet_count % SIZE_VSNET;
  if (chunk <= (SIZE_VSNET_SPLIT - SIZE_VSNET))
    chunk += SIZE_VSNET;

  /* Clip off, chain together, and store overflowing vsnet fragments */
  /* Note that 'cur_label' is 0 initially. */
  while (vstail) {
    chain_label = last_label;
    vshead = vstail;
    if (clip_vsnet(chunk,  &vstail, &chain_label) != SUCCESS)
      return FAIL;
    if (cur_label == (LABEL)0)
      nx->linkitem->vs = chain_label;
    else {
      if (store_fragment(vshead, cur_label, chain_label) != SUCCESS)
	return FAIL;
    }
    cur_label = chain_label;
    chunk = SIZE_VSNET;
  }

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: LABEL TranslateIndexToLabel(WS_LABEL index)
 *
 * Translates an index to a label using entries in the Interface
 * Table and, if empty entries, creates new labels.
 *
 * Modifications:
 *      <list mods with name and date>
 */
LABEL TranslateIndexToLabel(index)
  WS_LABEL index;
{
  if (!Allowed(index)) {
    Error(ER_ILLEGAL_INDEX, "Illegal index in TranslateIndexToLabel.");
    return (IT_NO_LABEL);
  }
  if (index == NULLABEL)
    return (0);

  if (InterfaceTable[index].checknumber == IT_DELETED) {
    /* This happens only when TranslateIndexToLabel is called from Store */
    if ((InterfaceTable[index].label = create_new_label()) == IT_NO_LABEL)
      return (IT_NO_LABEL);
    InterfaceTable[index].checknumber = IT_NOT_SENT;
  }
  return (InterfaceTable[index].label);
}

/*  */
/**********************************************************************
 * Function: WS_LABEL TranslateLabelToIndex(LABEL label)
 *
 * Scans the Interface Table for the label. If the label isn't found
 * an entry is made.
 *
 * Modifications:
 *      <list mods with name and date>
 */
WS_LABEL TranslateLabelToIndex(label)
  LABEL label;
{
  WS_LABEL index;

  index = ScanInterfaceTable(label);
  if (index == IT_NOT_FOUND)
    return (MakeEntry(label));	/* Including errors. */
  else
    return (index);
}

/*  */
/**********************************************************************
 * Function: int Allowed(WS_LABEL index)
 *
 * Checks that the index is within the bounds.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int Allowed(index)
  WS_LABEL index;
{
  if ((index >= -1) && (index < ITblsize))
    return (1);
  else
    return (NewITix(index) == SUCCESS);
}

/*  */
/**********************************************************************
 * Function: WS_LABEL ScanInterfaceTable(LABEL label)
 *
 * Looks for a label in the Interface Table.
 *
 * Modifications:
 *      <list mods with name and date>
 */
WS_LABEL ScanInterfaceTable(label)
  LABEL label;
{
  WS_LABEL i = 0;

  if (!label)
    return (NULLABEL);

  for (i = 0; i < ITblsize; i++) {
    if (InterfaceTable[i].label == label)
      break;
  }

  if (i >= ITblsize)
    return (IT_NOT_FOUND);
  else
    return (i);
}

/*  */
/**********************************************************************
 * Function: WS_LABEL MakeEntry(LABEL label)
 *
 * Looks for an empty slot in the Interface Table and puts the label
 * there.
 *
 * Modifications:
 *      <list mods with name and date>
 */
WS_LABEL MakeEntry(label)
  LABEL label;
{
  WS_LABEL i = 0;

  for (i =  ITtblfree; i < ITblsize; i++) {
    if (InterfaceTable[i].checknumber == IT_DELETED)
      break;
  }

  if (i >= ITblsize)
    if (NewITix(i) < 0)
      return (IT_NO_SPACE);

  /* ITtblfree points to next free entry ... */
  ITtblfree = i + 1;			

  InterfaceTable[i].label = label;
  InterfaceTable[i].checknumber = IT_NOT_SENT;

  return (i);
}

/*  */
/**********************************************************************
 * Function: static LABEL create_new_label()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static LABEL create_new_label()
{
  static N_ENTRY    new = {0, (LABEL)0, (char *)NULL,  0L, 0};
  static L_ENTRY    largs = {(LABEL)0, (UID)0};
  int st;

  largs.l_uid = uid;
  if (new.n_filename) {
    xdr_free(xdr_string, (char *)&new.n_filename);
    new.n_filename = NULL;
  }
  if ((st=callrpc(host, (u_long) MONITORPROGNO, lincksport, CREATE,
		  xdr_entry, (char *)&largs, xdr_new, (char *)&new))) {
    RPCTrouble(CREATE,st);
    return(IT_NO_LABEL);
  }
  return new.n_label;
}

/*  */
/**********************************************************************
 * Function: static int NewITix(WS_LABEL wanted)
 *
 * Mallocs IT table space and returns the requested index
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int NewITix(wanted)
  WS_LABEL wanted;
{
  WS_LABEL i;
  unsigned newsize;

  newsize = wanted + ITABLEINC;
  i = newsize * sizeof(IT_ENTRY);

  if (ITblsize == 0) {
    InterfaceTable = (IT_ENTRY *) calloc((ALLOC_T)newsize, sizeof(IT_ENTRY));
  } else {
    InterfaceTable = (IT_ENTRY *) realloc((FREEPTR *)InterfaceTable, (ALLOC_T)i);
  }

  if (!InterfaceTable) {
    Error(ER_MALLOC, "NewITix: no more memory");
    return (FAIL);
  }

  /* Initialize IT */
  for (i = ITblsize; i < newsize; ++i) {
    InterfaceTable[i].checknumber = IT_DELETED;
    InterfaceTable[i].label = NULLABEL;
  }
  /* Set new interface table size */
  ITblsize = newsize;

  /* Return SUCCESS */
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: int Check(WS_LABEL index, int insertflg)
 *
 * Checks if parallel editing occurs and optionally adds
 * new editing information
 *
 * Modifications:
 *      <list mods with name and date>
 */
int Check(index, insertflg)
  WS_LABEL index;
  int insertflg;
{
  static R_ENTRY r = {(LABEL)0, 0, (char *)NULL, 0};
  static E_ENTRY e = {(LABEL)0, (char *)NULL};
         int st;

  r.r_label = TranslateIndexToLabel(index);
  r.r_pid = getpid();
  r.r_username = username;	/* global username */
  r.r_flag = insertflg;

  if (e.e_usernames) {
    xdr_free(xdr_string, (char *)&e.e_usernames);
    e.e_usernames = NULL;
  }

  if ((st=callrpc(host, (u_long) MONITORPROGNO, lincksport, CHECK,
		  xdr_remove, (char *)&r, xdr_edit, (char *)&e))) {
    RPCTrouble(CHECK, st);
    return (FAIL);
  }

  if (e.e_usernames && *e.e_usernames)
    SendOOB(PEN_CHECK, index, e.e_usernames);

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: int Remove(WS_LABEL index)
 *
 * Removes edit information for a label
 *
 * Modifications:
 *      <list mods with name and date>
 */
int Remove(index)
  WS_LABEL index;
{
  static R_ENTRY r = {(LABEL)0, 0, (char *)NULL, 0};
  int st;

  r.r_label = TranslateIndexToLabel(index);
  r.r_pid = getpid();
  r.r_username = "";
  r.r_flag = 0;

  if ((st=callrpc(host, (u_long) MONITORPROGNO, lincksport, REMOVE,
		  xdr_remove, (char *)&r, xdr_void, (char *)NULL))) {
    RPCTrouble(REMOVE, st);
    return(FAIL);
  }

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: int RemoveAll()
 *
 * Removes editing information
 *
 * Modifications:
 *      <list mods with name and date>
 */
int RemoveAll()
{
  static R_ENTRY r = {(LABEL)0, 0, (char *)NULL, 0};
  int st;

  r.r_label = 0;		/* not used by monitor */
  r.r_pid = getpid();
  r.r_username = "";		/* not used by monitor*/
  r.r_flag = 1;

  if ((st=callrpc(host, (u_long) MONITORPROGNO, lincksport, REMOVE,
		  xdr_remove, (char *)&r, xdr_void, (char *)NULL))) {
    RPCTrouble(REMOVE, st);
    return (FAIL);
  }
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: int Poll()
 *
 * Loads parallel editing message from monitor.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int Poll()
{
  static E_ENTRY e = {(LABEL)0, (char *)NULL};
  WS_LABEL index;
  pid_t pid;
  int st;

  pid = getpid();

  if (e.e_usernames) {
    xdr_free(xdr_string, (char *)&e.e_usernames);
    e.e_usernames = NULL;
  }

  if ((st=callrpc(host, (u_long) MONITORPROGNO, lincksport, POLL,
		  xdr_pid, (char *)&pid, xdr_edit, (char *)&e))) {
    RPCTrouble(POLL, st);
    return (FAIL);
  }

  index = TranslateLabelToIndex(e.e_label);

  SendOOB(PEN_POLL, index, e.e_usernames);

  return (SUCCESS);
}
