/*
 * An external command facility for Pcomm.  All of the "sanity checking"
 * is done at this end, so virtually no checking is done at Pcomm's end.
 */

#include <stdio.h>
#include <ctype.h>
#include "config.h"
#include "cmd.h"

#ifdef BSD
#include <sys/file.h>
#else /* BSD */
#include <fcntl.h>
#endif /* BSD */

int cmd_ipc;

main(argc, argv)
int argc;
char *argv[];
{
	static char *cmds[] = {"SET", "QUERY", "IF", "DIAL", "REDIAL", "EXIT",
	"CLEAR_SCREEN", "CHG_DIR", "HANG_UP", "PRINTER", "MODEM_BREAK", "SEND",
	"RECEIVE", "SCREEN_DUMP", "DATA_LOG", 0};
	static char *set_args[] = {"BAUD", "PARITY", "DATA_BITS", "STOP_BITS",
	"DUPLEX", "AUX", "HOT_KEY", "ASCII_HOT", "FLOW_CTRL", "CR_IN",
	"CR_OUT", "LOGFILE", "DUMPFILE", "STRIP", "LOCAL_ECHO", "EXPAND",
	"CR_DELAY", "PACE", "CR_UP", "LF_UP", "TIMER", "CR_DN", "LF_DN", 0};
	static char *qry_args[] = {"TTY_NAME", "MODEM_NAME", 0};
	static char *if_args[] = {"CONNECTED", "LOG_STATUS", "PRINTER_STATUS",
	0};
	static char *xfer_args[] = {"XMODEM", "XMODEM_1K", "MODEM7", "YMODEM",
	"YMODEM_G", "ASCII", "EXT_1", "EXT_2", "EXT_3", 0};
	char *command, buf[40], *mkupper(), *arg1, *arg2, *s, *strchr();
	char *pcomm_cmd, *getenv();
	int cmd_num, i, arg_num, got_it;
	unsigned int ui;
	void send_cmd(), get_value(), exit(), cmd_exit();

					/* get the environmental variable */
	pcomm_cmd = getenv("PCOMM_CMD");
	if (argc == 1 || pcomm_cmd == NULL || *pcomm_cmd == '\0') {
		fprintf(stderr, "%s: This is a Pcomm support program.  It is intended for use inside\n", argv[0]);
		fprintf(stderr, "an auto-login shell script\n");
		exit(1);
	}
					/* parse the command line */
	command = mkupper(argv[1]);
	if (argc > 2)
		arg1 = mkupper(argv[2]);
	else
		arg1 = "";
	if (argc > 3)
		arg2 = mkupper(argv[3]);
	else
		arg2 = "";
					/* convert command to a number */
	cmd_num = 0;
	while (cmds[cmd_num] != NULL) {
		if (!strcmp(command, cmds[cmd_num]))
			break;
		cmd_num++;
	}
					/* attach to the IPC channel */
	if ((cmd_ipc = ipc_attach(pcomm_cmd, O_RDWR)) < 0) {
		fprintf(stderr, "%s: Can't attach to IPC channel\n", argv[0]);
		exit(1);
	}
					/* do it! */
	switch(cmd_num) {
		case SET:
			if (argc < 4) {
				fprintf(stderr, "%s: SET command requires 2 arguments\n", argv[0]);
				cmd_exit(1);
			}
				
			arg_num = 0;
			while (set_args[arg_num] != NULL) {
				if (!strcmp(arg1, set_args[arg_num]))
					break;
				arg_num++;
			}
			switch(arg_num) {
				case BAUD:
					ui = (unsigned int) atoi(arg2);
					switch(ui) {
						case 300:
						case 1200:
						case 2400:
						case 4800:
						case 9600:
						case 19200:
						case 38400:
							break;
						default:
							fprintf(stderr, "%s: bad value '%s' for SET BAUD command\n", argv[0], argv[3]);
							cmd_exit(1);
					}
					sprintf(buf, "%d", ui);
					send_cmd(SET, BAUD, buf);
					break;
				case PARITY:
					if (strcmp(arg2, "EVEN") && strcmp(arg2, "ODD") && strcmp(arg2, "NONE")) {
						fprintf(stderr, "%s: bad value '%s' for SET PARITY command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, PARITY, arg2);
					break;
				case DATA_BITS:
					i = atoi(arg2);
					if (i != 7 && i != 8) {
						fprintf(stderr, "%s: bad value '%s' for SET DATA_BITS command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					sprintf(buf, "%d", i);
					send_cmd(SET, DATA_BITS, buf);
					break;
				case STOP_BITS:
					i = atoi(arg2);
					if (i != 1 && i != 2) {
						fprintf(stderr, "%s: bad value '%s' for SET STOP_BITS command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					sprintf(buf, "%d", i);
					send_cmd(SET, STOP_BITS, buf);
					break;
				case DUPLEX:
					if (strcmp(arg2, "FULL") && strcmp(arg2, "HALF")) {
						fprintf(stderr, "%s: bad value '%s' for SET DUPLEX command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, DUPLEX, arg2);
					break;	
				case AUX:
					send_cmd(SET, AUX, argv[3]);
					break;
				case HOT_KEY:
					i = atoi(arg2);
					if (i <= 0 || i > 256) {
						fprintf(stderr, "%s: bad value '%s' for SET HOT_KEY command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, HOT_KEY, arg2);
					break;
				case ASCII_HOT:
					send_cmd(SET, ASCII_HOT, argv[3]);
					break;
				case FLOW_CTRL:
					if (strcmp(arg2, "XON/XOFF") && strcmp(arg2, "NONE")) {
						fprintf(stderr, "%s: bad value '%s' for SET FLOW_CTRL command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, FLOW_CTRL, arg2);
					break;
				case CR_IN:	
					if (strcmp(arg2, "CR") && strcmp(arg2, "CR/LF")) {
						fprintf(stderr, "%s: bad value '%s' for SET CR_IN command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, CR_IN, arg2);
					break;
				case CR_OUT:
					if (strcmp(arg2, "CR") && strcmp(arg2, "CR/LF")) {
						fprintf(stderr, "%s: bad value '%s' for SET CR_OUT command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, CR_OUT, arg2);
					break;
				case LOGFILE:
					send_cmd(SET, LOGFILE, argv[3]);
					break;
				case DUMPFILE:
					send_cmd(SET, DUMPFILE, argv[3]);
					break;
				case STRIP:	
					if (strcmp(arg2, "YES") && strcmp(arg2, "NO")) {
						fprintf(stderr, "%s: bad value '%s' for SET STRIP command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, STRIP, arg2);
					break;
				case LOCAL_ECHO:	
					if (strcmp(arg2, "YES") && strcmp(arg2, "NO")) {
						fprintf(stderr, "%s: bad value '%s' for SET LOCAL_ECHO command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, LOCAL_ECHO, arg2);
					break;
				case EXPAND:
					if (strcmp(arg2, "YES") && strcmp(arg2, "NO")) {
						fprintf(stderr, "%s: bad value '%s' for SET EXPAND command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, EXPAND, arg2);
					break;
				case CR_DELAY:
					i = atoi(arg2);
					if (i != 0 && i != 100 && i != 150) {
						fprintf(stderr, "%s: bad value '%s' for SET CR_DELAY command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					sprintf(buf, "%d", i);
					send_cmd(SET, CR_DELAY, buf);
					break;
				case PACE:
					if (strcmp(arg2, "YES") && strcmp(arg2, "NO")) {
						fprintf(stderr, "%s: bad value '%s' for SET PACE command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, PACE, arg2);
					break;
				case CR_UP:
					if (strcmp(arg2, "NONE") && strcmp(arg2, "ADD_LF") && strcmp(arg2, "STRIP")) {
						fprintf(stderr, "%s: bad value '%s' for SET CR_UP command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, CR_UP, arg2);
					break;
				case LF_UP:
					if (strcmp(arg2, "NONE") && strcmp(arg2, "ADD_CR") && strcmp(arg2, "STRIP")) {
						fprintf(stderr, "%s: bad value '%s' for SET LF_UP command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, LF_UP, arg2);
					break;
				case TIMER:
					i = atoi(arg2);
					if (i < 5 || i > 150) {
						fprintf(stderr, "%s: bad value '%s' for SET TIMER command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					sprintf(buf, "%d", i);
					send_cmd(SET, CR_DN, buf);
					break;
				case CR_DN:
					if (strcmp(arg2, "NONE") && strcmp(arg2, "ADD_LF") && strcmp(arg2, "STRIP")) {
						fprintf(stderr, "%s: bad value '%s' for SET CR_DN command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, CR_DN, arg2);
					break;
				case LF_DN:	
					if (strcmp(arg2, "NONE") && strcmp(arg2, "ADD_CR") && strcmp(arg2, "STRIP")) {
						fprintf(stderr, "%s: bad value '%s' for SET LF_DN command\n", argv[0], argv[3]);
						cmd_exit(1);
					}
					send_cmd(SET, LF_DN, arg2);
					break;
				default:
					fprintf(stderr, "%s: Illegal argument '%s' for SET command\n", argv[0], argv[2]);
					cmd_exit(1);
					break;
			}
			break;
		case QUERY:		/* the QUERY commands */
			if (argc < 3) {
				fprintf(stderr, "%s: QUERY command requires an argument\n", argv[0]);
				cmd_exit(1);
			}
				
			arg_num = 0;
			while (qry_args[arg_num] != NULL) {
				if (!strcmp(arg1, qry_args[arg_num]))
					break;
				arg_num++;
			}

			switch(arg_num) {
				case TTY_NAME:
					send_cmd(QUERY, TTY_NAME, "dummy");
					get_value();
					break;
				case MODEM_NAME:
					send_cmd(QUERY, MODEM_NAME, "dummy");
					get_value();
					break;
				default:
					fprintf(stderr, "%s: Illegal argument '%s' for QUERY command\n", argv[0], argv[2]);
					cmd_exit(1);
					break;
			}
			break;
		case IF:		/* the IF commands */
			if (argc < 3) {
				fprintf(stderr, "%s: IF command requires an argument\n", argv[0]);
				cmd_exit(1);
			}
				
			arg_num = 0;
			while (if_args[arg_num] != NULL) {
				if (!strcmp(arg1, if_args[arg_num]))
					break;
				arg_num++;
			}

			switch(arg_num) {
				case CONNECTED:
					send_cmd(IF, CONNECTED, "dummy");
					break;
				case LOG_STATUS:
					send_cmd(IF, LOG_STATUS, "dummy");
					break;
				case PRINTER_STATUS:
					send_cmd(IF, PRINTER_STATUS, "dummy");
					break;
				default:
					fprintf(stderr, "%s: Illegal argument '%s' for IF command\n", argv[0], argv[2]);
					cmd_exit(1);
					break;
			}
			break;
		case DIAL:
			if (argc < 3) {
				fprintf(stderr, "%s: DIAL command requires an argument\n", argv[0]);
				cmd_exit(1);
			}
			if (!strcmp(arg1, "MANUAL")) {
				if (argc < 4) {
					fprintf(stderr, "%s: DIAL MANUAL command requires an argument\n", argv[0]);
					cmd_exit(1);
				}
				send_cmd(DIAL, 1, argv[3]);
			}
			else {
				if (s = strchr("+-@#", *argv[2]))
					i = atoi(++s);
				else
					i = atoi(argv[2]);
				if (i < 1 || i > 100) {
					fprintf(stderr, "%s: bad value '%s' for DIAL command\n", argv[0], argv[2]);
					cmd_exit(1);
				}
				send_cmd(DIAL, 0, argv[2]);
			}
			break;
		case REDIAL:
			if (argc < 3) {
				fprintf(stderr, "%s: REDIAL command requires at least 1 argument\n", argv[0]);
				cmd_exit(1);
			}
			send_cmd(REDIAL, 0, argv[2]);
			break;
		case EXIT:
			send_cmd(EXIT, 0, "dummy");
			/*
			 * Don't wait for a return code... Pcomm won't be
			 * around to send it!  So, assume it got there OK.
			 */
			cmd_exit(0);
			break;
		case CLEAR_SCREEN:
			send_cmd(CLEAR_SCREEN, 0, "dummy");
			break;
		case CHG_DIR:
			if (argc < 3) {
				fprintf(stderr, "%s: CHG_DIR command requires an argument\n", argv[0]);
				cmd_exit(1);
			}
			send_cmd(CHG_DIR, 0, argv[2]);
			break;
		case HANG_UP:
			send_cmd(HANG_UP, 0, "dummy");
			break;
		case PRINTER:
			if (argc < 3) {
				fprintf(stderr, "%s: PRINTER command requires an argument\n", argv[0]);
				cmd_exit(1);
			}
			if (strcmp(arg1, "ON") && strcmp(arg1, "OFF")) {
				fprintf(stderr, "%s: bad value '%s' for PRINTER command\n", argv[0], argv[2]);
				cmd_exit(1);
			}
			if (!strcmp(arg1, "ON"))
				send_cmd(PRINTER, 1, "dummy");
			else
				send_cmd(PRINTER, 0, "dummy");
			break;
		case MODEM_BREAK:
			send_cmd(MODEM_BREAK, 0, "dummy");
			break;
		case SEND:
			if (argc < 4) {
				fprintf(stderr, "%s: SEND command requires 2 arguments\n", argv[0]);
				cmd_exit(1);
			}
			arg_num = 0;
			got_it = 0;
			while (xfer_args[arg_num] != NULL) {
				if (!strcmp(xfer_args[arg_num], arg1)) {
					got_it++;
					break;
				}
				arg_num++;
			}
			if (!got_it) {
				fprintf(stderr, "%s: Illegal argument '%s' for SEND command\n", argv[0], argv[2]);
				cmd_exit(1);
			}
			send_cmd(SEND, ++arg_num, argv[3]);
			break;
		case RECEIVE:
			if (argc < 3) {
				fprintf(stderr, "%s: RECEIVE command requires at least 1 argument\n", argv[0]);
				cmd_exit(1);
			}
			arg_num = 0;
			got_it = 0;
			while (xfer_args[arg_num] != NULL) {
				if (!strcmp(xfer_args[arg_num], arg1)) {
					got_it++;
					break;
				}
				arg_num++;
			}
			if (!got_it) {
				fprintf(stderr, "%s: Illegal argument '%s' for RECEIVE command\n", argv[0], argv[2]);
				cmd_exit(1);
			}
			if (argc < 4)
				send_cmd(RECEIVE, ++arg_num, "dummy");
			else
				send_cmd(RECEIVE, ++arg_num, argv[3]);
			break;
		case SCREEN_DUMP:
			send_cmd(SCREEN_DUMP, 0, "dummy");
			break;
		case DATA_LOG:
			if (!strcmp(arg1, "ON"))
				send_cmd(DATA_LOG, 1, "dummy");
			else
				send_cmd(DATA_LOG, 0, "dummy");
			break;
		default:
			fprintf(stderr, "%s: Illegal command '%s'\n", argv[0], argv[1]);
			cmd_exit(1);
			break;
	}
					/* get the return status */
	if (ipc_read(cmd_ipc, buf, 256)) {
#ifndef sparc
		fprintf(stderr, "Can't read from IPC\n");
#endif /* sparc */
		cmd_exit(1);
	}
	cmd_exit(atoi(buf));
}

/*
 * Copy and convert a string to all upper case 
 */

char *
mkupper(string)
char *string;
{
	int i;
	char buf[80], *ans, *strcpy(), *malloc();
	void exit();

	i = 0;
	while (*string != '\0') {
		if (islower(*string))
			buf[i++] = toupper(*string);
		else
			buf[i++] = *string;
		string++;
		if (i == 79)
			break;
	}
	buf[i] = '\0';
	if ((ans = malloc((unsigned int) strlen(buf)+1)) == NULL) {
		fprintf(stderr, "out of memory!\n");
		exit(1);
	}
	strcpy(ans, buf);
	
	return(ans);
}

/*
 * Get a return value from a command, (and therefore exit prematurely).
 */

void 
get_value()
{
	char *s, *strchr(), buf[256];
	void cmd_exit();

	if (ipc_read(cmd_ipc, buf, 256)) {
		fprintf(stderr, "Can't read from IPC\n");
		cmd_exit(1);
	}

	if (s = strchr(buf, '\n'))
		*++s = '\0';

	printf("%s", buf);
	cmd_exit(0);
}

/*
 * Send a command to the Pcomm process
 */

void
send_cmd(cmd, arg1, arg2)
int cmd, arg1;
char *arg2;
{
	char buf[256];
	void cmd_exit();

	sprintf(buf, "%d %d %249.249s\n", cmd, arg1, arg2);
	if (ipc_write(cmd_ipc, buf, 256)) {
		fprintf(stderr, "Can't write to IPC\n");
		cmd_exit(1);
	}
	return;
}

/*
 * Clean up and go home...
 */

void
cmd_exit(n)
int n;
{
	void exit();

	ipc_detach(cmd_ipc);
	exit(n);
}

#define MAIN
#include IPC
