char *revision = "3.20";	/* cannot be longer than 7 chars (blk0.sender) */

/* #define TABLE_CRC16 */
#ifdef M_XENIX
#define NO_SELECT
#endif

/*+-------------------------------------------------------------------------
	ecusea.c - SEAlink - Sliding window file transfer protocol

  Defined functions:
	Nap(msec)
	arg_token(parsestr,termchars)
	cancel_transaction(sig)
	crc_update(c,crc)
	fname_split(cmd,arg,arg_max_quan,narg_rtn)
	fname_too_long(fname)
	fname_truncated()
	getspeed(code)
	lgetc_timeout(tenths)
	lgetc_timeout_SIGALRM()
	main(argc,argv,envp)
	rdchk(fd)
	receive_block(buf)
	receive_file()
	send_comm_block(blk,blknum)
	send_file(name)
	send_file_block(fp,blknum)
	set_sf_state(place,new_state)
	set_utime_1980(filename,secs_since_1980)
	sf_state_text(state)
	wait_for_rcvr_response()
	xmit_ack(blknum)
	xmit_cancel()
	xmit_nak(blknum)

ecu adaptation by W. Tucker
modelled after MSDOS sealink.c, which carried the following proviso:

              MS-DOS Version 1.20, created on 08/05/87
              at 17:51:40 (C) COPYRIGHT 1986,87 by
              System Enhancement Associates; ALL RIGHTS
              RESERVED By: Thom Henderson

              You are granted a license to use this
              code in your programs, and to adapt it to
              your particular situation and needs,
              subject only to the following conditions:
              1) You must refer to it as the SEAlink
              protocol, and you must give credit to
              System Enhancement Associates.  2) If you
              modify it in such a way that your version
              cannot converse with the original code as
              supplied by us, then you should refer to
              it as "SEAlink derived", or as a
              "variation of SEAlink", or words to that
              effect.  In short, we're not asking for
              any money, but we'd like to get some
              credit for our work.

--------------------------------------------------------------------------*/
/*+:EDITS:*/
/*:09-10-1992-14:00-wht@n4hgf-ECU release 3.20 */
/*:08-22-1992-15:39-wht@n4hgf-ECU release 3.20 BETA */
/*:08-16-1992-03:08-wht@n4hgf-head off another POSIX plot */
/*:08-10-1992-04:01-wht@n4hgf-use init_Nap */
/*:07-20-1992-13:30-wht@n4hgf-put hzmsec in AGAIN ?? */
/*:07-17-1992-18:28-wht@n4hgf-remove Nap() and use common ../nap.o */
/*:05-11-1992-16:43-wht@gyro-fix WORKING_SELECT nap once and for all */
/*:05-08-1992-02:42-wht@n4hgf-select-based Nap was buggy */
/*:09-01-1991-14:22-wht@n4hgf2-on sun, use termios */
/*:08-30-1991-20:09-wht@n4hgf2-sun Nap was not returning a value */
/*:08-30-1991-02:34-jdeitch@jadpc.cts.com-fix no hzmsec */
/*:07-25-1991-12:59-wht@n4hgf-ECU release 3.10 */
/*:03-18-1991-22:49-wht@n4hgf-ISC 2.2 has struct timeval in sys/time.h */
/*:08-14-1990-20:40-wht@n4hgf-ecu3.00-flush old edit history */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <setjmp.h>
#include <fcntl.h>
#include <errno.h>
#if defined(sun)
#include <termios.h>
#define termio termios
#undef TCGETA
#undef TCSETA
#undef TCSETAW
#define TCGETA	TCGETS
#define TCSETA	TCSETS
#define TCSETAW	TCSETSW
#undef ECHO
#undef NL0
#undef NL1
#undef TAB0
#undef TAB1
#undef TAB2
#undef XTABS
#undef CR0
#undef CR1
#undef CR2
#undef CR3
#undef FF0
#undef FF1
#undef BS0
#undef BS1
#undef TOSTOP
#undef FLUSHO
#undef PENDIN
#undef NOFLSH
#else
#include <termio.h>
#endif /* sun */
#include <sys/param.h>
#include <sys/ioctl.h>
#include "../ecu_types.h"
#include "../ecu_stat.h"
#include <time.h>
#include <memory.h>
#if !defined(NO_SELECT)
#if defined(M_SYSV) && !defined(SCO32v4)	/* SCO pre 3.2v4 */
# include <sys/select.h>
#else
# include <sys/time.h>
#endif
#endif

#ifdef USE_PROTOS
# include "protos.h"
#endif

extern int errno;

/* Various system constants */
#define WINDOW		6					/* maximum size of window */
#define TIMEOUT	0x0FFF
#define OFFSET_1980	(time_t)315547200	/* time offset for 1970 <-> 1980 */

/*
 * The section of code that is compiled when NAKEOT is defined is in the
 * original MS-DOS version 1.16 routine.  Its purpose is to send a NAK when
 * an EOT is received during receive_file(), apparently to confirm that this is
 * indeed the end of file.  However, in certain (apparently non - standard)
 * versions of the protocol, it is possible that the program will report an
 * error when in fact there isn't one.  Comment this out at your discretion.
 */
#define NAKEOT

/* SEAlink block zero data structure */
typedef struct blk0
{
	long length;			/* length */
	time_t secs_since_1980;	/* creation/last mod in secs since 1/1/80 */
	char filename[17];		/* file name */
	char sender[15];		/* sending program */
	char send_no_acks;		/* true if rcvr need not ack */
	char filler[87];		/* fill to 128 bytes */
}	BLK0;

/* protocol characters */
#define SOH	0x01
#define EOT	0x04
#define ACK	0x06
#define NAK	0x15
#define CAN	0x18

/*  send_file state (sf_state) values */
#define SFS_GND		0	/* Ground state, ACK or NAK expected */
#define SFS_ACK		1	/* ACK received */
#define SFS_NAK		2	/* NAK received */
#define SFS_ACKW	3	/* ACK, block# received */
#define SFS_NAKW	4	/* NAK, block# received */
#define SFS_RGND	5	/* Returning to ground state */
int sf_state;

int allow_slide = 1;	/* sliding windows allowed */
int crc_in_use;			/* check type, 1 = CRC, 0 = checksum */
char *dfile = "/tmp/ecuSEA.log";
int error_count = 0;	/* total number of errors */
int iofd = 0;			/* file descriptor to use */
int no_ack_mode = 1;	/* true of ACKs not required */
int rf_done = 0;		/* receive file done */
int sf_ackw_count;		/* count of sliding ACKs seen */
int sf_ackblk;			/* number of last block ACKed */
int sf_blknum;			/* number of next block to send */
int sf_lastnum;			/* number of last block sent */
int sf_nakquan;			/* number of sequential NAKs */
int sf_slide;			/* true if sliding window */
int sigint = 0;			/* dummy for nap.c */

int sending_flag = -1;		/* send == 1, receive == 0, bad usage == -1 */
int log_packets = 0;
long rx_char_count = 0;
long tx_char_count = 0;
int Filcnt = 0;
int npaths = 0;
char curr_dir[256];
char s128[128];
unsigned baud_rate;
int exit_code;
int sent_EOT = 0;

struct termio tio;
struct termio tio0;

jmp_buf	lgetc_timeout_setjmp;

/* CRC16 routine; finish CRC calculation for compare */

#ifdef TABLE_CRC16

/* crctab calculated by Mark G. Mendel,Network Systems Corporation */
unsigned short crctab[256] = 
{
	0x0000,0x1021,0x2042,0x3063,0x4084,0x50a5,0x60c6,0x70e7,
	0x8108,0x9129,0xa14a,0xb16b,0xc18c,0xd1ad,0xe1ce,0xf1ef,
	0x1231,0x0210,0x3273,0x2252,0x52b5,0x4294,0x72f7,0x62d6,
	0x9339,0x8318,0xb37b,0xa35a,0xd3bd,0xc39c,0xf3ff,0xe3de,
	0x2462,0x3443,0x0420,0x1401,0x64e6,0x74c7,0x44a4,0x5485,
	0xa56a,0xb54b,0x8528,0x9509,0xe5ee,0xf5cf,0xc5ac,0xd58d,
	0x3653,0x2672,0x1611,0x0630,0x76d7,0x66f6,0x5695,0x46b4,
	0xb75b,0xa77a,0x9719,0x8738,0xf7df,0xe7fe,0xd79d,0xc7bc,
	0x48c4,0x58e5,0x6886,0x78a7,0x0840,0x1861,0x2802,0x3823,
	0xc9cc,0xd9ed,0xe98e,0xf9af,0x8948,0x9969,0xa90a,0xb92b,
	0x5af5,0x4ad4,0x7ab7,0x6a96,0x1a71,0x0a50,0x3a33,0x2a12,
	0xdbfd,0xcbdc,0xfbbf,0xeb9e,0x9b79,0x8b58,0xbb3b,0xab1a,
	0x6ca6,0x7c87,0x4ce4,0x5cc5,0x2c22,0x3c03,0x0c60,0x1c41,
	0xedae,0xfd8f,0xcdec,0xddcd,0xad2a,0xbd0b,0x8d68,0x9d49,
	0x7e97,0x6eb6,0x5ed5,0x4ef4,0x3e13,0x2e32,0x1e51,0x0e70,
	0xFF9f,0xefbe,0xdfdd,0xcffc,0xbf1b,0xaf3a,0x9f59,0x8f78,
	0x9188,0x81a9,0xb1ca,0xa1eb,0xd10c,0xc12d,0xf14e,0xe16f,
	0x1080,0x00a1,0x30c2,0x20e3,0x5004,0x4025,0x7046,0x6067,
	0x83b9,0x9398,0xa3fb,0xb3da,0xc33d,0xd31c,0xe37f,0xf35e,
	0x02b1,0x1290,0x22f3,0x32d2,0x4235,0x5214,0x6277,0x7256,
	0xb5ea,0xa5cb,0x95a8,0x8589,0xf56e,0xe54f,0xd52c,0xc50d,
	0x34e2,0x24c3,0x14a0,0x0481,0x7466,0x6447,0x5424,0x4405,
	0xa7db,0xb7fa,0x8799,0x97b8,0xe75f,0xf77e,0xc71d,0xd73c,
	0x26d3,0x36f2,0x0691,0x16b0,0x6657,0x7676,0x4615,0x5634,
	0xd94c,0xc96d,0xf90e,0xe92f,0x99c8,0x89e9,0xb98a,0xa9ab,
	0x5844,0x4865,0x7806,0x6827,0x18c0,0x08e1,0x3882,0x28a3,
	0xcb7d,0xdb5c,0xeb3f,0xfb1e,0x8bf9,0x9bd8,0xabbb,0xbb9a,
	0x4a75,0x5a54,0x6a37,0x7a16,0x0af1,0x1ad0,0x2ab3,0x3a92,
	0xfd2e,0xed0f,0xdd6c,0xcd4d,0xbdaa,0xad8b,0x9de8,0x8dc9,
	0x7c26,0x6c07,0x5c64,0x4c45,0x3ca2,0x2c83,0x1ce0,0x0cc1,
	0xef1f,0xFF3e,0xcf5d,0xdf7c,0xaf9b,0xbfba,0x8fd9,0x9ff8,
	0x6e17,0x7e36,0x4e55,0x5e74,0x2e93,0x3eb2,0x0ed1,0x1ef0
};

/*+-------------------------------------------------------------------------
  updcrc macro derived from article Copyright (C) 1986 Stephen Satchell. 
  NOTE: First argument must be in range 0 to 255.
        Second argument is referenced twice.
  Programmers may incorporate any or all code into their programs, giving
  proper credit within the source.  Publication of the source routines is
  permitted so long as proper credit is given to Stephen Satchell,
  Satchell Evaluations and Chuck Forsberg, Omen Technology.
--------------------------------------------------------------------------*/
#define crc_update(ch,crc) ( crctab[((crc >> 8) & 255)] ^ (crc << 8) ^ ch)

#else /* calculated crc */

/*+-------------------------------------------------------------------------
	crc_update(c,crc)
--------------------------------------------------------------------------*/
unsigned short
crc_update(c,crc)
register c;
register unsigned crc;
{
register count;

	for(count = 8; --count >= 0;)
	{
		if(crc & 0x8000)
		{
			crc <<= 1;
			crc += (((c <<= 1) & 0400) != 0);
			crc ^= 0x1021;
		}
		else 
		{
			crc <<= 1;
			crc += (((c <<= 1) & 0400) != 0);
		}
	}
	return(crc);
}	/* end of crc_update */
#endif /* crc calc selection */

/*+-------------------------------------------------------------------------
	rdchk(fd) - for systems without it but with FIONREAD
--------------------------------------------------------------------------*/
#if defined(sun) || defined(NO_RDCHK)
int
rdchk(fd)
int fd;
{
int chars_waiting;

	if(ioctl(fd,FIONREAD,&chars_waiting))
		return(0);
	else
		return(!!chars_waiting);
}	/* end of rdchk */
#endif

/*+-----------------------------------------------------------------------
	arg_token(parsestr,termchars)

Get next token from string parsestr ((char *)0 on 2nd, 3rd, etc.
calls), where tokens are nonempty strings separated by runs of chars
from termchars.  Writes nulls into parsestr to end tokens.
termchars need not remain constant from call to call.

Treats multiple occurrences of a termchar as one delimiter (does not
allow null fields).
------------------------------------------------------------------------*/
#if defined(M_UNIX)
static char *arg_token_static = (char *)0;
char *arg_token(parsestr,termchars)
char *parsestr;
char *termchars;
{
register char *parseptr;
char *token;

	if(parsestr == (char *)0 && arg_token_static == (char *)0)
		return((char *)0);

	if(parsestr)
		parseptr = parsestr;
	else
       parseptr = arg_token_static;

	while(*parseptr)
	{
		if(!strchr(termchars,*parseptr))
			break;
		parseptr++;
	}

	if(!*parseptr)
	{
		arg_token_static = (char *)0;
		return((char *)0);
	}

	token = parseptr;
	if(*token == '\'')
	{
		token++;
		parseptr++;
		while(*parseptr)
		{
			if(*parseptr == '\'')
			{
				arg_token_static = parseptr + 1;
				*parseptr = 0;
				return(token);
			}
			parseptr++;
		}
		arg_token_static = (char *)0;
		return(token);
	}
	while(*parseptr)
	{
		if(strchr(termchars,*parseptr))
		{
			*parseptr = 0;
			arg_token_static = parseptr + 1;
			while(*arg_token_static)
			{
				if(!strchr(termchars,*arg_token_static))
					break;
				arg_token_static++;
			}
			return(token);
		}
		parseptr++;
	}
	arg_token_static = (char *)0;
	return(token);
}	/* end of arg_token */
#endif

/*+-------------------------------------------------------------------------
	fname_split(cmd,arg,arg_max_quan,&narg)
--------------------------------------------------------------------------*/
#if defined(M_UNIX)
void
fname_split(cmd,arg,arg_max_quan,narg_rtn)
char *cmd;
char **arg;
int arg_max_quan;
int *narg_rtn;
{
register itmp;
register narg;

	for(itmp = 0; itmp < arg_max_quan; itmp++)
		arg[itmp] = (char *)0;
	arg[0] = arg_token(cmd,"/");

	for(narg = 1; narg < arg_max_quan; ++narg)
	{
		if((arg[narg] = arg_token((char *)0,"/")) == (char *)0) 
			break;
	}

	*narg_rtn = narg;

}	/* end of fname_split */
#endif

#if defined(M_UNIX)
#define MAX_COMPONENT_LEN	14
#define MAX_PATH_COMPONENTS	16
static char trunc_fname[257];
static char *trunc_components[MAX_PATH_COMPONENTS];
static int trunc_components_quan;
static int trunc_absolute_path;
#endif

/*+-------------------------------------------------------------------------
	fname_too_long(fname) - check for any pathname component too long
--------------------------------------------------------------------------*/
#if defined(M_UNIX)
int
fname_too_long(fname)
register char *fname;
{
register int itmp;
register char **cpptr;

	if(trunc_absolute_path = (*fname == '/'))
		fname++;
	strncpy(trunc_fname,fname,sizeof(trunc_fname) - 1);
	fname_split(trunc_fname,trunc_components,
		MAX_PATH_COMPONENTS,&trunc_components_quan);
	itmp = trunc_components_quan;
	cpptr = trunc_components;
	while(itmp--)
	{
		if(strlen(*cpptr) > MAX_COMPONENT_LEN)
			return(1);
		cpptr++;
	}
	return(0);
}	/* end of fname_too_long */
#endif

/*+-------------------------------------------------------------------------
	fname_truncated() - build truncated path last checked by fname_too_long
--------------------------------------------------------------------------*/
#if defined(M_UNIX)
char *
fname_truncated()
{
register int icomp;
char new_fname[257];
register char *cptr = new_fname;

	if(trunc_absolute_path)
	{
		*cptr = '/';
		*(cptr + 1) = 0;
	}
	else
		*cptr = 0;
	for(icomp = 0; icomp < trunc_components_quan; icomp++)
	{
		if(strlen(trunc_components[icomp]) > MAX_COMPONENT_LEN)
			*(trunc_components[icomp] + MAX_COMPONENT_LEN) = 0;
		strcat(cptr,trunc_components[icomp]);
		if(icomp < trunc_components_quan - 1)
			strcat(cptr,"/");
	}
	strcpy(trunc_fname,cptr);
	return(trunc_fname);

}	/* end of fname_truncated */
#endif

/*+-------------------------------------------------------------------------
	xmit_cancel()
--------------------------------------------------------------------------*/
void
xmit_cancel()
{
char *cancel_msg = "\030\030\030\030\030\030\030\030\b\b\b\b\b\b\b\b";

	ioctl(iofd,TCFLSH,(char *)1);
	write(iofd,cancel_msg,16);
	tx_char_count += 16;
	report_str("CANCELling transfer",1);
	report_last_txhdr("CAN",0);

}	/* end of xmit_cancel */

/*+-------------------------------------------------------------------------
	xmit_ack(blknum)
--------------------------------------------------------------------------*/
void
xmit_ack(blknum)
register int blknum;			/* block number */
{
char s16[16];

	sprintf(s16,"ACK %3d",blknum);
	report_last_txhdr(s16,0);

	s16[0] = ACK;
	s16[1] = blknum;			/* block number */
	s16[2] = blknum ^ 0xFF;	/* block number check */
	write(iofd,s16,3);
	tx_char_count += 3;
}	/* end of xmit_ack */

/*+-------------------------------------------------------------------------
	xmit_nak(blknum)
--------------------------------------------------------------------------*/
void
xmit_nak(blknum)
register int blknum;			/* block number */
{
char s16[16];

	sprintf(s16,"NAK %d",blknum);
	report_last_txhdr(s16,1);

	if(crc_in_use)
		s16[0] = 'C';
	else
		s16[0] = NAK;

	s16[1] = blknum;			/* block number */
	s16[2] = blknum ^ 0xFF;	/* block number check */
	write(iofd,s16,3);
	tx_char_count += 3;

}	/* end of xmit_nak */

/*+-------------------------------------------------------------------------
	lgetc_timeout_SIGALRM() - called when alarm is caught by lgetc_timeout
--------------------------------------------------------------------------*/
#if defined(NO_SELECT)
void
lgetc_timeout_SIGALRM(sig)
int sig;
{
	longjmp(lgetc_timeout_setjmp,TIMEOUT);
}	/* end of lgetc_timeout_SIGALRM */
#endif

/*+-------------------------------------------------------------------------
	lgetc_timeout(tenths)

 reads one character from line unless timeout in tenths passes
 with no receipt.
--------------------------------------------------------------------------*/
unsigned int
lgetc_timeout(tenths)
int tenths;
{
#if defined(NO_SELECT)
unsigned char rdchar;
long msec;
int seconds;
long Nap();
#else
int fdmask;
struct timeval tval;
unsigned char rdchar;
#endif

	if(!tenths)
	{
		if(!rdchk(iofd))
			return(TIMEOUT);
		else
		{
			read(iofd,&rdchar,1);
			rx_char_count++;
			return((unsigned int)rdchar);
		}
	}

#if defined(NO_SELECT)

/* there is a timeout ... if less than 2 secs, nap it out */
	if(tenths <= 20)
	{
		msec = (tenths < 6) ? 60L : (long)tenths * 10;
		while(msec)
		{
			msec -= Nap(20L);
			if(rdchk(iofd))
			{
				read(iofd,&rdchar,1);
				rx_char_count++;
				return((unsigned int)rdchar);
			}
		}
		report_last_rxhdr("TIMEOUT",0);
		return(TIMEOUT);
	}

/* timeout is > 2 seconds use sleep */

	seconds = (tenths / 10) + 1;

	if(setjmp(lgetc_timeout_setjmp))
	{
		report_last_rxhdr("TIMEOUT",0);
		return(TIMEOUT);
	}

	signal(SIGALRM,lgetc_timeout_SIGALRM);
	alarm(seconds);
	while(read(iofd,&rdchar,1) != 1)
		;
	alarm(0);
	signal(SIGALRM,SIG_DFL);

#else

	if(tenths < 6)
		tenths = 6;
	tval.tv_sec = tenths / 10L;
	tval.tv_usec = (tenths % 10L) * 100000L;
	fdmask = 1 << iofd;
	if(select(32,&fdmask,(int *)0,(int *)0,&tval) != 1)
	{
		report_last_rxhdr("TIMEOUT",0);
		return(TIMEOUT);
	}
	if((!rdchk(iofd)) || (read(iofd,&rdchar,1) < 0))
	{
		report_last_rxhdr("TIMEOUT",0);
		return(TIMEOUT);
	}

#endif

	rx_char_count++;
	return((unsigned int)rdchar);

}	/* end of lgetc_timeout */

/*+-------------------------------------------------------------------------
	sf_state_text(state)
--------------------------------------------------------------------------*/
char *
sf_state_text(state)
register state;
{
char unrecog[16];

	switch(state)
	{
		case SFS_GND:	return("GND");
		case SFS_ACK:	return("ACK");
		case SFS_NAK:	return("NAK");
		case SFS_ACKW:	return("ACKW");
		case SFS_NAKW:	return("NAKW");
		case SFS_RGND:	return("RGND");
		default:
			sprintf(unrecog,"SFS_%d",state);
			return(unrecog);
	}

}	/* end of sf_state_text */

/*+-------------------------------------------------------------------------
	set_sf_state(place,new_state)
--------------------------------------------------------------------------*/
void
set_sf_state(place,new_state)
int place;
int new_state;
{
	if(log_packets)
	{
		sprintf(s128,"state from %s to %s (%d)",
			sf_state_text(sf_state),sf_state_text(new_state),place);
		report_str(s128,0);
	}
	sf_state = new_state;
}	/* end of set_sf_state */

/*+-------------------------------------------------------------------------
	wait_for_rcvr_response() - check for ACK or NAK
 sets 'sf_state' to SFS_... value depending on response from file rcvr
 returns 1 if TIMEOUT at state other than ground, else 0
--------------------------------------------------------------------------*/
int
wait_for_rcvr_response()
{
int c;						/* one byte of data */
static int rawblk = 0;		/* raw block number */

	while((c = lgetc_timeout((sf_state == SFS_GND) ? 0 : 6)) != TIMEOUT)
	{
		if(c == CAN)
		{									/* CANcel received? */
			if((c = lgetc_timeout(20)) == CAN)
			{
				sf_nakquan = 11;
				report_last_rxhdr("CAN",0);	/* error counted at cancel time */
			}
			break;
		}
		if(sf_state == SFS_ACKW || sf_state == SFS_NAKW)	/* windowed */
		{
			sf_slide = 0;						/* assume this will fail */
			/* see if we believe the number */
			if(rawblk == (c ^ 0xFF))
			{
				rawblk = sf_blknum - ((sf_blknum - rawblk) & 0xFF);
				if((rawblk >= 0) && (rawblk <= sf_blknum) &&
					(rawblk > (sf_blknum - 128)))
				{				/* we have sliding window! */
					if(sf_state == SFS_ACKW)
					{
						sf_ackblk = (sf_ackblk > rawblk) ? sf_ackblk : rawblk;
						sf_slide = 1;
						if(no_ack_mode && (++sf_ackw_count > 10))
						{
							no_ack_mode = 0;
							report_str("Overdrive disengaged",0);
						}
					}
					else 
					{
						sf_blknum = (rawblk < 0) ? 0 : rawblk;
						sf_slide = (sf_nakquan < 4);
					}
					sprintf(s128,"%s %5d",
						(sf_state == SFS_ACKW) ? "ACKW" : "NAKW",rawblk);
					report_last_rxhdr(s128,(sf_state != SFS_ACKW) && rawblk);
				}
			}
			set_sf_state(1,SFS_RGND);	/* return to ground state */
		}

		if(sf_state == SFS_ACK || sf_state == SFS_NAK)
		{
			rawblk = c;
			if(sf_state == SFS_ACK)
				set_sf_state(2,SFS_ACKW);
			else
				set_sf_state(3,SFS_NAKW);
		}

		if(!sf_slide || sf_state == SFS_GND)
		{
			if(c == ACK)
			{
				if(!sf_slide)
				{
					sprintf(s128,"ACK %3d",sf_ackblk);
					report_last_rxhdr(s128,0);
					sf_ackblk++;
				}
				set_sf_state(4,SFS_ACK);
				sf_nakquan = 0;
			}
			else if(c == 'C' || c == NAK)
			{
				/* if method not determined yet */
				if(crc_in_use > 1)	/* then do what rcvr wants */
				{
					crc_in_use = (c == 'C');
					report_protocol_crc_type(crc_in_use ? "/CRC16" : "/CHK");
				}
				ioctl(iofd,TCFLSH,(char *)1);
				if(!sf_slide)
				{
					sf_blknum = sf_ackblk + 1;
					sprintf(s128,"NAK %3d",sf_blknum);
					report_last_rxhdr(s128,(!!sf_blknum));
				}
				set_sf_state(5,SFS_NAK);
				sf_nakquan++;
				if(sf_lastnum)
					error_count++;
			}
		}

		if(sf_state == SFS_RGND)
			set_sf_state(6,SFS_GND);
	}
	return((sf_state != SFS_GND) && (c == TIMEOUT));
}	/* end of wait_for_rcvr_response */

/*+-------------------------------------------------------------------------
	send_comm_block(blk,blknum) - format and transmit block
--------------------------------------------------------------------------*/
int
send_comm_block(blk,blknum)
char *blk;				/* data to be shipped */
int blknum;				/* number of block */
{
register unsigned short rUINT16 = 0;
register int itmp;
unsigned char chksum;
char *cptr = blk;
char s3[3];

	s3[0] = SOH;				/* block header */
	s3[1] = blknum;				/* block number */
	s3[2] = blknum ^ 0xFF;		/* block number check value */

/* calculate the crc or checksum */
	itmp = 128;
	if(crc_in_use)
	{
		while(itmp--)
		{
			rUINT16 = crc_update(*cptr,rUINT16);
			cptr++;
		}
		rUINT16 = crc_update(0,rUINT16);
		rUINT16 = crc_update(0,rUINT16);
	}
	else 
	{
		while(itmp--)
			rUINT16 += *cptr++;
	}

/* write the block */

	write(iofd,s3,3);						/* the header */
	write(iofd,blk,128);					/* the block */
	if(crc_in_use)							/* the crc or checksum */
	{
		s3[0] = rUINT16 >> 8;
		s3[1] = rUINT16 & 0xFF;
		write(iofd,s3,2);
		tx_char_count += 133;
	}
	else
	{
		chksum = rUINT16;
		write(iofd,&chksum,1);
		tx_char_count += 132;
	}

	return(1);
}	/* end of send_comm_block */

/*+-------------------------------------------------------------------------
	send_file_block(fp,blknum) - read a block from file and send it
--------------------------------------------------------------------------*/
void
send_file_block(fp,blknum)
FILE *fp;
int blknum;
{
long fileaddr;
char buf[128];

	fileaddr = (long)(blknum - 1) * 128L;
	if(blknum != sf_lastnum + 1)
		fseek(fp,fileaddr,0);	/* move where to */
	sf_lastnum = blknum;
	report_txpos(fileaddr);

	memset(buf,0x1A,sizeof(buf));	/* fill buffer with control Zs */
	fread(buf,1,sizeof(buf),fp);	/* read in some data */
	send_comm_block(buf,blknum);	/* pump it out to the receiver */
}	/* end of send_file_block */

/*+-------------------------------------------------------------------------
	send_file(name) - transmit a file
--------------------------------------------------------------------------*/
int
send_file(name)
char *name;
{
register int endblk;	/* block number of EOT */
FILE *fp = (FILE *)0;	/* file to send */
struct stat fst;
BLK0 blk0;
char *basename;			/* base filename */
char eot = EOT;

	Filcnt++;
	if(name && *name)			/* if sending a file */
	{
		if((fp = fopen(name,"r")) == NULL)
		{
			sprintf(s128,"Cannot open %s",name);
			report_str(s128,1);
			exit_code = 253;
			return(0);
		}

		memset((char *)&blk0,0,sizeof(blk0)); /* clear out data block */

		stat(name,&fst);	/* get file statistics */
		blk0.length = (long)fst.st_size;

		/* cnvt time from 1970 base to 1980 */
		if((blk0.secs_since_1980 = fst.st_mtime-OFFSET_1980) < 0L)
			blk0.secs_since_1980 = 0;

		if((basename = strrchr(name,'/')) == NULL) /* find basename */
			strcpy(blk0.filename,name);
		else 
		{
			basename++;
			strcpy(blk0.filename,basename);
		}

		strcpy(blk0.sender,"ecusea ");
		strcat(blk0.sender,revision);
		blk0.send_no_acks = no_ack_mode;

		endblk = (int)((blk0.length + 127L) / 128L) + 1;
		report_file_send_open(name,&fst);
	}
	else 
	{
		endblk = 0;						/* fake for no file */
		report_str("sending EOT indication",-1);
		report_txpos(blk0.length);
	}


	sf_blknum = 1;						/* set starting state */
	sf_ackblk = -1;
	sf_state = SFS_GND;
	sf_lastnum = 0;
	sf_slide = 0;
	sf_nakquan = 0;
	error_count = 0;
	sf_ackw_count = 0;
	crc_in_use = 2;						/* undetermined */

	while(sf_ackblk < endblk)			/* while not all there yet */
	{
		sent_EOT = 0;
		if(sf_blknum <= sf_ackblk + ((sf_slide && allow_slide) ? WINDOW : 1))
		{
			if(sf_blknum < endblk)
			{
				if(sf_blknum > 0)
				{
					sprintf(s128,"sending block %d",sf_blknum);
					report_last_txhdr(s128,0);
					send_file_block(fp,sf_blknum);
				}
				else
				{
					sprintf(s128,"sending filename",sf_blknum);
					report_last_txhdr(s128,0);
					send_comm_block((char *)&blk0,0);
					report_txpos(0L);
				}
				if(no_ack_mode && sf_slide && allow_slide)
					sf_ackblk = sf_blknum;
			}
			else if(sf_blknum == endblk)
			{
				report_last_txhdr("EOT",0);
				write(iofd,&eot,1);
				sent_EOT = 1;
				Nap(500L);
				tx_char_count++;
			}
			sf_blknum++;
		}

		if(wait_for_rcvr_response() && sent_EOT)
		{
			report_str("Receiver did not ACK our EOT",-1);
			break;
		}

		if(sf_nakquan > 10)
			goto CANCEL_TRANSFER;
	}

	if(endblk)	/* if sending file, not EOT */
		fclose(fp);
	return(1);							/* exit with good status */

CANCEL_TRANSFER:
	if(endblk)	/* if sending file, not EOT */
		fclose(fp);
	xmit_cancel();
	return(0);                          /* exit with bad status */
}	/* end of send_file */

/*+-------------------------------------------------------------------------
	set_utime_1980(filename,secs_since_1980)
--------------------------------------------------------------------------*/
void
set_utime_1980(filename,secs_since_1980)
char *filename;						/* file to set stamp on */
long secs_since_1980;
{
time_t times[2];
time_t time();

	times[0] = time((long *) 0);				/* accessed */
	times[1] = secs_since_1980 + OFFSET_1980;	/* modified (convert time) */
	utime(filename,times);
}	/* end of set_utime_1980 */

/*+-------------------------------------------------------------------------
	receive_block(buf) - get block from line
return 0 if good chk/CRC, 1 if bad
--------------------------------------------------------------------------*/
int
receive_block(buf)
char *buf;				/* data buffer */
{
register unsigned int rdchar;	
register unsigned short rUINT16 = 0;	/* calculated CRC or check value */
int itmp;
int timeout = no_ack_mode ? 200 : 5;	/* short block timeout */
unsigned short rcvd_crc;				/* received CRC or check value */

	itmp = 128;
	while(itmp--)
	{
		if((rdchar = lgetc_timeout(timeout)) == TIMEOUT)
			return(1);
		if(crc_in_use)
			rUINT16 = crc_update(rdchar,rUINT16);
		else
			rUINT16 += rdchar;
		*buf++ = rdchar;
	}

	if(crc_in_use)
	{
		rUINT16 = crc_update(0,rUINT16);
		rUINT16 = crc_update(0,rUINT16);
		rdchar = lgetc_timeout(timeout);
		rcvd_crc = (rdchar << 8) | lgetc_timeout(timeout);
	}
	else 
	{
		rUINT16 &= 0xFF;
		rcvd_crc = lgetc_timeout(timeout) & 0xFF;
	}

	if(rUINT16 != rcvd_crc)
	{
		sprintf(s128,"bad %s calc=%04x rcvd=%04x",
			crc_in_use ? "CRC" : "checksum",rcvd_crc,rUINT16);
		report_str(s128,-1);
	}
	return(rUINT16 != rcvd_crc);
}	/* end of receive_block */

/*+-------------------------------------------------------------------------
	receive_file()
--------------------------------------------------------------------------*/
char *
receive_file()
{
int rdchar;					/* received character */
int tries;					/* retry counter */
int blknum;					/* desired block number */
int inblk;					/* this block number */
FILE *fp;
char buf[128];				/* data buffer */
char tmpname[100];			/* name of temporary file */
static char outname[100];	/* name of final file */
BLK0 blk0;					/* file header data storage */
int endblk;					/* block number of EOT, if known */
long left = 0;				/* bytes left to output */
int itmp;					/* index */
int cnvrt;					/* flag -- convert filename? */
char *onp;					/* use to convert filename to l / rdchar */
long ftell();

	*outname = '\0';		/* get name from transmitter */
	cnvrt = 1;		/* convert to local is necessary */
	sprintf(tmpname,"./SEA%05d.tmp",getpid());	/* use a unique temp filename */

	if(!(fp = fopen(tmpname,"w")))
	{	/* open temporary file */
		sprintf(s128,"Cannot create temp file %s\n",tmpname);
		report_str(s128,0);
		xmit_cancel();
		return(NULL);
	}

	blknum = 0;
	tries = -10;				/* kludge for first time around */
	crc_in_use = 1;				/* try for CRC error checking */
	error_count = 0;			/* no errors yet */
	endblk = 0;					/* we don't know the size yet */
	no_ack_mode = 0;			/* we don't know about this yet */
	memset((char *)&blk0,0,sizeof(blk0));	/* or much of anything else */
	report_protocol_crc_type("/CRC16");

SEND_NAK:				/* we got a bad block */
	if(blknum > 1)
	{
		error_count++;
		report_str("bad block",1);
	}
	if(++tries > 10)
		goto CANCEL_TRANSFER;
	if(tries == 0)			/* if CRC isn't going */
	{
		crc_in_use = 0;		/* then give checksum a try */
		report_protocol_crc_type("/CHK");
	}

	xmit_nak(blknum);		/* send the NAK */
	if(no_ack_mode && error_count > 20)
	{	/* if no_ack_mode mode isn't working */
		no_ack_mode = 0;		/* then shut it off */
		report_str("Overdrive disengaged",0);
	}

RECEIVE_NEXT_BLOCK:				/* start of "get a block" */
	report_rxpos(ftell(fp));
	while((rdchar = lgetc_timeout(30)) != TIMEOUT)
	{
		if(rdchar == CAN)
		{
			if((rdchar = lgetc_timeout(30)) == CAN)
			{
				xmit_cancel();
				return(NULL);
			}
			break;
		}
		if(rdchar == EOT)
		{
			if(!endblk || endblk == blknum)
				goto RECEIVE_EOT_SEEN;
		}
		else if(rdchar == SOH)
		{
			if((inblk = lgetc_timeout(5)) == TIMEOUT)
				goto SEND_NAK;
			if(lgetc_timeout(5) == (inblk ^ 0xFF))
			{
				sprintf(s128,"receiving %d",inblk);
				report_last_rxhdr(s128,0);
				goto GOT_START_OF_BLOCK;	/* we found a start */
			}
		}
	}
	goto SEND_NAK;

GOT_START_OF_BLOCK:				/* start of block detected */
	rdchar = blknum & 0xFF;
	if(inblk == 0 && blknum <= 1)
	{	/* if this is the header */
		if(receive_block((char *)&blk0))
			goto SEND_NAK;		/* bad header block */
		else 
		{
			xmit_ack(inblk);	/* ack the header */

#if defined(M_UNIX)
			if(fname_too_long(blk0.filename))
			{
				strcpy(s128,"truncated: ");
				strncat(s128,blk0.filename,sizeof(s128) - 12);
				report_str(s128,-1);
				strcpy(outname,fname_truncated());
			}
			else
#endif
				strcpy(outname,blk0.filename);
			report_file_rcv_started(outname,blk0.length,
				blk0.secs_since_1980 + OFFSET_1980);
			if(left = blk0.length)	/* length to transfer */
				endblk=(int)((left + 127L)/128L)+1;
			if(no_ack_mode != blk0.send_no_acks)
			{
				sprintf(s128,"Overdrive %sengaged",
					(blk0.send_no_acks) ? "" : "dis");
				report_str(s128,0);
			}
			no_ack_mode = blk0.send_no_acks;
			blknum = 1;	/* now we want first data block */
			goto RECEIVE_NEXT_BLOCK;
		}
	}

	if(inblk == rdchar)
	{			/* if this is the one we want */
		if(!receive_block(buf))
		{		/* else if we get it okay */
			if(!no_ack_mode)		/* if we're sending ACKs */
				xmit_ack(inblk);	/* then ACK the data */
			for(itmp = 0; itmp < 128; itmp++)
			{
				if(endblk)
				{	/* limit file size if known */
					if(!left)
						break;
					left--;
				}
				if(fputc(buf[itmp],fp) == EOF)
				{
					report_str("FILE WRITE ERROR",0);
					goto CANCEL_TRANSFER;
				}
			}
			tries = 0;		/* reset try count */
			blknum++;		/* we want the next block */
			goto RECEIVE_NEXT_BLOCK;
		}
		goto SEND_NAK;		/* ask for a resend */
	}

	if(inblk < rdchar || inblk > rdchar + 100)
	{	/* if resending what we have */
		receive_block(buf);			/* ignore it */
		xmit_ack(inblk);			/* but ack it */
	}
	goto RECEIVE_NEXT_BLOCK;		/* else if running ahead */

RECEIVE_EOT_SEEN:
#ifdef NAKEOT
	xmit_nak(blknum);				/* NAK the EOT, make sure */
	if(lgetc_timeout(20) != EOT)	/* we're all done */
		goto SEND_NAK;
#endif /* NAKEOT */
	xmit_ack(blknum);				/* ACK it and clean up */
	report_last_rxhdr("EOT",0);
	if(blknum > 1)
	{				/* if we really got anything */
		fclose(fp);
		unlink(outname);		/* rename temp to proper name */
		for(onp = outname;cnvrt && *onp;onp++)
			/* find out if there's lower- */
			if(islower(*onp))	/* case letters filename */
				cnvrt = 0;	/*  there are, don't convert */
		if(cnvrt)			/* if there aren't, make all */
			for(onp = outname;*onp;onp++)	/* into uppercase */
				*onp = tolower(*onp);
		if(link(tmpname,outname) == 0)
			unlink(tmpname);
		if(blk0.secs_since_1980)		/* set stamp, if known */
			set_utime_1980(outname,blk0.secs_since_1980);
		return(outname);
	}
	else 
	{				/* else no real file */
		fclose(fp);
		unlink(tmpname);		/* discard empty file */
		report_str("end of transfer",0);
		rf_done = 1;
		return(NULL);
	}

CANCEL_TRANSFER:
	fclose(fp);
	xmit_cancel();
	rf_done = 2;
	return(NULL);
}	/* end of receive_file */

/*+-------------------------------------------------------------------------
	cancel_transaction(sig)
--------------------------------------------------------------------------*/
SIGTYPE
cancel_transaction(sig)
int sig;
{
	xmit_cancel();
	sprintf(s128,"signal %d ... exiting",sig);
	report_str(s128,1);
/*
	report_rx_ind(0);
	report_tx_ind(0);
*/
	report_uninit();
	if(sig == SIGQUIT)
		abort();
	exit(128+sig);
}	/* end of cancel_transaction */

/*+-------------------------------------------------------------------------
	getspeed(code)
--------------------------------------------------------------------------*/
struct B_to_baud { unsigned baud; int B_code; };
unsigned
getspeed(code)
int code;
{
register itmp;
static struct B_to_baud speeds[] = 
{
 50, B50, 75, B75, 110, B110, 300, B300, 600, B600, 1200, B1200,
 2400, B2400, 4800, B4800, 9600, B9600, 19200, EXTA, 38400, EXTB, 0
};

	code &= CBAUD;
	for(itmp = 0; speeds[itmp].baud; itmp++)
		if(speeds[itmp].B_code == code)
			return(speeds[itmp].baud);
	return(38400);	/* Assume fifo if ioctl failed */
}	/* end of getspeed */

/*+-------------------------------------------------------------------------
	main(argc,argv,envp)
--------------------------------------------------------------------------*/
main(argc,argv,envp)
int argc;
char **argv;
char **envp;
{
int ipaths;
int ok = 0;
#define MAX_PATHS 512
char *paths[MAX_PATHS];
char **ppaths = paths;
char *cptr;
char **gargv = argv;
int gargc = argc;

	exit_code = 254;
	while(--argc)
	{
		cptr = *++argv;
		if(*cptr == '-')
		{
			cptr++;
			switch(*cptr++)
			{
			case ',':
				log_packets = 1;
				break;
			case '/':
				if(--argc < 1)
					exit(255);
				strcpy(curr_dir,*++argv);
				break;
			case '.':
				if(--argc < 1)
					exit(255);
				iofd = atoi(*++argv);
				break;
			case 'r':
				sending_flag = 0;
				break;
			case 's':
				sending_flag = 1;
			}
		}
		else if(argc > 0)
		{
			if(npaths < MAX_PATHS)
			{
				*ppaths++ = cptr;
				npaths++;
			}
			else
			{
				printf("too many filenames to send\n");
				exit(255);
			}
		}
	}

	if(sending_flag == -1)
	{
		printf("no -r or -s\n");
		exit(255);
	}

	if((npaths < 1) && sending_flag)
		exit(253);

	if(npaths && !sending_flag)
		exit(255);

	if(log_packets)
	{
		char log_packets_name[64];
		FILE *ftmp;
		int iargv;
		sprintf(log_packets_name,"/tmp/sea%05d.plog",getpid());
		unlink(log_packets_name);
		ftmp = fopen(log_packets_name,"w");
		fclose(ftmp);
		log_packets = open(log_packets_name,O_WRONLY,0644);
		if(log_packets < 0)
			log_packets = 0;
		else
		{
			write(log_packets,"exec: ",6);
			for(iargv = 0; iargv < gargc; iargv++)
			{
				write(log_packets,gargv[iargv],strlen(gargv[iargv]));
				write(log_packets," ",1);
			}
			write(log_packets,"\n",1);
		}
	}

	sprintf(s128,"ecusea %s",revision);
	report_init(s128);
	report_top_line("System Enhancement Associates");
	signal(SIGHUP,cancel_transaction);
	signal(SIGQUIT,cancel_transaction);
	signal(SIGINT,cancel_transaction);
	signal(SIGTERM,cancel_transaction);
#if	defined(SIGSTOP)
	/*
	 * call Roto-Rooter on POSIX plots
	 */
	signal(SIGSTOP,SIG_IGN);
	signal(SIGTSTP,SIG_IGN);
	signal(SIGCONT,SIG_IGN);
	signal(SIGTTIN,SIG_IGN);
	signal(SIGTTOU,SIG_IGN);
#endif

	ioctl(iofd,TCGETA,&tio0);
	tio = tio0;

	tio.c_oflag = 0;
	tio.c_cflag &= ~PARENB;
	tio.c_cflag &= ~CSIZE;
	tio.c_cflag |= CS8;

	/*
	 * learn tick rate for various timers
	 */
	init_Nap();

	baud_rate = getspeed(tio.c_cflag);
	ioctl(iofd,TCSETA,&tio);
	report_line(baud_rate,"RAW");

	switch(sending_flag)
	{
		case 0:				/* receive files */
			while(receive_file() != NULL)
				Nap(1000L);
			ok = (rf_done == 1);
			break;

		case 1:				/* send files */
			ipaths = 0;
			while(ipaths < npaths)
			{
				if(!(ok = send_file(paths[ipaths])))
					break;
				Nap(1000L);
				ipaths++;
			}
			if(ok)		/* no errors, send end marker */
				send_file("");
			report_str("end of transfer",0);
			break;
	}

	ioctl(iofd,TCSETA,&tio0);
	report_line(baud_rate,"NORMAL");
	report_uninit();
	exit(ok ? 0 : exit_code);	/* and return error status */

}	/* end of main */

