/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 *  RCS: $Header: global.h,v 2.3 94/10/29 17:31:32 nau Exp $
 */

/* definition of types
 */

#ifndef	__GLOBAL_INCLUDED__
#define	__GLOBAL_INCLUDED__

#include "const.h"
#include "macro.h"

#include <X11/Xmd.h>
#include <X11/Xlib.h>
#include <X11/Xresource.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/keysym.h>

/* ---------------------------------------------------------------------------
 * some useful values of our widgets
 */
typedef struct						/* holds information about output window */
{
	Widget			Toplevel,		/* toplevel widget */
					StatusLine,		/* label widget of statusline */
					CursorPosition,	/* current cursor position */
					ScrollbarLeft,	/* viewport scrollbars */
					ScrollbarBottom,
					Control,		/* control widget */
					Menu,			/* popup menue */
					Message,		/* fields for user message */
					MasterForm,		/* the widgets thats hols all the others */
					Viewport,		/* viewport widget to scoll output */
					Output;			/* simple widget for drawing */
	Window			OutputWindow;	/* window ID of simple widget */
	GC				bgGC,			/* background and foreground; */
					fgGC;			/* changed from some routines */
	Position		OffsetX,		/* origin offset */
					OffsetY;
	Dimension		Width,			/* sizes of output window */
					Height;
	Cursor			XCursor;		/* used X cursor */
	unsigned int	XCursorShape;	/* and its shape (cursorfont) */
} OutputType, *OutputTypePtr;

/* ----------------------------------------------------------------------
 * layer group. A layer group identifies layers which are always switched
 * on/off together.
 */
typedef struct
{
	Cardinal		Number[MAX_LAYER],		/* number of entries per groups */
					Entries[MAX_LAYER][MAX_LAYER];
} LayerGroupType, *LayerGroupTypePtr;

typedef struct						/* a bounding box */
{
	Position		X1, Y1,			/* upper left */
					X2, Y2;			/* and lower right corner */
} BoxType, *BoxTypePtr;

/* ---------------------------------------------------------------------------
 * the basic type supported by PCB
 * there are only four combination of coordinates supported:
 * line straight up, right and up, straight right and down and right
 * check 'draw.c' and 'create.c' for details
 */
typedef struct						/* all objects start with an ID */
{
	int				ID;
} AnyObjectType, *AnyObjectTypePtr;

typedef struct						/* holds information about one line */
{
	int				ID,
					Flags;
	Position		X1,				/* start and endposition, */
					Y1,
					X2,
					Y2;
	Dimension		Thickness;		/* thickness */
} LineType, *LineTypePtr;

typedef struct
{
	int				ID,
					Flags;
	Position		X,				/* origin */
					Y;
	BoxType			BoundingBox;
	BYTE			Direction;
	Dimension		Scale;			/* text scaling in percent */
	char			*TextString;	/* string */
} TextType, *TextTypePtr;

typedef struct						/* a polygon point */
{
	int				ID;
	Position		X,
					Y;
} PolygonPointType, *PolygonPointTypePtr;

typedef struct						/* holds information about a polygon */
{
	int					ID,
						Flags;
	BoxType				BoundingBox;
	Cardinal			PointN,		/* number of points in polygon */
						PointMax;	/* max number from malloc() */
	PolygonPointTypePtr	Points;		/* data */
} PolygonType, *PolygonTypePtr;

typedef struct						/* holds information about one layer */
{
	char			*Name;			/* layer name */
	Cardinal		LineN,			/* number of lines */
					TextN,			/* labels */
					PolygonN,		/* and polygons */
					LineMax,		/* max number from malloc() */
					TextMax,
					PolygonMax;
	LineTypePtr		Line;			/* pointer to additional structures */
	TextTypePtr		Text;
	PolygonTypePtr	Polygon;
	Boolean			On;				/* visible flag */
	Pixel			Color;			/* color */
} LayerType, *LayerTypePtr;

typedef struct
{
	int				ID,
					Flags;
	Position		X,				/* start-, endposition and thickness */
					Y;
	Dimension		Thickness,
					DrillingHole;
	char			*Name;				
} PinType, *PinTypePtr;

typedef struct						/* used for arcs */
{
	Position		X,				/* center coordinates */
					Y;
	Dimension		Width,			/* length of axis */
					Height,
					Thickness;
	int				StartAngle,		/* the two limiting angles in degrees */
					Delta;	
} ArcType, *ArcTypePtr;

typedef struct
{
	int				ID,
					Flags;
	TextType		Name[2];		/* the elements names; canonical name */
									/* first, name on PCB second; see macro.h */
	Position		MarkX,			/* position mark */
					MarkY;
	BoxType			BoundingBox;
	Cardinal		PinN,			/* number of pins, lines and arcs */
					PinMax,
					LineN,
					LineMax,
					ArcN,
					ArcMax;
	PinTypePtr		Pin;			/* pointer to pin description */
	LineTypePtr		Line;
	ArcTypePtr		Arc;
} ElementType, *ElementTypePtr;

/* ---------------------------------------------------------------------------
 * symbol and font related stuff
 */
typedef struct						/* a single symbol */
{
	LineTypePtr		Line;
	Boolean			Valid;
	Cardinal		LineN,			/* number of lines */
					LineMax;
	Dimension		Width,			/* size of cell */
					Height,
					Delta;			/* distance to next symbol in 0.001'' */
} SymbolType, *SymbolTypePtr;

typedef struct						/* complete set of symbols */
{
	Dimension		MaxHeight,		/* maximum cell width and height */
					MaxWidth;
	BoxType			DefaultSymbol;	/* the default symbol is a filled box */
	SymbolType		Symbol[MAX_FONTPOSITION+1];
	Boolean			Valid;
} FontType, *FontTypePtr;

typedef struct				/* holds all objects */
{
	Cardinal		ViaN,			/* number of vias */
					ViaMax,			/* max number from malloc() */
					ElementN,		/* and elements */
					ElementMax;		/* max number from malloc() */
	PinTypePtr		Via;			/* pointer to object data */
	ElementTypePtr	Element;
	LayerType		Layer[MAX_LAYER];
} DataType, *DataTypePtr;

typedef struct				/* holds information about board layout */
{							/* most of the information is saved with layout */
							/* and initialized from resources when */
							/* a new board design is started */
							/* this struct is also used for the remove list */
							/* and for buffer handling */
	int				ID,				/* see macro.h */
					Flags;
	char			*Name,			/* name of board */
					*Filename;		/* name of file (from load) */
	Boolean			Changed,		/* layout has been changed */
					ViaOn,			/* visibility flags */
					ElementOn,
					PinOn;
	Pixel			ViaColor,		/* some colors */
					PinColor,
					ConnectedColor,
					SelectedColor,
					ElementColor;
	int				Zoom,			/* zoom factor (factor = 1 << zoom) */
					CursorX,		/* cursor position as saved with layout */
					CursorY,
					Grid,			/* used grid with offsets */
					GridOffsetX,	/* as saves with layout */
					GridOffsetY;
	Dimension		MaxWidth,		/* allowed size */
					MaxHeight;
	FontType		Font;
	LayerGroupType	LayerGroups;
	Pixmap			SaveUnder;		/* saves data during dialog handling */
	DataTypePtr		Data;
} PCBType, *PCBTypePtr;

typedef struct						/* information about the paste buffer */
{
	Position		X,				/* offset */
					Y;
	BoxType			BoundingBox;
	DataTypePtr		Data;			/* data; not all members of PCBType */
									/* are used */
} BufferType, *BufferTypePtr;

/* ---------------------------------------------------------------------------
 * some types for cursor drawing, setting of block and lines
 * as well as for merging of elements
 */
typedef struct						/* current marked line */
{
	Position		X1,				/* start- and endposition */
					Y1,
					X2,
					Y2;
	int				State;
} AttachedLineType, *AttachedLineTypePtr;

typedef struct						/* currently marked block */
{
	Position 		X1,				/* upper left */
					Y1,
					X2,				/* and lower right corner */
					Y2;
	int				State;
} AttachedBoxType, *AttachedBoxTypePtr;

typedef struct						/* currently attached object */
{
	Position		X,				/* saved position when MOVE_MODE */
					Y;				/* was entered */
	BoxType			BoundingBox;
	int				Type,			/* object type */
					State;
	void			*Ptr1,			/* two pointers to data, see */
					*Ptr2;			/* search.c */
} AttachedObjectType, *AttachedObjectTypePtr;

typedef struct						/* holds cursor information */
{
	GC					GC,			/* GC for cursor drawing */
						AttachGC;	/* and for displaying buffer contents */
	Position			X,			/* position in PCB coordinates */
						Y,
						MinX,		/* lowest and highest coordinates */
						MinY,
						MaxX,
						MaxY;
	Boolean				On;				/* flag for 'is visible' */
	AttachedLineType	AttachedLine;	/* data of new lines... */
	AttachedBoxType		AttachedBox;
	PolygonType			AttachedPolygon;
	AttachedObjectType	AttachedObject;	/* data of attached objects */
} CrosshairType, *CrosshairTypePtr;

/* ---------------------------------------------------------------------------
 * our resources
 * most of them is used as default when a new design is started
 */
typedef struct						/* some resources... */
{
	Pixel			bgColor,		/* background and cursor color ... */
					CrosshairColor,
					ViaColor,		/* different object colors */
					PinColor,
					ElementColor,
					ConnectedColor,
					SelectedColor,
					OffLimitColor,
					LayerColor[MAX_LAYER];
	Dimension		ViaThickness,	/* some preset values */
					ViaDrillingHole,
					LineThickness,
					MaxWidth,		/* maximum size of a layout */
					MaxHeight,
					MediaWidth,		/* size of output media */
					MediaHeight;
	int				Grid,			/* grid 0.001'' */
					Zoom,			/* number of shift operations for zooming */
					PinoutZoom,		/* same for pinout windows */
					PinoutNameLength,	/* max displayed length of a pinname */
					Volume,			/* the speakers volume -100..100 */
					CharPerLine,	/* width of an output line in characters */
					Mode,			/* currently active mode */
					BufferNumber;	/* number of the current buffer */
	long			BackupInterval;	/* time between two backups in seconds */
	String			FontCommand,	/* commands for file loading */
					FileCommand,
					ElementCommand,
					PrintCommand,
					SaveCommand,	/* and saving */
					FontFile,		/* name of default font file */
					Groups,			/* string with layergroups */
					FilePath,		/* colon seperated search paths */
					FontPath,
					ElementPath,
					Size,			/* geometry string for size */
					Media;			/* type of output media */
	Position		MediaMarginLeft,/* print margin, depends on device */
					MediaMarginBottom,
					PinoutOffsetX,	/* offset of origin */
					PinoutOffsetY,
					PinoutTextOffsetX,	/* offset of text from pin center */
					PinoutTextOffsetY;
	LayerGroupType	LayerGroups;	/* default layer groups */
	Boolean			AbsoluteGrid,	/* grid is relative to (0,0) */
					UseLogWindow,	/* use log window instead of dialog box */
					RaiseLogWindow,	/* raise log window if iconified */
					SaveLastCommand,/* save the last command entered by user */
					SaveInTMP,				/* always save data in /tmp */
					DrawGrid,				/* draw grid points */
					AllDirectionLines,		/* enable lines to all directions */
					ResetAfterElement,		/* reset connections after */
											/* each element */
					RingBellWhenFinished;	/* flag if a signal should be */
											/* produced when searching of */
											/* connections is done */
	XFontStruct		*PinoutFont;	/* font ID used for pin names */
} SettingType, *SettingTypePtr;

/* ----------------------------------------------------------------------
 * pointer to low-level copy, move and rotate functions
 */
typedef struct
{
	void	*(*Line)(LayerTypePtr, LineTypePtr);
	void	*(*Text)(LayerTypePtr, TextTypePtr);
	void	*(*Polygon)(LayerTypePtr, PolygonTypePtr);
	void	*(*Via)(PinTypePtr);
	void	*(*Element)(ElementTypePtr);
	void	*(*ElementName)(ElementTypePtr);
	void	*(*Pin)(ElementTypePtr, PinTypePtr);
	void	*(*PolygonPoint)(PolygonTypePtr, PolygonPointTypePtr);
} ObjectFunctionType, *ObjectFunctionTypePtr;

#endif
