/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: copy.c,v 2.1 94/09/28 14:26:06 nau Exp $";

/* functions used to copy pins, elements ...
 * it's necessary to copy data by calling create... since the base pointer
 * may change cause of dynamic memory allocation
 */

#include <stdlib.h>

#include "global.h"

#include "copy.h"
#include "create.h"
#include "data.h"
#include "draw.h"
#include "mymem.h"
#include "misc.h"
#include "move.h"
#include "select.h"
#include "undo.h"

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	*CopyVia(PinTypePtr);
static	void	*CopyLine(LayerTypePtr, LineTypePtr);
static	void	*CopyText(LayerTypePtr, TextTypePtr);
static	void	*CopyPolygon(LayerTypePtr, PolygonTypePtr);
static	void	*CopyElement(ElementTypePtr);

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Position			DeltaX, DeltaY;		/* movement vector */
static	ObjectFunctionType	CopyFunctions = {
	CopyLine,
	CopyText,
	CopyPolygon,
	CopyVia,
	CopyElement,
	NULL,
	NULL,
	NULL };

/* ---------------------------------------------------------------------------
 * copies data from one polygon to another
 * 'Dest' has to exist
 */
PolygonTypePtr CopyPolygonLowLevel(PolygonTypePtr Dest, PolygonTypePtr Src)
{
		/* release old memory if necessary */
	if (Dest)
		FreePolygonMemory(Dest);

		/* copy all data */
	POLYGONPOINT_LOOP(Src,
		CreateNewPointInPolygon(Dest, point->X, point->Y);
	);
	SetPolygonBoundingBox(Dest);
	return(Dest);
}

/* ---------------------------------------------------------------------------
 * copies data from one element to another and creates the destination 
 * if necessary
 */
ElementTypePtr CopyElementLowLevel(ElementTypePtr Dest, ElementTypePtr Src)
{
		/* release old memory if necessary */
	if (Dest)
		FreeElementMemory(Dest);

		/* both coordinates and flags are the same */
	Dest = CreateNewElement(PCB->Data, Dest, &PCB->Font, NOFLAG,
		CANONICAL_NAME(Src), NAMEONPCB_NAME(Src),
		CANONICAL_TEXT(Src).X, CANONICAL_TEXT(Src).Y,
		CANONICAL_TEXT(Src).Direction, CANONICAL_TEXT(Src).Scale,
		CANONICAL_TEXT(Src).Flags & MIRRORFLAG);

		/* abort on error */
	if (!Dest)
		return(Dest);

	ELEMENTLINE_LOOP(Src,
		CreateNewLineInElement(Dest, line->X1, line->Y1,
			line->X2, line->Y2, line->Thickness);
	);
	PIN_LOOP(Src,
		CreateNewPin(Dest, pin->X, pin->Y, pin->Thickness,
			pin->DrillingHole, pin->Name, pin->Flags & PINFLAG);
	);
	ARC_LOOP(Src,
		CreateNewArcInElement(Dest, arc->X, arc->Y, arc->Width, arc->Height,
			arc->StartAngle, arc->Delta, arc->Thickness);
	);

	Dest->MarkX = Src->MarkX;
	Dest->MarkY = Src->MarkY;
	SetElementBoundingBox(Dest);
	return(Dest);
}

/* ---------------------------------------------------------------------------
 * copies a via 
 */
static void *CopyVia(PinTypePtr Via)
{
	PinTypePtr	via;

	via = CreateNewVia(PCB->Data, Via->X, Via->Y, Via->Thickness,
			Via->DrillingHole, Via->Name, VIAFLAG);
	MOVE_VIA_LOWLEVEL(via, DeltaX, DeltaY);
	DrawVia(via);
	AddObjectToCopyUndoList(VIA_TYPE, NULL, via);
	return(via);
}

/* ---------------------------------------------------------------------------
 * copies a line 
 */
static void *CopyLine(LayerTypePtr Layer, LineTypePtr Line)
{
	LineTypePtr	line = CreateNewLineOnLayer(Layer,
					Line->X1, Line->Y1, Line->X2, Line->Y2,
					Line->Thickness, NOFLAG);

	MOVE_LINE_LOWLEVEL(line, DeltaX, DeltaY);
	DrawLine(Layer, line);
	AddObjectToCopyUndoList(LINE_TYPE, Layer, line);
	return(line);
}

/* ---------------------------------------------------------------------------
 * copies a text 
 */
static void *CopyText(LayerTypePtr Layer, TextTypePtr Text)
{
	TextTypePtr	text = CreateNewText(Layer, &PCB->Font,
					Text->X, Text->Y, Text->Direction, Text->Scale,
					Text->TextString, Text->Flags & MIRRORFLAG);

	MOVE_TEXT_LOWLEVEL(text, DeltaX, DeltaY);
	DrawText(Layer, text);
	AddObjectToCopyUndoList(TEXT_TYPE, Layer, text);
	return(text);
}

/* ---------------------------------------------------------------------------
 * copies a polygon 
 */
static void *CopyPolygon(LayerTypePtr Layer, PolygonTypePtr Polygon)
{
	PolygonTypePtr	polygon = CreateNewPolygon(Layer, NOFLAG);

	CopyPolygonLowLevel(polygon, Polygon);
	MovePolygonLowLevel(polygon, DeltaX, DeltaY);
	DrawPolygon(Layer, polygon);
	AddObjectToCopyUndoList(POLYGON_TYPE, Layer, polygon);
	return(polygon);
}

/* ---------------------------------------------------------------------------
 * copies a element 
 */
static void *CopyElement(ElementTypePtr Element)
{
	ElementTypePtr	element = CopyElementLowLevel(NULL, Element);

	MoveElementLowLevel(element, DeltaX, DeltaY);
	if (PCB->ElementOn)
	{
		DrawElementName(element);
		DrawElementPackage(element);
	}
	if (PCB->PinOn)
		DrawElementPins(element);
	AddObjectToCopyUndoList(ELEMENT_TYPE, NULL, element);
	return(element);
}

/* ---------------------------------------------------------------------------
 * pastes the contents of the buffer to the layout. Only visible objects
 * are handled by the routine.
 * returns True if any objects have benn added
 */
Boolean CopyPastebufferToLayout(Position X, Position Y)
{
	Cardinal	i;
	Boolean		changed = False;

		/* set movement vector */
	DeltaX = X -PASTEBUFFER->X,
	DeltaY = Y -PASTEBUFFER->Y;

		/* paste all layers */
	for (i = 0; i < MAX_LAYER; i++)
	{
		if (PCB->Data->Layer[i].On)
		{
			LayerTypePtr	layer = &PASTEBUFFER->Data->Layer[i];

			changed = changed ||
					  (layer->LineN != 0) ||
					  (layer->PolygonN != 0) ||
					  (layer->TextN != 0);
			LINE_LOOP(layer, CopyLine(&PCB->Data->Layer[i], line););
			TEXT_LOOP(layer, CopyText(&PCB->Data->Layer[i], text););
			POLYGON_LOOP(layer, CopyPolygon(&PCB->Data->Layer[i], polygon););
		}
	}

		/* paste elements */
	if (PCB->PinOn || PCB->ElementOn)
	{
		changed |= (PASTEBUFFER->Data->ElementN != 0);
		ELEMENT_LOOP(PASTEBUFFER->Data, CopyElement(element););
	}

		/* finally the vias */
	if (PCB->ViaOn)
	{
		changed |= (PASTEBUFFER->Data->ViaN != 0);
		VIA_LOOP(PASTEBUFFER->Data, CopyVia(via););
	}

	IncrementUndoSerialNumber(); 
	return(changed);
}

/* ---------------------------------------------------------------------------
 * copies the object identified by its data pointers and the type
 * the new objects is moved by DX,DY
 * I assume that the appropriate layer ... is switched on
 */
void CopyObject(int Type, void *Ptr1, void *Ptr2, Position DX, Position DY)
{
		/* setup movement vector */
	DeltaX = DX;
	DeltaY = DY;
	ObjectOperation(&CopyFunctions, Type, Ptr1, Ptr2);
	IncrementUndoSerialNumber();
}
