/* static char *SccsId = "@(#)write_layout.c 3.24 (Delft University of Technology) 03/25/94"; */
/**********************************************************

Name/Version      : nelsea/3.24

Language          : C
Operating system  : UNIX SYSTEM V
Host machine      : any

Author(s)         : Patrick Groeneveld and Paul Stravers
Creation date     : june, 1 1990
Modified by       : Patrick Groeneveld
Modification date : April 15, 1992


        Delft University of Technology
        Department of Electrical Engineering
        Mekelweg 4 - P.O.Box 5031
        2600 GA DELFT
        The Netherlands

        Phone : 015 - 786240

        COPYRIGHT (C) 1990 , All rights reserved
**********************************************************/
/*
 * 
 *    write_layout.c
 *
 *********************************************************/
#include <time.h> /* prototypes the time() syscall */

#include "def.h"
#include "nelsis.h"
#include "typedef.h"
#include "grid.h"
#include "tbox.h"
#include "prototypes.h"


#define  map_to_overlay_layout_coord(crd,ori) \
(((crd) / GridRepitition[ori]) * LayoutRepitition[ori]) + OverlayGridMapping[ori][(crd) % GridRepitition[ori]]


static void write_layout(LAYOUTPTR,int,int,long,long);
static void recursive_write(SLICEPTR);
static void write_image(LAYOUTPTR,DM_STREAM *,long,long);
static void write_mc(MAPTABLEPTR,DM_STREAM *,DM_STREAM *);
static void do_write_mc(SLICEPTR,DM_STREAM *,DM_STREAM *);
static void write_box(MAPTABLEPTR,DM_STREAM *);
static void write_terminals(MAPTABLEPTR,DM_STREAM *,DM_STREAM *);
static void write_vias(LAYOUTPTR,DM_STREAM *,DM_STREAM *);

/*
 */
extern char
   *primitive_str,
   *in_core_str,
   *not_in_core_str,
   *circuit_str,
   *layout_str,
   *written_str;
extern long
   Hierarchical_fish,      /* TRUE to perform hierarchical fishing */
   verbose; 
extern TBOXPTR
   *ViaBox;                /* array containing the box structure of each via */
extern long 
   Flat_vias,              /* TRUE to print the vias as boxes instead of mc's */
   Write_Floorplan,        /* TRUE to write a floorplan of the cell... */
   ChipSize[2],            /* number of elementary core cells in x and y */
   Chip_num_layer,         /* number of metal layers to be used */
   *GridMapping[2],        /* mapping of gridpoints on layout coordinates: size: GridRepitition[X] * GridRepitition[Y] */
   *OverlayGridMapping[2], /* overlaymapping of gridpoints to layout coordinates */
   OverlayBounds[2][2],    /* boundaries of overlaymapping, index 1 = orient, index2 = L/R */
   GridRepitition[2],      /* repitionvector (dx, dy) of grid core image (in grid points) */
   LayoutRepitition[2],    /* repitionvector (dx, dy) of LAYOUT image (in lambda) */ 
   ***ViaIndex,            /* Viaindex[x][y][z]: */
                           /* if z == 0: Index of via to core image in array ViaCellName (-1 if nonex) */
                           /* if z >  0: -1 if via between z and z-1 is not allowed, 1 otherwise */
   *LayerWidth,            /* array with wire width in each layer. size: Chip_num_layer */
   *LayerMaskNo,           /* contains NELSIS mask numbers of layer mask string in array LayerMaskName. size: ChipNumLayer */
   DummyMaskNo,
   *ViaCellBbx[4],         /* contains the boundingboxes of the vias. size: 4 * NumViaName */
   *ViaCellImported;       /* contains flag whether the Via Cell is imported or not */
extern char
   ImageName[],            /* NELSIS name of image to be repeated */
   **LayerMaskName,        /* array with mask names of each layer. size: Chip_num_layer */
   **ViaMaskName,          /* array with mask names of via to layer. size: NumViaName */
   **ViaCellName;          /* array with cell names of these vias . size: NumViaName */

extern DM_PROJECT
   *projectkey;

extern MAPTABLEPTR
   maptable;


/*
 * local vars
 */
static long
   bxl, bxr, byb, byt;     /* bounding box */


/*
 * this routine writes all layouts in the current datastruct
 */
void write_nelsis_layout(int  image_call, /* TRUE to add model call to image */
		    int recursive,
		    long  xsize,
		    long  ysize   /* size of image (valid if > 0 and if image_call == TRUE) */
)
{
MAPTABLEPTR
   map;

for(map = maptable; map != NULL; map = map->next)
   {
   if(map->seanelstatus != written_str &&
      map->internalstatus == in_core_str &&
      map->seanelstatus != primitive_str &&
      map->view == layout_str)
      { /* write it */ 
      if(map->overrule_status == TRUE)
         { /* it's the parent cell, do with image call if enabled */
         write_layout(map->layoutstruct, image_call, recursive, xsize, ysize);
         }
      else
         { /* always without image */
         write_layout(map->layoutstruct, FALSE, recursive, xsize, ysize);
         } 
      }
   }
}


/* * * * * * * * * 
 * 
 * This routine dumps the seadif datastructure in a proper NELSIS
 * database cell.
 * Before calling the database should be opened.
 */
static void write_layout(LAYOUTPTR  lay ,
			 int  image_call    /* TRUE to call the image */  ,
			 int  recursive,    /* TRUE for recursive write */
			 long  xsize ,
			 long  ysize   /* pre-specified size of image only used if image_call == TRUE */
)

{
MAPTABLEPTR
   map;              /* points to datastructure */
DM_STREAM
   *fp, 
   *vfp,
   *flp_fp = NULL;
DM_CELL
   *key,
   *flp_key = NULL;
long
   rbxl, rbxr, rbyb, rbyt;     /* real bounding box, without the image */

if(lay == NULL)
   {
   fprintf(stderr,"WARNING (write_layout): null struct.\n");
   return;
   }

/*
 * find out its mapping
 */
map = look_up_seadif_map(layout_str, 
                                lay->circuit->function->library->name,
                                lay->circuit->function->name,
                                lay->circuit->name,
                                lay->name);

if(map->seanelstatus == written_str ||
   map->seanelstatus == primitive_str)
   { /* cell is already written or primitive */
   return;
   }

if(map->internalstatus != in_core_str)
   {
/*
   fprintf(stderr,"ERROR: attempt to write cell '%s', which is not in core\n",
      map->layout);
 */
   return;
   }


/*
 * recursively write other cells
 */
if(recursive == TRUE)
   recursive_write(lay->slice);

/*
 * test: is the name too long? 
 */
if(strlen(map->cell) > DM_MAXNAME)
   { /* print warning */
   fprintf(stderr,"WARNING (write_layout): cell name %s too long, truncated\n",
      map->cell); 
   map->cell[DM_MAXNAME] = '\0';
   }

if(verbose == ON)
   {
   printf("------ writing sdflay '%s(%s(%s(%s)))' into nelsis '%s' ------\n", 
               lay->name, 
               lay->circuit->name,
               lay->circuit->function->name,
               lay->circuit->function->library->name,
               map->cell);  
   fflush(stdout);
   }

/*
 * open a new model file, called cell_name
 * Cell is written away as WORKING
 */
if((key = dmCheckOut (projectkey, map->cell, WORKING, DONTCARE, layout_str, UPDATE)) == NULL) 
   {
   error(ERROR,"Unable to open cell (cell not written)");
   return;
   }

/*
 * open same in floorplan
 */
if(Write_Floorplan == TRUE)
   { /* open floorplan */
   if((flp_key = dmCheckOut (projectkey, map->cell, 
			  WORKING, DONTCARE, FLOORPLAN, UPDATE)) == NULL) 
      error(WARNING,"Unable to open flp_cell");
   }
else
   flp_key = NULL;


/* initialize bounding box */

bxl = byb = BIGNUMBER;
bxr = byt = -BIGNUMBER;

/*
 * write terminals
 */
if((fp = dmOpenStream(key,"term","w")) == NULL)
   error(FATAL_ERROR, "write_to_nelsis");

if(flp_key != NULL)
   flp_fp = dmOpenStream(flp_key,"term","w");

write_terminals(map, fp, flp_fp);

dmCloseStream(fp, COMPLETE);
if(flp_fp != NULL)
   dmCloseStream(flp_fp, COMPLETE);   

/*
 * write box file= the wires in the grid
 */
if((fp = dmOpenStream(key,"box","w")) == NULL)
   error(FATAL_ERROR, "write_to_nelsis");

write_box(map, fp);

/*
 * write model calls
 */
if((vfp = dmOpenStream(key,"mc","w")) == NULL)
   error(FATAL_ERROR, "write_to_nelsis");

if(flp_key != NULL)
   flp_fp = dmOpenStream(flp_key,"mc","w");
  
/* vias */
write_vias(map->layoutstruct, vfp, fp); 

write_mc(map, vfp, flp_fp);

/* 
 * save the 'real' bounding box
 */
rbxl = bxl;
rbxr = bxr;
rbyb = byb;
rbyt = byt;

/* basic image */
if(image_call == TRUE) 
   write_image(map->layoutstruct, vfp, xsize, ysize);
  
dmCloseStream(vfp, COMPLETE);   /* mc */

dmCloseStream(fp, COMPLETE);    /* box */

if(flp_fp != NULL)
   dmCloseStream(flp_fp, COMPLETE);   

/*
 * write bounding box
 */
if((fp = dmOpenStream(key,"info","w")) == NULL)
   error(FATAL_ERROR, "write_to_nelsis");

if(flp_key != NULL)
   flp_fp = dmOpenStream(flp_key,"info","w");

if(bxl == BIGNUMBER)
   bxl = bxr = 0;  /* was unchanged */
if(byb == BIGNUMBER)
   byb = byt = 0;  /* was unchanged */
if(rbxl == BIGNUMBER)
   rbxl = rbxr = 0;  /* was unchanged */
if(byb == BIGNUMBER)
   rbyb = rbyt = 0;  /* was unchanged */

ginfo.bxl = finfo.bxl = bxl; 
ginfo.bxr = finfo.bxr = bxr;
ginfo.byb = finfo.byb = byb; 
ginfo.byt = finfo.byt = byt;

if(flp_fp != NULL)
   {
   dmPutDesignData(flp_fp, FLP_INFO);       
   dmCloseStream(flp_fp, COMPLETE); 
   }

dmPutDesignData(fp, GEO_INFO); /* 1 */
dmPutDesignData(fp, GEO_INFO); /* 2 */
dmPutDesignData(fp, GEO_INFO); /* 3 */
ginfo.bxl = rbxl; 
ginfo.bxr = rbxr;
ginfo.byb = rbyb; 
ginfo.byt = rbyt;
dmPutDesignData(fp, GEO_INFO); /* 4= real bounding box */

dmCloseStream(fp, COMPLETE); 
      
/*
 * make misterious files
 */
if(fp = dmOpenStream(key,"nor","w"))
   dmCloseStream(fp, COMPLETE); 

if(flp_key != NULL)
   {
   flp_fp = dmOpenStream(flp_key,"chan","w");
   if(flp_fp != NULL)
      dmCloseStream(flp_fp, COMPLETE); 
   }

dmCheckIn(key,COMPLETE);
if(flp_key != NULL)
   dmCheckIn(flp_key,COMPLETE);

#ifdef NELSIS_REL4
/*
 * make equivalence between floorplan cell and layout
 */
if(Write_Floorplan == TRUE &&
   dmPutMetaDesignData(CELLEQUIVALENCE, projectkey, map->cell, 
    FLOORPLAN, map->cell, layout_str) == (char *)-1)
   {
   fprintf(stderr,"Klote: problems with dmPutMetaDesignData\n");
   }
#endif

/*
 * set status
 */
map->nelsis_time = time(0);
map->seanelstatus = written_str; 
}

/* * * * * * *
 *
 * this is a recursive help routine to perform depth-first
 * write of the children
 */
static void recursive_write(SLICEPTR  slice )

{
LAYINSTPTR
   inst;

/*
 * depth-first write of children
 */
for(; slice != NULL; slice = slice->next)
   {
   /*
    * recursion??
    */
   if(slice->chld_type == SLICE_CHLD)
      {
      recursive_write(slice->chld.slice);
      return;
      }

   if(slice->chld_type != LAYINST_CHLD)
      {
      fprintf(stderr,"WARNING (recursive_write): strange type of slice.\n");
      return;
      }

   for(inst = slice->chld.layinst; inst != NULL; inst = inst->next)
      {
      /* always without image and no force */
      write_layout(inst->layout, FALSE, TRUE, (long) -1, (long) -1);
      }
   }
}


/* * * * * * * *
 *
 * This routine writes the proper nelsis model calls to generate the image 
 * below the circuit
 */
static void write_image(LAYOUTPTR  lay ,
			DM_STREAM  *fp ,
			long  xsize ,
			long  ysize    /* requested array size */
)

{
DM_PROJECT   
   *remote_projectkey;
DM_CELL
   *cell_key;
DM_STREAM
   *tempfp;
char
   *remote_cellname; 
long
   imported,
   bbx[4];

/*
 * look for this basic image cell
 */
if((imported = (long) exist_cell(ImageName, layout_str)) == -1)
   { /* it does not exist */
   fprintf(stderr,"ERROR: cannot find elementary core image/template cell '%s' in database\n", ImageName);
   return;
   }

/*
 * open it: key for project
 */
if((remote_projectkey = dmFindProjKey((int) imported, ImageName,
    projectkey, &remote_cellname, layout_str)) == NULL) 
   {  /* ? */
   error(ERROR, "cannot find nasty project key");
   return;
   }

/*
 * open it
 */
if((cell_key = dmCheckOut(remote_projectkey, remote_cellname, 
    ACTUAL, DONTCARE, layout_str, READONLY)) == NULL)
   {  /* ? */
   fprintf(stderr,"ERROR: cannot open core cell '%s' in database\n", ImageName);
   return;
   }

/*
 * read bounding box of image
 */
if(!(tempfp = dmOpenStream(cell_key, "info", "r"))) 
   error(FATAL_ERROR, "write_image");
 
if(dmGetDesignData(tempfp, GEO_INFO) == -1)
   error(FATAL_ERROR, "write_image");

bbx[L] = ginfo.bxl;
bbx[R] = ginfo.bxr;
bbx[B] = ginfo.byb;
bbx[T] = ginfo.byt;

/*
 * terminate
 */
dmCloseStream(tempfp, COMPLETE);
dmCheckIn(cell_key, COMPLETE);


strcpy(gmc.inst_name,"IMAGE");
strcpy(gmc.cell_name, ImageName);
gmc.imported = imported;
gmc.mtx[0] = 1; gmc.mtx[1] = 0; gmc.mtx[2] = 0;
gmc.mtx[3] = 0; gmc.mtx[4] = 1; gmc.mtx[5] = 0;

if(strncmp(ImageName, "fishbone", strlen("fishbone") - 1) == 0)
   { /* fistbone overlaps 1 in x and y */
   gmc.nx = MAX(0, (lay->bbx[X] + lay->off[X] - 1)/GridRepitition[X]);   
   gmc.ny = MAX(0, (lay->bbx[Y] + lay->off[Y] - 2)/GridRepitition[Y]);
   }
else
   {
   gmc.nx = (lay->bbx[X] + lay->off[X])/GridRepitition[X];
   gmc.ny = (lay->bbx[Y] + lay->off[Y])/GridRepitition[Y];
   if(strncmp(ImageName, "gatearray", 8) == 0 ||
      strncmp(ImageName, "pm25", 4) == 0)
      { /* hack to get image size of gatearray right */
      gmc.nx-- ;  gmc.ny--;
      if(gmc.nx < 0)
	 gmc.nx = 0;
      if(gmc.ny < 0)
	 gmc.ny = 0;
      }
   }

if(xsize > 0)
   { /* size pre-specified */
   gmc.nx = xsize - 1;
   }

if(ysize > 0)
   { /* size pre-specified */
   gmc.ny = ysize - 1;
   }

gmc.dx = LayoutRepitition[X]; gmc.dy = LayoutRepitition[Y];

gmc.bxl = bbx[L];
gmc.byb = bbx[B]; 
gmc.bxr = bbx[R] + (gmc.nx * LayoutRepitition[X]);
gmc.byt = bbx[T] + (gmc.ny * LayoutRepitition[Y]);

bxl = MIN(bxl, gmc.bxl);
bxr = MAX(bxr, gmc.bxr);
byb = MIN(byb, gmc.byb);
byt = MAX(byt, gmc.byt);

dmPutDesignData(fp, GEO_MC);
}

                    
/* * * * * * * * * 
 *
 * This routine writes the model-calls of the slice pointed at
 */ 
static void do_write_mc(SLICEPTR slice,
			DM_STREAM  *fp,
			DM_STREAM  *flp_fp   /* floorplan */
)

{
long
   xl, xr, yb, yt, imported,
   instcrd, orgcrd, x0, y0;
DM_CELL
   *cell_key;
DM_PROJECT   
   *remote_projectkey;
DM_STREAM
   *sfp;
int
   j, num; 
char
   *remote_cellname; 
LAYINSTPTR
   inst;
MAPTABLEPTR
   sonmap;

for(; slice != NULL; slice = slice->next)
   {
   /*
    * recursion??
    */
   if(slice->chld_type == SLICE_CHLD)
      {
      do_write_mc(slice->chld.slice, fp, flp_fp);
      continue;
      }

   if(slice->chld_type != LAYINST_CHLD)
      {
      fprintf(stderr,"WARNING (write_mc): strange type of slice.\n");
      return;
      }

   for(inst = slice->chld.layinst; inst != NULL; inst = inst->next)
      {

      if(inst->layout == NULL)
         {
         fprintf(stderr,"WARNING (write_mc): mc '%s' without layout.\n", inst->name);
         continue;
         }


      if(strncmp(inst->layout->name,"Error_Marker", 10) == 0)
	 { /* its'a error marker.. */
	 sonmap = look_up_map(layout_str, "Error_Marker");
	 }
      else
	 { 
	 sonmap = look_up_seadif_map(layout_str, 
                                inst->layout->circuit->function->library->name,
                                inst->layout->circuit->function->name,
                                inst->layout->circuit->name,
                                inst->layout->name);
	 }
      
   
      if(sonmap->internalstatus != in_core_str &&
	 sonmap->nelsis_time == 0)
	 {
	 if(sonmap->num_read_attempts >= 2)
	    continue;

	 sonmap->num_read_attempts++; /* to prevent many printing */

	 if(strncmp(inst->layout->name,"Error_Marker", 10) == 0)
	    {
	    fprintf(stderr,"ERROR: can't find error marker cell '%s'\n", 
		    inst->layout->name);
	    continue;
	    }

	 fprintf(stderr,"WARNING: layout contains a reference to non-existing\n");
	 fprintf(stderr,"         son-cell '%s' (skipped)\n", sonmap->cell);
	 continue;
	 }
      
      if(sonmap->bbx_found == FALSE)
	 { /* bbx was not yet found: do it... */
	 /*
	  * look for son cell in database, to find its bbx
	  */
	 if((imported = (long) exist_cell(sonmap->cell, layout_str)) == -1)
	    { /* it does not exist */
	    fprintf(stderr,"ERROR: cannot find son-cell '%s' (mapped '%s') of cell in database\n", 
		    inst->layout->name, sonmap->cell);
	    continue;
	    }
	 
	 /*
	  * open it: key for project
	  */
	 if((remote_projectkey = dmFindProjKey((int) imported, sonmap->cell,
					       projectkey, &remote_cellname, layout_str)) == NULL) 
	    {  /* ? */
	    fprintf(stderr,"ERROR: cannot find nasty project key\n");
	    continue;
	    }
	 
	 /*
	  * open it
	  */
	 if((cell_key = dmCheckOut (remote_projectkey, remote_cellname, 
				    ACTUAL, DONTCARE, layout_str, READONLY)) == NULL)
	    {  /* ? */
	    fprintf(stderr,"ERROR: cannot open son cell '%s' in database\n", sonmap->cell);
	    continue;
	    }

	 /*
	  * read bounding box
	  */
	 if (!(sfp = dmOpenStream (cell_key, "info", "r"))) 
	    {
	    fprintf(stderr,"ERROR: cannot open info of cell '%s'\n", sonmap->cell);
	    error(FATAL_ERROR, "write_mc()");
	    }
	 
	 num = 1;
	 while(dmGetDesignData(sfp, GEO_INFO) > 0 && num < 6)
	    {  /* take 4th bbx, if possible */
	    if(num == 1 || num == 4)
	       {	    
	       sonmap->bbx[L] = ginfo.bxl;	 sonmap->bbx[R] = ginfo.bxr;
	       sonmap->bbx[B] = ginfo.byb;	 sonmap->bbx[T] = ginfo.byt;
	       }
	    num++;
	    }
	 if(num <= 1)
	    error(ERROR, "cannot find bbx");
	 sonmap->bbx_found = TRUE;
	 sonmap->imported = imported;
	 dmCloseStream(sfp, COMPLETE);
	 dmCheckIn(cell_key, COMPLETE);
	 }

      xl = sonmap->bbx[L];
      xr = sonmap->bbx[R];
      yb = sonmap->bbx[B];
      yt = sonmap->bbx[T];

      strncpy(gmc.inst_name, inst->name, DM_MAXNAME);
      strncpy(gmc.cell_name, sonmap->cell, DM_MAXNAME);
    
      gmc.imported = sonmap->imported; 

      for(j=0; j != 6; j++)
         gmc.mtx[j] = inst->mtx[j];

      x0 = map_to_layout_coord(0, X);
      y0 = map_to_layout_coord(0, Y);

      instcrd = map_to_layout_coord(inst->mtx[2], X); 
      orgcrd = gmc.mtx[0] * x0 + gmc.mtx[1] * y0;
      gmc.mtx[2] = instcrd - orgcrd;

      instcrd = map_to_layout_coord(inst->mtx[5], Y); 
      orgcrd = gmc.mtx[3] * x0 + gmc.mtx[4] * y0;
      gmc.mtx[5] = instcrd - orgcrd;
 
      gmc.bxl = gmc.mtx[0] * xl + gmc.mtx[1] * yb + gmc.mtx[2];
      gmc.bxr = gmc.mtx[0] * xr + gmc.mtx[1] * yt + gmc.mtx[2];
      if(gmc.bxl > gmc.bxr)
         { /* swap */
         x0 = gmc.bxl; gmc.bxl = gmc.bxr; gmc.bxr = x0;
         }
      gmc.byb = gmc.mtx[3] * xl + gmc.mtx[4] * yb + gmc.mtx[5];
      gmc.byt = gmc.mtx[3] * xr + gmc.mtx[4] * yt + gmc.mtx[5];
      if(gmc.byb > gmc.byt)
         { /* swap */
         y0 = gmc.byb; gmc.byb = gmc.byt; gmc.byt = y0;
         }

      bxl = MIN(bxl, gmc.bxl);
      bxr = MAX(bxr, gmc.bxr);
      byb = MIN(byb, gmc.byb);
      byt = MAX(byt, gmc.byt);
 
      gmc.nx = gmc.ny = 0;       /* no repitition */

      /* copy to floorplan */
      strcpy(fmc.inst_name, gmc.inst_name);
      strcpy(fmc.cell_name, gmc.cell_name);
      fmc.imported = gmc.imported;
      for(j=0; j != 6; j++)
         fmc.mtx[j] = gmc.mtx[j];
      fmc.bxl = gmc.bxl; fmc.bxr = gmc.bxr; 
      fmc.byb = gmc.byb; fmc.byt = gmc.byt;
      fmc.nx = fmc.ny = 0;
      
      dmPutDesignData(fp, GEO_MC); 
     
      if(flp_fp != NULL)
         dmPutDesignData(flp_fp, FLP_MC);          
      }
   }
}


/* * * * * * * * * 
 *
 * This routine writes the model-calls of the slice pointed at
 */ 
static void write_mc(MAPTABLEPTR  map ,
		     DM_STREAM  *fp  ,
		     DM_STREAM  *flp_fp   /* floorplan */
)

{
NELSISOBJECTPTR
   no;
int
   j; 

/*
 * call the real recursive writing routine
 */
do_write_mc(map->layoutstruct->slice, fp, flp_fp);


/*
 * and now scan the nelsisobjects for MC's
 * which were not fished
 */
for(no = map->list_of_unfished_objects; no != NULL; no = no->next)
   {
   if(no->type != GMC_FISH)
      continue;
   
   strcpy(gmc.inst_name, no->name);
   strcpy(gmc.cell_name, no->cell_name);
   gmc.imported = no->imported;
   for(j = 0; j != 6; j++)
      gmc.mtx[j] = no->mtx[j];
   gmc.bxl = no->bxl; gmc.bxr = no->bxr;
   gmc.byb = no->byb; gmc.byt = no->byt;
   gmc.dx = no->dx; gmc.nx = no->nx;
   gmc.ny = no->ny; gmc.dy = no->dy;

   /* copy to floorplan */
   strcpy(fmc.inst_name, gmc.inst_name);
   strcpy(fmc.cell_name, gmc.cell_name);
   fmc.imported = gmc.imported;
   for(j=0; j != 6; j++)
      fmc.mtx[j] = gmc.mtx[j];
   fmc.bxl = gmc.bxl; fmc.bxr = gmc.bxr; 
   fmc.byb = gmc.byb; fmc.byt = gmc.byt;
   fmc.nx = fmc.ny = 0;

   bxl = MIN(bxl, gmc.bxl);
   bxr = MAX(bxr, gmc.bxr);
   byb = MIN(byb, gmc.byb);
   byt = MAX(byt, gmc.byt);

   dmPutDesignData(fp, GEO_MC);

   if(flp_fp != NULL)
      dmPutDesignData(flp_fp, FLP_MC);
   }

}


/*
 * writes the image to the database
 */
static void write_box(MAPTABLEPTR  map ,
		      DM_STREAM  *fp )

{
WIREPTR
   box;
LAYOUTPTR
   lay;
NELSISOBJECTPTR
   no;
long
   rest;
long 
   lwidth;
int
   layer,
   no_dummy,
   illegal_layer,
   negative_wire;

lay = map->layoutstruct;

illegal_layer = negative_wire = no_dummy = FALSE;

for(box = lay->wire; box != NULL; box = box->next)
   {

   if(box->layer < 0)
      {  /* cannot write negative wires */
      negative_wire = TRUE;
      continue;
      }

   if(box->layer >= 100 && box->layer < 200)
      continue;  /* not a box , but a via */
 
   if(box->layer >= 200)
      { /* dummy mask */ 
      if(box->layer == 200)
         { /* real dummy */
         if(DummyMaskNo == -1)
            {
            no_dummy = TRUE;
            continue;
            }
         layer = 0;  /* use width of layer 0 */
         gbox.layer_no = DummyMaskNo; 
         lwidth = LayerWidth[layer];
         } 
      else
         { /* overlay */
         layer = box->layer - 201;  
         if(layer < 0 || layer >= Chip_num_layer)
            { /* illegal layer */
            fprintf(stderr,"illegal: %d\n", layer);
            illegal_layer = TRUE;
            continue;
            } 
         gbox.layer_no = LayerMaskNo[layer];  
         lwidth = LayerWidth[layer];

         /* find out the width of a via */ 
         lwidth = MAX((ViaCellBbx[R][layer]  - ViaCellBbx[L][layer]), lwidth);
         }
      }
   else
      { /* metal mask */
      if(box->layer == 0 || box->layer > Chip_num_layer)
         { /* illegal layer */
         illegal_layer = TRUE;
         continue;
         }
         
      layer = box->layer - 1;
      gbox.layer_no = LayerMaskNo[box->layer - 1]; 
      lwidth = LayerWidth[layer];
      }
 
 
   rest = lwidth%2;
   if(OverlayGridMapping[X] == NULL ||
      box->crd[T]%GridRepitition[Y] < OverlayBounds[Y][L] ||
      box->crd[T]%GridRepitition[Y] > OverlayBounds[Y][R] ||
      box->crd[B]%GridRepitition[Y] < OverlayBounds[Y][L] ||
      box->crd[B]%GridRepitition[Y] > OverlayBounds[Y][R])
      { /* ordinary */
      gbox.xl = gbox.bxl = map_to_layout_coord(box->crd[L], X) - (lwidth/2); 
      gbox.xr = gbox.bxr = map_to_layout_coord(box->crd[R], X) + (lwidth/2) + rest;
      }
   else
      { /* overlay */
      gbox.xl = gbox.bxl = map_to_overlay_layout_coord(box->crd[L], X) - (lwidth/2); 
      gbox.xr = gbox.bxr = map_to_overlay_layout_coord(box->crd[R], X) + (lwidth/2) + rest; 
      }
   if(OverlayGridMapping[Y] == NULL ||
      box->crd[L]%GridRepitition[X] < OverlayBounds[X][L] ||
      box->crd[L]%GridRepitition[X] > OverlayBounds[X][R] ||
      box->crd[R]%GridRepitition[X] < OverlayBounds[X][L] ||
      box->crd[R]%GridRepitition[X] > OverlayBounds[X][R])
      { /* ordinary */
      gbox.yb = gbox.byb = map_to_layout_coord(box->crd[B], Y) - (lwidth/2);
      gbox.yt = gbox.byt = map_to_layout_coord(box->crd[T], Y) + (lwidth/2) + rest;
      }
   else 
      { /* overlay */
      gbox.yb = gbox.byb = map_to_overlay_layout_coord(box->crd[B], Y) - (lwidth/2);
      gbox.yt = gbox.byt = map_to_overlay_layout_coord(box->crd[T], Y) + (lwidth/2) + rest;
      }
   gbox.nx = gbox.ny = 0;

   bxl = MIN(bxl, gbox.bxl);
   bxr = MAX(bxr, gbox.bxr);
   byb = MIN(byb, gbox.byb);
   byt = MAX(byt, gbox.byt);

   dmPutDesignData(fp, GEO_BOX);
   } 

if(negative_wire == TRUE)
   {
   fprintf(stderr,"WARNING (write_box): negative cannot be written into nelsis.\n");
   }

if(illegal_layer == TRUE)
   {
   fprintf(stderr,"WARNING (write_box): found illegal metal layer.\n");
   } 

if(no_dummy == TRUE)
   {
   fprintf(stderr,"WARNING (write_box): found dummy layer, but no dummy mask was defined.\n");
   }

/*
 * and now scan the nelsisobjects for terminals
 */
for(no = map->list_of_unfished_objects; no != NULL; no = no->next)
   {
   if(no->type != GBOX_FISH)
      continue;
   
   gbox.layer_no = no->layer_no;
   gbox.xl = no->xl; gbox.xr = no->xr;
   gbox.yb = no->yb; gbox.yt = no->yt;
   gbox.bxl = no->bxl; gbox.bxr = no->bxr;
   gbox.byb = no->byb; gbox.byt = no->byt;
   gbox.dx = no->dx; gbox.nx = no->nx;
   gbox.ny = no->ny; gbox.dy = no->dy;

   bxl = MIN(bxl, gbox.bxl);
   bxr = MAX(bxr, gbox.bxr);
   byb = MIN(byb, gbox.byb);
   byt = MAX(byt, gbox.byt);

   dmPutDesignData(fp, GEO_BOX);
   }

}


/* * * * * * * * * *
 *
 * This routine writes a terminallist to the database
 */
static void write_terminals(MAPTABLEPTR  map ,
			    DM_STREAM  *fp    /* layout */  ,
			    DM_STREAM  *flp_fp  /* floorplan */
)

{
register LAYPORTPTR
   hterm;
LAYOUTPTR
   lay;
NELSISOBJECTPTR
   no;


lay = map->layoutstruct;

for(hterm = lay->layport; hterm != NULL; hterm = hterm->next)
   { 
   if(hterm->layer < 0 || hterm->layer >= Chip_num_layer)
      {
      fprintf(stderr,"WARING: terminal in illegal layer: '%s' (not written)\n",
            hterm->cirport->name);
      continue; /* do not write illegal terminals */
      }

   if(LayerMaskNo[hterm->layer] < 0)
      continue; /* layer not defined */

   if(OverlayGridMapping[X] == NULL ||
      hterm->pos[Y]%GridRepitition[Y] < OverlayBounds[Y][L] ||
      hterm->pos[Y]%GridRepitition[Y] > OverlayBounds[Y][R])
      { /* ordinary */
      gterm.xl = gterm.bxl = map_to_layout_coord(hterm->pos[X], X) - (LayerWidth[hterm->layer]/2);
      }
   else
      { /* overlay */
      gterm.xl = gterm.bxl = map_to_overlay_layout_coord(hterm->pos[X], X) - (LayerWidth[hterm->layer]/2);      
      }
   gterm.xr = gterm.bxr = gterm.xl + LayerWidth[hterm->layer];
   if(OverlayGridMapping[Y] == NULL ||
      hterm->pos[X]%GridRepitition[X] < OverlayBounds[X][L] ||
      hterm->pos[X]%GridRepitition[X] > OverlayBounds[X][R])
      { /* ordinary */
      gterm.yb = gterm.byb = map_to_layout_coord(hterm->pos[Y], Y) - (LayerWidth[hterm->layer]/2);
      }
   else
      { /* overlay */
      gterm.yb = gterm.byb = map_to_layout_coord(hterm->pos[Y], Y) - (LayerWidth[hterm->layer]/2);
      }
   gterm.yt = gterm.byt = gterm.yb + LayerWidth[hterm->layer];
   gterm.nx = gterm.ny = 0; 
   strcpy(gterm.term_name, hterm->cirport->name);
   gterm.layer_no = LayerMaskNo[hterm->layer];

   bxl = MIN(bxl, gterm.bxl);
   bxr = MAX(bxr, gterm.bxr);
   byb = MIN(byb, gterm.byb);
   byt = MAX(byt, gterm.byt);

   /* mcopy to flp terminal */
   strcpy(fterm.term_name, gterm.term_name);
   fterm.term_attribute = NULL;
   fterm.layer_no = gterm.layer_no;
   fterm.side = 0;
   fterm.xl = fterm.bxl = gterm.xl;
   fterm.xr = fterm.bxr = gterm.xr;
   fterm.yb = fterm.byb = gterm.yb;
   fterm.yt = fterm.byt = gterm.yt;
   fterm.nx = fterm.ny = 0; 

   dmPutDesignData(fp, GEO_TERM);

   if(flp_fp != NULL) 
      dmPutDesignData(flp_fp, FLP_TERM);
   }     

/*
 * and now scan the nelsisobjects for terminals
 */
for(no = map->list_of_unfished_objects; no != NULL; no = no->next)
   {
   if(no->type != GTERM_FISH)
      continue;
   
   strcpy(gterm.term_name, no->name);
   gterm.layer_no = no->layer_no;
   gterm.xl = no->xl; gterm.xr = no->xr;
   gterm.yb = no->yb; gterm.yt = no->yt;
   gterm.bxl = no->bxl; gterm.bxr = no->bxr;
   gterm.byb = no->byb; gterm.byt = no->byt;
   gterm.dx = no->dx; gterm.nx = no->nx;
   gterm.ny = no->ny; gterm.dy = no->dy;

   /* mcopy to flp terminal */
   strcpy(fterm.term_name, gterm.term_name);
   fterm.term_attribute = NULL;
   fterm.layer_no = gterm.layer_no;
   fterm.side = 0;
   fterm.xl = fterm.bxl = gterm.xl;
   fterm.xr = fterm.bxr = gterm.xr;
   fterm.yb = fterm.byb = gterm.yb;
   fterm.yt = fterm.byt = gterm.yt;
   fterm.nx = fterm.ny = 0;

   bxl = MIN(bxl, gterm.bxl);
   bxr = MAX(bxr, gterm.bxr);
   byb = MIN(byb, gterm.byb);
   byt = MAX(byt, gterm.byt);

   dmPutDesignData(fp, GEO_TERM);   
   if(flp_fp != NULL) 
      dmPutDesignData(flp_fp, FLP_TERM);
   }
}

/* * * * * * * * 
 *
 * This routine writes the via model calls of cell
 */
static void write_vias(LAYOUTPTR  lay ,
		       DM_STREAM  *fp  ,
		       DM_STREAM  *bfp   /* box stream */
)
{
int
   wrong_via,
   viacount; 
WIREPTR
   via;
TBOXPTR
   tbox;
long
   viaindex;
 
viacount = 0;
wrong_via = FALSE;

for(via = lay->wire; via != NULL; via = via->next)
   {
   if(via->layer < 100)
      continue;  /* not a via */
 
   if(via->layer >= 200)
      continue;  /* dummy mask */

   viaindex = via->layer - 100;

   if(viaindex == 0)
      { /* via to core (not between metal layers */
      viaindex = ViaIndex[to_core(via->crd[L], X)][to_core(via->crd[B], Y)][0]; 
      if(viaindex < 0)
         {    /* does not exist */
         wrong_via = TRUE;
         continue;
         }
      }
   else 
      { /* via between metal layers */
      viaindex--;
      if(viaindex >= Chip_num_layer - 1)
         {
         wrong_via = TRUE;
         continue;
         }
      if(ViaIndex[to_core(via->crd[L], X)][to_core(via->crd[B], Y)][viaindex+1] == -1)
         {
         wrong_via = TRUE;
/*         continue; */
         }
      } 

   if(ViaCellImported[viaindex] < 0)
      continue;     /* was not found in database */
 
   sprintf(gmc.inst_name,"via_%d", ++viacount);
   strcpy(gmc.cell_name, ViaCellName[viaindex]);
   gmc.imported = ViaCellImported[viaindex]; 
   gmc.nx = 0; gmc.ny = 0;

   /* set coordinates of placed via */ 
   if(OverlayGridMapping[X] == NULL ||
      via->crd[B]%GridRepitition[Y] < OverlayBounds[Y][L] ||
      via->crd[B]%GridRepitition[Y] > OverlayBounds[Y][R])
      { /* ordinary */
      gmc.bxl = map_to_layout_coord(via->crd[L], X) - 
                         ((ViaCellBbx[R][viaindex]  - ViaCellBbx[L][viaindex])/2);
      }
   else
      {  /* in overlay area */
      gmc.bxl = map_to_overlay_layout_coord(via->crd[L], X) - 
                         ((ViaCellBbx[R][viaindex]  - ViaCellBbx[L][viaindex])/2);
      }

   if(OverlayGridMapping[Y] == NULL ||
      via->crd[L]%GridRepitition[X] < OverlayBounds[X][L] ||
      via->crd[L]%GridRepitition[X] > OverlayBounds[X][R])
      {  
      gmc.byb = map_to_layout_coord(via->crd[B], Y) - 
                         ((ViaCellBbx[T][viaindex]  - ViaCellBbx[B][viaindex])/2);
      }
   else
      {  /* in overlay area */ 
      gmc.byb = map_to_overlay_layout_coord(via->crd[B], Y) - 
                         ((ViaCellBbx[T][viaindex]  - ViaCellBbx[B][viaindex])/2);
      }

   gmc.bxr = gmc.bxl + (ViaCellBbx[R][viaindex] - ViaCellBbx[L][viaindex]);
   gmc.byt = gmc.byb + (ViaCellBbx[T][viaindex] - ViaCellBbx[B][viaindex]);

   gmc.mtx[0] = 1; gmc.mtx[1] = 0; gmc.mtx[2] = gmc.bxl - ViaCellBbx[L][viaindex];
   gmc.mtx[3] = 0; gmc.mtx[4] = 1; gmc.mtx[5] = gmc.byb - ViaCellBbx[B][viaindex];

   bxl = MIN(bxl, gmc.bxl);
   bxr = MAX(bxr, gmc.bxr);
   byb = MIN(byb, gmc.byb);
   byt = MAX(byt, gmc.byt);
 
   if(Flat_vias == FALSE)
      { /* vias as model call */
      dmPutDesignData(fp, GEO_MC);
      }
   else
      { /* vias as boxes */
      
      for(tbox = ViaBox[viaindex]; tbox != NULL; tbox = tbox->next)
         {
         gbox.layer_no = tbox->layer_no;
         gbox.xl = tbox->xl + gmc.mtx[2]; gbox.xr = tbox->xr + gmc.mtx[2];
         gbox.yb = tbox->yb + gmc.mtx[5]; gbox.yt = tbox->yt + gmc.mtx[5];
         gbox.bxl = tbox->bxl + gmc.mtx[2]; gbox.bxr = tbox->bxr + gmc.mtx[2];
         gbox.byb = tbox->byb + gmc.mtx[5]; gbox.byt = tbox->byt + gmc.mtx[5];
         gbox.dx = tbox->dx; gbox.nx = tbox->nx;
         gbox.dy = tbox->dy; gbox.ny = tbox->ny;

         dmPutDesignData(bfp, GEO_BOX);
         }
      }
   }

if(wrong_via == TRUE)
   {
   fprintf(stderr,"WARNING: (write_vias)\n");
   fprintf(stderr,"         Some vias in an illegal layer or at an illegal position.\n");
   }
}



/* * * * * * * * 
 * 
 * This routine is called to make an empty array of 
 * the image
 */
void write_empty_image(char  *cell_name ,
                       long  xsize ,
                       long  ysize )

{
DM_PROJECT   
   *remote_projectkey;
DM_CELL
   *cell_key;
DM_STREAM
   *fp;
char
   *remote_cellname; 
long
   imported,
   bbx[4];


/*
 * 1: look for image 
 */
/*
 * look for this basic image cell
 */
if((imported = (long) exist_cell(ImageName, layout_str)) == -1)
   { /* it does not exist */
   fprintf(stderr,"ERROR: cannot find elementary core image/template cell '%s' in database\n", ImageName);
   return;
   }

/*
 * open it: key for project
 */
if((remote_projectkey = dmFindProjKey((int) imported, ImageName,
    projectkey, &remote_cellname, layout_str)) == NULL) 
   {  /* ? */
   error(ERROR, "cannot find nasty project key");
   return;
   }

/*
 * open it
 */
if((cell_key = dmCheckOut(remote_projectkey, remote_cellname, 
    ACTUAL, DONTCARE, layout_str, READONLY)) == NULL)
   {  /* ? */
   fprintf(stderr,"ERROR: cannot open core cell '%s' in database\n", ImageName);
   return;
   }

/*
 * read bounding box of image
 */
if(!(fp = dmOpenStream(cell_key, "info", "r"))) 
   error(FATAL_ERROR, "write_image");
 
if(dmGetDesignData(fp, GEO_INFO) == -1)
   error(FATAL_ERROR, "write_image");

bbx[L] = ginfo.bxl;
bbx[R] = ginfo.bxr;
bbx[B] = ginfo.byb;
bbx[T] = ginfo.byt;

/*
 * terminate
 */
dmCloseStream(fp, COMPLETE);
dmCheckIn(cell_key, COMPLETE);
 

/*
 * 2: make the actual cell
 */

/*
 * test: is the name too long? 
 */
if(strlen(cell_name) >= DM_MAXNAME)
   { /* print warning */
   fprintf(stderr,"WARNING (write_empty_image): cell name %s too long, truncated\n",
      cell_name); 
   cell_name[DM_MAXNAME - 1] = '\0';
   }

if(verbose == ON)
   {
   printf("\n------ writing empty array %ldx%ld into cell '%s' ------\n", 
            xsize, ysize, cell_name);
   fflush(stdout);
   }

/*
 * open a new model file, called cell_name
 */
if((cell_key = dmCheckOut (projectkey, cell_name, DERIVED, DONTCARE, layout_str, UPDATE)) == NULL) 
   {
   error(ERROR,"Unable to open cell (cell not written)");
   return;
   }

bxl = byb = BIGNUMBER;
bxr = byt = -BIGNUMBER;

/*
 * open mc
 */
if((fp = dmOpenStream(cell_key,"mc","w")) == NULL)
   error(FATAL_ERROR, "write_empty_image");


strcpy(gmc.inst_name,"IMAGE");
strcpy(gmc.cell_name, ImageName);
gmc.imported = imported;
gmc.mtx[0] = 1; gmc.mtx[1] = 0; gmc.mtx[2] = 0;
gmc.mtx[3] = 0; gmc.mtx[4] = 1; gmc.mtx[5] = 0;

gmc.nx = xsize - 1;
gmc.ny = ysize - 1;
gmc.dx = LayoutRepitition[X]; 
gmc.dy = LayoutRepitition[Y];

gmc.bxl = bbx[L];
gmc.byb = bbx[B]; 
gmc.bxr = bbx[R] + (gmc.nx * LayoutRepitition[X]);
gmc.byt = bbx[T] + (gmc.ny * LayoutRepitition[Y]);

bxl = gmc.bxl;
bxr = gmc.bxr;
byb = gmc.byb;
byt = gmc.byt;

dmPutDesignData(fp, GEO_MC);
dmCloseStream(fp, COMPLETE);


/*
 * write terminals
 */
if((fp = dmOpenStream(cell_key,"term","w")) == NULL)
   error(FATAL_ERROR, "write_empty_image");
dmCloseStream(fp, COMPLETE);

/*
 * write box 
 */
if((fp = dmOpenStream(cell_key,"box","w")) == NULL)
   error(FATAL_ERROR, "write_empty_image");
dmCloseStream(fp, COMPLETE);

/*
 * write bounding box
 */
if((fp = dmOpenStream(cell_key,"info","w")) == NULL)
   error(FATAL_ERROR, "write_empty_image");

ginfo.bxl = bxl; ginfo.bxr = bxr;
ginfo.byb = byb; ginfo.byt = byt;

dmPutDesignData(fp, GEO_INFO);

dmCloseStream(fp, COMPLETE); 

/*
 * make misterious files
 */
if(fp = dmOpenStream(cell_key,"nor","w"))
   dmCloseStream(fp, COMPLETE); 

dmCheckIn(cell_key,COMPLETE);
}

