/*
 * Automatic screen saver.  This is a combination of the original "xdim" and
 * "xrss" savers, with the original savers implemented as modules and some
 * provision for other modules to be added.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include "xmss.h"

static int (*oldErrorHandler)(Display *, XErrorEvent *);
static int blanktime, disabled, dying, doprop;
unsigned long hotspot, coldspot;
static _Xconst Module *module;
_Xconst HotGeometry *on, *off;
_Xconst HotKey *hotkey;
Atom XA_XMSS;
time_t now;

/*
 * The signal handler:  set a flag to make the main loop remove our property
 * and exit.  It has to use a flag because we could be in deep doo-doo if we
 * try to send a request to the server from here...
 */

static sigreturn_t
#if NeedFunctionPrototypes
croaked(int sig)
#else
croaked(sig)
    int sig;
#endif
{
    dying = True;
}

/*
 * Clean up by removing our property from the root window.  This is done
 * only if we have a reasonable expectation that it is ours.
 */

static void
#if NeedFunctionPrototypes
cleanup(void)
#else
cleanup()
#endif
{
    if (doprop)
    {
	XDeleteProperty(dpy, root, XA_XMSS);
	XSync(dpy, False);
    }
}

/*
 * Print a warning message.
 */

void
#if NeedFunctionPrototypes
warning(_Xconst char *s, _Xconst char *b)
#else
warning(s, b)
    char *s, *b;
#endif
{
    fprintf(stderr, "%s: %s%s%s\n", CLASS, s, (b? ": ": ""), (b? b: ""));
}

/*
 * A simple fatal error handler:  print a message and exit.
 */

void
#if NeedFunctionPrototypes
fatal(_Xconst char *s, _Xconst char *b)
#else
fatal(s, b)
    char *s, *b;
#endif
{
    warning(s, b);
    cleanup();
    exit(1);
}

/*
 * A void X error handler:  things can change out from under us.
 */

static int
#if NeedFunctionPrototypes
voidErrorHandler(Display *dpy, XErrorEvent *ev)
#else
voidErrorHandler(dpy, ev)
    Display *dpy;
    XErrorEvent *ev;
#endif
{
    return 0;
}

/*
 * Read (or re-read) the resource database and reinitialize all modules.
 */

void
#if NeedFunctionPrototypes
xmss_init(int reinit)
#else
xmss_init(reinit)
    int reinit;
#endif
{
    _Xconst char *cp;

    disabled = getBoolRes("disabled", 0, "Disabled");
    if (!(blanktime = getNumRes("blankTime", 0, "Time")))
	blanktime = 600;
    /* we really should free the old values... */
    on = getHotRes("hotSpots", 0, "HotGeometry");
    off = getHotRes("coldSpots", 0, "HotGeometry");
    hotkey = getKeyRes("hotKey", 0, "Key");
    hotspot = getColorRes("hotColor", 0, "Color");
    coldspot = getColorRes("coldColor", 0, "Color");
    if (!module_init(reinit))
	fatal("no modules could be initialized", 0);
    if (!(module = module_select((cp = getRes("module", 0, "Module")))))
	fatal("no such saver module", cp);
}

/*
 * The main program:  read resource databases, open connection to display,
 * establish event handlers, and loop on events and timer.
 */

int
#if NeedFunctionPrototypes
main(int argc, char **argv)
#else
main(argc, argv)
    int argc;
    char **argv;
#endif
{
    unsigned char pbuf[1024];
    _Xconst Module *curmodule;
    unsigned long zl, zr;
    unsigned char *prop;
    struct sigaction sa;
    struct timeval tv;
    long delay, nus;
    int blanked, t;
    time_t timer;
    fd_set iofs;
    Atom za;
    
    module_options();
    getResources(&argc, argv);
    if (argc > 2 || (argc == 2 && strcmp(argv[1], "-unlock") != 0))
	fatal("unrecognized argument", argv[1]);
    XA_XMSS = XInternAtom(dpy, "XMSS", False);
    if (argc == 2)
    {
	doprop = 1;
	cleanup();
	return 0;
    }
    /* we only need to do this once, so we don't fetch it from xmss_init() */
    XSync(dpy, False);
    if (!(delay = getNumRes("initialDelay", 0, "Delay")))
	delay = 60;
    sleep(delay);
    xmss_init(False);
    XSync(dpy, False);
    /* check for, then install, a property on the root window pointing to us */
    dying = False;
    sa.sa_flags = 0;
    sigemptyset(&sa.sa_mask);
    sa.sa_handler = croaked;
    sigaction(SIGHUP, &sa, 0);
    sigaction(SIGINT, &sa, 0);
    sigaction(SIGQUIT, &sa, 0);
    sigaction(SIGTERM, &sa, 0);
    gethostname(pbuf, sizeof pbuf - 1);
    sprintf(pbuf + strlen(pbuf), ":%d", getpid());
    XGrabServer(dpy);
    prop = 0;
    if (XGetWindowProperty(dpy, root, XA_XMSS, 0, 1024, False, XA_STRING, &za,
			   &t, &zl, &zr, &prop) == Success && zl)
    {
	XUngrabServer(dpy);
	fatal("already running from host:pid", prop);
    }
    XFree(prop);
    doprop = True;
    XChangeProperty(dpy, root, XA_XMSS, XA_STRING, 8, PropModeReplace,
		    pbuf, strlen(pbuf));
    XUngrabServer(dpy);
    time(&now);
    nus = 0;
    registerEvents(root, True);
    queueWindow(root);
    timer = now;
    delay = -1;
    blanked = 0;
    XSetErrorHandler(voidErrorHandler);
    while (!dying)
    {
	XFlush(dpy);
	FD_ZERO(&iofs);
	FD_SET(ConnectionNumber(dpy), &iofs);
	if (delay == -1 || delay > 1000000)
	    t = 1000000;
	else
	    t = delay;
	tv.tv_sec = (t == 1000000);
	tv.tv_usec = (tv.tv_sec? 0: t);
	select(FD_SETSIZE, &iofs, 0, 0, &tv);
	/* not perfect, but should be good enough --- for now, at least */
	if (!FD_ISSET(ConnectionNumber(dpy), &iofs))
	{
	    if ((nus += t) > 1000000)
	    {
		now++;
		nus -= 1000000;
	    }
	    if (delay != -1)
		delay -= t;
	}
	if (checkActive())
	    timer = now;
	if (delay)
	    runQueue();
	switch (checkPointer())
	{
	case -1:
	    return 0;
	case 0:
	    break;
	case 1:
	    timer = now;
	    break;
	case 2:
	    if (!blanked && delay == -1)
		timer = 0;
	    break;
	}
	/* check to see if it's time to act */
	if (disabled)
	{
	    if (blanked)
		delay = (*curmodule->shutdown)();
	    blanked = 0;
	}
	else if ((blanked || delay == -1) &&
		 blanked != (now - timer > blanktime))
	{
	    if (blanked)
		delay = (*curmodule->shutdown)();
	    else
		delay = (*(curmodule = module)->startup)();
	    blanked = !blanked;
	}
	if (!delay && !FD_ISSET(ConnectionNumber(dpy), &iofs))
	{
	    if ((delay = (*curmodule->work)(blanked)) == -2)
	    {
		delay = -1;
		if (blanked)
		    blanked = 0; /* it unblanked... */
		timer = now;	/* prevent immediate reblank */
	    }
	    else if (delay == -1 && !blanked)
		timer = now;
	}
    }
    cleanup();
    return 0;
}
