/*
 * Resource management for xmss.
 */

#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/Xatom.h>
#include "xmss.h"

static XrmOptionDescRec Options[NOPTION] =
{
    "-display",		".display",	 XrmoptionSepArg, (caddr_t) 0,
    "-initial",		".initialDelay", XrmoptionSepArg, (caddr_t) 0,
    "-blanktime",	".blankTime",	 XrmoptionSepArg, (caddr_t) 0,
    "-bt",		".blankTime",	 XrmoptionSepArg, (caddr_t) 0,
    "-name",		".name",	 XrmoptionSepArg, (caddr_t) 0,
    "-xrm",		0,		 XrmoptionResArg, (caddr_t) 0,
    "-hotspots",	".hotSpots",	 XrmoptionSepArg, (caddr_t) 0,
    "-hs",		".hotSpots",	 XrmoptionSepArg, (caddr_t) 0,
    "-coldspots",	".coldSpots",	 XrmoptionSepArg, (caddr_t) 0,
    "-cs",		".coldSpots",	 XrmoptionSepArg, (caddr_t) 0,
    "-hotkey",		".hotKey",	 XrmoptionSepArg, (caddr_t) 0,
    "-hk",		".hotKey",	 XrmoptionSepArg, (caddr_t) 0,
    "-hotcolor",	".hotColor",	 XrmoptionSepArg, (caddr_t) 0,
    "-hc",		".hotColor",	 XrmoptionSepArg, (caddr_t) 0,
    "-coldcolor",	".coldColor",	 XrmoptionSepArg, (caddr_t) 0,
    "-cc",		".coldColor",	 XrmoptionSepArg, (caddr_t) 0,
    "-module",		".module",	 XrmoptionSepArg, (caddr_t) 0,
};

Display *dpy;
Window root;
int scr;

static XrmDatabase appRes;
static _Xconst char *av0;

/*
 * Fetch a raw (string) resource.  We allow the database to be specified
 * because the first call pulls the display from the command line and the
 * second pulls the name from the same place.  All other calls use the
 * complete database.
 *
 * _getRes() is the internal version, and takes a resource database.  It is
 * only used in this file, as part of reading the full application resource
 * context.
 */

static _Xconst char *
#if NeedFunctionPrototypes
_getRes(XrmDatabase db, _Xconst char *res, _Xconst char *class, _Xconst char *cres)
#else
_getRes(db, res, class, cres)
    XrmDatabase db;
    char *res, *class, *cres;
#endif
{
    static char rn[1024], rc[1024];
    XrmValue val;
    char *cp;

    strcpy(rn, av0);
    strcat(rn, ".");
    strcat(rn, res);
    strcpy(rc, CLASS);
    strcat(rc, ".");
    if (class)
    {
	strcat(rc, class);
	strcat(rc, ".");
    }
    strcat(rc, cres);
    if (!XrmGetResource(db, rn, rc, &cp, &val))
	return 0;
    strncpy(rn, val.addr, val.size);
    rn[val.size] = 0;
    return rn;
}

/*
 * The external version uses the full application database.
 */

_Xconst char *
#if NeedFunctionPrototypes
getRes(_Xconst char *res, _Xconst char *class, _Xconst char *cres)
#else
getRes(res, class, cres)
    char *res, *class, *cres;
#endif
{
    return _getRes(appRes, res, class, cres);
}

/*
 * Fetch a numeric resource.
 */

int
#if NeedFunctionPrototypes
getNumRes(_Xconst char *res, _Xconst char *class, _Xconst char *cres)
#else
getNumRes(res, class, cres)
    char *res, *class, *cres;
#endif
{
    _Xconst char *cp;

    if (!(cp = getRes(res, class, cres)))
	return 0;
    /* should use strtol()... */
    return atoi(cp);
}

/*
 * Fetch a boolean resource.
 */

static _Xconst char *bools[] =
{
    "on",
    "true",
    "1",
    "yes",
    0,
};

int
#if NeedFunctionPrototypes
getBoolRes(_Xconst char *res, _Xconst char *class, _Xconst char *cres)
#else
getBoolRes(res, class, cres)
    char *res, *class, *cres;
#endif
{
    _Xconst char *cp;
    _Xconst char **x;

    if (!(cp = getRes(res, class, cres)))
	return 0;
    for (x = bools; *x; x++)
    {
	if (strcasecmp(*x, cp) == 0)
	    return 1;
    }
    return 0;
}

/*
 * Fetch a geometry list resource - two of them, but important
 * I hope XWMGeometry() doesn't query the WM... it's easier to use in some ways
 * than XGeometry(), but the geometries handled here *aren't* for the window
 * manager to use!
 */

_Xconst HotGeometry *
#if NeedFunctionPrototypes
getHotRes(_Xconst char *res, _Xconst char *class, _Xconst char *cres)
#else
getHotRes(res, class, cres)
    char *res, *class, *cres;
#endif
{
    HotGeometry *gp, *hot;
    int x, y, w, h, g;
    _Xconst char *cp;
    char *bp, *buf;
    XSizeHints sh;

    if (!(cp = getRes(res, class, cres)))
	return 0;
    hot = 0;
    if (!(buf = strdup(cp)))	/* writeable version, so we can strtok() it */
	fatal("can't dup resource", cp);
    sh.base_width = 5;		/* only needed to support minimum size */
    sh.base_height = 5;
    sh.flags = PBaseSize;
    for (bp = buf; (cp = strtok(bp, ", ")); bp = 0)
    {
	XWMGeometry(dpy, scr, cp, "5x5", 0, &sh, &x, &y, &w, &h, &g);
	/* possibly should check this more carefully... */
	if (w > 0 && h > 0)
	{
	    if (!(gp = malloc(sizeof *gp)))
		fatal("can't allocate HotGeometry", 0);
	    gp->next = hot;
	    gp->x = x;
	    gp->y = y;
	    gp->w = w;
	    gp->h = h;
	    hot = gp;
	}
    }
    free(buf);
    return hot;
}

/*
 * Fetch a color resource
 */

unsigned long
#if NeedFunctionPrototypes
getColorRes(_Xconst char *res, _Xconst char *class, _Xconst char *cres)
#else
getColorRes(res, class, cres)
    char *res, *class, *cres;
#endif
{
    _Xconst char *cp;
    XColor c1, c2;

    if (!(cp = getRes(res, class, cres)))
	return BAD;
    if (!XAllocNamedColor(dpy, DefaultColormap(dpy, scr), cp, &c1, &c2))
	return BAD;
    return c1.pixel;
}

/*
 * Fetch a keysym resource.  Slightly tricky; I wish I could use Xt's converter
 * to do this for me, but...  We recognize a key as a keysym with zero or more
 * modifiers in front of it; those modifiers are the known X modifiers (Alt,
 * Meta, etc.) with optional "!" or "~" to indicate that the modifier should
 * *not* be present.  Example:  "Control ~Meta slash".
 *
 * Note that only key modifiers can be used, the <Key> token cannot appear, and
 * Xt's ":" modifier cannot be used.
 */

static unsigned int
#if NeedFunctionPrototypes
key_is_modifier(_Xconst char *key)
#else
key_is_modifier(key)
    char *key;
#endif
{
    struct xmod
    {
	struct xmod *next;
	char *name;
	unsigned long mod;
    } *xmp;
    static XModifierKeymap *map;
    static struct xmod *xmod;
    char *cp, *bp;
    KeySym k;
    int i;

    if (strcasecmp(key, "shift") == 0)
	return ShiftMask;
    if (strcasecmp(key, "lock") == 0)
	return LockMask;
    if (strcasecmp(key, "control") == 0)
	return ControlMask;
    if (strcasecmp(key, "mod1") == 0)
	return Mod1Mask;
    if (strcasecmp(key, "mod2") == 0)
	return Mod2Mask;
    if (strcasecmp(key, "mod3") == 0)
	return Mod3Mask;
    if (strcasecmp(key, "mod4") == 0)
	return Mod4Mask;
    if (strcasecmp(key, "mod5") == 0)
	return Mod5Mask;
    if (!map)
    {
	map = XGetModifierMapping(dpy);
	for (i = 8 * map->max_keypermod; i--; )
	{
	    if (!map->modifiermap[i])
		continue;
	    if ((k = XKeycodeToKeysym(dpy,map->modifiermap[i],0)) == NoSymbol)
		continue;
	    if (!(cp = XKeysymToString(k)))
		continue;
	    if (!(cp = strdup(cp)))
		fatal("can't dup keysym", cp);
	    for (bp = cp; *bp; bp++)
		;
	    if (bp[-2] == '_' && (bp[-1] == 'L' || bp[-1] == 'R'))
		bp[-2] = '\0';
	    if (!(xmp = malloc(sizeof *xmp)))
		fatal("can't allocate modifier translation", 0);
	    xmp->next = xmod;
	    xmod = xmp;
	    xmp->name = cp;
	    switch (i / map->max_keypermod)
	    {
	    case ShiftMapIndex:
		xmod->mod = ShiftMask;
		break;
	    case LockMapIndex:
		xmod->mod = LockMask;
		break;
	    case ControlMapIndex:
		xmod->mod = ControlMask;
		break;
	    case Mod1MapIndex:
		xmod->mod = Mod1Mask;
		break;
	    case Mod2MapIndex:
		xmod->mod = Mod2Mask;
		break;
	    case Mod3MapIndex:
		xmod->mod = Mod3Mask;
		break;
	    case Mod4MapIndex:
		xmod->mod = Mod4Mask;
		break;
	    case Mod5MapIndex:
		xmod->mod = Mod5Mask;
		break;
	    default:
		fatal("key_is_modifier: internal: unrecognized modifier", 0);
	    }
	}
    }
    for (xmp = xmod; xmp; xmp = xmp->next)
    {
	if (strcasecmp(xmp->name, key) == 0)
	    return xmp->mod;
    }
    return 0;
}

_Xconst HotKey *
#if NeedFunctionPrototypes
getKeyRes(_Xconst char *res, _Xconst char *class, _Xconst char *cres)
#else
getKeyRes(res, class, cres)
    char *res, *class, *cres;
#endif
{
    int not, mod, exact;
    _Xconst char *cp;
    char *buf, *bp;
    HotKey *ks;
    KeyCode c;
    KeySym k;

    if (!(cp = getRes(res, class, cres)))
	return 0;
    if (!(buf = strdup(cp)))	/* writeable version, so we can strtok() it */
	fatal("can't dup resource", cp);
    if (!(ks = malloc(sizeof *ks)))
	fatal("can't allocate key event", 0);
    ks->avoid = 0;
    ks->require = 0;
    ks->key = 0;
    bp = buf;
    if ((exact = (*bp == '!')))
	bp++;
    for (; (cp = strtok(bp, ", <>")); bp = 0)
    {
	if (ks->key != 0)
	{
	    warning("only one non-modifier allowed in key event", cp);
	    free(buf);
	    free(ks);
	    return 0;
	}
	if ((not = (*cp == '~')))
	{
	    if (exact)
	    {
		warning("can't combine ~ and !", 0);
		free(buf);
		free(ks);
		return 0;
	    }
	    cp++;
	}
	/* ignore these just in case we get handed an Xt event specifier */
	if (strcasecmp(cp, "Key") == 0 || strcasecmp(cp, "KeyPress") == 0)
	    continue;
	if ((mod = key_is_modifier(cp)) == 0)
	{
	    if ((k = XStringToKeysym(cp)) == NoSymbol)
	    {
		warning("unknown keysym", cp);
		free(buf);
		free(ks);
		return 0;
	    }
	    if (!(c = XKeysymToKeycode(dpy, k)))
	    {
		warning("keysym not mapped to keycode", cp);
		free(buf);
		free(ks);
		return 0;
	    }
	    if (not)
	    {
		warning("keysym cannot be ~'ed", cp);
		free(buf);
		free(ks);
		return 0;
	    }
	    ks->key = c;
	}
	else if (strcasecmp(cp, "None") == 0)
	{
	    if (not)
	    {
		warning("modifier ~None is illegal", 0);
		free(buf);
		free(ks);
		return 0;
	    }
	    if (exact)
	    {
		warning("modifier !None is illegal", 0);
		free(buf);
		free(ks);
		return 0;
	    }
	    if (ks->avoid || ks->require)
	    {
		warning("can't combine None with modifiers", 0);
		free(buf);
		free(ks);
		return 0;
	    }
	    ks->avoid = ~0;
	}
	else if (not)
	    ks->avoid |= mod;
	else
	    ks->require |= mod;
    }
    if (!ks->key)
    {
	warning("a non-modifier must be included in the key event", 0);
	free(buf);
	free(ks);
	return 0;
    }
    if (exact)
	ks->avoid = ~ks->require;
    free(buf);
    return ks;
}

/*
 * Add the specified options to the master options list.  Duplicate options
 * will be ignored; some modules (e.g. 2600 and dim) use common options via
 * class resources, and making them share option records is difficult.  Maybe
 * this means a redesign is needed...
 */

void
#if NeedFunctionPrototypes
addOptions(_Xconst XrmOptionDescRec *opt, int cnt, _Xconst char *src)
#else
addOptions(opt, cnt, src)
    XrmOptionDescRec *opt;
    int cnt;
    char *src;
#endif
{
    int nr, l;

    if (!cnt)
	return;
    if ((l = strlen(src)) < strlen(SCLASS))
	l = strlen(SCLASS);
    l += 3;
    for (nr = 0; Options[nr].option; nr++)
	;
    while (cnt--)
    {
	Options[nr].option = opt->option;
	if (!(Options[nr].specifier = malloc(strlen(opt->specifier) + l)))
	    fatal("addOptions: out of memory", 0);
	strcpy(Options[nr].specifier, ".");
	/* kluge.  Definitely we need to redesign for classes... */
	if (isupper(opt->specifier[1]))
	    strcat(Options[nr].specifier, SCLASS);
	else
	    strcat(Options[nr].specifier, src);
	strcat(Options[nr].specifier, opt->specifier);
	Options[nr].argKind = opt->argKind;
	Options[nr].value = opt->value;
	opt++;
	nr++;
    }
}

/*
 * Process options and resource files.  Return the -name specified on the
 * command line, or one created from the basename otherwise.  This opens the
 * display as a side effect.
 */

void
#if NeedFunctionPrototypes
getResources(int *argc, char **argv)
#else
getResources(argc, argv)
    int *argc;
    char **argv;
#endif
{
    XrmDatabase cmdLine, defaults;
    char rmbuf[1024];
    _Xconst char *cp;
    int nr;

    XrmInitialize();
    cmdLine = 0;
    if ((av0 = strrchr(argv[0], '/')))
	av0++;
    else
	av0 = argv[0];
    for (nr = 0; Options[nr].option; nr++)
	;
    XrmParseCommand(&cmdLine, Options, nr, av0, argc, argv);
    if ((cp = _getRes(cmdLine, "name", 0, "Name")))
	av0 = cp;
    if (!(dpy = XOpenDisplay(cp = _getRes(cmdLine, "display", 0, "Display"))))
	fatal("can't open display", cp);
    scr = DefaultScreen(dpy);
    root = RootWindow(dpy, scr);
#ifdef sun
    /* Sun throws a monkey wrench into the works... */
    if (!(cp = getenv("OPENWINHOME")))
#endif
	strcpy(rmbuf, "/usr/lib/X11");
#ifdef sun
    else
    {
	strcpy(rmbuf, cp);
	strcat(rmbuf, "/lib");
    }
#endif
    strcat(rmbuf, "/app-defaults/");
    strcat(rmbuf, CLASS);
    defaults = XrmGetFileDatabase(rmbuf);
    XrmMergeDatabases(defaults, &appRes);
    if ((cp = XResourceManagerString(dpy)))
	defaults = XrmGetStringDatabase(cp);
    else
    {
	/* some use one, others use the other... */
	strcpy(rmbuf, getenv("HOME"));
	strcat(rmbuf, "/.Xdefaults");
	defaults = XrmGetFileDatabase(rmbuf);
	XrmMergeDatabases(defaults, &appRes);
	strcpy(rmbuf, getenv("HOME"));
	strcat(rmbuf, "/.Xresources");
	defaults = XrmGetFileDatabase(rmbuf);
    }
    XrmMergeDatabases(defaults, &appRes);
    if ((cp = getenv("XENVIRONMENT")))
	defaults = XrmGetFileDatabase(cp);
    else
    {
	strcpy(rmbuf, getenv("HOME"));
	strcat(rmbuf, ".Xdefaults-");
	gethostname(rmbuf + strlen(rmbuf), sizeof rmbuf - strlen(rmbuf) - 1);
	defaults = XrmGetFileDatabase(rmbuf);
    }
    XrmMergeDatabases(defaults, &appRes);
    XrmMergeDatabases(cmdLine, &appRes);
}

/*
 * Update the resource database by reading the current value of the
 * XA_RESOURCE_MANAGER property, then reinitialize all active modules so they
 * can update their resources.
 */

void
#if NeedFunctionPrototypes
update_resources(void)
#else
update_resources()
#endif
{
    unsigned long cnt, rest;
    unsigned char *prop;
    XrmDatabase res;
    Atom z;
    int n;

    if (XGetWindowProperty(dpy, root, XA_RESOURCE_MANAGER, 0, 131072, False,
			   XA_STRING, &z, &n, &cnt, &rest, &prop) != Success)
	fatal("can't read XA_RESOURCE_MANAGER property on root window", 0);
    if (rest)
	fatal("ridiculously long XA_RESOURCE_MANAGER property", 0);
    res = XrmGetStringDatabase(prop);
    XFree(prop);
    XrmMergeDatabases(res, &appRes);
    xmss_init(True);
}
