/*
 * xmss window queues and events.
 */

#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include "xmss.h"

typedef struct wqueue
{
    struct wqueue *next;
    Window w;
    time_t time;
} Queue;

static int x = -1, y = -1;
static Queue *q;

/*
 * Recursively register interest in events on windows.  Generally we are called
 * with the root window and progress downward; this allows us to find out
 * about all mouse and key events, so we can reset the timer.
 *
 * This was pretty much taken from xautolock.  The major difference is that
 * only windows in the screen specified in $DISPLAY are registered, since
 * colormap-mangling screen savers only work on a single screen.
 */

void
#if NeedFunctionPrototypes
registerEvents(Window w, int no_action)
#else
registerEvents(w, no_action)
    Window w;
    int no_action;
#endif
{
    XWindowAttributes attrs;
    unsigned long mask;
    Window rw, par;
    Window *child;
    int nchild, i;

    if (!XQueryTree(dpy, w, &rw, &par, &child, &nchild))
	return;
    if (rw != root)
	return;
    mask = SubstructureNotifyMask;
    if (w == root)
	mask |= PropertyChangeMask;
    if (no_action)
	XSelectInput(dpy, w, mask);
    else
    {
	if (par == None)
	{
	    attrs.all_event_masks = attrs.do_not_propagate_mask =
		KeyPressMask|KeyReleaseMask;
	}
	else if (!XGetWindowAttributes(dpy, w, &attrs))
	    return;
	XSelectInput(dpy, w, mask |
		     ((attrs.all_event_masks|attrs.do_not_propagate_mask) &
		      (KeyPressMask|KeyReleaseMask)));
    }
    for (i = nchild; i--; )
	registerEvents(child[i], no_action);
    if (nchild)
	XFree(child);
}

/*
 * Add a window creation event to the queue.
 */

void
#if NeedFunctionPrototypes
queueWindow(Window w)
#else
queueWindow(w)
    Window w;
#endif
{
    Queue *nq;

    if (!(nq = malloc(sizeof *nq)))
	fatal("can't allocate queue entry", 0);
    nq->w = w;
    nq->time = now + 30;
    nq->next = q;
    q = nq;
}

/*
 * Process events and queues.
 */

void
#if NeedFunctionPrototypes
runQueue(void)
#else
runQueue()
#endif
{
    Queue *nq;

    while (q && q->time < now)
    {
	registerEvents(q->w, False);
	nq = q;
	q = q->next;
	free(nq);
    }
}

/*
 * Check for window activity and hotkeys.
 */

int
#if NeedFunctionPrototypes
checkActive(void)
#else
checkActive()
#endif
{
    int active;
    XEvent ev;

    active = False;
    while (XPending(dpy))
    {
	XNextEvent(dpy, &ev);
	if (ev.type == CreateNotify && !is_hot(ev.xcreatewindow.window))
	    queueWindow(ev.xcreatewindow.window);
	else if (ev.type == PropertyNotify &&
		 ev.xproperty.state == PropertyNewValue &&
		 ev.xproperty.atom == XA_RESOURCE_MANAGER)
	    update_resources();
	else if (ev.type == PropertyNotify &&
		 ev.xproperty.atom == XA_XMSS)
	    fatal("lock on root window broken", 0);
	else if (ev.type == KeyPress && !ev.xany.send_event)
	{
	    if (hotkey &&
		(ev.xkey.state & hotkey->require) == hotkey->require &&
		!(ev.xkey.state & hotkey->avoid) &&
		hotkey->key == ev.xkey.keycode)
	    {
		add_hot(on, hotspot);
		add_hot(off, coldspot);
	    }
	    else
	    {
		del_hot(on);
		del_hot(off);
	    }
	    active = True;
	}
	else if (ev.type == KeyRelease && !ev.xany.send_event)
	{
	    del_hot(on);
	    del_hot(off);
	    active = True;
	}
    }
    return active;
}

/*
 * Check for pointer activity and hotspots.  This returns -1, 0, 1, or 2;
 * -1 is a lost connection, 0 is no activity, 1 for activity, 2 to force a
 * screen save because of a hotspot.
 */

int
#if NeedFunctionPrototypes
checkPointer(void)
#else
checkPointer()
#endif
{
    int active, nx, ny, z;
    _Xconst HotGeometry *hot;
    Window nr, t;

    active = 0;
    if (!XQueryPointer(dpy, root, &nr, &t, &nx, &ny, &z, &z, &z))
	return -1;
    if (root != nr)
    {
	del_hot(on);
	del_hot(off);
	hide_hot();
	nx = ny = -1;
    }
    else if ((hot = ck_hot(nx, ny, on)))
    {
	del_hot(on);
	del_hot(off);
	show_hot(hot, hotspot);
	active = 2;
    }
    else if ((hot = ck_hot(nx, ny, off)))
    {
	del_hot(on);
	del_hot(off);
	show_hot(hot, coldspot);
	active = 1;
    }
    else if (nx != x || ny != y)
    {
	del_hot(on);
	del_hot(off);
	hide_hot();
	active = 1;
    }
    else
	hide_hot();
    x = nx;
    y = ny;
    return active;
}
