/* digital clock for X  by Li Wei Jih */

#include <stdio.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "position.c"
#include "dclock.xbm"
#include "dclock_nosec.xbm"
#include "digit0.xbm"
#include "digit1.xbm"
#include "digit2.xbm"
#include "digit3.xbm"
#include "digit4.xbm"
#include "digit5.xbm"
#include "digit6.xbm"
#include "digit7.xbm"
#include "digit8.xbm"
#include "digit9.xbm"
#include "space.xbm"
#include "colon.xbm"
#include "nocolon.xbm"
#include "sec0.xbm"
#include "sec1.xbm"
#include "sec2.xbm"
#include "sec3.xbm"
#include "sec4.xbm"
#include "sec5.xbm"
#include "sec6.xbm"
#include "sec7.xbm"
#include "sec8.xbm"
#include "sec9.xbm"
#include "am.xbm"
#include "pm.xbm"
#include "hr24.xbm"
#include "am_nosec.xbm"
#include "pm_nosec.xbm"
#include "dash.xbm"
#include "su.xbm"
#include "mo.xbm"
#include "tu.xbm"
#include "we.xbm"
#include "th.xbm"
#include "fr.xbm"
#include "sa.xbm"
#include "sound.c"

char copyright[]= "digital clock for X  by Li Wei Jih";
char appname[]= "clock", iconname[]= "digital clock";
Pixmap dclock_pixmap, dclock_nosec_pixmap, digit_pixmap[11],
       colon_pixmap, nocolon_pixmap, sec_pixmap[10],
       ampm_pixmap[3], ampm_nosec_pixmap[2], dash_pixmap, day_pixmap[7];
char hr12= 0;
char no_second= 0;
char displayname[255]= "";
pid_t child= 0;
XSizeHints hint;

static time_t getdigitstr(timearray)
char *timearray;
{
  time_t cal;
  struct tm *l_time;
  static int l_hr= -1;
  
  cal= time((time_t*) NULL);
  l_time= localtime(&cal);

  if (l_time->tm_min==0)
    if (l_hr!=l_time->tm_hour)
    {
      child= play_sound();
      l_hr= l_time->tm_hour;
    }
    
  if (hr12)
  {					/* 12-hour */
    timearray[6]= l_time->tm_hour>=12;	/* AM/PM */
    l_time->tm_hour= (l_time->tm_hour+11)%12+1;
  }
  else
    timearray[6]= 2;			/* 24-hour */    
  timearray[0]= l_time->tm_hour/10;  timearray[1]= l_time->tm_hour%10;
  timearray[2]= l_time->tm_min/10;   timearray[3]= l_time->tm_min%10;
  timearray[4]= l_time->tm_sec/10;   timearray[5]= l_time->tm_sec%10;

  return (cal);
}

static void showtime(display, window, gc, timearray)
Display *display;
Window window;
GC gc;
char *timearray;
{
  int i;

  if (timearray[0]==0)
    timearray[0]= 10;

  for (i=0; i<4; i++)
    XCopyPlane(display, digit_pixmap[timearray[i]], window, gc,	/* hour & min */
               0, 0, digit0_width, digit0_height, digit_x[i], digit_y, 1);
  if (no_second)
  {
    static char last_sec= -1, show_colon= 0;
    if (timearray[6]<2)
      XCopyPlane(display, ampm_nosec_pixmap[timearray[6]],	/* AM/PM */
                 window, gc, 0, 0, am_nosec_width, am_nosec_height,
                 ampm_nosec_x, ampm_nosec_y, 1);
    if (last_sec!=timearray[5])
    {
      last_sec= timearray[5];
      if (show_colon= !show_colon)
        XCopyPlane(display, colon_pixmap, window, gc,
                   0, 0, colon_width, colon_height, colon_x, colon_y, 1);
      else
        XCopyPlane(display, nocolon_pixmap, window, gc,
                   0, 0, colon_width, colon_height, colon_x, colon_y, 1);        
    }
  }
  else
  {
    XCopyPlane(display, sec_pixmap[timearray[4]], window, gc,	/* second */
               0, 0, sec0_width, sec0_height, sec_x[0], sec_y, 1);
    XCopyPlane(display, sec_pixmap[timearray[5]], window, gc,
               0, 0, sec0_width, sec0_height, sec_x[1], sec_y, 1);
    XCopyPlane(display, ampm_pixmap[timearray[6]], window, gc,	/* AM/PM */
               0, 0, am_width, am_height, ampm_x, ampm_y, 1);
    XCopyPlane(display, colon_pixmap, window, gc,
               0, 0, colon_width, colon_height, colon_x, colon_y, 1);
  }
}

static void showdate(display, window, gc)
Display *display;
Window window;
GC gc;
{
  time_t cal;
  struct tm *l_time;
  char datearray[4];
  int i;
  
  cal= time((time_t*) NULL);
  l_time= localtime(&cal);
  datearray[0]= (l_time->tm_mon>=9)? 1:10;
  datearray[1]= (l_time->tm_mon+1)%10;
  datearray[2]= l_time->tm_mday/10;  datearray[3]= l_time->tm_mday%10;
  if (datearray[2]==0)
    datearray[2]= 10;
  for (i=0; i<4; i++)
    XCopyPlane(display, digit_pixmap[datearray[i]], window, gc,	/* date */
               0, 0, digit0_width, digit0_height, digit_x[i], digit_y, 1);
  XCopyPlane(display, day_pixmap[l_time->tm_wday], window, gc,	/* day */
             0, 0, su_width, su_height, day_x, day_y, 1);
  XCopyPlane(display, dash_pixmap, window, gc,
             0, 0, dash_width, dash_height, dash_x, dash_y, 1);
  XCopyPlane(display, ampm_pixmap[2], window, gc,
             0, 0, am_width, am_height, ampm_x, ampm_y, 1);
}

void changesize(display, window)
Display *display;
Window window;
{
  if (no_second)
    hint.width= hint.min_width= hint.max_width= dclock_nosec_width;
  else
    hint.width= hint.min_width= hint.max_width= dclock_width;
  XResizeWindow(display, window, hint.width, hint.height);
  XSetNormalHints(display, window, &hint);
}

void loadpixmap(display, window)
Display *display;
Window window;
{
  char *digit_bits[11], *sec_bits[10], *ampm_bits[3],
       *ampm_nosec_bits[3], *day_bits[7];

  int i;
					/* pixmaps to show hour & minute */
  digit_bits[0]= digit0_bits;  digit_bits[1]= digit1_bits;
  digit_bits[2]= digit2_bits;  digit_bits[3]= digit3_bits;
  digit_bits[4]= digit4_bits;  digit_bits[5]= digit5_bits;
  digit_bits[6]= digit6_bits;  digit_bits[7]= digit7_bits;
  digit_bits[8]= digit8_bits;  digit_bits[9]= digit9_bits;
  digit_bits[10]= space_bits;
  for (i=0; i<11; i++)
    digit_pixmap[i]= XCreateBitmapFromData(display, window, digit_bits[i],
                                           digit0_width, digit0_height);

  					/* pixmaps to show second */
  sec_bits[0]= sec0_bits;  sec_bits[1]= sec1_bits;
  sec_bits[2]= sec2_bits;  sec_bits[3]= sec3_bits;
  sec_bits[4]= sec4_bits;  sec_bits[5]= sec5_bits;
  sec_bits[6]= sec6_bits;  sec_bits[7]= sec7_bits;
  sec_bits[8]= sec8_bits;  sec_bits[9]= sec9_bits;
  for (i=0; i<10; i++)
    sec_pixmap[i]= XCreateBitmapFromData(display, window, sec_bits[i],
                                         sec0_width, sec0_height);

  					/* pixmaps to show AM/PM */
  ampm_bits[0]= am_bits;  ampm_bits[1]= pm_bits;  ampm_bits[2]= hr24_bits;
  for (i=0; i<3; i++)
    ampm_pixmap[i]= XCreateBitmapFromData(display, window, ampm_bits[i],
                                          am_width, am_height);

  				/* pixmaps to show AM/PM without second */
  ampm_nosec_bits[0]= am_nosec_bits; ampm_nosec_bits[1]= pm_nosec_bits;
  for (i=0; i<2; i++)
    ampm_nosec_pixmap[i]=
      XCreateBitmapFromData(display, window, ampm_nosec_bits[i],
                            am_nosec_width, am_nosec_height);

					/* pixmap to show clock */
  dclock_pixmap= XCreateBitmapFromData(display, window, dclock_bits,
                                       dclock_width, dclock_height);

				/* pixmap to show clock without second */
  dclock_nosec_pixmap=
    XCreateBitmapFromData(display, window, dclock_nosec_bits,
                          dclock_nosec_width, dclock_nosec_height);

					/* pixmap to show colon */
  colon_pixmap= XCreateBitmapFromData(display, window, colon_bits,
                                      colon_width, colon_height);
  nocolon_pixmap= XCreateBitmapFromData(display, window, nocolon_bits,
                                        colon_width, colon_height);
                                      
  dash_pixmap= XCreateBitmapFromData(display, window, dash_bits,
                                      dash_width, dash_height);
  day_bits[0]= su_bits;  day_bits[1]= mo_bits;  day_bits[2]= tu_bits;
  day_bits[3]= we_bits;  day_bits[4]= th_bits;  day_bits[5]= fr_bits;
  day_bits[6]= sa_bits;
  for (i=0; i<7; i++)
    day_pixmap[i]= XCreateBitmapFromData(display, window, day_bits[i],
                                         su_width, su_height);  
}

void show_usage()
{
  char usage_text[]=
  {
    "\nUsage: dclock [option]\n\n"
    "  -1, --12                      Display 12 hour time.\n"
    "  -2, --24                      Display 24 hour time.\n"
    "  -n, --nosecond                Don't display 'second'.\n"
    "  -p, --position x,y            Position to display clock.\n"
    "  -d, --display displayname     The X server to contact.\n"
    "  -h, --help                    Display this help.\n"
  };
  puts(usage_text);
}

int args(argc, argv)
int argc;
char **argv;
{
  int c, option_index= 0;
  static struct option long_options[]=
  {
    {"12", 0, 0, '1'},
    {"24", 0, 0, '2'},
    {"nosecond", 0, 0, 'n'},
    {"position", 1, 0, 'p'},
    {"display", 1, 0, 'd'},
    {"help", 0, 0, 'h'},
  };
  opterr= 0;
  while ((c= getopt_long(argc, argv, "12np:d:h",
                         long_options, &option_index))!=EOF)
  {
    switch (c)
    {
      case '1':
        hr12= 1;
        break;
      case '2':
        hr12= 0;
        break;
      case 'n':
        no_second= 1;
        break;
      case 'p':
        sscanf(optarg, "%d,%d", &hint.x, &hint.y);
        break;
      case 'd':
        strcpy(displayname, optarg);
        break;
      case 'h':
        show_usage();
        return (1);
      case '?':
      default:
        puts("dclock: illegal option");
        puts("Try `dclock --help' for more information.\n");
        return (1);
    }
  }
  return (0);
}

Window initializewindow(display, window, argc, argv, gc)
Display **display;
Window *window;
int argc;
char **argv;GC *gc;
{
  int screen;
  Colormap cmap;
  XColor light;
  unsigned long foreground, background;
  Pixmap clock_pixmap;

  hint.x= hint.y= 100;
  if (args(argc, argv))
    exit(0);

  if (!(*display= XOpenDisplay(displayname)))
  {
    printf("Error: can't open display: %s\n\n", displayname);
    exit(0);
  }

  if (no_second)
  {
    hint.width= hint.min_width= hint.max_width= dclock_nosec_width;
    hint.height= hint.min_height= hint.max_height= dclock_nosec_height;
  }
  else
  {
    hint.width= hint.min_width= hint.max_width= dclock_width;
    hint.height= hint.min_height= hint.max_height= dclock_height;
  }
  hint.flags= PPosition | PSize | PMinSize | PMaxSize;

  cmap= DefaultColormap(*display, screen= DefaultScreen(*display));
  light.red= 65535; light.green= 65535; light.blue= 0;
  XAllocColor(*display, cmap, &light);  
  background= BlackPixel(*display, screen);
  foreground= light.pixel;		/* bg: black, fg: yellow */

  *window= XCreateSimpleWindow(*display, DefaultRootWindow(*display),
                               hint.x, hint.y, hint.width, hint.height,
                               0, foreground, background);

  loadpixmap(*display, *window);	/* load pixmaps */

  *gc= XCreateGC(*display, *window, 0, 0);
  XSetBackground(*display, *gc, background);
  XSetForeground(*display, *gc, foreground);

  if (no_second)
    clock_pixmap= dclock_nosec_pixmap;
  else
    clock_pixmap= dclock_pixmap;

  XSetStandardProperties(*display, *window, appname, iconname,
                         clock_pixmap, argv, argc, &hint);

  XSelectInput(*display, *window, ExposureMask | KeyPressMask | KeyReleaseMask |
                                  VisibilityChangeMask | StructureNotifyMask |
                                  ButtonPressMask | Button1MotionMask |
                                  FocusChangeMask);
  XMapRaised(*display, *window);
}


static void processevent(display, window, gc, timer, timearray)
Display *display;
Window window;
GC gc;
int *timer;
char *timearray;
{
  while (XEventsQueued(display, QueuedAfterReading))
  {
    XEvent event;
    KeySym key;
    char buffer[8];
    static int org_x, org_y;

    XNextEvent(display, &event);
    switch(event.type)
    {
      case Expose:			/* updates clock display */
        XCopyPlane(display, colon_pixmap, window, gc,
                   0, 0, colon_width, colon_height, colon_x, colon_y, 1);
        showtime(display, window, gc,  timearray);
        break;
      case UnmapNotify:
        XMapWindow(display, window);	/* uniconfies clock if iconfied */
        break;
      case ButtonPress:
        if (event.xbutton.button>1)
        {				/* button 2/3 pressed to lower clock */
          XLowerWindow(display, window);
          *timer= 7;			/* lowers clock for 7 sec */
        }
        else
        {				/* button 1 pressed to move clock */
          XRaiseWindow(display, window);
          org_x= event.xbutton.x;	/* gets mouse position */
          org_y= event.xbutton.y;	/* befroe moves clock */
        }
        break;
      case MotionNotify:		/* moves clock */
        XMoveWindow(display, window, 
                    event.xmotion.x_root- org_x, event.xmotion.y_root- org_y);
        break;
      case VisibilityNotify:
        switch (event.xvisibility.state)
        {
          case VisibilityPartiallyObscured:
          case VisibilityFullyObscured:
            if (*timer==0)		/* raises clock if obscured */
              XRaiseWindow(display, window);
            break;
          default:
            getdigitstr(timearray);
            showtime(display, window, gc, timearray);
        }
        break;
      case KeyPress:
        if (XLookupString(&event.xkey, buffer, sizeof(buffer), &key, NULL))
          switch(buffer[0])
          {
            case ' ':
              showdate(display, window, gc);
              do
                XNextEvent(display, &event);
              while (event.type!=KeyRelease && event.type!=FocusOut);
              break;
            case 't':
            case 'T':
            case '\t':			/* 12/24-hr switch */
              hr12= !hr12;
              getdigitstr(timearray);
              showtime(display, window, gc, timearray);
              break;
            case 's':
            case 'S':
              no_second= !no_second;
              changesize(display, window);
              showtime(display, window, gc, timearray);
              break;
            case 'q':
            case 'Q':			/* quits program */
              XFreeGC(display, gc);
              XDestroyWindow(display, window);
              XCloseDisplay(display);
              exit(0);
          }
        break;
    }
  }
}
  

static void runclock(display, window, gc)
Display *display;
Window window;
GC gc;
{
  int timer= 0;  
  char timearray[7];

  getdigitstr(timearray);
  processevent(display, window, gc, &timer, timearray);
  for (;;)
  {
    static time_t l_time= 0;
    
    if (l_time!=time((time_t*) NULL))	/* updates time per second*/
    {
      l_time= getdigitstr(timearray);
      showtime(display, window, gc, timearray);
      if (child>0)
        child= waitpid(child, NULL, WNOHANG) +1;
      if (timer>0)
      {
        if (timer--==1)
          XRaiseWindow(display, window);
      }
    }
    XFlush(display);
    usleep(100000L);			/* 1/10 sec */
    processevent(display, window, gc, &timer, timearray);
  }
}

main(argc, argv)
int argc;
char **argv;
{
  Display *display;
  Window window;
  GC gc;
  
  initializewindow(&display, &window, argc, argv, &gc);
  
  runclock(display, window, gc);
}
