#include "astro.h"

#if defined(__STDC__) || defined(__cplusplus)
#define P_(s) s
#else
#define P_(s) ()
#endif

extern void cal_mjd P_((int mn, double dy, int yr, double *Mjd));
extern void mjd_cal P_((double Mjd, int *mn, double *dy, int *yr));
extern void range P_((double *v, double r));

void utc_gst P_((double mjd, double utc, double *gst));
void gst_utc P_((double mjd, double gst, double *utc));
static double tnaught P_((double mjd));

#undef P_



/* given a modified julian date, mjd, and a universally coordinated time, utc,
 * return greenwich mean siderial time, *gst.
 * N.B. mjd must be at the beginning of the day.
 */
void
utc_gst (mjd, utc, gst)
double mjd;
double utc;
double *gst;
{
	static double lastmjd = -10000;
	static double t0;

	if (mjd != lastmjd) {
	    t0 = tnaught (mjd);
	    lastmjd = mjd;
	}
	*gst = (1.0/SIDRATE)*utc + t0;
	range (gst, 24.0);
}

/* given a modified julian date, mjd, and a greenwich mean siderial time, gst,
 * return universally coordinated time, *utc.
 */
void
gst_utc (mjd, gst, utc)
double mjd;
double gst;
double *utc;
{
	static double lastmjd = -10000;
	static double t0;

	if (mjd != lastmjd) {
	    t0 = tnaught (mjd);
	    range (&t0, 24.0);
	    lastmjd = mjd;
	}
	*utc = gst - t0;
	range (utc, 24.0);
	*utc *= SIDRATE;
}

static double
tnaught (mjd)
double mjd;	/* julian days since 1900 jan 0.5 */
{
	double dmjd;
	int m, y;
	double d;
	double t, t0;

	mjd_cal (mjd, &m, &d, &y);
	cal_mjd (1, 0., y, &dmjd);
	t = dmjd/36525;
	t0 = 6.57098e-2 * (mjd - dmjd) - 
	     (24 - (6.6460656 + (5.1262e-2 + (t * 2.581e-5))*t) -
		   (2400 * (t - (((double)y - 1900)/100))));
	return (t0);
}
