#include "surface.h"

/* Limit scope of EM and MtEt to this source file */
static matrix EM, MtEt;

/* Calculate ME and EtMt matrices */
void calcmatrices(int numlines)
{
   const FLOATTYPE delta = 1.0 / (numlines - 1),
           deltasq = delta * delta, 
           deltacb = delta * deltasq;

   /* LINT complains that E is being redefined here.  Why? */
   matrix E, M =
   {
      {-1, 3, -3, 1},
      {3, -6, 3, 0},
      {-3, 3, 0, 0},
      {1, 0, 0, 0}
   };

   /* E was initialised in declaration, but only gcc seems to support
      this, so I've moved it into the body of the procedure 16/11/94 */
   E[0][0] = 0;
   E[0][1] = 0;
   E[0][2] = 0;
   E[0][3] = 1;
   E[1][0] = deltacb;
   E[1][1] = deltasq;
   E[1][2] = delta;
   E[1][3] = 0;
   E[2][0] = 6*deltacb;
   E[2][1] = 2*deltasq;
   E[2][2] = 0;
   E[2][3] = 0;
   E[3][0] = 6*deltacb;
   E[3][1] = 0;
   E[3][2] = 0;
   E[3][3] = 0;
   

   matrixmultiply(E, M, EM);
   transpose(EM, MtEt);
}

/* Calculates the difference matrices used in DrawBezierPatch */
void calcbezier(objectdata * object)
{
   extern Itemsmode itemsmode;

   patch *patchptr, *finish;
   int i, j, k;
   matrix temp;
   matrix control[3];

   /* Set loop for all patches/current patch only */
   if (itemsmode == ALL) {
      patchptr = object->firstpatch;
      finish = NULL;
   } else {
      patchptr = object->currentpatch;
      finish = patchptr->next;
   }

   do {
      /* Copy control points for patch into matrix */
      for (i = 0; i < 3; i++)
         for (j = 0; j < 4; j++)
            for (k = 0; k < 4; k++)
               control[i][j][k] = patchptr->data[j][k]->world[i];

      /* Calculate difference matrices for x,y,z */
      /* dn =  M * E * Gn * E' * M'              */
      for (i = 0; i < 3; i++) {
         matrixmultiply(EM, control[i], temp);
         matrixmultiply(temp, MtEt, patchptr->differences[i]);
      }

      patchptr = patchptr->next;
   }
   while (patchptr != finish);
}

/* Runs through the patchlist, calling DrawBezierPatch */
void displaybezier(objectdata * object)
{
   extern Window bezierwindow;
   extern Display *bezierdisplay;
   extern Itemsmode itemsmode;

   patch *patchptr;

#ifdef DOUBLEBUFFERING
   extern Pixmap bezierpixmap; 
   extern GC gc, gcclear;
   extern windowdata bswindow;

   /* Erase old pixmap */
   XCopyArea(bezierdisplay, bezierpixmap, bezierpixmap, 
             gcclear, 0, 0, bswindow.width, bswindow.height, 0, 0); 
#else
   /* Erase old window */ 
   XClearWindow(bezierdisplay, bezierwindow);
#endif

   /* Draw all patches/current patch only */
   if (itemsmode == ALL) {
      patchptr = object->firstpatch;
      do {
         drawbezierpatch(patchptr);
         patchptr = patchptr->next;
      }
      while (patchptr != NULL);
   } else
      drawbezierpatch(object->currentpatch);

#ifdef DOUBLEBUFFERING
   /* Copy new pixmap to window */
   XCopyArea(bezierdisplay, bezierpixmap, bezierwindow, 
             gc, 0, 0, bswindow.width, bswindow.height, 0, 0); 
#endif
}


/* Draws Bezier patch with forward difference algorithm.  */
void drawbezierpatch(patch * patchptr)
{
   extern int numlines;
   extern Display *bezierdisplay;
   extern windowdata bswindow;
   extern GC gc;
#ifdef DOUBLEBUFFERING
   extern Pixmap bezierpixmap; 
#else 
   extern Window bezierwindow;
#endif

   int i, j;
   matrix dx, dy, dz;
   vertex coord;
   screenpoint oldpoint;
   FLOATTYPE delta1x, delta2x, delta3x, delta1y, delta2y, delta3y, delta1z,
          delta2z, delta3z;

   /* Make copy of dx, dy, dz matrices */
   for (i = 0; i < 4; i++)
      for (j = 0; j < 4; j++) {
         dx[i][j] = patchptr->differences[0][i][j];
         dy[i][j] = patchptr->differences[1][i][j];
         dz[i][j] = patchptr->differences[2][i][j];
      }

   /* Draw curves of constant s */
   for (i = 0; i < numlines; i++) {
      coord.world[0] = dx[0][0];
      delta1x = dx[1][0];
      delta2x = dx[2][0];
      delta3x = dx[3][0];
      coord.world[1] = dy[0][0];
      delta1y = dy[1][0];
      delta2y = dy[2][0];
      delta3y = dy[3][0];
      coord.world[2] = dz[0][0];
      delta1z = dz[1][0];
      delta2z = dz[2][0];
      delta3z = dz[3][0];
      coord.world[3] = 1.0;

      transformpoint(&coord, &bswindow);
      oldpoint = coord.screen;

      for (j = 1; j < numlines; j++) {
         coord.world[0] += delta1x;
         delta1x += delta2x;
         delta2x += delta3x;
         coord.world[1] += delta1y;
         delta1y += delta2y;
         delta2y += delta3y;
         coord.world[2] += delta1z;
         delta1z += delta2z;
         delta2z += delta3z;

         transformpoint(&coord, &bswindow);
         clipline(&oldpoint, &coord.screen, bezierdisplay, BEZIERDRAWABLE, gc, 
                  &bswindow); 
         oldpoint = coord.screen;
      }

      for (j = 0; j < 4; j++) {
         dx[j][0] += dx[j][1];
         dx[j][1] += dx[j][2];
         dx[j][2] += dx[j][3];

         dy[j][0] += dy[j][1];
         dy[j][1] += dy[j][2];
         dy[j][2] += dy[j][3];

         dz[j][0] += dz[j][1];
         dz[j][1] += dz[j][2];
         dz[j][2] += dz[j][3];
      }
   }

   /* Recopy dx, dy, dz matrices */
   for (i = 0; i < 4; i++)
      for (j = 0; j < 4; j++) {
         dx[i][j] = patchptr->differences[0][i][j];
         dy[i][j] = patchptr->differences[1][i][j];
         dz[i][j] = patchptr->differences[2][i][j];
      }

   /* Draw curves of constant t */
   for (i = 0; i < numlines; i++) {

      coord.world[0] = dx[0][0];
      delta1x = dx[0][1];
      delta2x = dx[0][2];
      delta3x = dx[0][3];
      coord.world[1] = dy[0][0];
      delta1y = dy[0][1];
      delta2y = dy[0][2];
      delta3y = dy[0][3];
      coord.world[2] = dz[0][0];
      delta1z = dz[0][1];
      delta2z = dz[0][2];
      delta3z = dz[0][3];
      coord.world[3] = 1.0;

      transformpoint(&coord, &bswindow);
      oldpoint = coord.screen;

      for (j = 1; j < numlines; j++) {
         coord.world[0] += delta1x;
         delta1x += delta2x;
         delta2x += delta3x;
         coord.world[1] += delta1y;
         delta1y += delta2y;
         delta2y += delta3y;
         coord.world[2] += delta1z;
         delta1z += delta2z;
         delta2z += delta3z;

         transformpoint(&coord, &bswindow);
         clipline(&oldpoint, &coord.screen, bezierdisplay, BEZIERDRAWABLE, gc, 
                  &bswindow);
         oldpoint = coord.screen;
      }

      for (j = 0; j < 4; j++) {
         dx[0][j] += dx[1][j];
         dx[1][j] += dx[2][j];
         dx[2][j] += dx[3][j];

         dy[0][j] += dy[1][j];
         dy[1][j] += dy[2][j];
         dy[2][j] += dy[3][j];

         dz[0][j] += dz[1][j];
         dz[1][j] += dz[2][j];
         dz[2][j] += dz[3][j];
      }
   }
}
