/*
 * imageh.c: image handlers and interface between the main program
 *		and the gif library
 */


#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <gif_lib.h>

#include "imageh.h"

#define PROGRAM_NAME	"XWarp"
#define ABS(x)		((x) > 0 ? (x) : (-(x)))

/* Make some variables global, so we could access them faster */

static int
    ColorMapSize = 0,
    OldCMapSize = 0,
    BackGround = 0,

    /* The way interlaced images shold be read - offset and jumps */

    InterlacedOffset[] = { 0, 4, 2, 1 },
    InterlacedJumps[] = { 8, 8, 4, 2 };

static GifColorType
    *ColorMap;

/* X specific staff goes here. XColorTable will hold the gif image colors, */
/* while XPixelTable will hold the pixel number, so we can redirect through */
/* it when forming the image bitmap in X format. Note the table has 256 */
/* entry which is the maximum allowed in gif format */

extern XColor XColorTable[256];
extern unsigned long XPixelTable[256];
extern Display *XDisplay; 
extern int XScreen;
extern Window Xroot;
extern Colormap XColorMap;
extern GC XGraphContext;
extern Visual *XVisual;
extern int WWidth,WHeight;
extern char* XImageData1;
extern char* XImageData2;

extern void xwarning(char*);

/* Color allocation function */

static void AllocateColors1(void);

/* Read an image from disk */

GifRowType* load_gif(char *gif_file_name)
	{
	int i,j,ImageNum=0,Size,Row,Col,Width,Height,ExtCode,Count;
	GifRecordType RecordType;
	GifByteType *Extension;
	GifRowType *ScreenBuffer;
	GifFileType *GifFile;

	if((GifFile=DGifOpenFileName(gif_file_name))==NULL)
		return(NULL);

	WWidth=GifFile->SWidth;
	WHeight=GifFile->SHeight;

	/* Allocate the screen as vector of column of rows. We can't */
	/* allocate the all screen at once for the sake of */
	/* compatibility */

	if((ScreenBuffer=(GifRowType*)
		malloc(GifFile->SHeight*sizeof(GifRowType*)))==NULL)
			return(NULL);

	/* First row */

	Size=GifFile->SWidth*sizeof(GifPixelType);
	if((ScreenBuffer[0]=(GifRowType)malloc(Size))==NULL)
		return(NULL);

	/* Set the row to background color */

	for(i=0;i<GifFile->SWidth;i++)
		ScreenBuffer[0][i]=GifFile->SBackGroundColor;

	/* Allocate other rows */

	for(i=1;i<GifFile->SHeight;i++)
		{
		if((ScreenBuffer[i]=(GifRowType)malloc(Size))==NULL)
			return(NULL);
		memcpy(ScreenBuffer[i], ScreenBuffer[0], Size);
		}

	/* A gif file contains sub-images: load all of them */

	do
		{
		if(DGifGetRecordType(GifFile,&RecordType)==GIF_ERROR)
			return(NULL);

			/* <-- */

switch (RecordType)
	{
	case IMAGE_DESC_RECORD_TYPE:

	/* This record contains a sub-image */

	if(DGifGetImageDesc(GifFile)==GIF_ERROR)
		{
		PrintGifError();
		return(NULL);
		}
	
	/* Position of the image in the main screen */

	Row=GifFile->ITop;
	Col=GifFile->ILeft;
	Width=GifFile->IWidth;
	Height=GifFile->IHeight;
	if(GifFile->ILeft+GifFile->IWidth>GifFile->SWidth ||
		GifFile->ITop+GifFile->IHeight >
		GifFile -> SHeight)
		{
		fprintf(stderr,
			"Image not confined to screen dimension, aborted.\n");
		return(NULL);
		}

	/* Let's see if the image is interlaced */

	if(GifFile->IInterlace)
		{
		
		/* We have to do 4 pass */

		for(Count=i=0;i<4;i++)
			for(j=Row+InterlacedOffset[i];j<Row+Height;
				j+=InterlacedJumps[i])
				{
				if(DGifGetLine(GifFile, &ScreenBuffer[j][Col],
					Width)==GIF_ERROR)
					{
					PrintGifError();
					return(NULL);
					}
				}

	/* If the image is not interlaced */

		}
	else
		{

		/* Read each row */

		for(i=0;i<Height;i++)
			{
			if(DGifGetLine(GifFile,&ScreenBuffer[Row++][Col],
				Width)==GIF_ERROR)
				{
				PrintGifError();
				return(NULL);
				}
			}

		}
	break;

	case EXTENSION_RECORD_TYPE:

	/* Skip all extension records */

	if(DGifGetExtension(GifFile,&ExtCode,&Extension)==GIF_ERROR)
		{
		PrintGifError();
		return(NULL);
		}
	while(Extension!=NULL)
		{
		if(DGifGetExtensionNext(GifFile,&Extension)==GIF_ERROR)
			{
			PrintGifError();
			return(NULL);
			}
		}
	break;

	case TERMINATE_RECORD_TYPE:
	break;

	default:
	break;
}

		/* --> */

		}
	while(RecordType!=TERMINATE_RECORD_TYPE);

	/* Modify global variables */

	BackGround=GifFile->SBackGroundColor;
	ColorMap=(GifFile->IColorMap ? GifFile -> IColorMap :
				       GifFile -> SColorMap) ;
	OldCMapSize=ColorMapSize;
	ColorMapSize=1<<(GifFile->IColorMap ? GifFile -> IBitsPerPixel :
					      GifFile -> SBitsPerPixel);

	if(DGifCloseFile(GifFile)==GIF_ERROR) /* Not very safe!!!! */
		PrintGifError();
	return(ScreenBuffer);
	}

/* Convert ScreenBuffer (gif image stored as vector of column of rows)
   into two XImage (source and destination) */
 
void Screen2X(GifRowType *ScreenBuffer,XImage** XImageSource,
	XImage** XImageDest,int ScreenWidth,int ScreenHeight)
	{
	int i,j,c,Size,x,y,MinIntensity,MaxIntensity,AvgIntensity;
	GifPixelType *Line;
	GifColorType *ColorMapEntry=ColorMap;

	/* Find the intensities in the color map */

	MaxIntensity=0;
	MinIntensity=256*100;
	for(i=0;i<ColorMapSize;i++)
		{
		c=ColorMapEntry[i].Red * 30 +
			ColorMapEntry[i].Green * 59 +
			ColorMapEntry[i].Blue * 11;
		if(c>MaxIntensity) MaxIntensity=c;
		if(c<MinIntensity) MinIntensity=c;
		}
	AvgIntensity=(MinIntensity+MaxIntensity)/2;

	/* The big trick here is to select the colors so let's do this */
	/* first */

	AllocateColors1();

	/* Create two XImage */

	if((XImageData1=(char *)malloc(ScreenWidth*ScreenHeight))==NULL)
		{
		xwarning("Failed to allocate memory required");
		return;
		}
	if((XImageData2=(char *)malloc(ScreenWidth*ScreenHeight))==NULL)
		{
		xwarning("Failed to allocate memory required");
		return;
		}
	for(i=0;i<ScreenHeight;i++)
		{
		y=i*ScreenWidth;
		for(j=0;j<ScreenWidth;j++)
		XImageData1[y+j]=XImageData2[y+j]=
			XPixelTable[ScreenBuffer[i][j]];
		}
	*XImageSource=XCreateImage(XDisplay,XVisual,8,ZPixmap,0,
				XImageData1,ScreenWidth,ScreenHeight,
				8,ScreenWidth);
	*XImageDest=XCreateImage(XDisplay,XVisual,8,ZPixmap,0,
				XImageData2,ScreenWidth,ScreenHeight,
				8,ScreenWidth);
	}

/* Routine to allocate the requested colors from the X server. */
/* Colors are allocated until success by stripping off the least bits */
/* of the colors. */

static void AllocateColors1(void)
	{
	int Strip,Msk,i,j;
	char Msg[80];
	int cnt;

	for(i=0;i<256;i++)
		XPixelTable[i]=0;

	cnt=0;
	for(Strip=0,Msk=0xff;Strip<8;Strip++,Msk<<=1)
		{
		for(i=0;i<ColorMapSize;i++)
			{
			
			/* Prepare color entry in X format */

			XColorTable[i].red=(ColorMap[i].Red & Msk) << 8;
			XColorTable[i].green=(ColorMap[i].Green & Msk) << 8;
			XColorTable[i].blue=(ColorMap[i].Blue & Msk) << 8;
			XColorTable[i].flags=DoRed | DoGreen | DoBlue;

			/* Try to allocate color */

			if(XAllocColor(XDisplay,XColorMap,&XColorTable[i]))
				XPixelTable[i]=XColorTable[i].pixel;
			else
				break;
			}

		if(i<ColorMapSize)
			XFreeColors(XDisplay,XColorMap,XPixelTable,i,0L);
		else
			break;
		}
	if(Strip==8)
	GIF_EXIT("Can not display the image - not enough colors available.");
	}

/* Deallocate colors */

void DeallocateColors(void)
	{              
	XFreeColors(XDisplay,XColorMap,XPixelTable,OldCMapSize,0L);
	}


