#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <termios.h>
#include "rbase.h"

char which_composer[NAME_LEN];
char **which_title, **which_other;
int  page_left;
int  send_to_file = 0;
int  ttymode = 1;         /* Normal mode: I/O to text terminal */
FILE *outfile = NULL;
char *calloc();

void list_composers(struct composer *);


void usage(char *pname)
{
  fprintf(stderr, "Usage is %s [-m] [dbFileName]\n", pname);
  fprintf(stderr, "  -m produces output in reccat form\n");
  fprintf(stderr, "  dbFileName defaults to \"%s\"\n", FILENAME);
  exit(ERR_CMDLINE_SYNTAX);
}


main(int argc, char *argv[])
{
FILE	*infile, *fopen();
struct composer *root, *setup();
int	key;
char    *infilename=FILENAME;

        if (argc==2) {
	  if(argv[1][0]=='-' && argv[1][1]=='m')
	    ttymode = 0;
	  else if (argv[1][0]=='-')
	    usage(argv[0]);
	  else
	    infilename=argv[1];
	}
        else if (argc==3) {
	  if (argv[1][0]=='-' && argv[1][1]=='m') {
	    ttymode = 0;
	    infilename=argv[2];
	  }
	  else {
	    usage(argv[0]);
	  }
	}
        else if (argc != 1) {
	  usage(argv[0]);
	}

	if ((infile = fopen(infilename, "r")) == NULL) {
		fprintf(stderr,
			"rbase: Failed to open data file\"%s\"\n",
			infilename);
		exit(ERR_NO_DBASE);
	}
	else {
		root = setup(infile);
		clear_cond();
/*		xbios(21,1,0);		/* Atari: Turn on cursor */
		while ((key = get_cmd()) != CMD_QUIT) {
			switch(key) {
				case CMD_COMPOSER:
					set_composer();
					break;
				case CMD_TITLE:
					parse_list(&which_title);
					break;
				case CMD_GLOBAL:
					parse_list(&which_other);
					break;
				case CMD_CLEAR:
					clear_cond();
					break;
				case CMD_SEARCH:
					find_entry(root);
					break;
				case CMD_FILE:
					file_output();
					break;
				case CMD_HELP:
					help_message();
					break;
				case CMD_RESCAN:
					wipe_tree(root);
					rewind(infile);
					root = setup(infile);
					break;
				case CMD_LISTCMP:
					list_composers(root);
					break;
				default:
					break;
			}
		}
		puts("");
/*		xbios(21, 0, 0);				/* Turn off cursor */
		if (outfile)
			fclose(outfile);
		exit(ERR_NONE);
	}
}




int get_cmd()    /* Get command key */
{
int c;

      if (ttymode) {
	fputs("?]", stdout);
	c =getkey();
	return(c&0x7f);
      }
      else {
	return getchar();
      }
}




struct composer *setup(source)		/* Read in entire file */
FILE	*source;
{
char buf[MAX_LINE], *stuff;
int  first = 1, c;
struct composer *begin, *end, *last;
struct work     *this_work_ptr, *new_work_ptr;

	while ((c = fgetc(source)) != EOF) {
		if (c != F_COMPOSER)
			puts("WARNING: Illegal file format.");
		else {
			this_work_ptr = NULL;
			last = end;
			if ((end = (struct composer *)calloc(1,sizeof(struct composer)))==NULL) {
				puts("RBASE: Out of memory.");
				exit(ERR_NO_MEM);
			}
			if (first) {
				begin = end;
				first = 0;
			}
			else {
				last->next = end;
			}
			fgets(end->name, NAME_LEN, source);
			*strchr(end->name, '\n') = '\0';				/* Replace nl with terminator */
			/* puts(end->name); /* Print composers' names */
					/* (slows down a sun too much) */
			c = fgetc(source);
			do {
				buf[0]=c;
				fgets(buf+1, MAX_LINE-2, source);
				*strchr(buf, '\n') = '\0';				/* Replace nl with terminator */
				if ((new_work_ptr = (struct work *)calloc(1, sizeof(struct work))) == NULL)  {
					puts("RBASE: Out of memory.");
					exit(ERR_NO_MEM);
				}

				if ((new_work_ptr->info = (char *)calloc(1, strlen(buf)+1)) == NULL) {
					puts("RBASE: Out of memory.");
					exit(ERR_NO_MEM);
				}
				if (this_work_ptr) {
					this_work_ptr->next_work = new_work_ptr;
				}
				else {
					end->works = new_work_ptr;
				}
				this_work_ptr = new_work_ptr;
				strcpy(this_work_ptr->info, buf);
			} while ((c = fgetc(source)) != F_COMPOSER && c != EOF);
			if (c != EOF) {
				ungetc(c, source);
			}
			this_work_ptr->next_work = NULL;
		}
	}
	end->next = NULL;
	return(begin);
}




parse_list(start)	/* Read the line and break into words */
char ***start;
{
char *temp;
int words=0;

	if (ttymode) {
	  fputs(":=", stdout);
	}
	if (*start) {	/* Clear the current list */
		free(**start);
		free(*start);
	}
	*start = (char **)calloc(1, MAX_WORDS * sizeof(char **));
	(*start)[0] = (char *)calloc(1, MAX_LINE);
	temp = (*start)[0];
	gets(temp);
	while (*temp && words < MAX_WORDS) {
		if (*temp == ' ') {
			*temp = '\0';
			(*start)[++words] = ++temp;
			(*start)[words+1] = NULL;
		}
		else
			++temp;
	}
	if (temp == (*start)[0]) {		/* No words => null list */
		free(**start);
		free(*start);
		*start=NULL;
	}
	if (ttymode) {
	  show_set();
	}
}



find_entry(root)						/* List the matches */
struct composer *root;
{
struct work *here;
struct composer *now;

	if (send_to_file)				/* Write target on output file */
		fshow_set();
	page_left = 2;
	if (*which_composer)				/* Scan composer list */
		while (strncmp(which_composer, root->name, strlen(which_composer))) {
			root = root->next;
			if (!root) {
			        if (ttymode) {
				  puts("");
				  puts("RBASE could not find any works for that composer.");
				} else {
				  puts("*");
				  fflush(stdout);
				}
				*which_composer = '\0';
				return(0);
			}
		}
	here = root->works;
	now = root;
	while(here && now) {
		if ((which_other && instr(here->info, which_other, 0)) ||
	    	(which_title && instr(here->info, which_title, 1))  ||
	    	(!which_title && !which_other))
				if (!output(now, here->info) && ttymode) {
					puts("\nSearch interrupted by user");
					return(0);
				}
		here = here->next_work;
		if (!here) {
			now = now->next;
			if(!*which_composer && now)	/* Skip to next composer only in general search */
				here = now->works;
		}
	}
	if (ttymode) {
          putchar('\n');
	  puts("Search complete.");
	}
        else {
	  puts("*");
	  fflush(stdout);
	}
	return(1);
}



void wipe_works(struct work *work)
{
  if (work->next_work) {
    wipe_works(work->next_work);
  }
  free (work->info);
  free (work);
}

wipe_tree(struct composer *root)
{
  wipe_works(root->works);
  if (root->next) {
    wipe_tree(root->next);
  }
  free(root);
}



output(cmp, string)								/* Display the string */
struct composer *cmp;
char *string;
{
char *s = string;
static char outfcmp[NAME_LEN] = {'\0'};

    if (ttymode) {
	if (!send_to_file && !page_left--) {
		int c;
		puts("Press space to continue, [Q] to exit...");
		while((c = (getkey() & 0x7f)) != ' ' && c != 'q');
		page_left = 1;
		if (c == 'q')
			return(0);
	}

	puts("\n*** MATCH FOUND...");
	fputs("Composer:\t\t", stdout);
	puts(cmp->name);
	fputs("Title:\t\t\t", stdout);
	while(outch(*s++)!=F_NL);
	fputs("\rOther information:\t", stdout);
	while(outch(*s++));
	puts("");

	if (send_to_file) {
		if (outfile == NULL && ((outfile = fopen(OUTFILENAME, "w")) == NULL))
			puts("rbase: Warning: failed to write this record to file\n");
		else {
			s = string;
			if (strcmp(cmp->name, outfcmp) != 0) {
				fputs(strcpy(outfcmp, cmp->name), outfile);
				putc('\n', outfile);
			}
			while(*s && *s!=F_NL)
				putc(*s++, outfile);
			putc('\n', outfile);
			fputs(++s, outfile);
			putc('\n', outfile);
		}
	}
    } else {
      int fieldc;

      /* Build a string in a format suitable for the reccat program */
      printf("{{%s} {", cmp->name);

      fieldc = 0;
      while (*s) {
	if (*s!=F_NL) {
	  putchar(*s++);
	}
	else {
	  printf("} {");
	  ++s; ++fieldc;
	}
      }
      putchar('}');
      while (fieldc++ < MAX_AWK_FIELDS) {
        printf(" {}");
      }
      puts("}");
      fflush(stdout);
    }
    return(1);
}




outch(c)
char c;
{
static int pos = 24;

	if ((c == F_NL) || ((c == '\t' || c == ' ') && pos > T_WIDTH-5)) {
		pos = 24;
		fputs("\n\t\t\t", stdout);
	}
	else {
		++pos;
		putchar(c);
	}
	return(c);
}




instr(arg1, args2, title_only)
char *arg1, **args2;						/* Find all of args2 in arg1 */
int title_only;							/* Only search to \n if true */
{
char *scan, *s2 = *args2;
int i = 0, s2len;

	while (s2) {
		scan = arg1;
		s2len = strlen(s2)-1;
		for(;;) {
			while(scan[s2len] && toupper(*scan++) != toupper(*s2)) /* Skip non-matching chars */
				if (title_only && *scan == F_NL)	/* May abort at end of line */
					return(0);
			if (!scan[s2len])
				return(0);								/* Still no match at end of string */
			if (compare(scan-1, s2)) { 			/* Match found */
				s2 = args2[++i];						/*    so try next word */
				break;
			}
		}
	}
	return(1);											/* All tests passed */
}




compare(s1, s2)
char *s1, *s2;
{
	while(*s2)
		if (toupper(*s1++) != toupper(*s2++))
			return(0);
	return(1);
}




clear_cond()							/* Clear the current conditions */
{
	which_composer[0] = '\0';
	if (which_title)
		free(which_title);
	if (which_other)
		free(which_other);
	which_title = NULL;
	which_other = NULL;
	if (ttymode) show_set();
}




fshow_set()					/* Record current conditions on output file */
{
	if (outfile == NULL && ((outfile = fopen(OUTFILENAME, "w")) == NULL))
		return;
	fputs("\n*** New search: target as follows:\n", outfile);
	fputs("Composer: ", outfile);
	if (*which_composer)
		fputs(which_composer, outfile);
	else
		fputs("<any>", outfile);
	fputs("\nTitle:    ", outfile);
	if (which_title)
		plist(which_title, outfile);
	else
		fputs("<any>\n", outfile);
	fputs("Other:    ", outfile);
	if (which_other)
		plist(which_other, outfile);
	else
		fputs("<any>\n", outfile);
	fputs("\n", outfile);
}



show_set()								/* Display the current conditions */
{
	puts("");
	fputs("Composer: ", stdout);
	if (*which_composer)
		puts(which_composer);
	else
		puts("<any>");
	fputs("Title:    ", stdout);
	if (which_title)
		plist(which_title, stdout);
	else
		puts("<any>");
	fputs("Other:    ", stdout);
	if (which_other)
		plist(which_other, stdout);
	else
		puts("<any>");
	puts("");
	if (send_to_file) {
		fputs("Recording results of search in file \"", stdout);
		fputs(OUTFILENAME, stdout);
		puts("\"");
	}
}




plist(start, opf)	    /* Print a list of strings, separated */
char **start;		    /* by spaces */
FILE *opf;
{
	while (*start) {
		fputs(*start, opf);
		putc(' ', opf);
		start++;
	}
	fputs("\n", opf);
}




set_composer()							/* Read the new composer name */
{
char *p;

        if (ttymode) fputs(":=", stdout);
	p = which_composer;
	gets(p);
	while (*p) {
		*p = toupper(*p);
		p++;
	}
	if (ttymode) show_set();
}


file_output()							/* Echo the results of the search to a file */
{
	send_to_file = !send_to_file;
	if (send_to_file) {
		fputs("\nNow writing output to file \"", stdout);
		fputs(OUTFILENAME, stdout);
		puts("\"");
	}
	else {
		fputs("\nNo longer writing output to file \"", stdout);
		fputs(OUTFILENAME, stdout);
		puts("\"");
	}
}


help_message()
{
	puts("");
	puts("RBASE by Nick Bailey, May 1989.\n");
	puts("Commands available:");
	putchar('\t'); putchar(CMD_COMPOSER); puts(": set composer for search");
	putchar('\t'); putchar(CMD_TITLE);    puts(": set title for search");
	putchar('\t'); putchar(CMD_GLOBAL);   puts(": set any other conditions");
	putchar('\t'); putchar(CMD_SEARCH);   puts(": start looking");
	putchar('\t'); putchar(CMD_QUIT);     puts(": exit to desktop");
	putchar('\t'); putchar(CMD_CLEAR);    puts(": delete current conditions");
	putchar('\t'); putchar(CMD_FILE);     fputs(": record results
of searches in file \"", stdout); fputs(OUTFILENAME, stdout); puts("\"");
	putchar('\t'); putchar(CMD_LISTCMP);  puts(": list composers on file");
	putchar('\t'); putchar(CMD_RESCAN);   puts(": rescan database file");
	putchar('\t'); putchar(CMD_HELP);     puts(": print this help page");

	show_set();
}


getkey()
{
static struct termios default_termio, key_termio;
static int called = 0;
int c;

	if (!called) {
		called = (ioctl(0, TCGETS, &default_termio) != -1);
		key_termio = default_termio;
		key_termio.c_cc[VMIN] = 1;
		key_termio.c_cc[VTIME] = 0;
		key_termio.c_lflag &= ~ICANON;
	}

	if (called) {
		ioctl(0, TCSETS, &key_termio);
		c = getchar();
		ioctl(0, TCSETS, &default_termio);
		return c;
	}
	else
		return(getchar());
}



void list_composers(struct composer *here)
{
  do {
      puts(here->name);
  } while (here = here->next);
  if(!ttymode) puts("*");
  fflush(stdout);
}

debug(begin)	         /* Print out the entire database */
struct composer *begin;	 /* starting at *begin */
{			 /* (a debugging aid) */
struct work *here;
puts("Debugging print");
do {
	puts("new composer");
	puts(begin->name);
	here = begin->works;
	do {
		puts(here->info);
		here = here->next_work;
	} while (here);
	begin = begin->next;
} while(begin);
}

