/*
 * Copyright 1992 the Board of Trustees of the Leland Stanford Junior
 * University. Official permission to use this software is included in
 * the documentation. It authorizes you to use this file for any
 * non-commercial purpose, provided that this copyright notice is not
 * removed and that any modifications made to this file are commented
 * and dated in the style of the example below.
 */

/*
 *
 *  source file:   ./xtpanel/object.c
 *
 * Steve Cole, Dave Nichols (SEP), August 31 1992
 *      Inserted this sample edit history entry.
 *      Please log any further modifications made to this file:
 * Steve Cole, Dave Nichols (SEP), November 20 1992 -  version 2.00
 *      objects can have multiple actions.
 * Steve Cole (SEP), January 14 1993
 *      Added call to object->sync after update_tag, to make sure
 *	that object values are kept up to date.
 */

#include <stdio.h>

#include <X11/Xatom.h>
#include <X11/Intrinsic.h>

#include <X11/Xaw/Label.h>

#include "object.h"
#include "tree.h"

static Objdef* topobj=0;

/*
 * Function name: new_object
 * Description: creates a structure for a new object
 * Arguments: none
 * Returns: pointer to new structure of type objdef
 */
Objdef* new_object()
{
    Objdef *a, *b;
    
    if( topobj == (Objdef*)0 ){
	/* make the head of the list */
	topobj = (Objdef*) malloc( sizeof( Objdef ) );
	topobj->name = strdupl("top");
	topobj->action = (action *) 0;
	topobj->value = (char *) 0;
	topobj->widgetname = (Widget) 0;
	topobj->info = (void *) 0;
	topobj->updater = (void *) 0;
	topobj->sync = (void *) 0;
	topobj->next = (Objdef*)0;
    }
    
    a = (Objdef*) malloc( sizeof( Objdef ) );
    a->name= (char *)0;
    a->action= (action *)0;
    a->value= (char *)0;
    a->widgetname= (Widget)0;
    a->info= (void *)0;
    a->updater= (void *)0;
    a->sync= (void *)0;
    a->next= (Objdef*)0;
    
    /* find the tail of the list */
    for( b=topobj; b->next != (Objdef*)0; b = b->next );
    
    b->next = a;
    
    return a;
}

/*
 * Function name: find_by_name
 * Description: loop over all objects, find the one with
 *		the right name.
 * Arguments: name - name to match
 * Returns: pointer to object structure for matching object
 */
Objdef* find_by_name(name)
    char* name;
{
    int iobj;
    Objdef* obj;
    
    /* loop over objects */
    for ( obj=topobj; obj != (Objdef*) 0; obj=obj->next )
      {
	  if (!strcmp(obj->name,name)) {
	      return obj;
	  }
      }
    
    fprintf(stderr,"find_by_name(): cannot find object %s\n",name);
    fprintf(stderr,"Trying to find the object refered to by \"$%s\"\n",name);
    fprintf(stderr," but this object does not yet exist\n");
    exit(-1);  
}

/*
 * Function name: find_widget
 * Description: loop over all objects, find the one with
 *		the right widget name.
 * Arguments: w - widget id to match
 * Returns: pointer to object structure for matching object
 */
Objdef*
  find_by_widget(w)
Widget w;
{
    Objdef* obj;
    
    /* loop over objects */
    for ( obj=topobj; obj != (Objdef*) 0; obj=obj->next )
      {
	  if (obj->widgetname == w) {
	      return obj;
	  }
      }
    
    fprintf(stderr,"find_by_widget cannot find widget %d\n", (int)w );
    exit(-1);  
}

/*
 * Function name: get_string
 * Description: find the value of an object, given its name
 * Arguments: name - object name
 * Returns: pointer to string containing objects value
 */
char*
  get_string(name)
char* name;
{
char * val;
    if( (val = find_by_name(name)->value) == (char*) 0 ){
      return(strdupl(""));  
    }else{
      return(strdupl(val));  
    }
}

/*
 * Function name: set_string
 * Description: set the value of an object
 * Arguments: name - object name
 *	      string - pointer to string containing new object value
 * Returns: none
 */
void set_string(name,string)
     char* name,*string;
{
    Objdef *obj;
    obj = find_by_name(name);
    if( obj->value != (char*)0 ) free(obj->value);
    obj->value = strdupl(string);
}

/*
 * Function name: all_actions
 * Description: do the actions of all objects whose action type
 *		is STRING. This makes sure defaults are set.
 * Arguments: none
 * Returns: none
 */
void all_actions()
{
    Objdef* obj;
    
    if( topobj == (Objdef*)0 ) return;
    
    /* loop over objects, skip the top one */
    for ( obj=topobj->next; obj != (Objdef*) 0; obj=obj->next )
      {
	  if (obj->action != (action *) 0) {
		 perform_actions(obj->name,obj->action,0);
	  }
      }
}

/*
 * Function name: update_object
 * Description: update the value of an object
 * Arguments: name - object name
 *	      string - pointer to string containing new object value
 * Returns: none
 */
void update_object(name,string)
     char* name,*string;
{
    Objdef *object;

    object = find_by_name(name);

    if( object->value != (char*)0 ) free(object->value);
    object->value = strdupl(string);

    if( object->updater != 0 ) object->updater(object,string);
}

/*
 * Function name: update_tag
 * Description: update the value of an object's tag
 * Arguments: name - object name
 *            tag  - object tag that is changed.
 *	      value - pointer to string containing new object value
 * Returns: none
 */
void update_tag(name,tag,value)
     char* name,*tag,*value;
{
    Objdef *object;
    Arg args[20];
    int narg;

    object = find_by_name(name);

    if( object->sync == 0 ) {
        /* the default sync action is just to update the
         * corresponding widget resources 
         */
        narg = 0;
        if( object->widgetname != 0 ){
          SetTag(object->widgetname,args,&narg,tag,value);
          XtSetValues(object->widgetname,args,narg);
        }
    }else{

        /*   
         * object->sync is for objects who may need to do
         * something different after a SET command (such as toggles, whose
         * state may have changed). 
         * Note that they may have to do the SetTag/SetValues as part of
         * their action.
         */
        object->sync(object,tag,value);
    }
}
