#include <stdio.h>
/*
 * 			Copyright 1993, 1994 by AT&T
 * 
 * 			 All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of AT&T not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * AT&T DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * AT&T BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * 
 * AT&T's dontation of this software does not imply a licence granted for
 * patents nor transfer of ownership of any patents which may inadvertently
 * be implemented in this code.
 * 
 */

#include "tiffio.h"
#include <math.h>
#include <assert.h>

typedef struct Filterprog Filterprog;
struct Filterprog {
	short *coef;
	short count, decr;
	Filterprog *next;
};

#include "C_P_args.h"

C_PROTOS_BEGIN_EXTERN

static char *
resample_malloc C_P_ARGS((unsigned size));

extern void
set_hitachi_to_luminance C_P_ARGS((double gamma));

extern void
set_luminance_to_hitachi C_P_ARGS((double gamma));

static double
bspline C_P_ARGS((double x));

static double
ease C_P_ARGS((double x));

static double
crisp C_P_ARGS((double x));

extern double
filter C_P_ARGS((double x));

static Filterprog *
impulse_response C_P_ARGS((int j));

static int
gcd C_P_ARGS((int u, int v));

static void
compile_resampling_program C_P_ARGS((int m, int n));

extern void
antivignette C_P_ARGS((int n));

extern void
resample C_P_ARGS((short *input, short *output, int m, int n, int init));

extern void
main C_P_ARGS((int argc, char **argv));

extern void
usage C_P_ARGS((char *program));

C_PROTOS_END_EXTERN

/*
 * a mixture of code from Sam Lefler's tiff and Don Mitchell's sampling
 */

#define	howmany(x, y)	(((x)+((y)-1))/(y))
#define	streq(a,b)	(strcmp(a,b) == 0)
#define	CopyField(tag, v) \
    if (TIFFGetField(in, tag, &v)) TIFFSetField(out, tag, v)
#define	CopyField3(tag, v1, v2, v3) \
    if (TIFFGetField(in, tag, &v1, &v2, &v3)) TIFFSetField(out, tag, v1, v2, v3)

#define MAX(X,Y) ((X) > (Y) ? (X) : (Y))
#define MIN(X,Y) ((X) < (Y) ? (X) : (Y))
#define HALFWIDTH 2.0

static char *
resample_malloc (size)
unsigned size;
{
	extern char *malloc();
	char *ptr = malloc (size);

	if (ptr == (char *) NULL)
	{
		fprintf (stderr, "resample: cannot allocate memory.\n");
		exit (0);
	}

	return ptr;
}

/*
 *	Accurate linearization tables for SUN's hitachi monitor.
 *	D. P. Mitchell  85/09/05.
 */

short hitachi_to_luminance[256] = {
	   0,   0,   0,   0,   0,   0,   0,   0,
	   0,   0,   0,   0,   0,   0,   0,   0,
	   0,   0,   1,   2,   3,   4,   4,   5,
	   5,   5,   6,   7,   8,   9,  11,  12,
	  13,  15,  17,  19,  20,  22,  24,  26,
	  28,  29,  31,  34,  37,  40,  43,  46,
	  49,  52,  54,  57,  60,  64,  68,  72,
	  76,  80,  84,  88,  92,  96, 101, 106,
	 111, 116, 121, 125, 130, 135, 140, 147,
	 153, 159, 165, 171, 178, 184, 191, 199,
	 206, 213, 221, 229, 237, 245, 253, 262,
	 271, 280, 288, 297, 305, 313, 321, 330,
	 339, 348, 358, 368, 379, 389, 400, 411,
	 423, 434, 446, 456, 467, 479, 491, 503,
	 516, 529, 543, 557, 570, 583, 597, 610,
	 623, 636, 650, 663, 678, 692, 705, 718,
	 731, 744, 759, 774, 790, 806, 821, 838,
	 854, 871, 887, 904, 921, 939, 957, 975,
	 993,1012,1032,1051,1070,1088,1106,1125,
	1144,1164,1184,1203,1223,1242,1262,1281,
	1300,1319,1339,1360,1381,1403,1426,1448,
	1469,1490,1512,1535,1558,1581,1606,1631,
	1658,1684,1709,1733,1757,1782,1807,1833,
	1858,1883,1908,1934,1960,1987,2015,2043,
	2071,2098,2124,2152,2180,2209,2238,2267,
	2295,2323,2350,2377,2407,2437,2468,2498,
	2529,2560,2591,2622,2651,2680,2710,2742,
	2774,2806,2837,2868,2899,2930,2962,2995,
	3028,3061,3094,3127,3159,3193,3227,3261,
	3294,3325,3357,3390,3426,3464,3503,3542,
	3581,3618,3654,3688,3721,3755,3790,3825,
	3861,3898,3935,3972,4008,4042,4071,4095,
};

unsigned char luminance_to_hitachi[4096] = {
	  0, 18, 19, 20, 21, 23, 26, 27, 28, 28, 29, 30, 31, 31, 32, 33,
	 33, 34, 34, 35, 36, 36, 37, 37, 38, 38, 39, 39, 40, 41, 41, 42,
	 42, 42, 43, 43, 43, 44, 44, 45, 45, 45, 46, 46, 46, 47, 47, 47,
	 47, 48, 48, 49, 49, 49, 50, 50, 50, 51, 51, 51, 52, 52, 52, 53,
	 53, 53, 53, 54, 54, 54, 54, 55, 55, 55, 55, 56, 56, 56, 56, 57,
	 57, 57, 57, 58, 58, 58, 58, 59, 59, 59, 59, 60, 60, 60, 60, 61,
	 61, 61, 61, 61, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 64, 64,
	 64, 64, 64, 65, 65, 65, 65, 66, 66, 66, 66, 66, 67, 67, 67, 67,
	 68, 68, 68, 68, 68, 69, 69, 69, 69, 69, 69, 70, 70, 70, 70, 70,
	 70, 71, 71, 71, 71, 71, 71, 72, 72, 72, 72, 72, 72, 73, 73, 73,
	 73, 73, 73, 74, 74, 74, 74, 74, 74, 75, 75, 75, 75, 75, 75, 76,
	 76, 76, 76, 76, 76, 76, 77, 77, 77, 77, 77, 77, 77, 78, 78, 78,
	 78, 78, 78, 78, 79, 79, 79, 79, 79, 79, 79, 80, 80, 80, 80, 80,
	 80, 80, 80, 81, 81, 81, 81, 81, 81, 81, 82, 82, 82, 82, 82, 82,
	 82, 82, 83, 83, 83, 83, 83, 83, 83, 83, 84, 84, 84, 84, 84, 84,
	 84, 84, 85, 85, 85, 85, 85, 85, 85, 85, 86, 86, 86, 86, 86, 86,
	 86, 86, 86, 87, 87, 87, 87, 87, 87, 87, 87, 88, 88, 88, 88, 88,
	 88, 88, 88, 88, 89, 89, 89, 89, 89, 89, 89, 89, 89, 90, 90, 90,
	 90, 90, 90, 90, 90, 91, 91, 91, 91, 91, 91, 91, 91, 91, 92, 92,
	 92, 92, 92, 92, 92, 92, 93, 93, 93, 93, 93, 93, 93, 93, 94, 94,
	 94, 94, 94, 94, 94, 94, 95, 95, 95, 95, 95, 95, 95, 95, 95, 96,
	 96, 96, 96, 96, 96, 96, 96, 96, 97, 97, 97, 97, 97, 97, 97, 97,
	 97, 97, 98, 98, 98, 98, 98, 98, 98, 98, 98, 98, 99, 99, 99, 99,
	 99, 99, 99, 99, 99, 99,100,100,100,100,100,100,100,100,100,100,
	100,101,101,101,101,101,101,101,101,101,101,102,102,102,102,102,
	102,102,102,102,102,102,103,103,103,103,103,103,103,103,103,103,
	103,103,104,104,104,104,104,104,104,104,104,104,104,104,105,105,
	105,105,105,105,105,105,105,105,105,106,106,106,106,106,106,106,
	106,106,106,106,107,107,107,107,107,107,107,107,107,107,107,108,
	108,108,108,108,108,108,108,108,108,108,109,109,109,109,109,109,
	109,109,109,109,109,109,110,110,110,110,110,110,110,110,110,110,
	110,110,111,111,111,111,111,111,111,111,111,111,111,111,112,112,
	112,112,112,112,112,112,112,112,112,112,112,112,113,113,113,113,
	113,113,113,113,113,113,113,113,113,114,114,114,114,114,114,114,
	114,114,114,114,114,114,114,115,115,115,115,115,115,115,115,115,
	115,115,115,115,116,116,116,116,116,116,116,116,116,116,116,116,
	116,116,117,117,117,117,117,117,117,117,117,117,117,117,117,118,
	118,118,118,118,118,118,118,118,118,118,118,118,119,119,119,119,
	119,119,119,119,119,119,119,119,119,120,120,120,120,120,120,120,
	120,120,120,120,120,120,121,121,121,121,121,121,121,121,121,121,
	121,121,121,121,122,122,122,122,122,122,122,122,122,122,122,122,
	122,123,123,123,123,123,123,123,123,123,123,123,123,123,123,123,
	124,124,124,124,124,124,124,124,124,124,124,124,124,124,125,125,
	125,125,125,125,125,125,125,125,125,125,125,126,126,126,126,126,
	126,126,126,126,126,126,126,126,127,127,127,127,127,127,127,127,
	127,127,127,127,127,128,128,128,128,128,128,128,128,128,128,128,
	128,128,129,129,129,129,129,129,129,129,129,129,129,129,129,129,
	130,130,130,130,130,130,130,130,130,130,130,130,130,130,130,131,
	131,131,131,131,131,131,131,131,131,131,131,131,131,131,131,132,
	132,132,132,132,132,132,132,132,132,132,132,132,132,132,132,133,
	133,133,133,133,133,133,133,133,133,133,133,133,133,133,134,134,
	134,134,134,134,134,134,134,134,134,134,134,134,134,134,134,135,
	135,135,135,135,135,135,135,135,135,135,135,135,135,135,135,136,
	136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,137,
	137,137,137,137,137,137,137,137,137,137,137,137,137,137,137,137,
	138,138,138,138,138,138,138,138,138,138,138,138,138,138,138,138,
	139,139,139,139,139,139,139,139,139,139,139,139,139,139,139,139,
	139,140,140,140,140,140,140,140,140,140,140,140,140,140,140,140,
	140,140,140,141,141,141,141,141,141,141,141,141,141,141,141,141,
	141,141,141,141,141,142,142,142,142,142,142,142,142,142,142,142,
	142,142,142,142,142,142,142,143,143,143,143,143,143,143,143,143,
	143,143,143,143,143,143,143,143,143,144,144,144,144,144,144,144,
	144,144,144,144,144,144,144,144,144,144,144,144,145,145,145,145,
	145,145,145,145,145,145,145,145,145,145,145,145,145,145,145,146,
	146,146,146,146,146,146,146,146,146,146,146,146,146,146,146,146,
	146,146,147,147,147,147,147,147,147,147,147,147,147,147,147,147,
	147,147,147,147,147,148,148,148,148,148,148,148,148,148,148,148,
	148,148,148,148,148,148,148,148,149,149,149,149,149,149,149,149,
	149,149,149,149,149,149,149,149,149,149,150,150,150,150,150,150,
	150,150,150,150,150,150,150,150,150,150,150,150,151,151,151,151,
	151,151,151,151,151,151,151,151,151,151,151,151,151,151,151,151,
	152,152,152,152,152,152,152,152,152,152,152,152,152,152,152,152,
	152,152,152,153,153,153,153,153,153,153,153,153,153,153,153,153,
	153,153,153,153,153,153,153,154,154,154,154,154,154,154,154,154,
	154,154,154,154,154,154,154,154,154,154,154,155,155,155,155,155,
	155,155,155,155,155,155,155,155,155,155,155,155,155,155,156,156,
	156,156,156,156,156,156,156,156,156,156,156,156,156,156,156,156,
	156,156,157,157,157,157,157,157,157,157,157,157,157,157,157,157,
	157,157,157,157,157,158,158,158,158,158,158,158,158,158,158,158,
	158,158,158,158,158,158,158,158,159,159,159,159,159,159,159,159,
	159,159,159,159,159,159,159,159,159,159,159,160,160,160,160,160,
	160,160,160,160,160,160,160,160,160,160,160,160,160,160,160,161,
	161,161,161,161,161,161,161,161,161,161,161,161,161,161,161,161,
	161,161,162,162,162,162,162,162,162,162,162,162,162,162,162,162,
	162,162,162,162,162,162,162,163,163,163,163,163,163,163,163,163,
	163,163,163,163,163,163,163,163,163,163,163,164,164,164,164,164,
	164,164,164,164,164,164,164,164,164,164,164,164,164,164,164,164,
	164,165,165,165,165,165,165,165,165,165,165,165,165,165,165,165,
	165,165,165,165,165,165,165,166,166,166,166,166,166,166,166,166,
	166,166,166,166,166,166,166,166,166,166,166,166,166,166,167,167,
	167,167,167,167,167,167,167,167,167,167,167,167,167,167,167,167,
	167,167,167,168,168,168,168,168,168,168,168,168,168,168,168,168,
	168,168,168,168,168,168,168,168,168,169,169,169,169,169,169,169,
	169,169,169,169,169,169,169,169,169,169,169,169,169,169,170,170,
	170,170,170,170,170,170,170,170,170,170,170,170,170,170,170,170,
	170,170,170,170,170,171,171,171,171,171,171,171,171,171,171,171,
	171,171,171,171,171,171,171,171,171,171,171,171,172,172,172,172,
	172,172,172,172,172,172,172,172,172,172,172,172,172,172,172,172,
	172,172,172,173,173,173,173,173,173,173,173,173,173,173,173,173,
	173,173,173,173,173,173,173,173,173,173,173,174,174,174,174,174,
	174,174,174,174,174,174,174,174,174,174,174,174,174,174,174,174,
	174,174,174,174,175,175,175,175,175,175,175,175,175,175,175,175,
	175,175,175,175,175,175,175,175,175,175,175,175,175,176,176,176,
	176,176,176,176,176,176,176,176,176,176,176,176,176,176,176,176,
	176,176,176,176,176,176,176,176,177,177,177,177,177,177,177,177,
	177,177,177,177,177,177,177,177,177,177,177,177,177,177,177,177,
	177,178,178,178,178,178,178,178,178,178,178,178,178,178,178,178,
	178,178,178,178,178,178,178,178,178,178,179,179,179,179,179,179,
	179,179,179,179,179,179,179,179,179,179,179,179,179,179,179,179,
	179,179,180,180,180,180,180,180,180,180,180,180,180,180,180,180,
	180,180,180,180,180,180,180,180,180,180,180,181,181,181,181,181,
	181,181,181,181,181,181,181,181,181,181,181,181,181,181,181,181,
	181,181,181,181,182,182,182,182,182,182,182,182,182,182,182,182,
	182,182,182,182,182,182,182,182,182,182,182,182,182,183,183,183,
	183,183,183,183,183,183,183,183,183,183,183,183,183,183,183,183,
	183,183,183,183,183,183,184,184,184,184,184,184,184,184,184,184,
	184,184,184,184,184,184,184,184,184,184,184,184,184,184,184,185,
	185,185,185,185,185,185,185,185,185,185,185,185,185,185,185,185,
	185,185,185,185,185,185,185,185,186,186,186,186,186,186,186,186,
	186,186,186,186,186,186,186,186,186,186,186,186,186,186,186,186,
	186,186,187,187,187,187,187,187,187,187,187,187,187,187,187,187,
	187,187,187,187,187,187,187,187,187,187,187,187,188,188,188,188,
	188,188,188,188,188,188,188,188,188,188,188,188,188,188,188,188,
	188,188,188,188,188,188,189,189,189,189,189,189,189,189,189,189,
	189,189,189,189,189,189,189,189,189,189,189,189,189,189,189,189,
	189,189,190,190,190,190,190,190,190,190,190,190,190,190,190,190,
	190,190,190,190,190,190,190,190,190,190,190,190,190,190,191,191,
	191,191,191,191,191,191,191,191,191,191,191,191,191,191,191,191,
	191,191,191,191,191,191,191,191,191,191,192,192,192,192,192,192,
	192,192,192,192,192,192,192,192,192,192,192,192,192,192,192,192,
	192,192,192,192,192,193,193,193,193,193,193,193,193,193,193,193,
	193,193,193,193,193,193,193,193,193,193,193,193,193,193,193,193,
	194,194,194,194,194,194,194,194,194,194,194,194,194,194,194,194,
	194,194,194,194,194,194,194,194,194,194,194,195,195,195,195,195,
	195,195,195,195,195,195,195,195,195,195,195,195,195,195,195,195,
	195,195,195,195,195,195,195,196,196,196,196,196,196,196,196,196,
	196,196,196,196,196,196,196,196,196,196,196,196,196,196,196,196,
	196,196,196,197,197,197,197,197,197,197,197,197,197,197,197,197,
	197,197,197,197,197,197,197,197,197,197,197,197,197,197,197,197,
	198,198,198,198,198,198,198,198,198,198,198,198,198,198,198,198,
	198,198,198,198,198,198,198,198,198,198,198,198,198,199,199,199,
	199,199,199,199,199,199,199,199,199,199,199,199,199,199,199,199,
	199,199,199,199,199,199,199,199,199,199,200,200,200,200,200,200,
	200,200,200,200,200,200,200,200,200,200,200,200,200,200,200,200,
	200,200,200,200,200,200,201,201,201,201,201,201,201,201,201,201,
	201,201,201,201,201,201,201,201,201,201,201,201,201,201,201,201,
	201,202,202,202,202,202,202,202,202,202,202,202,202,202,202,202,
	202,202,202,202,202,202,202,202,202,202,202,202,202,203,203,203,
	203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,203,
	203,203,203,203,203,203,203,203,203,204,204,204,204,204,204,204,
	204,204,204,204,204,204,204,204,204,204,204,204,204,204,204,204,
	204,204,204,204,204,204,204,205,205,205,205,205,205,205,205,205,
	205,205,205,205,205,205,205,205,205,205,205,205,205,205,205,205,
	205,205,205,205,205,206,206,206,206,206,206,206,206,206,206,206,
	206,206,206,206,206,206,206,206,206,206,206,206,206,206,206,206,
	206,206,206,206,207,207,207,207,207,207,207,207,207,207,207,207,
	207,207,207,207,207,207,207,207,207,207,207,207,207,207,207,207,
	207,207,207,208,208,208,208,208,208,208,208,208,208,208,208,208,
	208,208,208,208,208,208,208,208,208,208,208,208,208,208,208,208,
	208,208,209,209,209,209,209,209,209,209,209,209,209,209,209,209,
	209,209,209,209,209,209,209,209,209,209,209,209,209,209,209,209,
	209,210,210,210,210,210,210,210,210,210,210,210,210,210,210,210,
	210,210,210,210,210,210,210,210,210,210,210,210,210,210,210,211,
	211,211,211,211,211,211,211,211,211,211,211,211,211,211,211,211,
	211,211,211,211,211,211,211,211,211,211,211,211,211,212,212,212,
	212,212,212,212,212,212,212,212,212,212,212,212,212,212,212,212,
	212,212,212,212,212,212,212,212,212,212,213,213,213,213,213,213,
	213,213,213,213,213,213,213,213,213,213,213,213,213,213,213,213,
	213,213,213,213,213,213,213,213,214,214,214,214,214,214,214,214,
	214,214,214,214,214,214,214,214,214,214,214,214,214,214,214,214,
	214,214,214,214,214,214,214,215,215,215,215,215,215,215,215,215,
	215,215,215,215,215,215,215,215,215,215,215,215,215,215,215,215,
	215,215,215,215,215,215,215,216,216,216,216,216,216,216,216,216,
	216,216,216,216,216,216,216,216,216,216,216,216,216,216,216,216,
	216,216,216,216,216,216,216,217,217,217,217,217,217,217,217,217,
	217,217,217,217,217,217,217,217,217,217,217,217,217,217,217,217,
	217,217,217,217,217,217,217,218,218,218,218,218,218,218,218,218,
	218,218,218,218,218,218,218,218,218,218,218,218,218,218,218,218,
	218,218,218,218,218,218,219,219,219,219,219,219,219,219,219,219,
	219,219,219,219,219,219,219,219,219,219,219,219,219,219,219,219,
	219,219,219,219,220,220,220,220,220,220,220,220,220,220,220,220,
	220,220,220,220,220,220,220,220,220,220,220,220,220,220,220,220,
	220,220,220,220,221,221,221,221,221,221,221,221,221,221,221,221,
	221,221,221,221,221,221,221,221,221,221,221,221,221,221,221,221,
	221,221,221,222,222,222,222,222,222,222,222,222,222,222,222,222,
	222,222,222,222,222,222,222,222,222,222,222,222,222,222,222,222,
	222,222,222,223,223,223,223,223,223,223,223,223,223,223,223,223,
	223,223,223,223,223,223,223,223,223,223,223,223,223,223,223,223,
	223,223,223,223,224,224,224,224,224,224,224,224,224,224,224,224,
	224,224,224,224,224,224,224,224,224,224,224,224,224,224,224,224,
	224,224,224,224,224,224,225,225,225,225,225,225,225,225,225,225,
	225,225,225,225,225,225,225,225,225,225,225,225,225,225,225,225,
	225,225,225,225,225,225,225,226,226,226,226,226,226,226,226,226,
	226,226,226,226,226,226,226,226,226,226,226,226,226,226,226,226,
	226,226,226,226,226,226,226,227,227,227,227,227,227,227,227,227,
	227,227,227,227,227,227,227,227,227,227,227,227,227,227,227,227,
	227,227,227,227,227,227,227,227,228,228,228,228,228,228,228,228,
	228,228,228,228,228,228,228,228,228,228,228,228,228,228,228,228,
	228,228,228,228,228,228,228,228,228,229,229,229,229,229,229,229,
	229,229,229,229,229,229,229,229,229,229,229,229,229,229,229,229,
	229,229,229,229,229,229,229,229,229,229,229,230,230,230,230,230,
	230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,230,
	230,230,230,230,230,230,230,230,230,230,230,230,230,231,231,231,
	231,231,231,231,231,231,231,231,231,231,231,231,231,231,231,231,
	231,231,231,231,231,231,231,231,231,231,231,231,231,231,232,232,
	232,232,232,232,232,232,232,232,232,232,232,232,232,232,232,232,
	232,232,232,232,232,232,232,232,232,232,232,232,232,232,233,233,
	233,233,233,233,233,233,233,233,233,233,233,233,233,233,233,233,
	233,233,233,233,233,233,233,233,233,233,233,233,233,233,234,234,
	234,234,234,234,234,234,234,234,234,234,234,234,234,234,234,234,
	234,234,234,234,234,234,234,234,234,234,234,234,234,234,235,235,
	235,235,235,235,235,235,235,235,235,235,235,235,235,235,235,235,
	235,235,235,235,235,235,235,235,235,235,235,235,235,235,235,235,
	235,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,
	236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,236,
	236,236,236,236,236,236,237,237,237,237,237,237,237,237,237,237,
	237,237,237,237,237,237,237,237,237,237,237,237,237,237,237,237,
	237,237,237,237,237,237,237,237,237,237,237,237,238,238,238,238,
	238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
	238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,238,
	238,238,238,239,239,239,239,239,239,239,239,239,239,239,239,239,
	239,239,239,239,239,239,239,239,239,239,239,239,239,239,239,239,
	239,239,239,239,239,239,239,239,239,239,240,240,240,240,240,240,
	240,240,240,240,240,240,240,240,240,240,240,240,240,240,240,240,
	240,240,240,240,240,240,240,240,240,240,240,240,240,240,240,240,
	241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,
	241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,241,
	241,241,241,241,241,242,242,242,242,242,242,242,242,242,242,242,
	242,242,242,242,242,242,242,242,242,242,242,242,242,242,242,242,
	242,242,242,242,242,242,242,242,243,243,243,243,243,243,243,243,
	243,243,243,243,243,243,243,243,243,243,243,243,243,243,243,243,
	243,243,243,243,243,243,243,243,243,244,244,244,244,244,244,244,
	244,244,244,244,244,244,244,244,244,244,244,244,244,244,244,244,
	244,244,244,244,244,244,244,244,244,244,244,245,245,245,245,245,
	245,245,245,245,245,245,245,245,245,245,245,245,245,245,245,245,
	245,245,245,245,245,245,245,245,245,245,245,245,245,246,246,246,
	246,246,246,246,246,246,246,246,246,246,246,246,246,246,246,246,
	246,246,246,246,246,246,246,246,246,246,246,246,246,246,246,246,
	247,247,247,247,247,247,247,247,247,247,247,247,247,247,247,247,
	247,247,247,247,247,247,247,247,247,247,247,247,247,247,247,247,
	247,247,247,247,248,248,248,248,248,248,248,248,248,248,248,248,
	248,248,248,248,248,248,248,248,248,248,248,248,248,248,248,248,
	248,248,248,248,248,248,248,248,248,249,249,249,249,249,249,249,
	249,249,249,249,249,249,249,249,249,249,249,249,249,249,249,249,
	249,249,249,249,249,249,249,249,249,249,249,249,249,249,250,250,
	250,250,250,250,250,250,250,250,250,250,250,250,250,250,250,250,
	250,250,250,250,250,250,250,250,250,250,250,250,250,250,250,250,
	250,250,251,251,251,251,251,251,251,251,251,251,251,251,251,251,
	251,251,251,251,251,251,251,251,251,251,251,251,251,251,251,251,
	251,251,251,251,251,251,251,252,252,252,252,252,252,252,252,252,
	252,252,252,252,252,252,252,252,252,252,252,252,252,252,252,252,
	252,252,252,252,252,252,252,252,252,252,253,253,253,253,253,253,
	253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,253,
	253,253,253,253,253,253,253,253,253,253,254,254,254,254,254,254,
	254,254,254,254,254,254,254,254,254,254,254,254,254,254,254,254,
	254,254,254,255,255,255,255,255,255,255,255,255,255,255,255,255,
};

/*
 *	Build gamma adjustment tables.
 *	D. P. Mitchell  85/07/16.
 *
 *	RGB camera in video lab has gamma of about 1.5.  Hitachi SUN
 *	color monitors have gamma of about 2.3.
 */

extern double exp(), log();

void
set_hitachi_to_luminance(gamma)
double gamma;
{
	int i;
	double x;

	hitachi_to_luminance[0] = 0;
	for (i = 1; i < 256; i++) {
		x = (double)i / 255.0;
		x = exp(log(x) * gamma);
		hitachi_to_luminance[i] = 4095.0 * x + 0.5;
	}
}

void
set_luminance_to_hitachi(gamma)
double gamma;
{
	int i;
	double x;

	luminance_to_hitachi[0] = 0;
	for (i = 1; i < 4096; i++) {
		x = (double)i / 4095.0;
		x = exp(log(x) / gamma);
		luminance_to_hitachi[i] = 255.0 * x + 0.5;
	}
}

static float itox, jtox, xtoi, xtoj;
static int startprog;
static Filterprog *prog;
static long *vignette;

double B = 0.3333333333;
double C = 0.3333333333;

static double
bspline(x)
double x;
{

	if (x < 1.0)
		return 0.5*x*x*x - x*x + 2.0/3.0;
	else
		return -x*x*x/6.0 + x*x - 2.0*x + 4.0/3.0;
}

static double
ease(x)
double x;
{

	if (x < 1.0)
		return 2.0*x*x*x - 3.0*x*x + 1.0;
	else
		return 0.0;
}

static double
crisp(x)
double x;
{

	if (x < 1.0)
		return -x*x*x + x*x;
	else
		return -x*x*x + 5.0*x*x - 8.0*x + 4.0;
}

double
filter(x)
double x;
{

	if (x < 0.0) x = -x;
	if (x >= 2.0) return 0.0;
	return (1.0 -B)*ease(x) + B*bspline(x) + C*crisp(x);
}

static Filterprog *
impulse_response(j)
int j;
{
	int start, stop, i;
	float x, y;
	Filterprog *fp;

	fp = (Filterprog *) resample_malloc (sizeof (Filterprog));
	x = jtox*((float)j + 0.5);
	start = floor(xtoi*(x - HALFWIDTH) - 0.5);
	start++;
	stop = floor(xtoi*(x + HALFWIDTH) - 0.5);
	fp->count = 1 + stop - start;
	fp->coef = (short *) resample_malloc (fp->count * sizeof(short));
	fp->decr = start;
	fp->next = 0;
	for (i = start; i <= stop; i++) {
		y = itox*((float)i + 0.5);
		fp->coef[i - start] = 4096.0*filter(x - y) + 0.5;
/*
fprintf(stderr, "%3d - %d\n", i, fp->coef[i - start]);
*/
	}
	return fp;
}

static int
gcd(u, v)
int u; int v;
{

	if (v > u)
		return gcd(v, u);
	else if (v == 0)
		return u;
	else
		return gcd(v, u % v);
}

static void
compile_resampling_program(m, n)
int m; int n;
{
	int j, u, period;
	Filterprog *temp, *fp, **fpp;
	float x;

/*
for (x = 0.0; x < HALFWIDTH; x += 0.25)
	fprintf(stderr, "f(%f) == %f\n", x, filter(x));
*/
	u = MIN(m, n);
	itox = (float)u/(float)m;
	jtox = (float)u/(float)n;
	xtoi = 1.0/itox;
	xtoj = 1.0/jtox;
	period = n / gcd(m, n);
	fpp = &fp;
	for (j = 0; j < period + 1; j++) {
		*fpp = impulse_response(j);
		fpp = &(*fpp)->next;
	}
	startprog = fp->decr;
	for (temp = fp, j = 0; j < period; j++) {
		temp->decr = temp->count - temp->next->decr + temp->decr;
		if (j < period - 1)
			temp = temp->next;
	}
	temp->next = fp;			/* circular */
	prog = fp;
}

void
antivignette(n)
int n;
{
	register short *coef;
	register i, offset;
	register sig;
	register short temp;
	Filterprog *fp;
	long *vp;
	int width;

	vp = vignette = (long *) resample_malloc (sizeof (long) * n);
	width = n;
	fp = prog;
	offset = startprog;
	for (fp = prog; --n >= 0; fp = fp->next) {
		i = fp->count;
		coef = fp->coef;
		for (sig = 0; --i >= 0; ) {
			temp = offset >= 0 && offset < width;
			offset++;
			sig += temp * *coef++;
		}
		*vp++ = sig;
		offset -= fp->decr;
	}
}

void
resample(input, output, m, n, init)
short *input; short *output;
int m; int n;
int init;
{
	register short *src;
	register short *coef;
	register i;
	register sig;
	register short temp;
	Filterprog *fp;
	long *vp;

	if (init == 0) {
		itox = jtox = xtoi = xtoj = 0.0;
		startprog = 0;
		prog = (Filterprog *) NULL;
		vignette = (long *) NULL;
		init = 1;
		compile_resampling_program(m, n);
	/*	antivignette(n);	*/
	}
	vp = vignette;
	fp = prog;
	src = input + startprog;
	for (fp = prog; --n >= 0; fp = fp->next) {
		i = fp->count;
		coef = fp->coef;
		for (sig = 0; --i >= 0; ) {
			temp = *src++;		/* bug in SUN compiler */
			sig += temp * *coef++;
		}
	/*	sig /= *vp++;		*/
		if (m > n)
			sig = itox*(double)sig;
		sig >>= 12;
		if (sig < 0)
			sig = 0;
		if (sig > 4095)
			sig = 4095;
		*output++ = sig;
		src -= fp->decr;
	}
}

void
main(argc, argv)
	int argc;
	char **argv;
{
	short bitspersample, samplesperpixel, shortv, config;
	long in_imagewidth, in_imagelength;
	unsigned short compression = (unsigned short) -1;
	long rowsperstrip = -1;
	float floatv;
	char *stringv;
	int row, s;
	TIFF *in, *out;
	int out_imagewidth = -1;
	int out_imagelength = -1;

	int n;
	register i;
	register unsigned char *cp;
	register short *sp;
	short *in_chan, *out_chan;
	unsigned char *in_buf, *out_buf;
	int init;
	unsigned char *complete_image;
	int out_tiff_width;
	char *program = argv[0];
	int xargc = argc;
	char **xargv = argv;

	if (argc < 3)
		usage(program);
	argc--, argv++;
	for (; argc > 2 && argv[0][0] == '-'; argc--, argv++) {
		if (streq(argv[0], "-none")) {
			compression = COMPRESSION_NONE;
			continue;
		}
		if (streq(argv[0], "-packbits")) {
			compression = COMPRESSION_PACKBITS;
			continue;
		}
		if (streq(argv[0], "-lzw")) {
			compression = COMPRESSION_LZW;
			continue;
		}
		if (streq(argv[0], "-rowsperstrip")) {
			argc--, argv++;
			rowsperstrip = atoi(argv[0]);
			continue;
		}
		if (streq(argv[0], "-width")) {
			argc--, argv++;
			out_imagewidth = atoi(argv[0]);
			continue;
		}
		if (streq(argv[0], "-height")) {
			argc--, argv++;
			out_imagelength = atoi(argv[0]);
			continue;
		}
		if (streq(argv[0], "-B")) {
			argc--, argv++;
			B = atof(argv[0]);
			continue;
		}
		if (streq(argv[0], "-C")) {
			argc--, argv++;
			C = atof(argv[0]);
			continue;
		}
	}

	in = TIFFOpen(argv[0], "r");
	if (in == NULL)
		exit(-1);

	out = TIFFOpen(argv[1], "w");
	if (out == NULL)
		exit(-2);

	CopyField(TIFFTAG_SUBFILETYPE, shortv);

	CopyField(TIFFTAG_IMAGEWIDTH, in_imagewidth);
	if (out_imagewidth != -1)
		TIFFSetField(out, TIFFTAG_IMAGEWIDTH, out_imagewidth);
	else
		out_imagewidth = in_imagewidth;

	CopyField(TIFFTAG_IMAGELENGTH, in_imagelength);
	if (out_imagelength != -1)
		TIFFSetField(out, TIFFTAG_IMAGELENGTH, out_imagelength);
	else
		out_imagelength = in_imagelength;

	CopyField(TIFFTAG_BITSPERSAMPLE, bitspersample);
	if (compression != (unsigned short)-1)
		TIFFSetField(out, TIFFTAG_COMPRESSION, compression);
	else
		CopyField(TIFFTAG_COMPRESSION, compression);

	CopyField(TIFFTAG_PHOTOMETRIC, shortv);
	CopyField(TIFFTAG_THRESHHOLDING, shortv);
	CopyField(TIFFTAG_ORIENTATION, shortv);
	CopyField(TIFFTAG_SAMPLESPERPIXEL, samplesperpixel);
	CopyField(TIFFTAG_PREDICTOR, shortv);
	CopyField(TIFFTAG_MINSAMPLEVALUE, shortv);
	CopyField(TIFFTAG_MAXSAMPLEVALUE, shortv);
	CopyField(TIFFTAG_XRESOLUTION, floatv);
	CopyField(TIFFTAG_YRESOLUTION, floatv);
	CopyField(TIFFTAG_RESOLUTIONUNIT, shortv);

	CopyField(TIFFTAG_PLANARCONFIG, config);

	if (rowsperstrip <= 0)
		rowsperstrip = (8*1024)/TIFFScanlineSize(out);
	TIFFSetField(out, TIFFTAG_ROWSPERSTRIP,
	    rowsperstrip == 0 ? 1 : rowsperstrip);

	CopyField(TIFFTAG_XPOSITION, floatv);
	CopyField(TIFFTAG_YPOSITION, floatv);
	CopyField(TIFFTAG_GRAYRESPONSEUNIT, shortv);
	{ unsigned short *graycurve;
	  CopyField(TIFFTAG_GRAYRESPONSECURVE, graycurve);
	}
	CopyField(TIFFTAG_COLORRESPONSEUNIT, shortv);
	CopyField(TIFFTAG_MATTEING, shortv);
	{ unsigned short *red, *green, *blue;
	  CopyField3(TIFFTAG_COLORMAP, red, green, blue);
	}
	CopyField(TIFFTAG_ARTIST, stringv);
	stringv = (char *) NULL;
	TIFFGetField(in, TIFFTAG_IMAGEDESCRIPTION, &stringv);
	if (stringv)
	{
		i = strlen (stringv) + 2;
	}
	else
	{
		i = 2;
		stringv = "";
	}
	for (n = 0; n < xargc; n++)
		i += strlen (xargv[n]) + 1;
	cp = (unsigned char *) resample_malloc (i);
	sprintf ((char *) cp, "%s\n", stringv);
	i = strlen ((char *) cp);
	for (n = 0; n < xargc; n++)
	{
		sprintf ((char *) &cp[i], "%s ", xargv[n]);
		i += strlen (xargv[n]) + 1;
	}
	TIFFSetField(out, TIFFTAG_IMAGEDESCRIPTION, cp);
	
	(void) TIFFGetField(in, TIFFTAG_PLANARCONFIG, &shortv);
	if (shortv != PLANARCONFIG_CONTIG)
	{
	fprintf (stderr, "%s only works with planar contiguous images.\n",
		 program);
		exit(-1);
	}
	if (shortv != config && bitspersample != 8 && samplesperpixel > 1)
	{
		fprintf(stderr,
    "%s cannot handle different planar configurations with bits/sample != 8\n",
		 program);
		exit(-1);
	}

#define	pack(a,b)	((a)<<16)|(b)
	switch (pack(shortv, config))
	{
	case pack(PLANARCONFIG_CONTIG, PLANARCONFIG_CONTIG):
		in_chan = (short *) resample_malloc (sizeof (short) *
						     (in_imagewidth + 2000));
		for (i = in_imagewidth + 2000; --i > 0;)
			in_chan[i] = 0;
		in_chan += 1000;
		out_chan = (short *) resample_malloc (sizeof (short) *
						      out_imagewidth);

		out_tiff_width = MAX (TIFFScanlineSize (in),
				      TIFFScanlineSize (out));
		complete_image = (unsigned char *)
			resample_malloc (out_tiff_width *
					 MAX (in_imagelength,
					      out_imagelength));
							   
		in_buf = (unsigned char *)
			resample_malloc (TIFFScanlineSize (in));

		init = 0;
		for (row = 0; row < in_imagelength; row++)
		{
			if (TIFFReadScanline(in, in_buf, row, 0) < 0)
				break;

			if (in_imagewidth == out_imagewidth)
			{
				memcpy (complete_image + row * out_tiff_width,
					in_buf,
					out_imagewidth * samplesperpixel);
				continue;
			}

			out_buf = complete_image + row * out_tiff_width;
			for (n = 0; n < samplesperpixel; n++)
			{
				cp = in_buf + n;
				sp = in_chan;
				for (i = 0; i < in_imagewidth; i++)
				{
					*sp++ = hitachi_to_luminance[*cp];
					cp += samplesperpixel;
				}
				resample(in_chan, out_chan, in_imagewidth,
					 out_imagewidth, init);
				if (init == 0)
					init++;
				sp = out_chan;
				cp = out_buf + n;
				for (i = 0; i < out_imagewidth; i++)
				{
					*cp = luminance_to_hitachi[*sp++];
					cp += samplesperpixel;
				}
			}

		}

		if (in_imagelength == out_imagelength)
		{
			cp = complete_image;
			for (row = 0; row < out_imagelength; row++)
			{
				if (TIFFWriteScanline(out, cp, row, 0) < 0)
					goto done;
				cp += out_tiff_width;
			}
			break;
		}

		free (in_chan - 1000);
		free (out_chan);
		in_chan = (short *) resample_malloc (sizeof (short) *
						     (in_imagelength + 2000));
		for (i = in_imagelength + 2000; --i > 0;)
			in_chan[i] = 0;
		in_chan += 1000;
		out_chan = (short *) resample_malloc (sizeof (short) * out_imagelength);

		init = 0;
		for (row = 0; row < out_imagewidth * samplesperpixel; row++)
		{
			sp = in_chan;
			for (i = 0; i < in_imagelength; i++)
			{
				*sp++ = hitachi_to_luminance
					[complete_image[i * out_tiff_width +
							row]];
			}
			resample(in_chan, out_chan, in_imagelength,
				 out_imagelength, init);
			if (init == 0)
				init++;
			sp = out_chan;
			for (i = 0; i < out_imagelength; i++)
			{
				complete_image[i * out_tiff_width + row] =
					luminance_to_hitachi[*sp++];
			}
		}

		cp = complete_image;
		for (row = 0; row < out_imagelength; row++)
		{
			if (TIFFWriteScanline(out, cp, row, 0) < 0)
				goto done;
			cp += out_tiff_width;
		}
		break;
	}
done:
	(void) TIFFClose(in);
	(void) TIFFClose(out);
}

void
usage(program)
char *program;
{
	fprintf(stderr, "usage: %s [options] input output\n", program);
	fprintf(stderr, "where options are:\n");
	fprintf(stderr,
	    " -lzw\t\tcompress output with Lempel-Ziv & Welch encoding\n");
	fprintf(stderr,
	    " -packbits\tcompress output with packbits encoding\n");
	fprintf(stderr,
	    " -none\t\tuse no compression algorithm on output\n");
	fprintf(stderr, "\n");
	fprintf(stderr,
	    " -rowsperstrip #\tmake each strip have no more than # rows\n");
	fprintf(stderr, "\n");
	fprintf(stderr, " -width #\twidth of the output image\n");
	fprintf(stderr, " -height #\theight of the output image\n");
	fprintf(stderr, "\n");
	fprintf(stderr, " -B #\tB factor for resampling the image\n");
	fprintf(stderr, " -C #\tC factor for resampling the image\n");
	exit(-1);
}
