static char TMXIpcServer_c[] = "<%W%	%D% %T%>";
/*
 * 			Copyright 1993, 1994 by AT&T
 * 
 * 			 All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of AT&T not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * AT&T DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * AT&T BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * 
 * AT&T's dontation of this software does not imply a licence granted for
 * patents nor transfer of ownership of any patents which may inadvertently
 * be implemented in this code.
 * 
 */

#include <sys/time.h>		/* RAD - added time.h and resource.h to use */
#include <sys/resource.h>	/* getrlimit instaead of getdtablesize  */
#include "XIpc.h"

#ifdef DoesNotLookRight
#if defined(__STDC__)
#include <sys/systeminfo.h>	/* RAD - using sysinfo instead of gethostname*/
#endif
#endif
#if defined(__STDC__)
#include <sys/file.h>		/* RAD - to pick up FNDELAY */
#endif

/*
 * function prototypes in this file
 */
#include "C_P_args.h"

C_PROTOS_BEGIN_EXTERN

extern XIpcServer *
XIpcSetupServer C_P_ARGS((char *server_display_name, caddr_t server_display,
			  caddr_t server_window, char *environment_name));

extern XIpcServer *
XIpcSetupPortServer C_P_ARGS((char *server_display_name,
			      caddr_t server_display, caddr_t server_window,
			      char *environment_name, int port_number));

extern void
XIpcCloseServer C_P_ARGS((XIpcServer *server));

extern void
XIpcFlushServer C_P_ARGS((XIpcServer *server));

extern void
XIpcServerCloseClient C_P_ARGS((XIpcServer *server,
				C_Promoted_UShort client_id));

extern int
XIpcServerMonitor C_P_ARGS((XIpcServer *server, int which, int timeout));

extern void
XIpcAcceptNewClientSocket C_P_ARGS((XIpcServer *server));

extern void
XIpcServerProcessMessage C_P_ARGS((XIpcServer *server, XIpcMessage *message));

extern int
XIpcFindClient C_P_ARGS((XIpcServer *server, XIpcChannel channel));

extern int
XIpcSendToClient C_P_ARGS((XIpcServer *server,
			   C_Promoted_UShort client_id,
			   XIpcMessage *message));

extern XIpcMessage *
XIpcRecvFromClients C_P_ARGS((XIpcServer *server));

C_PROTOS_END_EXTERN

/*
 * interface routines for servers using XIpc
 */
XIpcServer *
XIpcSetupServer (server_display_name, server_display, server_window,
		 environment_name)
char *server_display_name;
caddr_t server_display;
caddr_t server_window;
char *environment_name;
{
    extern XIpcServer *_XIpcSetupServer ();

    return XIpcSetupPortServer (server_display_name, server_display,
				server_window, environment_name, 0);
}

XIpcServer *
XIpcSetupPortServer (server_display_name, server_display, server_window,
		     environment_name, port_number)
char *server_display_name;
caddr_t server_display;
caddr_t server_window;
char *environment_name;
int port_number;
{
    char name_buffer[EBUFSIZ];
    /* XIPC=cinnamon:1234 */
    char machine[MACHINESIZ];
    char *machine_ptr = machine;
    char *ptr;
    int length;
    int clientsize;
    struct sockaddr_in saddr;
    struct sockaddr_in nsaddr;
    register XIpcServer *server;
    static int max_clients = 0;

    /*
     * initailize the clients
     */
#ifdef RLIMIT_NOFILE
    /* RAD */
    if (max_clients == 0)
    {
	struct rlimit rl;	/* RAD */

	getrlimit(RLIMIT_NOFILE, &rl);
	max_clients = (int) rl.rlim_cur;
    }
#else
    if (max_clients == 0)
	max_clients = getdtablesize ();
#endif

    length = XIpcToNextWord (sizeof (XIpcServer));
    clientsize = max_clients * sizeof (XIpcClient);
    if ((ptr = (char *) malloc (length + clientsize)) == (char *) NULL)
    {
	fprintf (stderr,
		 "Memory error: cannot create the clients array.\n");
	return (XIpcServer *) NULL;
    }
    server = (XIpcServer *) ptr;
    ptr += length;
    server -> clients = (XIpcClient *) ptr;
    server -> max_clients = max_clients;
    FD_ZERO (&server -> select_mask);
    memset (ptr, (char) 0, clientsize);

    /*
     * determine our machine name - default to unix on error
     */
    if (server_display_name != (char *) NULL)
	machine_ptr = server_display_name;
    else
/* RAD */
/*
#if defined(__STDC__)
	if ( sysinfo(SI_HOSTNAME, machine, MACHINESIZ) == -1 )
#else
	if (gethostname (machine, MACHINESIZ))
#endif
*/
	if (gethostname (machine, MACHINESIZ))
	{
	    fprintf (stderr, "Cannot determine the machine name.\n");
	    strcpy (machine, "unix");	/* \(tm */
	}

    /*
     * create the server socket
     */
    if ((server -> channel = socket (AF_INET, SOCK_STREAM, 0)) < 0)
    {
	fprintf (stderr, "Cannot create the server socket.\n");
	XIpcCloseServer (server);
	return (XIpcServer *) NULL;
    }

    memset ((char *) &saddr, (char) 0, sizeof (saddr));
    saddr.sin_family = AF_INET;
    saddr.sin_addr.s_addr = htonl (INADDR_ANY);
    saddr.sin_port = htonl (port_number);

    if (bind (server -> channel, (struct sockaddr *) &saddr, 
			sizeof (saddr)) < 0)
    {
	fprintf (stderr, "cannot bind the socket.\n");
	XIpcCloseServer (server);
	return (XIpcServer *) NULL;
    }

    length = sizeof (nsaddr);
    if (getsockname (server -> channel, (struct sockaddr *) &nsaddr, 
			&length) == -1)
    {
	fprintf (stderr, "Cannot get the socket name.\n");
	XIpcCloseServer (server);
	return (XIpcServer *) NULL;
    }

    if (listen (server -> channel, 5) < 0)	/* max 5 connections allowed */
    {
	fprintf (stderr, "Error listening for socket connections.\n");
	XIpcCloseServer (server);
	return (XIpcServer *) NULL;
    }

    /*
     * add the server name to the environment
     */
    sprintf (name_buffer, "%s=%s:%d",
	     (environment_name ? environment_name : SERVER_ENV_NAME),
	     machine_ptr, ntohs (nsaddr.sin_port));
    if (XIpcSetEnviron (name_buffer) == FALSE)
    {
	fprintf (stderr, "Could not set the server environment.\n");
	XIpcCloseServer (server);
	return (XIpcServer *) NULL;
    }

    return server;
}

void
XIpcCloseServer (server)
register XIpcServer *server;
{
    register int i;

    for (i = 0; i < server -> max_clients; i++)
	if (server -> clients[i].recv_buffer)
	    XIpcServerCloseClient (server, i);

    close (server -> channel);
    free (server);
}

void
XIpcFlushServer (server)
XIpcServer *server;
{
    register XIpcClient *client = server -> clients;
    register int i;
    register int max = server -> max_clients;

    for (i = 0; i < max; i++, client++)
	if (client -> recv_buffer)
	    XIpcSendQueue (client);
}

void
XIpcServerCloseClient (server, client_id)
XIpcServer *server;
XIpcID client_id;
{
    if ((int) client_id >= server -> max_clients ||
	server -> clients[client_id].recv_buffer == (unsigned char *) NULL)
	return;

    XIpcClose (&server -> clients[client_id]);
}

int
XIpcServerMonitor (server, which, timeout)
register XIpcServer *server;
int which;
int timeout;
{
    extern void XIpcAcceptNewClientSocket ();
    fd_set write_select_mask;
    fd_set *writefds;
    int write_flag;
    fd_set read_select_mask;
    fd_set *readfds;
    struct timeval local_timeout;
    struct timeval *time_ptr;
    int rc;
    register int i;

    if ((which & XIPC_MONITOR_BOTH) == 0 && timeout < 0)
	return (XIPC_MONITOR_ERROR);

    if (which & XIPC_MONITOR_MESSAGES)
    {
	for (i = 0; i < server -> max_clients; i++)
	    if (server -> clients[i].recv_buffer &&
		FD_ISSET (server -> clients[i].channel,
			  &server -> select_mask))
		return XIPC_MONITOR_MESSAGES;

	readfds = &server -> select_mask;
    }
    else
	readfds = &read_select_mask;

 ContinueLoop:

    do {
	FD_ZERO (readfds);
	writefds = (fd_set *) NULL;
	if (which & XIPC_MONITOR_MESSAGES)
	{
	    FD_SET (server -> channel, readfds);
	    for (i = 0; i < server -> max_clients; i++)
	    {
		if (server -> clients[i].recv_buffer)
		    FD_SET (server -> clients[i].channel,
			    readfds);

		if (server -> clients[i].message_queue)
		{
		    if (writefds == (fd_set *) NULL)
		    {
			writefds = &write_select_mask;
			FD_ZERO (writefds);
		    }
		    FD_SET (server -> clients[i].channel,
			    writefds);
		}
	    }
	}
	if (which & XIPC_MONITOR_KEYBOARD)
	    FD_SET (0, readfds);

	if (timeout >= 0)
	{
	    local_timeout.tv_sec = timeout;
	    local_timeout.tv_usec = 0;
	    time_ptr = &local_timeout;
	}
	else
	    time_ptr = (struct timeval *) NULL;

    } while ((rc = select (server -> max_clients, readfds, writefds, NULL,
			   time_ptr)) < 0);

    write_flag = FALSE;
    for (i = 0; i < server -> max_clients; i++)
    {
	if (server -> clients[i].message_queue &&
	    FD_ISSET (server -> clients[i].channel,
		      &write_select_mask))
	{
	    write_flag = TRUE;
	    if (XIpcSendQueue (&server -> clients[i]) == FALSE)
	    {
		FD_CLR (server -> clients[i].channel,
			&server -> select_mask);
		XIpcServerCloseClient (server, i);
	    }
	}
    }
    if (FD_ISSET (server -> channel, readfds))
	XIpcAcceptNewClientSocket (server);
    if (rc == 0 && timeout >= 0)
	return XIPC_MONITOR_TIMEOUT;

    if (which & XIPC_MONITOR_KEYBOARD && FD_ISSET (0, readfds))
    {
	FD_CLR (server -> channel, &server -> select_mask);
	FD_CLR (0, &server -> select_mask);
	return XIPC_MONITOR_KEYBOARD;
    }

    for (i = 0; i < server -> max_clients; i++)
	if (server -> clients[i].recv_buffer &&
	    FD_ISSET (server -> clients[i].channel,
		      &server -> select_mask))
	{
	    FD_CLR (server -> channel, &server -> select_mask);
	    return XIPC_MONITOR_MESSAGES;
	}

    if (FD_ISSET (server -> channel, readfds))
    {
	FD_CLR (server -> channel, &server -> select_mask);
	goto ContinueLoop;
    }
    if (write_flag)
	goto ContinueLoop;

    return XIPC_MONITOR_ERROR;
}

void
XIpcAcceptNewClientSocket (server)
register XIpcServer *server;
{
    struct sockaddr_in caddr;
    int caddr_len = sizeof (caddr);
    register int rc;
    int i;
    int flags;
    char *error_message = "cannot open client channel.\n";

    if ((rc = accept (server -> channel, (struct sockaddr *) &caddr, 
			&caddr_len)) < 0)
	return;

    if ((i = XIpcFindClient (server, (XIpcChannel) NULL)) ==
	server -> max_clients)
    {
	close (rc);
	fprintf (stderr, error_message);
	return;
    }

    flags = fcntl (rc, F_GETFL, 0);
    if (flags >= 0 && (flags & FNDELAY) == 0)
	if (fcntl (rc, F_SETFL, flags | FNDELAY) < 0)
	{
	    close (rc);
	    fprintf (stderr, error_message);
	    return;
	}

    memset (&server -> clients[i], (char) 0, sizeof (XIpcClient));
    server -> clients[i].channel = rc;
    if (xipc_fdopen (&server -> clients[i]) == FALSE)
    {
	close (rc);
	fprintf (stderr, error_message);
	return;
    }
}

void
XIpcServerProcessMessage (server, message)
register XIpcServer *server;
register XIpcMessage *message;
{
    XIpcID client_id;
    char buf[sizeof (XIpcMessage) + 16];
    XIpcMessage *return_message_ptr = (XIpcMessage *) buf;

    if (message == (XIpcMessage *) NULL)
	return;

    client_id = message -> client_id;

    if ((message -> type != XIPC_SERVER_NEW_CLIENT &&
	 (int) client_id >= server -> max_clients) ||
	server -> clients[client_id].recv_buffer == (unsigned char *) NULL)
	return;

    switch (message -> type)
    {
    case XIPC_SERVER_NEW_CLIENT:
	return_message_ptr -> client_id = client_id;
	return_message_ptr -> length = 0;
	XIpcSend (&server -> clients[client_id], return_message_ptr);
	XIpcSendQueue (&server -> clients[client_id]);
	server -> clients[client_id].client_id = client_id;
	if (message -> length)
	    server -> clients[client_id].client_key =
		atol (message -> buffer);
	else
	    server -> clients[client_id].client_key = 0;
	return;

    case XIPC_SERVER_DELETE_CLIENT:
	XIpcServerCloseClient (server, client_id);
	return;

    case XIPC_SERVER_ECHO:
	XIpcPrintMessage (stdout, "ECHO request", message);
	return;

    case XIPC_SERVER_PING:
	XIpcSendToClient (server, client_id, message);
	XIpcSendQueue (&server -> clients[client_id]);
	return;
    }
}

int
XIpcFindClient (server, channel)
register XIpcServer *server;
register XIpcChannel channel;
{
    register int client_id;

    for (client_id = 0; client_id < server -> max_clients; client_id++)
	if (server -> clients[client_id].channel == channel)
	    return client_id;

    return client_id;
}

int
XIpcSendToClient (server, client_id, message)
register XIpcServer *server;
register XIpcID client_id;
XIpcMessage *message;
{
    if ((int) client_id >= server -> max_clients ||
	server -> clients[client_id].recv_buffer == (unsigned char *) NULL)
	return FALSE;

    message -> client_id = XIPC_SERVER_CLIENT_ID;

    if (XIpcSend (&server -> clients[client_id], message) == FALSE)
    {
	XIpcServerCloseClient (server, client_id);
	return FALSE;
    }
    return TRUE;
}

XIpcMessage *
XIpcRecvFromClients (server)
register XIpcServer *server;
{
    XIpcMessage *message;
    register XIpcID client_id;

 ContinueLoop:
    for (client_id = 0; (int) client_id < server -> max_clients; client_id++)
	if (server -> clients[client_id].recv_buffer &&
	    FD_ISSET (server -> clients[client_id].channel,
		      &server -> select_mask))
	    break;

    if (client_id == server -> max_clients)
	return (XIpcMessage *) NULL;

    message = XIpcRecv (&server -> clients[client_id]);

    if (message == (XIpcMessage *) -1)
    {
	FD_CLR (server -> clients[client_id].channel, &server -> select_mask);
	/*
	  fprintf (stderr, "server closing dead client #%d\n",
	  client_id);
	  */
	XIpcServerCloseClient (server, client_id);
	goto ContinueLoop;
    }

    if (message == (XIpcMessage *) NULL)
    {
	FD_CLR (server -> clients[client_id].channel, &server -> select_mask);
	goto ContinueLoop;
    }

    if (XIpcBufferEmpty (&server -> clients[client_id]))
	FD_CLR (server -> clients[client_id].channel, &server -> select_mask);

    if (message -> type == XIPC_SERVER_NEW_CLIENT)
	message -> client_id = client_id;
    else
	if (client_id != message -> client_id)
	    return (XIpcMessage *) NULL;

    return message;
}
