;;;
;;; standard macro for AKCL C inclusion
;;;
(eval-when (compile eval)
	   (set-macro-character #\% #'(lambda (stream char)
					(values (read-line stream)))))

;;;
;;; confine our work to the XIpc package
;;;
(in-package 'xipc)

;;;
;;; include Lisp constants
;;;
(require 'xipc "XIpcDefs.lsp")

;;;
;;; define the package
;;;
(provide 'xipc)
(export '(
;;;
;;;	structure creation functions
;;;
	  new-xipc-message new-xipc-server new-xipc-client
;;;
;;; 	XIpc general support functions
;;;
	  XIpcPrintMessage XIpcCatchErrors XIpcCopyMessage XIpcIsClientActive
;;;
;;;	XIpc server process support functions
;;;
	  XIpcSetupServer XIpcCloseServer XIpcFlushServer XIpcServerCloseClient
	  XIpcServerMonitor XIpcServerProcessMessage XIpcSendToClient
	  XIpcRecvFromClients
;;;
;;;	XIpc client process support functions
;;;
	  XIpcSetupClient XIpcCloseClient XIpcFlushClient XIpcClientMonitor
	  XIpcSendToServer XIpcRecvFromServer
;;;
;;;	support function
;;;
	  getenv
))

;;;
;;; add a documentation line to the AKCL defentry function
;;;
(defmacro defentry* (f a doc fm)
  `(progn
     (defentry ,f ,a ,fm)
     (setf (documentation (quote ,f) 'function) (quote ,doc))))

;;;
;;; include C constants and typedefs
;;;
(clines
%#include "XIpc.h"
)

;;;
;;; Lisp version of the XIpc message struct and support functions
;;;
(defstruct xipc-message
  "Lisp accessible message structure for the XIpc package."
  (type      0   :type fixnum)
  (client_id 0   :type fixnum)
  (buffer    ""  :type string))

(defun new-xipc-message (type client_id buffer)
  "Syntax: (new-xipc-message type client_id buffer)\
Argument (rather than keyword) message constructor function.  Useful from\
defCfun functions."
  (make-xipc-message :type type :client_id client_id :buffer buffer))

(defun xipc-messagep (message)
  "Syntax: (xipc-messagep message)\
Check if the argument is an xipc-message structure.  Mainly useful\
for type checking from defCfun fucntions."
  (typep message 'xipc-message))

;;;
;;; Blind pointer to the XIpc server structure
;;;
(defstruct xipc-server
  "Lisp blind pointer to the server structure for the XIpc package."
  (pointer 0 :type fixnum))

(defun new-xipc-server (server)
  "Syntax: (new-xipc-server server)\
Argument (rather than keyword) message constructor function.  Useful from\
defCfun functions."
  (make-xipc-server :pointer server))

(defun xipc-serverp (server)
  "Syntax: (xipc-serverp server)\
Check if the argument is an xipc-server structure.  Mainly useful\
for type checking from defCfun fucntions."
  (typep server 'xipc-server))

;;;
;;; Blind pointer to the XIpc client structure
;;;
(defstruct xipc-client
  "Lisp blind pointer to the client structure for the XIpc package."
  (pointer 0 :type fixnum))

(defun new-xipc-client (client)
  "Syntax: (new-xipc-client client)\
Argument (rather than keyword) message constructor function.  Useful from\
defCfun functions."
  (make-xipc-client :pointer client))

(defun xipc-clientp (client)
  "Syntax: (xipc-clientp client)\
Check if the argument is an xipc-client structure.  Mainly useful\
for type checking from defCfun fucntions."
  (typep client 'xipc-client))

;;;
;;; XIpc general support functions
;;;

;;;
;;;  (XIpcPrintMessage fp str message) - print a message - debugging tool
;;;
(defun XIpcPrintMessage (fp str message)
  "Syntax: (XIpcPrintMessage fp str message)\
Print an XIpcMessage to a specified stream (uses format).  str is prepended\
to the items in the message."
  (cond ((null message)
	 (format fp "~a: NULL message.~%" str))
	(t
	 (when (typep message 'xipc-message)
	       (format fp "~a: xipc-message-type: ~a~%"
		       str (xipc-message-type message))
	       (format fp "~a: xipc-message-client_id: ~a~%"
		       str (xipc-message-client_id message))
	       (format fp "~a: xipc-message-buffer: ->~a<-~%"
		       str (xipc-message-buffer message))))))

;;;
;;;  (XIpcCatchErrors t | nil) - catch ipc related errors
;;;
(defCfun "_XIpcCatchErrors (type) object type;" 0
%	if (type == Cnil)
%		XIpcCatchErrors (FALSE, (void *) NULL);
%	else
%		XIpcCatchErrors (TRUE, (void *) NULL);
%	Creturn (type);
)
(defentry* XIpcCatchErrors (object)
  "Syntax: (XIpcCatchErrors type)\
Enable (t) or disable (nil) catching IPC related errors."
  (object "_XIpcCatchErrors"))

;;;
;;;  (XIpcCopyMessage message) - make a copy of an XIpc message
;;;
(defun XIpcCopyMessage (message)
  "Syntax: (XIpcCopyMessage message)\
Make a copy of an XIpc message."
  (if (typep message 'xipc-message)
      (make-xipc-message :type (xipc-message-type message)
			 :client_id (xipc-message-client_id message)
			 :buffer (xipc-message-buffer message))))

;;;
;;;  (XIpcIsClientActive client) - Check if the XIpc client is valid
;;;
(defCfun "_XIpcIsClientActive (client) object client;" 1
%	object client_pointer;
%
	((xipc-clientp client) client_pointer)
%	if (client_pointer == Cnil)
%		Creturn (Cnil);
	((xipc-client-pointer client) client_pointer)
%	if (XIpcIsClientActive (fix (client_pointer)))
%		Creturn (Ct);
%	else
%		Creturn (Cnil);
)
(defentry* XIpcIsClientActive (object)
  "Syntax: (XIpcIsClientActive client)\
Check if the XIpc client is still logged in."
  (object "_XIpcIsClientActive"))

;;;
;;;  (XIpcSynchronize client_struct type) - 
;;;
(defCfun "_XIpcSynchronize (client, type) object client; object type;" 1
%	object client_pointer;
%	int rc;
%
	((xipc-clientp client) client_pointer)
%	if (client_pointer == Cnil)
%		Creturn (Cnil);
	((xipc-client-pointer client) client_pointer)
%	if (type == Cnil)
%		rc = XIpcSynchronize (fix (client_pointer), FALSE);
%	else
%		rc = XIpcSynchronize (fix (client_pointer), TRUE);
%	if (rc)
%		Creturn (Ct);
%	else
%		Creturn (Cnil);
)
(defentry* XIpcSynchronize (object object)
  "Syntax: (XIpcSynchronize client_struct type)\
If type is nil then buffer messages."
  (object "_XIpcSynchronize"))

;;;
;;; XIpc server process support functions
;;;

;;;
;;;  (XIpcSetupServer &optional server_display_name server_display
;;;	server_window server_env)
;;;   setup an XIpc server.
;;;
(defCfun "__XIpcSetupServer (server_display_name, server_display, server_window, server_env) object server_display_name; object server_display; object server_window; object server_env;" 2
%	XIpcServer *server;
%	char tempc1, tempc2;
%	int length1, length2;
%	char *ptr1, *ptr2;
%	object server_pointer;
%	object server_struct;
%
%	if ((length1 = server_display_name -> st.st_fillp) > 0)
%	{
%		ptr1 = server_display_name -> st.st_self;
%		tempc1 = ptr1[length1]; ptr1[length1] = '\0';
%	}
%	else
%		ptr1 = (char *) NULL;
%	if ((length2 = server_display_name -> st.st_fillp) > 0)
%	{
%		ptr2 = server_display_name -> st.st_self;
%		tempc2 = ptr2[length2]; ptr2[length2] = '\0';
%	}
%	else
%		ptr2 = (char *) NULL;
%	server = XIpcSetupServer (ptr1, fix (server_display),
%					fix (server_window), ptr2);
%	if (length1 > 0)
%		ptr1[length1] = tempc1;
%	if (length2 > 0)
%		ptr2[length2] = tempc2;
%	server_pointer = make_fixnum ((long) server);
	((new-xipc-server server_pointer) server_struct)
%	Creturn (server_struct);
)
(defentry* _XIpcSetupServer (object object object object)
  "Syntax: (XIpcSetupServer server_display_name server_display server_window server_env)\
Setup a server XIpc channel."
  (object "__XIpcSetupServer"))

(defun XipcSetupServer (&optional (display_name "") (server_display 0) (server_window 0) (server_env ""))
  "Syntax: (XIpcSetupServer &optional server_display_name server_display server_environment_name)\
Setup a server XIpc channel."
  (_XIpcSetupServer display_name server_display server_window server_env))

;;;
;;;  (XIpcCloseServer server) - shutdown the server.
;;;
(defCfun "_XIpcCloseServer (server) object server;" 1
%	object server_pointer;
%
	((xipc-serverp server) server_pointer)
%	if (server_pointer == Cnil)
%		Creturn (Cnil);
	((xipc-server-pointer server) server_pointer)
%	XIpcCloseServer (fix (server_pointer));
%	Creturn (Cnil);
)
(defentry* XIpcCloseServer (object)
  "Syntax: (XIpcCloseServer server)\
Close down the XIpc server."
  (object "_XIpcCloseServer"))

;;;
;;;  (XIpcFlushServer server) - Flush out all pending outgoing messages.
;;;
(defCfun "_XIpcFlushServer (server) object server;" 1
%	object server_pointer;
%
	((xipc-serverp server) server_pointer)
%	if (server_pointer == Cnil)
%		Creturn (Cnil);
	((xipc-server-pointer server) server_pointer)
%	XIpcFlushServer (fix (server_pointer));
%	Creturn (Cnil);
)
(defentry* XIpcFlushServer (object)
  "Syntax: (XIpcFlushServer server)\
Flush out all pending outgoing messages."
  (void "_XIpcFlushServer"))

;;;
;;;  (XIpcServerCloseClient server client_id) - Close the client
;;;    connection.
;;;
(defCfun "_XIpcServerCloseClient (server, client_id) object server; int client_id;" 1
%	object server_pointer;
%
	((xipc-serverp server) server_pointer)
%	if (server_pointer == Cnil)
%		Creturn (Cnil);
	((xipc-server-pointer server) server_pointer)
%	XIpcServerCloseClient (fix (server_pointer), client_id);
%	Creturn (Cnil);
)
(defentry* XIpcServerCloseClient (object int)
  "Syntax: (XIpcServerCloseClient server client_id)\
Close the specified client connection."
  (object "_XIpcServerCloseClient"))

;;;
;;;  (XIpcServerMonitor server_struct &optional which timeout) - Monitor
;;;    the keyboard and the XIpc channels to this server with the associated
;;;    timeout in seconds.
;;;
(defCfun "__XIpcServerMonitor (server, which, timeout) object server; int which, timeout;" 1
%	object server_pointer;
%	int rc;
%
	((xipc-serverp server) server_pointer)
%	if (server_pointer == Cnil)
%		Creturn (0);
	((xipc-server-pointer server) server_pointer)
%	rc = XIpcServerMonitor (fix (server_pointer), which, timeout);
%	Creturn (rc);
)
(defentry* _XIpcServerMonitor (object int int)
  "Syntax: (XIpcServerMonitor server_struct which timeout)\
Monitor messages and keyboard with the associated timeout in seconds.\
If timeout is equal to -1 it is ignored."
  (int "__XIpcServerMonitor"))
(defun XipcServerMonitor (server &optional (which XIPC-MONITOR-BOTH)
				 (timeout -1))
  "Syntax: (XIpcServerMonitor server_struct &optional which timeout)\
Monitor messages and keyboard with the associated timeout in seconds.\
If timeout is equal to -1 it is ignored."
  (_XIpcServerMonitor server which timeout))

;;;
;;;  (XIpcServerProcessMessage server message) - Perform standard server
;;;    operations on a message from a client.
;;;
(defCfun "_XIpcServerProcessMessage (server, message) object server, message;" 1
%	object translate;
%	char buffer[BUFSIZ];
%	XIpcMessage *message_ptr = (XIpcMessage *) buffer;
%	int length;
%
	((xipc-messagep message) translate)
%	if (translate == Cnil)
%		Creturn (Cnil);
	((xipc-message-type message) translate)
%	message_ptr -> type = fix (translate);
	((xipc-message-client_id message) translate)
%	message_ptr -> client_id = fix (translate);
	((xipc-message-buffer message) translate)
%	if ((length = translate -> st.st_fillp) > 0)
%	{
%		if (length >= XIPC_MAX_MESSAGE_SIZE)
%			length = XIPC_MAX_MESSAGE_SIZE - 1;
%		message_ptr -> length = length;
%		bcopy (translate -> st.st_self, message_ptr -> buffer, length);
%		message_ptr -> buffer[length] = '\0';
%	}
%	else
%		message_ptr -> length = 0;

	((xipc-serverp server) translate)
%	if (translate == Cnil)
%		Creturn (Cnil);
	((xipc-server-pointer server) translate)
%	XIpcServerProcessMessage (fix (translate), message_ptr);
%	Creturn (Cnil);
)
(defentry* XIpcServerProcessMessage (object object)
  "Syntax: (XIpcServerProcessMessage server message)\
Process the specified message.  This function does all of the client\
management."
  (void "_XIpcServerProcessMessage"))

;;;
;;;  (XIpcSendToClient server client_id message) - send a message to a client
;;;    specified by server ID.
;;;
(defCfun "_XIpcSendToClient (server, client_id, message) object server; int client_id; object message;" 1
%	object translate;
%	int length;
%	char buffer[BUFSIZ];
%	XIpcMessage *message_ptr = (XIpcMessage *) buffer;
%
	((xipc-messagep message) translate)
%	if (translate == Cnil)
%		Creturn (Cnil);
	((xipc-message-type message) translate)
%	message_ptr -> type = fix (translate);
	((xipc-message-client_id message) translate)
%	message_ptr -> client_id = fix (translate);
	((xipc-message-buffer message) translate)
%	if ((length = translate -> st.st_fillp) > 0)
%	{
%		if (length >= XIPC_MAX_MESSAGE_SIZE)
%			length = XIPC_MAX_MESSAGE_SIZE - 1;
%		message_ptr -> length = length;
%		bcopy (translate -> st.st_self, message_ptr -> buffer, length);
%		message_ptr -> buffer[length] = '\0';
%	}
%	else
%		message_ptr -> length = 0;

	((xipc-serverp server) translate)
%	if (translate == Cnil)
%		Creturn (Cnil);
	((xipc-server-pointer server) translate)
%	XIpcSendToClient (fix (translate), client_id, message_ptr);
%	Creturn (Cnil);
)
(defentry* XIpcSendToClient (object int object)
  "Syntax: (XIpcSendToClient server client_id message)\
Send the specified message to the client specified by the id number."
  (void "_XIpcSendToClient"))

;;;
;;;  (XIpcRecvFromClients server) - receive a message from one of the
;;;    clients.
;;;
(defCfun "_XIpcRecvFromClients (server) object server;" 5
%	object server_pointer;
%	XIpcMessage *message_ptr;
%	object type, client_id, buffer;
%	object message;
%
	((xipc-serverp server) server_pointer)
%	if (server_pointer == Cnil)
%		Creturn (Cnil);
	((xipc-server-pointer server) server_pointer)
%	message_ptr = XIpcRecvFromClients (fix (server_pointer));
%	if (message_ptr == NULL)
%		Creturn (Cnil);
%
%	type = make_fixnum (message_ptr -> type);
%	client_id = make_fixnum (message_ptr -> client_id);
%	buffer = make_simple_string (message_ptr -> buffer);
	((new-xipc-message type client_id buffer) message)
%	Creturn (message);
)
(defentry* XIpcRecvFromClients (object)
  "Syntax: (XIpcRecvFromClients server)\
Receive a message from a client process.  This should be called after\
the XIpcServerMonitor has indicated that an incoming message has arrived."
  (object "_XIpcRecvFromClients"))

;;;
;;; XIpc client process support functions
;;;

;;;
;;; (XIpcSetupClient &optional xipc_name client_display client_window) -
;;;   setup an XIpc client.
;;;
(defCfun "__XIpcSetupClient (xipc_name, client_display, client_window) object xipc_name; object client_display; object client_window;" 2
%	XIpcClient *client;
%	char tempc;
%	int length;
%	char *ptr;
%	object client_pointer;
%	object client_struct;
%
%	if ((length = xipc_name -> st.st_fillp) > 0)
%	{
%		ptr = xipc_name -> st.st_self;
%		tempc = ptr[length]; ptr[length] = '\0';
%	}
%	else
%		ptr = (char *) NULL;
%
%	client = XIpcSetupClient (ptr, fix (client_display),
%				       fix (client_window));
%	if (client == NULL)
%		Creturn (Cnil);
%
%	if (length > 0)
%		ptr[length] = tempc;
%	client_pointer = make_fixnum ((long) client);
%
	((new-xipc-client client_pointer) client_struct)
%
%	Creturn (client_struct);
)
(defentry* _XIpcSetupClient (object object object)
  "Syntax: (XIpcSetupClient xipc_name client_display client_window)\
Setup a client XIpc channel."
  (object "__XIpcSetupClient"))

(defun XIpcSetupClient (&optional (xipc_name "") (client_display 0)
				  (client_window 0))
  "Syntax: (XIpcSetupClient &optional xipc_name client_display client_window)\
Setup a client XIpc channel."
  (_XIpcSetupClient xipc_name client_display client_window))

;;;
;;;  (XIpcCloseClient client) - shutdown the client.
;;;
(defCfun "_XIpcCloseClient (client) object client;" 1
%	object client_pointer;
%
	((xipc-clientp client) client_pointer)
%	if (client_pointer == Cnil)
%		Creturn (Cnil);
	((xipc-client-pointer client) client_pointer)
%	XIpcCloseClient (fix (client_pointer));
%	Creturn (Cnil);
)
(defentry* XIpcCloseClient (object)
  "Syntax: (XIpcCloseClient client)\
Close the specified XIpc client."
  (object "_XIpcCloseClient"))

;;;
;;;  (XIpcFlushClient client_struct) - Send queued outgoing messages
;;;
(defCfun "_XIpcFlushClient (client) object client;" 1
%	object client_pointer;
%
	((xipc-clientp client) client_pointer)
%	if (client_pointer == Cnil)
%		Creturn (Cnil);
	((xipc-client-pointer client) client_pointer)
%	XIpcFlushClient (fix (client_pointer));
%	Creturn (Cnil);
)
(defentry* XIpcFlushClient (object)
  "Syntax: (XIpcFlushClient client_struct)\
Flush (send) queued outgoing messages of the specified XIpc client."
  (void "_XIpcFlushClient"))

;;;
;;;  (XIpcClientMonitor client_struct &optional which timeout) - monitor
;;;    messages and keyboard with the associated timeout in seconds.
;;;
(defCfun "__XIpcClientMonitor (client, which, timeout) object client; int which, timeout;" 1
%	object client_pointer;
%	int rc;
%
	((xipc-clientp client) client_pointer)
%	if (client_pointer == Cnil)
%		Creturn (Cnil);
	((xipc-client-pointer client) client_pointer)
%	rc = XIpcClientMonitor (fix (client_pointer), which, timeout);
%	Creturn (rc);
)
(defentry* _XIpcClientMonitor (object int int)
  "Syntax: (XIpcClientMonitor client_struct which timeout)\
Monitor messages and keyboard with the associated timeout in seconds.\
If timeout is equal to -1 it is ignored."
  (int "__XIpcClientMonitor"))
(defun XipcClientMonitor (client &optional (which XIPC-MONITOR-BOTH)
				 (timeout -1))
  "Syntax: (XIpcClientMonitor client_struct &optional which timeout)\
Monitor messages and keyboard with the associated timeout in seconds.\
If timeout is equal to -1 it is ignored."
  (_XIpcClientMonitor client which timeout))

;;;
;;;  (XIpcSendToServer client message) - Send the message to the server.
;;;
(defCfun "_XIpcSendToServer (client, message) object client; object message;" 1
%	object translate;
%	int length;
%	char buffer[BUFSIZ];
%	XIpcMessage *message_ptr = (XIpcMessage *) buffer;
%
	((xipc-messagep message) translate)
%	if (translate == Cnil)
%		Creturn (Cnil);
	((xipc-message-type message) translate)
%	message_ptr -> type = fix (translate);
	((xipc-message-client_id message) translate)
%	message_ptr -> client_id = fix (translate);
	((xipc-message-buffer message) translate)
%	if ((length = translate -> st.st_fillp) > 0)
%	{
%		if (length >= XIPC_MAX_MESSAGE_SIZE)
%			length = XIPC_MAX_MESSAGE_SIZE - 1;
%		message_ptr -> length = length;
%		bcopy (translate -> st.st_self, message_ptr -> buffer, length);
%		message_ptr -> buffer[length] = '\0';
%	}
%	else
%		message_ptr -> length = 0;

	((xipc-clientp client) translate)
%	if (translate == Cnil)
%		Creturn (Cnil);
	((xipc-client-pointer client) translate)
%	XIpcSendToServer (fix (translate), message_ptr);
%	Creturn (Cnil);
)
(defentry* XIpcSendToServer (object object)
  "Syntax: (XIpcSendToServer client message)\
Send the specified message to the server specified in the client struct."
  (void "_XIpcSendToServer"))

;;;
;;;  (XIpcRecvFromServer client) - Receive a message from the server
;;;
(defCfun "_XIpcRecvFromServer (client) object client;" 5
%	object client_pointer;
%	XIpcMessage *message_ptr;
%	object type, client_id, buffer;
%	object message;
%
	((xipc-clientp client) client_pointer)
%	if (client_pointer == Cnil)
%		Creturn (Cnil);
	((xipc-client-pointer client) client_pointer)
%	message_ptr = XIpcRecvFromServer (fix (client_pointer));
%	if (message_ptr == NULL)
%		Creturn (Cnil);
%
%	type = make_fixnum (message_ptr -> type);
%	client_id = make_fixnum (message_ptr -> client_id);
%	buffer = make_simple_string (message_ptr -> buffer);
	((new-xipc-message type client_id buffer) message)
%	Creturn (message);
)
(defentry* XIpcRecvFromServer (object)
  "Syntax: (XIpcRecvFromServer client_struct)\
Receive a message from the server process.  This should be called after\
the XIpcClientMonitor has indicated that an incoming message has arrived."
  (object "_XIpcRecvFromServer"))

;;;
;;;	support function
;;;

(defCfun "object _getenv (arg) object arg;" 1
%	object temp;
%	char tempc;
%	char *rc;
%
%	temp = coerce_to_string (arg);
%	tempc = temp -> st.st_self[temp -> st.st_fillp];
%	temp -> st.st_self[temp -> st.st_fillp] = '\0';
%	rc = getenv (temp -> st.st_self);
%	temp -> st.st_self[temp -> st.st_fillp] = tempc;
%	if (rc == NULL)
%		rc = "";
%	temp = make_simple_string (rc);
%	return (temp);
)

;;;
;;;  (getenv variable) - akcl getenv
;;;
(defentry* getenv (object)
  "Syntax: (getenv object)\
Get the value of environment variable specified by object."
  (object "_getenv"))
