static char TMXIpc_c[] = "<%W%	%D% %T%>";
/*
 * 			Copyright 1993, 1994 by AT&T
 * 
 * 			 All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of AT&T not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * AT&T DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * AT&T BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * 
 * AT&T's dontation of this software does not imply a licence granted for
 * patents nor transfer of ownership of any patents which may inadvertently
 * be implemented in this code.
 * 
 */

#include <signal.h>
#include "XIpc.h"

/* RAD */
#if defined(__STDC__)			/* for write() */
#       define VOIDPTR_CAST (void *)
#	define SIZE_T_CAST (size_t)
#else
#       define VOIDPTR_CAST
#	define SIZE_T_CAST
#endif
/* RAD */

/*
 *  function prototypes in this file
 */
#include "C_P_args.h"

C_PROTOS_BEGIN_EXTERN

extern int
xipc_fdopen C_P_ARGS((XIpcClient *client));

extern void
xipc_fdclose C_P_ARGS((XIpcClient *client));

extern int
xipc_fdread C_P_ARGS((XIpcClient *client, char *ptr, int count));

extern int
xipc_fdwrite C_P_ARGS((XIpcClient *client, char *ptr, int count));

extern int
xipc_fdflush C_P_ARGS((XIpcClient *client));

extern void
XIpcPrintMessage C_P_ARGS((FILE *fp, char *str, XIpcMessage *message));

static void
XIpcDiscardSocketErrors C_P_ARGS((int sig));

extern void XIpcCatchErrors C_P_ARGS((int type, void (*user_routine)()));

extern XIpcMessage *
XIpcCopyMessage C_P_ARGS((XIpcMessage *message));

extern int
XIpcSetEnviron C_P_ARGS((char *new_env_variable));

extern int
XIpcUnsetEnviron C_P_ARGS((char *env_variable));

static int
bytes_to_value C_P_ARGS((unsigned char *ptr));

static void
value_to_bytes C_P_ARGS((unsigned int value, unsigned char *ptr));

extern XIpcMessage *
XIpcRecv C_P_ARGS((XIpcClient *client));

static int
XIpcQueueMessage C_P_ARGS((XIpcClient *client, XIpcMessage *message,
			   int send_count));

extern int
XIpcSendQueue C_P_ARGS((XIpcClient *client));

extern int
XIpcSend C_P_ARGS((XIpcClient *client, XIpcMessage *message));

extern int
XIpcSynchronize C_P_ARGS((XIpcClient *client, int type));

extern int
XIpcBufferEmpty C_P_ARGS((XIpcClient *client));

extern void
XIpcClose C_P_ARGS((XIpcClient *client));

extern int
XIpcIsClientActive C_P_ARGS((XIpcClient *client));

C_PROTOS_END_EXTERN

/*
 * buffer I/O routines for support of XIPC reads and writes.
 */
int
xipc_fdopen (client)
register XIpcClient *client;
{
    unsigned char *base = (unsigned char *) malloc (XIPCBUFSIZ + 4 + BUFSIZ);

    if (base == (unsigned char *) NULL)
	return FALSE;

    client -> recv_buffer = base;
    client -> recv_count = 0;

    client -> message_queue = (XIpcMessageLink *) NULL;

    client -> recv_fp.base_ptr = client -> recv_fp.base =
	base + XIPCBUFSIZ + 4;
    client -> recv_fp.base_byte_count = 0;
    client -> recv_fp.status = 0;

    client -> send_fp.base_ptr = client -> send_fp.base =
	(unsigned char *) NULL;
    client -> send_fp.base_byte_count = 0;
    client -> send_fp.status = 0;

    return TRUE;
}

void
xipc_fdclose (client)
register XIpcClient *client;
{
    XIpcChannel channel = client -> channel;

    if (client -> recv_buffer == (unsigned char *) NULL)
	return;

    if (client -> send_fp.status & XIPC_WRITE_BUFFERED)
	if (xipc_fdflush (client) == EOF)
	    fprintf (stderr,"Unable to flush IPC when closing the channel.\n");

    if (client -> message_queue)
	switch (XIpcSendQueue (client))
	{
	case XIPC_SENDQUEUE_DEAD:
	case XIPC_SENDQUEUE_BLOCKED:
	    fprintf (stderr,
		    "Cannot send queued messages when closing the channel.\n");
	}

    free (client -> recv_buffer);
    client -> recv_buffer = (unsigned char *) NULL;

    client -> recv_fp.base_ptr = client -> recv_fp.base =
	(unsigned char *) NULL;
    client -> recv_fp.base_byte_count = 0;
    client -> recv_fp.status = 0;

    if ((client -> send_fp.status & XIPC_WRITE_BUFFERED) &&
	client -> send_fp.base)
	free (client -> send_fp.base);

    client -> send_fp.base_ptr = client -> send_fp.base =
	(unsigned char *) NULL;
    client -> send_fp.base_byte_count = 0;
    client -> send_fp.status = 0;
}

int
xipc_fdread (client, ptr, count)
XIpcClient *client;
char *ptr;
register int count;
{
    extern int errno;
    XIpcChannel channel = client -> channel;
    register xipc_iobuf *iop = &client -> recv_fp;
    register int local_count;
    register int ndone = 0;

    errno = 0;
    if (iop -> status & XIPC_IOEOF)
	return (EOF);

    if (count < 1)
	return ndone;

    while (count)
    {
	if (iop -> base_byte_count >= count)
	{
	    local_count = count;
	    memcpy (ptr, iop -> base_ptr, local_count);
	    iop -> base_ptr += local_count;
	    iop -> base_byte_count -= local_count;

	    ndone += local_count;

	    return ndone;
	}

	if (iop -> base_byte_count && iop -> base_byte_count < count)
	{
	    local_count = iop -> base_byte_count;
	    memcpy (ptr, iop -> base_ptr, local_count);
	    iop -> base_ptr += local_count;
	    iop -> base_byte_count -= local_count;

	    ndone += local_count;

	    ptr += local_count;
	    count -= local_count;
	}

	if (iop -> base_byte_count == 0 && count)
	{
	    iop -> base_ptr = iop -> base;
	    if (count >= BUFSIZ) /* avoid the copy on big messages */
	    {
		local_count = read (channel, ptr, count);
		if (local_count > 0)
		{
		    ndone += local_count;

		    ptr += local_count;
		    count -= local_count;
		}
	    }
	    else
	    {
		local_count = iop -> base_byte_count =
		    read (channel, iop -> base, BUFSIZ);
	    }

	    if (local_count <= 0)
	    {
		if (local_count == 0)
		{
		    if (errno != EWOULDBLOCK && errno != EAGAIN)
			iop -> status |= XIPC_IOEOF;
		}
		else
		{
		    iop -> base_byte_count = 0;
		}
		return ndone;
	    }
	}
    }

    return ndone;
}

int
xipc_fdwrite (client, ptr, count)
XIpcClient *client;
char *ptr;
register int count;
{
    extern int errno;
    XIpcChannel channel = client -> channel;
    register xipc_iobuf *iop = &client -> send_fp;
    register int local_count;
    int ndone = 0;

    errno = 0;
    if (count < 1)
	return ndone;

    if ((iop -> status & XIPC_WRITE_BUFFERED) == 0)
	return write (channel, VOIDPTR_CAST ptr, SIZE_T_CAST count);  /* RAD */

    while (count)
    {
	if (iop -> base_byte_count >= count)
	{
	    local_count = count;
	    memcpy (iop -> base_ptr, ptr, local_count);
	    iop -> base_ptr += local_count;
	    iop -> base_byte_count -= local_count;

	    ndone += local_count;

	    return ndone;
	}

	if (iop -> base_byte_count && iop -> base_byte_count < count)
	{
	    local_count = iop -> base_byte_count;
	    memcpy (iop -> base_ptr, ptr, local_count);
	    iop -> base_ptr += local_count;
	    iop -> base_byte_count -= local_count;

	    ndone += local_count;

	    ptr += local_count;
	    count -= local_count;
	}

	if (iop -> base_byte_count == 0)
	{
	    errno = 0;
	    local_count = write (channel, VOIDPTR_CAST iop -> base, 
					SIZE_T_CAST XIPCBUFSIZ);  /* RAD */
	    if (local_count != XIPCBUFSIZ)
	    {
		if (local_count <= 0)
		    return ndone;

		iop -> base_byte_count = local_count;
		iop -> base_ptr = iop -> base + (XIPCBUFSIZ - local_count);
		memcpy (iop -> base, iop -> base + local_count, 
		       XIPCBUFSIZ - local_count);
		return ndone;
	    }

	    iop -> base_ptr = iop -> base;
	    iop -> base_byte_count = XIPCBUFSIZ;
	}
    }

    return ndone;
}

int
xipc_fdflush (client)
XIpcClient *client;
{
    extern int errno;
    XIpcChannel channel = client -> channel;
    register xipc_iobuf *iop = &client -> send_fp;
    register n;

    errno = 0;
    if (client -> recv_buffer == (unsigned char *) NULL)
	return (EOF);

    if ((client -> send_fp.status & XIPC_WRITE_BUFFERED) == 0)
	return 0;

    n = iop -> base_ptr - iop -> base;
    if (n && n == XIPCBUFSIZ - iop -> base_byte_count)
    {
	/* RAD */
	if (write (channel, VOIDPTR_CAST iop -> base, SIZE_T_CAST n) != n)
	    return (EOF);
    }
    iop -> base_ptr = iop -> base;
    iop -> base_byte_count = XIPCBUFSIZ;
    return 0;
}

/*
 * a print function for debugging
 */
void
XIpcPrintMessage (fp, str, message)
FILE *fp;
char *str;
XIpcMessage *message;
{
    if (message == (XIpcMessage *) NULL)
    {
	fprintf (fp, "%s: NULL message.\n", str);
	fflush (fp);
	return;
    }

    fprintf (fp, "%s: message -> type: %d\n",
	     str, message -> type);
    fprintf (fp, "%s: message -> client_id: %d\n",
	     str, message -> client_id);
    fprintf (fp, "%s: message -> length: %d\n",
	     str, message -> length);
    fprintf (fp, "%s: message -> buffer: ->%s<-\n",
	     str, message -> buffer);
    fflush (fp);
}

/*
 * functions to handle common X missing window errors
 */
/* RAD */
static void
XIpcDiscardSocketErrors (sig)
int sig;
{
    fprintf (stderr,
	     "signal %d: an ipc connected process probably died.\n", sig);
    return;
}

void
XIpcCatchErrors (type, user_routine)
int type;
void (*user_routine)();		/* RAD */
{
    struct sigaction act, oact;
 
    memset ((char *) &act, (char) 0, sizeof (struct sigaction));
#if defined(__STDC__)
#ifndef SA_RESTART
#define SA_RESTART	NULL
#endif
    act.sa_flags = SA_RESTART;
#endif

    if (type)
        act.sa_handler =
	    (user_routine) ? user_routine : XIpcDiscardSocketErrors;
    else
	act.sa_handler = SIG_DFL;
 
    sigaction (SIGPIPE, &act, &oact);

}

/*
 * utility function to allocate and copy a message.  This message maybe
 * deallocated with free().
 */
XIpcMessage *
XIpcCopyMessage (message)
XIpcMessage *message;
{
    XIpcMessage *new;
    int length = XIPC_HEADER_SIZE + message -> length;

    if ((new = (XIpcMessage *) malloc (length)) == NULL)
	return (XIpcMessage *) NULL;

    memcpy (new, (char *) message, length);

    return new;
}

/*
 * ultility functions for managing environmental variables.
 */
int
XIpcSetEnviron (new_env_variable)
char *new_env_variable;
{
    extern char **environ;
    static char **old_local_environ = (char **) NULL;
    char **local_environ;
    char **lptr;
    char **cptr;
    int length;
    char *ptr;
    char *env_variable;

    ptr = new_env_variable;
    while (*ptr && *ptr != '=')
	ptr++;
    if (*ptr != '=')
	return FALSE;

    length = 0;
    lptr = environ;
    while (*lptr++)
	length++;
    length += 2;
    local_environ = (char **) malloc (length * sizeof (char *));
    env_variable = (char *) malloc (strlen (new_env_variable) + 1);
    if (local_environ == NULL)
	return FALSE;
    if (env_variable == NULL)
    {
	free (local_environ);
	return FALSE;
    }

    cptr = environ;
    lptr = local_environ;
    strcpy (env_variable, new_env_variable);
    length = ptr - new_env_variable;
    while (*lptr = *cptr++)
    {
	if (strncmp (*lptr, env_variable, length) == 0)
	{
	    while (*lptr = *cptr++)
		lptr++;
	    break;
	}
	lptr++;
    }

    *lptr++ = env_variable;
    *lptr = (char *) NULL;
    environ = local_environ;
    if (old_local_environ)
	free (old_local_environ);
    old_local_environ = local_environ;

    return TRUE;
}

int
XIpcUnsetEnviron (env_variable)
char *env_variable;
{
    extern char **environ;
    char **cptr = environ;
    char **lptr = environ;

    while (*lptr = *cptr++)
    {
	if (strcmp (*lptr, env_variable) == 0 &&
	    *lptr[strlen (env_variable)] == '=')
	{
	    while (*lptr++ = *cptr++)
		;
	    return TRUE;
	}
	lptr++;
    }

    return FALSE;
}

/*
 * functions for handling machine independent short int's
 */
static int
bytes_to_value (ptr)
unsigned char *ptr;
{
    unsigned int value;

    value = *ptr++;
    value = (value << 8) | *ptr;
    return (value);
}

static void
value_to_bytes (value, ptr)
unsigned int value;
unsigned char *ptr;
{
    unsigned int low;

    low = value & 0377;
    *ptr++ = (value >> 8) & 0377;
    *ptr = low;
}

/*
 * function for reading incoming messages
 */
XIpcMessage *
XIpcRecv (client)		/* receive that avoids malloc */
register XIpcClient *client;
{
    extern int errno;
    XIpcMessage *message = (XIpcMessage *) client -> recv_buffer;
    XIpcMessageRaw *rmessage = (XIpcMessageRaw *) client -> recv_buffer;
    register int rc;

    errno = 0;
    if (client -> recv_buffer == (unsigned char *) NULL)
		return (XIpcMessage *) -1;

    if (client -> recv_count < XIPC_HEADER_SIZE)
    {
	rc = xipc_fdread (client, 
			  (char *) &client -> recv_buffer[client -> recv_count],
			  XIPC_HEADER_SIZE - client -> recv_count);

	if (rc > 0)
	    client -> recv_count += rc;

	if (client -> recv_count != XIPC_HEADER_SIZE)
	{
	    if (errno == EWOULDBLOCK)
	    {
		client -> recv_fp.status &= ~(XIPC_IOEOF);
		return (XIpcMessage *) NULL;
	    }
	    else
	    {
		return (XIpcMessage *) -1;
	    }
	}

	message -> type = bytes_to_value (rmessage -> type);
	message -> client_id = bytes_to_value (rmessage -> client_id);
	message -> length = bytes_to_value (rmessage -> length);
	if (message -> length >= XIPC_MAX_MESSAGE_SIZE)
	    message -> length = XIPC_MAX_MESSAGE_SIZE - 1;
    }

    rc = xipc_fdread (client, 
		      (char *) &client -> recv_buffer[client -> recv_count],
		      message -> length + XIPC_HEADER_SIZE -
		      client -> recv_count);

    if (rc !=  message -> length + XIPC_HEADER_SIZE - client -> recv_count)
    {
	if (rc > 0)
	    client -> recv_count += rc;

	if (errno == EWOULDBLOCK)
	{
	    client -> recv_fp.status &= ~(XIPC_IOEOF);
	    return (XIpcMessage *) NULL;
	}
	else
	{
	    return (XIpcMessage *) -1;
	}
    }

    client -> recv_count = 0;
    message -> buffer[message -> length] = '\0';
    return message;
}

/*
 * functions for sending messages
 */
static int
XIpcQueueMessage (client, message, send_count)
XIpcClient *client;
XIpcMessage *message;
int send_count;
{
    register char *ptr;
    char *mptr;
    int length;
    int linklen;
    register XIpcMessageLink *link;

    linklen = XIpcToNextWord (sizeof (XIpcMessageLink));
    length = (message -> length + XIPC_HEADER_SIZE - send_count);
    ptr = (char *) malloc (length + linklen);
    if (ptr == (char *) NULL)
    {
	fprintf (stderr,
		 "Memory exhausted, cannot queue a message.\n");
	return FALSE;
    }

    link = (XIpcMessageLink *) ptr;
    ptr += linklen;
    link -> message = ptr;
    mptr = (char *) message;
    memcpy (ptr, &mptr[send_count], length);
    link -> length = length;
    link -> next = client -> message_queue;
    client -> message_queue = link;

    return TRUE;
}

int
XIpcSendQueue (client)
register XIpcClient *client;
{
    extern int errno;
    register XIpcMessageLink *link;
    XIpcMessageLink *link_prev;
    int rc;

    errno = 0;
    if (client -> recv_buffer == (unsigned char *) NULL)
	return XIPC_SENDQUEUE_DEAD;

    if (xipc_fdflush (client) == EOF)
    {
	if (errno == EWOULDBLOCK)
	    return XIPC_SENDQUEUE_BLOCKED;
	return XIPC_SENDQUEUE_DEAD;
    }

    if (client -> message_queue == (XIpcMessageLink *) NULL)
	    return XIPC_SENDQUEUE_SENT;

    do
    {
	link = client -> message_queue;
	link_prev = (XIpcMessageLink *) NULL;
	while (link -> next)
	{
	    link_prev = link;
	    link = link -> next;
	}

	rc = xipc_fdwrite (client, link -> message, link -> length);
	if (rc <= 0)
	{
	    if (errno == EWOULDBLOCK)
		return XIPC_SENDQUEUE_BLOCKED;
	    return XIPC_SENDQUEUE_DEAD;
	}
	if (rc != link -> length)
	{
	    link -> message += rc;
	    link -> length -= rc;
	    return XIPC_SENDQUEUE_BLOCKED;
	}

	free (link);

    } while (link_prev &&
	     (link_prev -> next = NULL, TRUE)); /* side effect only */

    if (link == client -> message_queue)
	client -> message_queue = (XIpcMessageLink *) NULL;

    if (xipc_fdflush (client) == EOF)
    {
	if (errno == EWOULDBLOCK)
	    return XIPC_SENDQUEUE_BLOCKED;
	return XIPC_SENDQUEUE_DEAD;
    }

    return XIPC_SENDQUEUE_SENT;
}

int
XIpcSend (client, message)
register XIpcClient *client;
register XIpcMessage *message;
{
    extern int errno;
    XIpcMessageRaw *rmessage = (XIpcMessageRaw *) message;
    int length;
    int rc;

    errno = 0;
    value_to_bytes (message -> type, rmessage -> type);
    value_to_bytes (message -> client_id, rmessage -> client_id);
    if (message -> length >= XIPC_MAX_MESSAGE_SIZE)
	message -> length = XIPC_MAX_MESSAGE_SIZE - 1;
    length = message -> length + XIPC_HEADER_SIZE;
    value_to_bytes (message -> length, rmessage -> length);

    if (client -> recv_buffer == (unsigned char *) NULL)
	return FALSE;

    if (client -> message_queue)
	switch (XIpcSendQueue (client))
	{
	case XIPC_SENDQUEUE_DEAD:
	    if (client -> queue_update != NULL)
		(*client -> queue_update) (client, XIPC_SENDQUEUE_DEAD);
	    return FALSE;
	case XIPC_SENDQUEUE_BLOCKED:
	    XIpcQueueMessage (client, message, 0);
	    if (client -> queue_update != NULL)
		(*client -> queue_update) (client, XIPC_SENDQUEUE_BLOCKED);
	    return TRUE;
	case XIPC_SENDQUEUE_SENT:
	    if (client -> queue_update != NULL)
		(*client -> queue_update) (client, XIPC_SENDQUEUE_SENT);
	    break;
	}

    if ((client -> send_fp.status & XIPC_WRITE_BUFFERED) &&
	(length + (XIPCBUFSIZ - client -> send_fp.base_byte_count)) >
	XIPCBUFSIZ)
    {
	if (xipc_fdflush (client) == EOF)
	{
	    if (errno == EWOULDBLOCK)
	    {
		XIpcQueueMessage (client, message, 0);
		if (client -> queue_update != NULL)
		    (*client -> queue_update) (client, XIPC_SENDQUEUE_BLOCKED);
		return TRUE;
	    }
	    return FALSE;
	}
    }

    rc = xipc_fdwrite (client, (char *) message, length);
    if (rc <= 0)
    {
	if (errno == EWOULDBLOCK)
	{
	    XIpcQueueMessage (client, message, 0);
	    if (client -> queue_update != NULL)
		(*client -> queue_update) (client, XIPC_SENDQUEUE_BLOCKED);
	    return TRUE;
	}
	return FALSE;
    }
    if (rc != length)
    {
	XIpcQueueMessage (client, message, rc);
	if (client -> queue_update != NULL)
	    (*client -> queue_update) (client, XIPC_SENDQUEUE_BLOCKED);
	return TRUE;
    }

    return TRUE;
}

int
XIpcSynchronize (client, type)
register XIpcClient *client;
register int type;
{
    if (client -> recv_buffer == (unsigned char *) NULL)
	return FALSE;

    if (type == FALSE)
    {
	if ((client -> send_fp.status & XIPC_WRITE_BUFFERED) == 0)
	{
	    unsigned char *base = (unsigned char *) malloc (XIPCBUFSIZ);

	    if (base == (unsigned char *) NULL)
		return FALSE;

	    client -> send_fp.base_ptr = client -> send_fp.base = base;
	    client -> send_fp.base_byte_count = XIPCBUFSIZ;
	    client -> send_fp.status |= XIPC_WRITE_BUFFERED;
	}
	return TRUE;
    }

    if (xipc_fdflush (client) == EOF)
	return FALSE;

    free (client -> send_fp.base);
    client -> send_fp.base_ptr = client -> send_fp.base = 
	(unsigned char *) NULL;
    client -> send_fp.base_byte_count = 0;
    client -> send_fp.status &= ~(XIPC_WRITE_BUFFERED);

    return TRUE;
}

int
XIpcBufferEmpty (client)
register XIpcClient *client;
{
    if (client -> recv_buffer == (unsigned char *) NULL)
	return TRUE;

    return (client -> recv_fp.base_byte_count <= 0);
}

/*
 * function to close IPC clients
 */
void
XIpcClose (client)
register XIpcClient *client;
{
    register XIpcMessageLink *link;
    register XIpcMessageLink *next;

    xipc_fdclose (client);
    close (client -> channel);
    client -> channel = (XIpcChannel) NULL;

    link = client -> message_queue;
    while (link)
    {
	next = link -> next;
	free (link);
	link = next;
    }
    client -> message_queue = (XIpcMessageLink *) NULL;
    if (client -> cleanup)
    {
	(*client -> cleanup) (client);
	client -> cleanup = (void (*)()) NULL;
    }
    client -> data = (caddr_t) NULL;
    if (client -> malloc_data)
    {
	free (client -> malloc_data);
	client -> malloc_data = (caddr_t) NULL;
    }
}

int
XIpcIsClientActive (client)
register XIpcClient *client;
{
    if (client == (XIpcClient *) NULL)
	return FALSE;

    return (client -> recv_buffer != (unsigned char *) NULL);
}
