#include <stdio.h>
#include "XIpc.h"
#include "remote.h"

typedef struct RemoteClient
{
	unsigned short client_id;
	XIpcServer *server;
	struct RemoteClient *next;
} RemoteClient;

typedef struct RemoteClientServer
{
	RemoteClient *ClientList;
	double LoadAverage;
} RemoteClientServer;

#include "C_P_args.h"

C_PROTOS_BEGIN_EXTERN

extern void
main C_P_ARGS((int argc, char **argv));

static int
RemoteStartClients C_P_ARGS((int argc, char **argv));

static void
RemoteExec C_P_ARGS((XIpcServer *server, XIpcMessage *message));

static void
RemoteForward C_P_ARGS((XIpcServer *server, XIpcMessage *message));

static void
RemoteServerCleanupOnExit C_P_ARGS((XIpcServer *server, XIpcMessage *message,
				    int client_id));

static void
RemoteServerCleanup C_P_ARGS((XIpcClient *client));

static void
RemoteServerLogin C_P_ARGS((XIpcServer *server, XIpcMessage *message));

static void
RemoteLoad C_P_ARGS((XIpcServer *server, XIpcMessage *message));

C_PROTOS_END_EXTERN

/*
 * s - server for remote execution
 */
void
main (argc, argv)
int argc;
char **argv;
{
    extern char *getenv ();
    extern void RemoteServerLogin ();
    extern void RemoteExec ();
    extern void RemoteForward ();
    extern void RemoteLoad ();

    XIpcMessage *message;
    char *name = REMOTE_SERVER_ENV_NAME;
    XIpcServer *server;
    FILE *fp;

    /*
     * setup the server
     */
    if ((server = XIpcSetupServer ((char *) NULL, (caddr_t) NULL,
				   (caddr_t) NULL, name)) != NULL)
    {
	fprintf (stderr, "%s=%s; export %s\n",
		 name, getenv (name), name);
	if ((fp = fopen (REMOTE_ID_FILE, "w")) != NULL)
	{
	    fprintf (fp, "%s\n", getenv (name));
	    fclose (fp);
	}
    }
    else
    {
	fprintf (stderr, "%s: Could not initialize the server.\n",
		 argv[0]);
	exit (1);
    }

    /*
     * disable errors from dead clients
     */
    XIpcCatchErrors (TRUE, NULL);

    /*
     * start the remote servers - 1 per remote machine
     */
    if (RemoteStartClients (argc, argv) == FALSE)
	exit (1);

    /*
     * loop servicing requests
     */
    for (;;)
    {
	switch (XIpcServerMonitor (server, XIPC_MONITOR_MESSAGES, -1))
	{
	case XIPC_MONITOR_MESSAGES:
	    message = XIpcRecvFromClients (server);
	    if (message == (XIpcMessage *) NULL)
		continue;

	    switch (message -> type)
	    {
	    case REMOTE_SERVER_LOGIN:
		RemoteServerLogin (server, message);
		continue;
	    case REMOTE_EXEC:
		RemoteExec (server, message);
		continue;
	    case REMOTE_FORWARD:
		RemoteForward (server, message);
		continue;
	    case REMOTE_LOAD:
		RemoteLoad (server, message);
		continue;
	    default:
		XIpcServerProcessMessage (server, message);
	    }
	}
    }
}

/*
 * start the remote servers - one per remote machine.  This assumes
 * that the clients can all be started prior to any of them timing out.
 * This routine has to complete and login the remote servers before
 * the login timeout value is reached.
 *
 * Remote servers can also be started from any shell.  Starting the
 * remote servers from a shell avoids all of the timing headaches and
 * is a much more incremental method for managing the environement.
 *
 */
static int
RemoteStartClients (argc, argv)
int argc;
char **argv;
{
    FILE *pp;
    int len;
    int i;
    char *name = REMOTE_SERVER_ENV_NAME;

    /*
     * open a pipe to a shell to improve our rsh startup time
     */
    if ((pp = popen ("ksh 2>/dev/null", "w"))
	== NULL)
    {
	fprintf (stderr, "%s: cannot start processes.\n", argv[0]);
	return FALSE;
    }

    /*
     * read the list of machines and fire up a remote server for
     * each machine.
     */
    for (i = 1; i < argc; i++)
	fprintf (pp,
		 "rsh %s %s=$%s DISPLAY=$DISPLAY PATH=$PATH %s &\n",
		 argv[i], name, name, REMOTE_CLIENT_SERVER_RSH_START);

    pclose (pp);

    return TRUE;
}

/*
 * run a command remotely by sending a message to a waiting remote
 * server.  This routine hands out commands by selecting the processor
 * with the lowest reported load average.
 */
static void
RemoteExec (server, message)
XIpcServer *server;
XIpcMessage *message;
{
    static char buffer[XIPCBUFSIZ];
    XIpcMessage *lmessage = (XIpcMessage *) buffer;
    int len;
    static int client_id = 0;
    RemoteClient *rc = (RemoteClient *) malloc (sizeof (RemoteClient));
    double load_average = 10000.0;
    int winning_client_server = -1;
	
    for (client_id = 0; client_id < server -> max_clients; client_id++)
    {
	if (server -> clients[client_id].data !=
	    (caddr_t) REMOTE_SERVER_TOKEN)
	    continue;
		
	if (XIpcIsClientActive (&server -> clients[client_id]) ==
	    FALSE)
	    continue;

	if (((struct RemoteClientServer *)
	     server -> clients [client_id].malloc_data) -> LoadAverage <
	    load_average)
	{
	    winning_client_server = client_id;
	    load_average = ((struct RemoteClientServer *)
			    server -> clients [client_id].malloc_data) -> LoadAverage;
	}
    }
	
    if (winning_client_server == -1)
    {
		
	strcpy ((char *) lmessage -> buffer,
		"[ No remote servers available. ]\n");
	len = strlen ((char *) lmessage -> buffer);
	lmessage -> type = REMOTE_ECHO;
	lmessage -> length = len;
	XIpcSendToClient (server, message -> client_id,
			  lmessage);
		
	strcpy ((char *) lmessage -> buffer, "1");
	len = strlen ((char *) lmessage -> buffer);
	lmessage -> type = REMOTE_EXIT;
	lmessage -> length = len;
	XIpcSendToClient (server, message -> client_id,
			  lmessage);
		
	if (rc)
	    free (rc);
	return;
    }
	
    client_id = winning_client_server;
    if (rc)
    {
	rc -> client_id = message -> client_id;
	rc -> server = server;
		
	rc -> next = ((struct RemoteClientServer *)
		      server -> clients[client_id].malloc_data) -> ClientList;
	((struct RemoteClientServer *)
	 server -> clients[client_id].malloc_data) -> ClientList = rc;
	((struct RemoteClientServer *)
	 server -> clients [client_id].malloc_data) -> LoadAverage += 1.0;
    }
	
    message -> type = message -> client_id;
    XIpcSendToClient (server, client_id, message);
    client_id++;
    return;
}

/*
 * forward the output and return value from the remotely executed
 * command back to process that initiated the request.
 */
static void
RemoteForward (server, message)
XIpcServer *server;
XIpcMessage *message;
{
    extern void RemoteServerCleanupOnExit ();
    static char buffer[XIPCBUFSIZ];
    XIpcMessage *forward;
    int client_id;
    char *ptr;

    ptr = (char *) message -> buffer;
    while (*ptr != ';' && *ptr != '\0')
	ptr++;
    if (*ptr != ';')
	return;
    *ptr++ = '\0';
    client_id = atoi (message -> buffer);
    forward = (XIpcMessage *) buffer;
    switch (*ptr)
    {
    case '0':
	forward -> type = REMOTE_ECHO;
	break;
    case '1':
	forward -> type = REMOTE_EXIT;
	RemoteServerCleanupOnExit (server, message, client_id);
	break;
    default:
	return;
    }
    if (*(++ptr) != ';')
	return;
    ptr++;
    forward -> length = message -> length -
	(ptr - (char *) message -> buffer);
    bcopy (ptr, forward -> buffer, (int) forward -> length);
    XIpcSendToClient (server, client_id, forward);
}

static void
RemoteServerCleanupOnExit (server, message, client_id)
XIpcServer *server;
XIpcMessage *message;
int client_id;
{
    RemoteClient *prc, *rc;
    int cs_id = message -> client_id;

    if (XIpcIsClientActive (&server -> clients[cs_id]) &&
	server -> clients[cs_id].data == (caddr_t) REMOTE_SERVER_TOKEN)
    {
	prc = (RemoteClient *) NULL;
	rc = ((struct RemoteClientServer *)
	      server -> clients[cs_id].malloc_data) -> ClientList;
	while (rc)
	{
	    if (rc -> client_id = client_id)
	    {
		if (prc)
		    prc -> next = rc -> next;
		else
		    ((struct RemoteClientServer *)
	     server -> clients[cs_id].malloc_data) -> ClientList = rc -> next;
		free (rc);
		return;
	    }
	    rc = rc -> next;
	}
    }
}

static void
RemoteServerCleanup (client)
XIpcClient *client;
{
    RemoteClient *rc;
    RemoteClient *crc;
    static char buffer[XIPCBUFSIZ];
    XIpcMessage *message = (XIpcMessage *) buffer;
    int len;

    strcpy ((char *) message -> buffer, "[ The remote server died. ]\n");
    len = strlen ((char *) message -> buffer);
    rc = ((struct RemoteClientServer *) client -> malloc_data) -> ClientList;
    while (rc)
    {
	message -> type = REMOTE_ECHO;
	message -> length = len;
	XIpcSendToClient (rc -> server, rc -> client_id, message);
	rc = rc -> next;
    }

    strcpy ((char *) message -> buffer, "1");
    len = strlen ((char *) message -> buffer);
    rc = ((struct RemoteClientServer *) client -> malloc_data) -> ClientList;
    while (rc)
    {
	message -> type = REMOTE_EXIT;
	message -> length = len;
	XIpcSendToClient (rc -> server, rc -> client_id, message);
	crc = rc;
	rc = rc -> next;
	free (crc);
    }

    free (client -> malloc_data);
    client -> malloc_data = (caddr_t) NULL;
}

static void
RemoteServerLogin (server, message)
XIpcServer *server;
XIpcMessage *message;
{
    server -> clients[message -> client_id].malloc_data =
	(caddr_t) malloc (sizeof (RemoteClientServer));
    if (server -> clients[message -> client_id].malloc_data == NULL)
    {
	XIpcMessage ret;

	fprintf (stderr, "Server: cannot allocate remote server.\n");
	ret.type = REMOTE_EXIT;
	ret.length = 0;
	XIpcSendToClient (server, message -> client_id, &ret);
	return;
    }
    ((struct RemoteClientServer *)
     server -> clients [message -> client_id].malloc_data) -> LoadAverage = 100;
    server -> clients[message -> client_id].data =
	(caddr_t) REMOTE_SERVER_TOKEN;
    server -> clients[message -> client_id].cleanup = RemoteServerCleanup;
}

static void
RemoteLoad (server, message)
XIpcServer *server;
XIpcMessage *message;
{
    extern double atof ();

    if (server -> clients[message -> client_id].malloc_data != NULL)
    {
	((struct RemoteClientServer *)
 server -> clients [message -> client_id].malloc_data) -> LoadAverage = atof (message -> buffer);
    }
}
