/*
 * File:	bwin.cc
 * Purpose:	wxWindows GUI builder -- base window stuff
 * Author:	Julian Smart
 * Created:	1993
 * Updated:	
 * Copyright:	(c) 1993, AIAI, University of Edinburgh
 */

static const char sccsid[] = "%W% %G%";

#include "wx.h"
#include "wx_help.h"
#include <ctype.h>
#include <stdlib.h>

#include "wxbuild.h"
#include "namegen.h"
#include "bapp.h"
#include "bwin.h"
#include "bframe.h"
#include "btoolbar.h"
#include "bactions.h"

BuildWindowData *NewObject = NULL;

/*
 * BuildWindow class for storing window information (size, position, colour etc.).
 * This is for all derivatives from wxWindow, and probably more besides.
 *
 */

BuildWindowData::BuildWindowData(BuildWindowData *theParent)
{
  userWindow = NULL;
  windowObject = NULL; // Representation in editor
  buildParent = theParent;
  name = NULL;
  description = NULL;
  className = NULL;
  memberName = NULL;
  title = NULL;
  helpString = NULL;
  intValue = 0;
  stringValue = NULL;
  windowStyle = 0;
  x = 0;
  y = 0;
  width = 0;
  height = 0;
  windowFont = NULL;
  windowColour = NULL;
  windowType = 0;
  dontResize = FALSE;
  id = NewId();

  if (theParent)
    theParent->children.Append(this);
}

BuildWindowData::~BuildWindowData(void)
{
  if (MainFrame && windowObject)
  {
    windowObject->Select(FALSE);
    windowObject->Erase();
    MainFrame->canvas->RemoveObject(windowObject);
    delete windowObject;
    windowObject = NULL;
  }

  if (name) delete[] name;
  if (description) delete[] description;
  if (className) delete[] className;
  if (memberName) delete[] memberName; 
  if (title) delete[] title;
  if (helpString) delete[] helpString;

  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();
    delete child;
    node = children.First();
  }
  if (buildParent)
    buildParent->children.DeleteObject(this);

  node = buildActions.First();
  while (node)
  {
    BuildAction *action = (BuildAction *)node->Data();
    wxNode *next = node->Next();
    DeleteAction(action);
    node = next;
  }

  // Remove dangling pointers -- windows that rely on this window
  // for an action.
  node = actionWindows.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();
    // Have to delete the actions that refer to this window.
    wxNode *node1 = child->buildActions.First();
    while (node1)
    {
      BuildAction *action = (BuildAction *)node1->Data();
      wxNode *next1 = node1->Next();
      if (action->actionWindow == this)
        child->DeleteAction(action);
      node1 = next1;
    }
      
    node = node->Next();
  }
}

void BuildWindowData::MakeRealWindowChildren(void)
{
  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();
    child->MakeRealWindow();
    node = node->Next();
  }
}

 // Set userWindow to NULL in all children, since
// we're deleting this real window.
void BuildWindowData::NullUserWindows(void)
{
  userWindow = NULL;
  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();
    child->NullUserWindows();
    node = node->Next();
  }
}

void BuildWindowData::AddAction(BuildAction *action)
{
  buildActions.Append(action);
  if (action->actionWindow)
    action->actionWindow->actionWindows.Append(this);
}

void BuildWindowData::DeleteAction(BuildAction *action)
{
  buildActions.DeleteObject(action);
  if (action->actionWindow)
    action->actionWindow->actionWindows.DeleteObject(this);
  delete action;
}

// Find an action for this window, for a specific event.
// There may be zero or more actions for each type of event relevant
// to this window. This function is only relevant when there
// is only one action per event (e.g. a button, as opposed to a frame
// where there may be many command actions for the menu command event).
BuildAction *BuildWindowData::FindAction(WXTYPE eventType)
{
  wxNode *node = buildActions.First();
  while (node)
  {
    BuildAction *action = (BuildAction *)node->Data();
    if (action->eventType == eventType)
      return action;
    node = node->Next();
  }
  return NULL;
}

/*
Bool BuildWindowData::EditAttributes(void)
{
  return TRUE;
}
*/

Bool BuildWindowData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  expr->AssignAttributeValue("id", &id);
  RegisterId(id);
  
  expr->AssignAttributeValue("name", &name);
  expr->AssignAttributeValue("description", &description);
  expr->AssignAttributeValue("class", &className);
  expr->AssignAttributeValue("title", &title);
  expr->AssignAttributeValue("help_string", &helpString);

  expr->AssignAttributeValue("window_style", &windowStyle);
  expr->AssignAttributeValue("x", &x);
  expr->AssignAttributeValue("y", &y);
  expr->AssignAttributeValue("width", &width);
  expr->AssignAttributeValue("height", &height);
  return TRUE;
}

Bool BuildWindowData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  // Get the window position in case it's changed without being saved
  if (userWindow)
    userWindow->GetPosition(&x, &y);
    
  expr->AddAttributeValue("id", id);
  expr->AddAttributeValue("type", (long)windowType);
  if (buildParent)
    expr->AddAttributeValue("parent", buildParent->id);
  if (name)
    expr->AddAttributeValueString("name", name);
  if (description)
    expr->AddAttributeValueString("description", description);
  if (className)
    expr->AddAttributeValueString("class", className);
  if (title)
    expr->AddAttributeValueString("title", title);
  if (helpString)
    expr->AddAttributeValueString("help_string", helpString);

  expr->AddAttributeValue("window_style", (long)windowStyle);
  expr->AddAttributeValue("x", (long)x);
  expr->AddAttributeValue("y", (long)y);
  expr->AddAttributeValue("width", (long)width);
  expr->AddAttributeValue("height", (long)height);

  return TRUE;
}

Bool BuildWindowData::WriteRecursively(PrologDatabase *database)
{
  PrologExpr *expr = new PrologExpr("window");
  WritePrologAttributes(expr, database);
  WriteActions(expr, database);
  database->Append(expr);
  
  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();
    child->WriteRecursively(database);
    node = node->Next();
  }
  return TRUE;
}

void BuildWindowData::WriteClassImplementation(ostream& stream)
{
}

void BuildWindowData::WriteClassDeclaration(ostream& stream)
{
}

/*
 * Add window and children to canvas as box representations
 *
 */
void BuildWindowData::AddWindowObject(ObjectEditorCanvas *canvas, Bool recurse)
{
  windowObject = new WindowObject(10.0, 10.0);

  if ((wxSubType(windowType, wxTYPE_PANEL) && !wxSubType(windowType, wxTYPE_DIALOG_BOX)) ||
       wxSubType(windowType, wxTYPE_TEXT_WINDOW) ||
       wxSubType(windowType, wxTYPE_CANVAS))
    windowObject->SetDraggable(FALSE);

  windowObject->buildWindow = this;

  canvas->AddObject(windowObject);
  windowObject->Show(TRUE);
//  windowObject->Move(40.0, 40.0);
  windowObject->FormatText(name);
  PositionWindowObject();

  if (!recurse)
    return;

  wxNode *node = children.First();
  while (node)
  {
    BuildWindowData *child = (BuildWindowData *)node->Data();
    if (!child->windowObject)
      child->AddWindowObject(canvas, TRUE);
    node = node->Next();
  }
  if (windowType == wxTYPE_FRAME)
  {
    BuildFrameData *data = (BuildFrameData *)this;
    if (data->toolbar && !data->toolbar->windowObject)
      data->toolbar->AddWindowObject(canvas, TRUE);
  }
}

void BuildWindowData::PositionWindowObject(void)
{
}

/*
 * WindowObject
 *
 */
WindowObject::WindowObject(float w, float h):
 RectangleShape(w, h)
{
  buildWindow = NULL;
  __type = 999;
}

void WindowObject::OnLeftClick(float x, float y, int keys)
{
    if ((EditorPaletteFrame->palette->currentlySelected > -1) &&
        (EditorPaletteFrame->palette->currentlySelected != PALETTE_ARROW) && !(keys & KEY_SHIFT))
    {
      Select(TRUE);
      NewObject = NULL;
      switch (EditorPaletteFrame->palette->currentlySelected)
      {
        case PALETTE_FRAME:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_FRAME);
          break;
        case PALETTE_DIALOG_BOX:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_DIALOG);
          break;
        case PALETTE_PANEL:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_PANEL);
          break;
        case PALETTE_CANVAS:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_CANVAS);
          break;
        case PALETTE_TEXT_WINDOW:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_TEXT_WINDOW);
          break;
        case PALETTE_BUTTON:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_BUTTON);
          break;
        case PALETTE_MESSAGE:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_MESSAGE);
          break;
        case PALETTE_TEXT:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_TEXT);
          break;
        case PALETTE_MULTITEXT:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_MULTITEXT);
          break;
        case PALETTE_CHOICE:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_CHOICE);
          break;
        case PALETTE_CHECKBOX:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_CHECKBOX);
          break;
        case PALETTE_RADIOBOX:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_RADIOBOX);
          break;
        case PALETTE_LISTBOX:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_LISTBOX);
          break;
        case PALETTE_SLIDER:
          MainFrame->OnMenuCommand(OBJECT_EDITOR_NEW_SLIDER);
          break;
        default:
          break;
      }
      Select(FALSE);
      if (NewObject &&
          !(wxSubType(NewObject->windowType, wxTYPE_PANEL) ||
            wxSubType(NewObject->windowType, wxTYPE_CANVAS) ||
            wxSubType(NewObject->windowType, wxTYPE_TEXT_WINDOW) ||
            wxSubType(NewObject->windowType, wxTYPE_DIALOG_BOX)))
      {
        int cx, cy;
        NewObject->FindClientPosition((int)x, (int)y, &cx, &cy);
        NewObject->userWindow->SetSize(cx, cy, -1, -1);
        NewObject->PositionWindowObject();
        MainFrame->canvas->Redraw();
        NewObject = NULL;
      }
      else MainFrame->canvas->Redraw();
    }
    else
    {
      // Selection is a concept the library knows about
      if (Selected())
      {
        Select(FALSE);
        canvas->Redraw(); // Redraw because bits of objects will be are missing
        if (buildWindow)
          MainFrame->SetStatusText("");
      }
      else
      {
        Select(TRUE);
        if (buildWindow)
          MainFrame->SetStatusText(buildWindow->name);
      }
    }
}

void WindowObject::OnRightClick(float x, float y, int keys)
{
  if (buildWindow)
  {
    // Can't use buildWindow directly since 'this' gets destroyed
    // at DestroyRealWindow!! - mustn't access member variables.
    BuildWindowData *data = buildWindow;
    wxBeginBusyCursor();
    data->EditAttributes();
    data->DestroyRealWindow();
    data->MakeRealWindow();
    if (wxSubType(data->windowType, wxTYPE_DIALOG_BOX) ||
        wxSubType(data->windowType, wxTYPE_FRAME))
      buildApp.AssociateObjectWithEditor(data);
    wxEndBusyCursor();
  }
}

void WindowObject::OnDragLeft(Bool draw, float x, float y, int keys)
{
  if (!buildWindow->userWindow)
    return;

  if (!wxSubType(buildWindow->userWindow->__type, wxTYPE_DIALOG_BOX) &&
      buildWindow->buildParent &&
      (wxSubType(buildWindow->userWindow->__type, wxTYPE_PANEL) ||
       wxSubType(buildWindow->userWindow->__type, wxTYPE_TEXT_WINDOW) ||
       wxSubType(buildWindow->userWindow->__type, wxTYPE_CANVAS)))
  {
    buildWindow->buildParent->windowObject->OnDragLeft(draw, x, y, keys);
    return;
  }

  RectangleShape::OnDragLeft(draw, x, y, keys);
}

void WindowObject::OnBeginDragLeft(float x, float y, int keys)
{
  if (!wxSubType(buildWindow->userWindow->__type, wxTYPE_DIALOG_BOX) &&
      buildWindow->buildParent &&
      (wxSubType(buildWindow->userWindow->__type, wxTYPE_PANEL) ||
       wxSubType(buildWindow->userWindow->__type, wxTYPE_TEXT_WINDOW) ||
       wxSubType(buildWindow->userWindow->__type, wxTYPE_CANVAS)))
  {
    buildWindow->buildParent->windowObject->OnBeginDragLeft(x, y, keys);
    return;
  }

  RectangleShape::OnBeginDragLeft(x, y, keys);
}

void WindowObject::OnEndDragLeft(float x, float y, int keys)
{
  if (!wxSubType(buildWindow->userWindow->__type, wxTYPE_DIALOG_BOX) &&
      buildWindow->buildParent &&
      (wxSubType(buildWindow->userWindow->__type, wxTYPE_PANEL) ||
       wxSubType(buildWindow->userWindow->__type, wxTYPE_TEXT_WINDOW) ||
       wxSubType(buildWindow->userWindow->__type, wxTYPE_CANVAS)))
  {
    buildWindow->buildParent->windowObject->OnEndDragLeft(x, y, keys);
    return;
  }

  wxBeginBusyCursor();

  canvas->quick_edit_mode = TRUE;
  RectangleShape::OnEndDragLeft(x, y, keys);
  canvas->quick_edit_mode = FALSE;

  if (buildWindow)
  {
    // Move the actual window, and then reposition all simulated
    // subwindows as appropriate.
    int wX, wY;
    float oW, oH;
    GetBoundingBoxMax(&oW, &oH);
    wX = (int)(xpos - (oW/2.0));
    wY = (int)(ypos - (oH/2.0));
    int cX, cY;
    
    // Need to translate this screen position into a client
    // position: but first, subtract the difference between
    // simulated top-level window and real top-level window,
    // in case the user has moved the frame or dialog.
    // UNLESS, of course, this IS the top level window, in
    // which case we really want to move it.
    if (buildWindow != MainFrame->currentWindow)
    {
      int topLevelX, topLevelY, topSimX, topSimY;
      MainFrame->currentWindow->userWindow->GetPosition(&topLevelX, &topLevelY);
      topSimX = (int)(MainFrame->currentWindow->windowObject->xpos - (MainFrame->currentWindow->windowObject->width/2.0));
      topSimY = (int)(MainFrame->currentWindow->windowObject->ypos - (MainFrame->currentWindow->windowObject->height/2.0));
      wX -= (topSimX - topLevelX);
      wY -= (topSimY - topLevelY);
    }
    
    buildWindow->FindClientPosition(wX, wY, &cX, &cY);
    if (!buildWindow->CanSizeWidth())
      oW = -1;
    if (!buildWindow->CanSizeHeight())
      oH = -1;

    buildWindow->userWindow->SetSize(cX, cY, (int)oW, (int)oH);
    
    buildWindow->PositionWindowObject(); // Should position all subwindows
  }
  canvas->Redraw();

  wxEndBusyCursor();
}

void WindowObject::SetSize(float x, float y, Bool recursive)
{
  RectangleShape::SetSize(x, y, recursive);
/*
  buildWindow->width = (int)x;
  buildWindow->height = (int)y;
  buildWindow->dontResize = TRUE;
  int xp, yp;
  buildWindow->userWindow->GetPosition(&xp, &yp);
  if (!buildWindow->CanSizeWidth())
    x = -1;
  if (!buildWindow->CanSizeHeight())
    y = -1;
  if (!(x == -1 && y == -1))
    buildWindow->userWindow->SetSize(xp, yp, (int)x, (int)y);
  buildWindow->dontResize = FALSE;
*/
}

void WindowObject::OnBeginSize(float width, float height)
{
  canvas->quick_edit_mode = TRUE;
}

void WindowObject::OnEndSize(float width, float height)
{
  canvas->quick_edit_mode = FALSE;
  buildWindow->width = (int)width;
  buildWindow->height = (int)height;
  buildWindow->dontResize = TRUE;
  int xp, yp;
  buildWindow->userWindow->GetPosition(&xp, &yp);
  if (!buildWindow->CanSizeWidth())
    width = -1;
  if (!buildWindow->CanSizeHeight())
    height = -1;
  if (!(width == -1 && height == -1))
    buildWindow->userWindow->SetSize(xp, yp, (int)width, (int)height);
  buildWindow->dontResize = FALSE;
  buildWindow->PositionWindowObject(); // Should position all subwindows
  MakeModified();
  canvas->Redraw();
}

void WindowObject::Select(Bool select)
{
  RectangleShape::Select(select);
  if (select)
  {
    if (!BuildSelections.Member(buildWindow))
      BuildSelections.Append(buildWindow);
  }
  else
  {
    if (BuildSelections.Member(buildWindow))
      BuildSelections.DeleteObject(buildWindow);
  }
}

/*
 * Default action reading/writing
 *
 */

Bool BuildWindowData::WriteActions(PrologExpr *expr, PrologDatabase *database)
{
  char buf[100];
  int i = 1;
  wxNode *node = buildActions.First();
  while (node)
  {
    BuildAction *action = (BuildAction *)node->Data();
    sprintf(buf, "action%d", i);
    PrologExpr *listExpr = new PrologExpr(PrologList);
    action->WritePrologAttributes(listExpr, database);
    expr->AddAttributeValue(buf, listExpr);
    i ++;
    node = node->Next();
  }
  return TRUE;
}

Bool BuildWindowData::ReadActions(PrologExpr *expr, PrologDatabase *database)
{
  char buf[100];
  int i = 1;
  sprintf(buf, "action%d", i);
  PrologExpr *actionExpr = NULL;
  while (actionExpr = expr->AttributeValue(buf))
  {
    BuildAction *action = new BuildAction;
    action->ReadPrologAttributes(actionExpr, database);
    buildActions.Append(action);
    i ++;
    sprintf(buf, "action%d", i);
  }
  return TRUE;
}

