/*
 * File:	bitem.cc
 * Purpose:	wxWindows GUI builder - panel items
 * Author:	Julian Smart
 * Created:	1993
 * Updated:	
 * Copyright:	(c) 1993, AIAI, University of Edinburgh
 */

static const char sccsid[] = "%W% %G%";

#include "wx.h"
#include "wx_form.h"

#ifdef wx_msw
#include "dib.h"
#else
#include "wx_image.h"
#endif

#include <ctype.h>
#include <stdlib.h>

#include "wxbuild.h"
#include "bapp.h"
#include "namegen.h"
#include "bframe.h"
#include "bsubwin.h"
#include "bitem.h"
#include "bactions.h"

BuildItemData::BuildItemData(BuildPanelData *theParent):
  BuildWindowData(theParent)
{
  labelPosition = TRUE;
  autoSize = TRUE;
  functionName = NULL;
}

BuildItemData::~BuildItemData(void)
{
}

Bool BuildItemData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildWindowData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValue("label_position", &labelPosition);
  expr->AssignAttributeValue("auto_size", &autoSize);
  return TRUE;
}

Bool BuildItemData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildWindowData::WritePrologAttributes(expr, database);
  expr->AddAttributeValue("label_position", (long)labelPosition);
  expr->AddAttributeValue("auto_size", (long)autoSize);
  return TRUE;
}

void BuildItemData::FindScreenPosition(int *sx, int *sy)
{
  if (!userWindow)
  {
    *sx = x; *sy = y; return;
  }

  if (buildParent)
  {
    userWindow->GetPosition(sx, sy);
    buildParent->userWindow->ClientToScreen(sx, sy);
  }
}

void BuildItemData::FindClientPosition(int sx, int sy, int *cx, int *cy)
{
  if (buildParent)
  {
    *cx = sx;
    *cy = sy;
    buildParent->userWindow->ScreenToClient(cx, cy);
    x = *cx; y = *cy;
  }
}

void BuildItemData::PositionWindowObject(void)
{
  if (!userWindow || !windowObject)
    return;

  windowObject->Erase();

  int screenX, screenY;
  FindScreenPosition(&screenX, &screenY);

  // Normalize to take into account difference between
  // real and simulated top-level window positions.
  int topLevelX, topLevelY, topSimX, topSimY;
  MainFrame->currentWindow->userWindow->GetPosition(&topLevelX, &topLevelY);
  topSimX = (int)(MainFrame->currentWindow->windowObject->xpos - (MainFrame->currentWindow->windowObject->width/2.0));
  topSimY = (int)(MainFrame->currentWindow->windowObject->ypos - (MainFrame->currentWindow->windowObject->height/2.0));
  screenX -= (topLevelX - topSimX);
  screenY -= (topLevelY - topSimY);

  userWindow->GetSize(&width, &height);
  userWindow->GetPosition(&x, &y);
  if (!dontResize)
  {
    dontResize = TRUE;
    windowObject->SetSize(width, height, TRUE);
    dontResize = FALSE;
  }

  windowObject->xpos = (float)(screenX + (width/2.0));
  windowObject->ypos = (float)(screenY + (height/2.0));
  windowObject->Move(windowObject->GetX(), windowObject->GetY());
}

// Generation
void BuildItemData::WriteClassImplementation(ostream& stream)
{
}

void BuildItemData::WriteClassDeclaration(ostream& stream)
{
}

void BuildItemData::GenerateWindowStyleString(char *buf)
{
  buf[0] = 0;

  if (windowStyle & wxALWAYS_SB)
  {
    if (strlen(buf) > 0)
      strcat(buf, " | ");
    strcat(buf, "wxALWAYS_SB");
  }
  if (windowStyle & wxVSCROLL)
  {
    if (strlen(buf) > 0)
      strcat(buf, " | ");
    strcat(buf, "wxVSCROLL");
  }
  if (windowStyle & wxHSCROLL)
  {
    if (strlen(buf) > 0)
      strcat(buf, " | ");
    strcat(buf, "wxHSCROLL");
  }
  if (windowStyle & wxPROCESS_ENTER)
  {
    if (strlen(buf) > 0)
      strcat(buf, " | ");
    strcat(buf, "wxPROCESS_ENTER");
  }
  if (strlen(buf) == 0)
    strcat(buf, "0");
}

// Do any common pre-constructor stuff, such as setting the
// label orientation and fonts.
void BuildItemData::GenerateWindowInitialisationPre(ostream& stream, char *instanceName, char *indentation)
{
  Bool defaultIsHorizontal = ((BuildPanelData *)buildParent)->horizLabelPosition;
  if (labelPosition == wxVERTICAL)
  {
    if (defaultIsHorizontal)
      stream << indentation << "SetLabelPosition(wxVERTICAL);\n";
  }
  else if (labelPosition == wxHORIZONTAL)
  {
    if (!defaultIsHorizontal)
      stream << indentation << "SetLabelPosition(wxHORIZONTAL);\n";
  }
}

void BuildItemData::GenerateWindowInitialisationPost(ostream& stream, char *instanceName, char *indentation)
{
  // Reset to default.
  Bool defaultIsHorizontal = ((BuildPanelData *)buildParent)->horizLabelPosition;
  if (labelPosition == wxVERTICAL)
  {
    if (defaultIsHorizontal)
      stream << indentation << "SetLabelPosition(wxHORIZONTAL);\n";
  }
  else if (labelPosition == wxHORIZONTAL)
  {
    if (!defaultIsHorizontal)
      stream << indentation << "SetLabelPosition(wxVERTICAL);\n";
  }
}

Bool BuildItemData::EditAttributes(void)
{
  char nameBuf[200];
  sprintf(nameBuf, "Properties for %s", name);
  wxDialogBox *dialog = new wxDialogBox(NULL, nameBuf, TRUE, 10, 10);
  dialog->SetLabelFont(SmallLabelFont);
  dialog->SetButtonFont(SmallButtonFont);

  BuildForm *form = new BuildForm;

  if (!(wxSubType(windowType, wxTYPE_BUTTON) ||
        wxSubType(windowType, wxTYPE_MESSAGE)))
  {
    if (labelPosition == wxHORIZONTAL)
      labelPositionString = copystring("Horizontal");
    else if (labelPosition == wxVERTICAL)
      labelPositionString = copystring("Vertical");
    else
      labelPositionString = copystring("Default");
  }
  
  AddFormItems(form);

  form->AssociatePanel(dialog);
  form->dialog = dialog;
  AddDialogItems(dialog);

  dialog->Fit();
  dialog->Centre(wxBOTH);

//  form->RevertValues();

  dialog->Show(TRUE);

  if (!(wxSubType(windowType, wxTYPE_BUTTON) ||
        wxSubType(windowType, wxTYPE_MESSAGE)))
  {
    if (strcmp(labelPositionString, "Horizontal") == 0)
      labelPosition = wxHORIZONTAL;
    else if (strcmp(labelPositionString, "Vertical") == 0)
      labelPosition = wxVERTICAL;
    else
      labelPosition = wxDEFAULT_POSITION;
  }

  // An empty title/label string should be NULLed
  if (title && (strcmp(title, "") == 0))
  {
    delete[] title;
    title = NULL;
  }

  PositionWindowObject();
  return TRUE;
}

// Add class-specific items to form
void BuildItemData::AddFormItems(wxForm *form)
{
  SetHelpTopic("Procedures");
  form->Add(wxMakeFormString("Name", &name, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL,
               300));
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormString("Description", &description, wxFORM_MULTITEXT, NULL, NULL, wxVERTICAL,
               300, 100));
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormString("Label", &title, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL,
               300));
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormBool("Auto size", &autoSize, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL));
  if (!(wxSubType(windowType, wxTYPE_BUTTON) || wxSubType(windowType, wxTYPE_MESSAGE)))
  {
    form->Add(wxMakeFormString("Label position", &labelPositionString, wxFORM_CHOICE,
              new wxList(wxMakeConstraintStrings("Horizontal", "Vertical", "Default", NULL), NULL, wxVERTICAL),
    NULL, wxVERTICAL));
  }
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
void BuildItemData::AddDialogItems(wxDialogBox *dialog)
{
}

/*
 * Button
 *
 */
  
BuildButtonData::BuildButtonData(BuildPanelData *theParent):
  BuildItemData(theParent)
{
  x = -1;
  y = -1;
  width = -1;
  height = -1;
  name = copystring(GetNewObjectName("button"));
  memberName = copystring(name);
  className = copystring("wxButton");
  title = copystring("untitled");
  windowStyle = 0;
  windowType = wxTYPE_BUTTON;
  functionName = copystring(GetNewObjectName("ButtonProc"));

// bitmapButton = FALSE;
  bitmapFile = NULL;
  buttonBitmap = NULL;
}

BuildButtonData::~BuildButtonData(void)
{
  if (bitmapFile)
    delete[] bitmapFile;
  if (buttonBitmap)
    delete buttonBitmap;
}

Bool BuildButtonData::EditAttributes(void)
{
  return BuildItemData::EditAttributes();
}

// Add class-specific items to form
// Handles for us to associate the wxButton with client data
// AddFormItems
/*
static wxFormItem *BitmapFileItem = NULL;
static wxFormItem *BitmapLoadItem = NULL;
static wxFormItem *BitmapRemoveItem = NULL;

static void ButtonBitmapLoad(wxButton& but, wxCommandEvent& ev)
{
  BuildButtonData *data = (BuildButtonData *)but.GetClientData();
  char *s = wxFileSelector("Select a bitmap file", NULL, NULL, NULL, "*.*", 0, but.GetParent());
  if (s && FileExists(s))
  {
    wxColourMap *colourMap = NULL;
    wxBitmap *bitMap = wxLoadBitmap(s, &colourMap);
    if (colourMap)
      delete colourMap;
      
    if (bitMap)
    {
      if (data->bitmapFile)
        delete[] data->bitmapFile;
      data->bitmapFile = copystring(s);
      
      if (data->buttonBitmap)
        delete data->buttonBitmap;
      data->buttonBitmap = bitMap;
      BitmapFileItem->RevertValue();

      wxButton *theButton = (wxButton *)data->userWindow;
      if (theButton && data->buttonBitmap)
      {
        theButton->SetLabel(data->buttonBitmap);
      }
    }
    else
    {
      wxMessageBox("Could not load bitmap file!", "Error", wxOK, but.GetParent());
    }
  }
}

static void ButtonBitmapRemove(wxButton& but, wxCommandEvent& ev)
{
  BuildButtonData *data = (BuildButtonData *)but.GetClientData();
  if (data->bitmapFile)
    delete[] data->bitmapFile;
  data->bitmapFile = NULL;
  if (data->buttonBitmap)
    delete data->buttonBitmap;
  data->buttonBitmap = NULL;

  BitmapFileItem->RevertValue();

  wxButton *theButton = (wxButton *)data->userWindow;
  if (theButton)
  {
    theButton->SetLabel(data->title);
  }
}
*/

void BuildButtonData::AddFormItems(wxForm *form)
{
  BuildItemData::AddFormItems(form);
  SetHelpTopic("Creating a button item");
#if defined(wx_x) || FAFA_LIB
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormString("Bitmap name", &bitmapFile, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL, 300));
#endif
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
void BuildButtonData::AddDialogItems(wxDialogBox *dialog)
{
  // We take advantage of this function because at this point
  // the panel items have been created (by AssociatePanel)
//  BitmapLoadItem->GetPanelItem()->SetClientData((char *)this);
//  BitmapRemoveItem->GetPanelItem()->SetClientData((char *)this);
}

Bool BuildButtonData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::ReadPrologAttributes(expr, database);
//  expr->AssignAttributeValue("is_bitmap", &bitmapButton);
  expr->AssignAttributeValue("bitmap_file", &bitmapFile);
  return TRUE;
}

Bool BuildButtonData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::WritePrologAttributes(expr, database);
//  expr->AddAttributeValue("is_bitmap", (long)bitmapButton);
  expr->AddAttributeValueString("bitmap_file", bitmapFile);
  return TRUE;
}

Bool BuildButtonData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  wxPanel *panel = (wxPanel *)buildParent->userWindow;
  if (labelPosition == wxDEFAULT_POSITION)
    panel->SetLabelPosition((((BuildPanelData *)buildParent)->horizLabelPosition ? wxHORIZONTAL : wxVERTICAL));
  else
    panel->SetLabelPosition(labelPosition);

  if (autoSize)
  {
    width = -1;
    height = -1;
  }
  
  UserButton *button = NULL;

  char buf3[400];
#if defined(wx_x) || FAFA_LIB
  if (bitmapFile && (strlen(bitmapFile) > 0))
  {
#ifdef wx_msw
    strcpy(buf3, buildApp.projectDirDOS);
    strcat(buf3, "\\");
#endif
#ifdef wx_x
    strcpy(buf3, buildApp.projectDirUNIX);
    strcat(buf3, "/");
#endif
    strcat(buf3, bitmapFile);    
#ifdef wx_x
    strcat(buf3, ".xbm");
#else
    strcat(buf3, ".bmp");
#endif
    if (!buttonBitmap && FileExists(buf3))
    {
      wxColourMap *colourMap = NULL;
      buttonBitmap = wxLoadBitmap(buf3, &colourMap);
      if (colourMap)
        delete colourMap;
    }
    button = new UserButton((wxPanel *)buildParent->userWindow,
                 (wxFunction)UserButtonCallback, buttonBitmap, x, y,
                 width, height, windowStyle);
  }
  else
#endif  
    button = new UserButton((wxPanel *)buildParent->userWindow,
                 (wxFunction)UserButtonCallback, title, x, y,
                 width, height, windowStyle);


  userWindow = button;
  button->buildWindow = this;

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildButtonData::DestroyRealWindow(void)
{
  if (userWindow)
  {
    userWindow->GetPosition(&x, &y);
    userWindow->GetSize(&width, &height);
    delete userWindow;
    userWindow = NULL;
  }
  return TRUE;
}

void BuildButtonData::WriteClassDeclaration(ostream& stream)
{
  stream << "void " << functionName << "(wxButton& but, wxCommandEvent& event);\n\n";
}

// Generation
void BuildButtonData::WriteClassImplementation(ostream& stream)
{
  stream << "void " << functionName << "(wxButton& but, wxCommandEvent& event)\n";
  stream << "{\n";
  wxNode *node = buildActions.First();
  while (node)
  {
    BuildAction *action = (BuildAction *)node->Data();
    action->GenerateActionCPP(stream, "  ", this);
    node = node->Next();
  }
  stream << "}\n\n";
}

void BuildButtonData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << "wxButton" << "(" << parentName;
  stream << ", (wxFunction)" << (functionName ? functionName : "NULL") << ", ";

  if (bitmapFile && *bitmapFile)
  {
    // Generate name for the bitmap
    char buf1[50];
    strcpy(buf1, bitmapFile);
    strcat(buf1, "Bitmap");
    stream << buf1 << ", ";
  }
  else
    stream << SafeString(title) << ", ";
  stream << x << ", " << y << ", ";
  stream << width << ", " << height << ", ";
  
  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}

// Generate bitmap, icon etc. resource entries for RC file.
void BuildButtonData::GenerateResourceEntries(ostream &stream)
{
  char buf2[100];
  char buf3[300];
  if (bitmapFile && (strlen(bitmapFile) > 0))
  {
    strcpy(buf2, bitmapFile);
    strcat(buf2, ".bmp");
    stream << bitmapFile << " BITMAP " << buf2 << "\n";

    // Check if it exists
#ifdef wx_msw
    strcpy(buf3, buildApp.projectDirDOS);
    strcat(buf3, "\\");
#endif
#ifdef wx_x
    strcpy(buf3, buildApp.projectDirUNIX);
    strcat(buf3, "/");
#endif
    strcat(buf3, bitmapFile);    
    strcat(buf3, ".bmp");
    if (!FileExists(buf3))
    {
      Report("Warning: bitmap file "); Report(buf3); Report(" not found.\n");
    }
  }
}

/*
 * Checkbox
 *
 */
  
BuildCheckBoxData::BuildCheckBoxData(BuildPanelData *theParent):
  BuildItemData(theParent)
{
  x = -1;
  y = -1;
  width = -1;
  height = -1;
  name = copystring(GetNewObjectName("checkbox"));
  memberName = copystring(name);
  className = copystring("wxCheckBox");
  title = copystring("untitled");
  windowStyle = 0;
  windowType = wxTYPE_CHECK_BOX;
  functionName = copystring(GetNewObjectName("CheckBoxProc"));

  intValue = 0;
}

BuildCheckBoxData::~BuildCheckBoxData(void)
{
}

Bool BuildCheckBoxData::EditAttributes(void)
{
  return BuildItemData::EditAttributes();
}

// Add class-specific items to form
void BuildCheckBoxData::AddFormItems(wxForm *form)
{
  BuildItemData::AddFormItems(form);
  SetHelpTopic("Creating a checkbox item");
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
void BuildCheckBoxData::AddDialogItems(wxDialogBox *dialog)
{
}

Bool BuildCheckBoxData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValue("value", &intValue);
  return TRUE;
}

Bool BuildCheckBoxData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::WritePrologAttributes(expr, database);
  expr->AddAttributeValue("value", (long)intValue);
  return TRUE;
}

Bool BuildCheckBoxData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  wxPanel *panel = (wxPanel *)buildParent->userWindow;
  if (labelPosition == wxDEFAULT_POSITION)
    panel->SetLabelPosition((((BuildPanelData *)buildParent)->horizLabelPosition ? wxHORIZONTAL : wxVERTICAL));
  else
    panel->SetLabelPosition(labelPosition);

  if (autoSize)
  {
    width = -1;
    height = -1;
  }
  
  UserCheckBox *button = new UserCheckBox((wxPanel *)buildParent->userWindow,
                 (wxFunction)NULL, title, x, y,
                 width, height, windowStyle);
  userWindow = button;
  button->buildWindow = this;

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildCheckBoxData::DestroyRealWindow(void)
{
  if (userWindow)
  {
    userWindow->GetPosition(&x, &y);
    userWindow->GetSize(&width, &height);
    delete userWindow;
    userWindow = NULL;
  }
  return TRUE;
}

void BuildCheckBoxData::WriteClassDeclaration(ostream& stream)
{
  stream << "void " << functionName << "(wxCheckBox& check, wxCommandEvent& event);\n\n";
}

// Generation
void BuildCheckBoxData::WriteClassImplementation(ostream& stream)
{
  stream << "void " << functionName << "(wxCheckBox& but, wxCommandEvent& event)\n";
  stream << "{\n";
  stream << "}\n\n";
}

void BuildCheckBoxData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << "wxCheckBox" << "(" << parentName;
  stream << ", (wxFunction)" << (functionName ? functionName : "NULL") << ", ";
  stream << SafeString(title) << ", " << x << ", " << y << ", ";
  stream << width << ", " << height << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}

/*
 * Message
 *
 */
  
BuildMessageData::BuildMessageData(BuildPanelData *theParent):
  BuildItemData(theParent)
{
  x = -1;
  y = -1;
  width = -1;
  height = -1;
  name = copystring(GetNewObjectName("message"));
  memberName = copystring(name);
  className = copystring("wxMessage");
  title = copystring("untitled");
  windowStyle = 0;
  windowType = wxTYPE_MESSAGE;
}

BuildMessageData::~BuildMessageData(void)
{
}

Bool BuildMessageData::EditAttributes(void)
{
  return BuildItemData::EditAttributes();
}

// Add class-specific items to form
void BuildMessageData::AddFormItems(wxForm *form)
{
  BuildItemData::AddFormItems(form);
  SetHelpTopic("Creating a message item");
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
void BuildMessageData::AddDialogItems(wxDialogBox *dialog)
{
}

Bool BuildMessageData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::ReadPrologAttributes(expr, database);
  return TRUE;
}

Bool BuildMessageData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::WritePrologAttributes(expr, database);
  return TRUE;
}

Bool BuildMessageData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  wxPanel *panel = (wxPanel *)buildParent->userWindow;
  if (labelPosition == wxDEFAULT_POSITION)
    panel->SetLabelPosition((((BuildPanelData *)buildParent)->horizLabelPosition ? wxHORIZONTAL : wxVERTICAL));
  else
    panel->SetLabelPosition(labelPosition);

  if (autoSize)
  {
    width = -1;
    height = -1;
  }
  
  UserMessage *message = new UserMessage((wxPanel *)buildParent->userWindow,
                 title, x, y,
                 windowStyle);
  userWindow = message;
  message->buildWindow = this;

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildMessageData::DestroyRealWindow(void)
{
  if (userWindow)
  {
    userWindow->GetPosition(&x, &y);
    userWindow->GetSize(&width, &height);
    delete userWindow;
    userWindow = NULL;
  }
  return TRUE;
}

void BuildMessageData::WriteClassDeclaration(ostream& stream)
{
}

// Generation
void BuildMessageData::WriteClassImplementation(ostream& stream)
{
}

void BuildMessageData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << "wxMessage" << "(" << parentName;
  stream << ", " << SafeString(title) << ", " << x << ", " << y << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}

/*
 * Text item
 *
 */
 
BuildTextData::BuildTextData(BuildPanelData *theParent):
  BuildItemData(theParent)
{
  x = -1;
  y = -1;
  width = -1;
  height = -1;
  name = copystring(GetNewObjectName("text"));
  memberName = copystring(name);
  className = copystring("wxText");
  title = NULL;
  windowStyle = 0;
  windowType = wxTYPE_TEXT;
  stringValue = NULL;
  functionName = copystring(GetNewObjectName("TextProc"));
}

BuildTextData::~BuildTextData(void)
{
}

Bool BuildTextData::EditAttributes(void)
{
  return BuildItemData::EditAttributes();
}

// Add class-specific items to form
void BuildTextData::AddFormItems(wxForm *form)
{
  BuildItemData::AddFormItems(form);
  SetHelpTopic("Creating a text item");
  form->Add(wxMakeFormNewLine());
  form->Add(wxMakeFormString("Value", &stringValue, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL,
               300));
}

// Add class-specific items to dialog, since forms can't
// cope with everything.

void BuildTextData::AddDialogItems(wxDialogBox *dialog)
{
}

Bool BuildTextData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValue("value", &stringValue);
  return TRUE;
}

Bool BuildTextData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::WritePrologAttributes(expr, database);
  expr->AddAttributeValueString("value", stringValue);
  return TRUE;
}

Bool BuildTextData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  wxPanel *panel = (wxPanel *)buildParent->userWindow;
  if (labelPosition == wxDEFAULT_POSITION)
    panel->SetLabelPosition((((BuildPanelData *)buildParent)->horizLabelPosition  ? wxHORIZONTAL : wxVERTICAL));
  else
    panel->SetLabelPosition(labelPosition);

  if (autoSize)
  {
    width = -1;
    height = -1;
  }

  UserText *text = new UserText((wxPanel *)buildParent->userWindow, (wxFunction)NULL,
                 title, stringValue, x, y,
                 width, height, windowStyle);
  userWindow = text;
  text->buildWindow = this;

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildTextData::DestroyRealWindow(void)
{
  if (userWindow)
  {
    userWindow->GetPosition(&x, &y);
    userWindow->GetSize(&width, &height);
    delete userWindow;
    userWindow = NULL;
  }
  return TRUE;
}

void BuildTextData::WriteClassDeclaration(ostream& stream)
{
  stream << "void " << functionName << "(wxText& text, wxCommandEvent& event);\n\n";
}

// Generation
void BuildTextData::WriteClassImplementation(ostream& stream)
{
  stream << "void " << functionName << "(wxText& text, wxCommandEvent& event)\n";
  stream << "{\n";
  stream << "}\n\n";
}

void BuildTextData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << "wxText" << "(" << parentName;
  stream << ", (wxFunction)" << (functionName ? functionName : "NULL") << ", ";
  stream << SafeString(title) << ", ";
  stream << SafeString(stringValue) << ", ";
  stream << x << ", " << y << ", ";
  stream << width << ", " << height << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}

/*
 * MultiText item
 *
 */
 
BuildMultiTextData::BuildMultiTextData(BuildPanelData *theParent):
  BuildTextData(theParent)
{
  x = -1;
  y = -1;
  width = -1;
  height = -1;
  name = copystring(GetNewObjectName("multitext"));
  memberName = copystring(name);
  className = copystring("wxMultiText");
  title = NULL;
  windowStyle = 0;
  windowType = wxTYPE_MULTI_TEXT;
  stringValue = NULL;
  functionName = copystring(GetNewObjectName("MultiTextProc"));
}

BuildMultiTextData::~BuildMultiTextData(void)
{
}

void BuildMultiTextData::AddFormItems(wxForm *form)
{
  BuildTextData::AddFormItems(form);
  SetHelpTopic("Create a multi-line text item");
}

Bool BuildMultiTextData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  wxPanel *panel = (wxPanel *)buildParent->userWindow;
  if (labelPosition == wxDEFAULT_POSITION)
    panel->SetLabelPosition((((BuildPanelData *)buildParent)->horizLabelPosition  ? wxHORIZONTAL : wxVERTICAL));
  else
    panel->SetLabelPosition(labelPosition);

  if (autoSize)
  {
    width = -1;
    height = -1;
  }

  UserMultiText *text = new UserMultiText((wxPanel *)buildParent->userWindow, (wxFunction)NULL,
                 title, stringValue, x, y,
                 width, height, windowStyle);
  userWindow = text;
  text->buildWindow = this;

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

void BuildMultiTextData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << "wxMultiText" << "(" << parentName;
  stream << ", (wxFunction)" << (functionName ? functionName : "NULL") << ", ";
  stream << SafeString(title) << ", ";
  stream << SafeString(stringValue) << ", ";
  stream << x << ", " << y << ", ";
  stream << width << ", " << height << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}

/*
 * List box
 *
 */
 
BuildListBoxData::BuildListBoxData(BuildPanelData *theParent):
  BuildItemData(theParent)
{
  x = -1;
  y = -1;
  width = 200;
  height = 100;
  name = copystring(GetNewObjectName("listbox"));
  memberName = copystring(name);
  className = copystring("wxListBox");
  title = NULL;
  windowStyle = 0;
  windowType = wxTYPE_LIST_BOX;

  multipleSel = FALSE;
  scrollAlways = FALSE;
  functionName = copystring(GetNewObjectName("ListBoxProc"));
  tmpStringArray = NULL;
}

BuildListBoxData::~BuildListBoxData(void)
{
}

Bool BuildListBoxData::EditAttributes(void)
{
  return BuildItemData::EditAttributes();
}

// Add class-specific items to form
void BuildListBoxData::AddFormItems(wxForm *form)
{
  BuildItemData::AddFormItems(form);
  SetHelpTopic("Creating a listbox item");
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
static void ListBoxStringsCallback(wxListBox& lbox, wxCommandEvent& event)
{
}

static void ListBoxAddCallback(wxButton& but, wxCommandEvent& event)
{
  BuildListBoxData *data = (BuildListBoxData *)but.GetClientData();
  char *s = (data->currentTextBox->GetValue());
  if (s && (strlen(s) > 0))
  {
    wxNode *node = data->strings.Add(s);
    data->currentStringsBox->Append(s, (char *)node);
    data->currentTextBox->SetValue("");
  }
}

static void ListBoxDeleteCallback(wxButton& but, wxCommandEvent& event)
{
  BuildListBoxData *data = (BuildListBoxData *)but.GetClientData();
  int sel = data->currentStringsBox->GetSelection();
  if (sel > -1)
  {
    wxNode *node = (wxNode *)data->currentStringsBox->GetClientData(sel);
    delete[] (char *)node->Data();
    delete node;
    data->currentStringsBox->Delete(sel);
  }
}

void BuildListBoxData::AddDialogItems(wxDialogBox *dialog)
{
  dialog->NewLine();
  dialog->SetLabelPosition(wxVERTICAL);
  currentStringsBox = new wxListBox(dialog, (wxFunction)ListBoxStringsCallback,
                           "Values", wxSINGLE,-1, -1, 300, 100);
  currentStringsBox->SetClientData((char *)this);
  wxNode *node = strings.First();
  while (node)
  {
    char *s = (char *)node->Data();
    currentStringsBox->Append(s, (char *)node);
    node = node->Next();
  }
  dialog->NewLine();
  dialog->SetLabelPosition(wxHORIZONTAL);
  currentTextBox = new wxText(dialog, (wxFunction)NULL, "Value", "", -1, -1, 200);
//  dialog->NewLine();
  wxButton *addBut = new wxButton(dialog, (wxFunction)ListBoxAddCallback, "Add");
  wxButton *deleteBut = new wxButton(dialog, (wxFunction)ListBoxDeleteCallback, "Delete");
  addBut->SetClientData((char *)this);
  deleteBut->SetClientData((char *)this);
}

Bool BuildListBoxData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValue("multiple_sel", &multipleSel);
  expr->AssignAttributeValue("scroll_always", &scrollAlways);
  expr->AssignAttributeValueStringList("values", &strings);
  return TRUE;
}

Bool BuildListBoxData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::WritePrologAttributes(expr, database);
  expr->AddAttributeValue("multiple_sel", (long)multipleSel);
  expr->AddAttributeValue("scroll_always", (long)scrollAlways);
  expr->AddAttributeValueStringList("values", &strings);
  return TRUE;
}

Bool BuildListBoxData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  wxPanel *panel = (wxPanel *)buildParent->userWindow;
  if (labelPosition == wxDEFAULT_POSITION)
    panel->SetLabelPosition((((BuildPanelData *)buildParent)->horizLabelPosition ? wxHORIZONTAL : wxVERTICAL));
  else
    panel->SetLabelPosition(labelPosition);

  if (autoSize)
  {
    width = -1;
    height = -1;
  }

  windowStyle = 0;
  if (scrollAlways)
    windowStyle |= wxALWAYS_SB;

  char **theStrings = new char*[strings.Number()];
  for (int i = 0; i < strings.Number(); i++)
    theStrings[i] = (char *)(strings.Nth(i))->Data();

  UserListBox *listbox = new UserListBox((wxPanel *)buildParent->userWindow,
                 (wxFunction)NULL, title, (multipleSel ? wxMULTIPLE : wxSINGLE), x, y,
                 width, height, strings.Number(), theStrings, windowStyle);

  delete[] theStrings;

  userWindow = listbox;
  listbox->buildWindow = this;

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildListBoxData::DestroyRealWindow(void)
{
  if (userWindow)
  {
    userWindow->GetPosition(&x, &y);
    userWindow->GetSize(&width, &height);
    delete userWindow;
    userWindow = NULL;
  }
  return TRUE;
}

void BuildListBoxData::WriteClassDeclaration(ostream& stream)
{
  stream << "void " << functionName << "(wxListBox& lbox, wxCommandEvent& event);\n\n";
}

// Generation
void BuildListBoxData::WriteClassImplementation(ostream& stream)
{
  stream << "void " << functionName << "(wxListBox& lbox, wxCommandEvent& event)\n";
  stream << "{\n";
  stream << "}\n\n";
}

void BuildListBoxData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << "wxListBox" << "(" << parentName;
  stream << ", " << "(wxFunction)" << (functionName ? functionName : "NULL") << ", " << SafeString(title) << ", ";
  stream << (multipleSel ? "wxMULTIPLE" : "wxSINGLE") << ", " << x << ", " << y << ", ";
  stream << width << ", " << height << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << strings.Number() << ", " << ((strings.Number() == 0) ? "NULL" : tmpStringArray)
         << ", " << buf << ", " << "\"" << name << "\")";
}

void BuildListBoxData::GenerateWindowInitialisationPre(ostream& stream, char *instanceName, char *indentation)
{
  BuildItemData::GenerateWindowInitialisationPre(stream, instanceName, indentation);
  if (strings.Number() != 0)
  {
    if (tmpStringArray) delete[] tmpStringArray;
    tmpStringArray = GetNewObjectName("stringArray");
    stream << indentation << "char *" << tmpStringArray << "[] = { ";
    wxNode *node = strings.First();
    while (node)
    {
      char *s = (char *)node->Data();
      stream << "\"" << s << "\"";
      node = node->Next();
      if (node) stream << ", "; else stream << " ";
    }
    stream << "};\n";
  }
}

/*
 * Choice
 *
 */

BuildChoiceData::BuildChoiceData(BuildPanelData *theParent):
  BuildItemData(theParent)
{
  x = -1;
  y = -1;
  width = 200;
  height = 100;
  name = copystring(GetNewObjectName("choice"));
  memberName = copystring(name);
  className = copystring("wxChoice");
  title = NULL;
  windowStyle = 0;
  windowType = wxTYPE_CHOICE;
  functionName = copystring(GetNewObjectName("ChoiceProc"));
  tmpStringArray = NULL;
}

BuildChoiceData::~BuildChoiceData(void)
{
}

Bool BuildChoiceData::EditAttributes(void)
{
  return BuildItemData::EditAttributes();
}

// Add class-specific items to form
void BuildChoiceData::AddFormItems(wxForm *form)
{
  BuildItemData::AddFormItems(form);
  SetHelpTopic("Creating a choice item");
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
static void ChoiceStringsCallback(wxListBox& lbox, wxCommandEvent& event)
{
}

static void ChoiceAddCallback(wxButton& but, wxCommandEvent& event)
{
  BuildChoiceData *data = (BuildChoiceData *)but.GetClientData();
  char *s = (data->currentTextBox->GetValue());
  if (s && (strlen(s) > 0))
  {
    wxNode *node = data->strings.Add(s);
    data->currentStringsBox->Append(s, (char *)node);
    data->currentTextBox->SetValue("");
  }
}

static void ChoiceDeleteCallback(wxButton& but, wxCommandEvent& event)
{
  BuildChoiceData *data = (BuildChoiceData *)but.GetClientData();
  int sel = data->currentStringsBox->GetSelection();
  if (sel > -1)
  {
    wxNode *node = (wxNode *)data->currentStringsBox->GetClientData(sel);
    delete[] (char *)node->Data();
    delete node;
    data->currentStringsBox->Delete(sel);
  }
}

void BuildChoiceData::AddDialogItems(wxDialogBox *dialog)
{
  dialog->NewLine();
  dialog->SetLabelPosition(wxVERTICAL);
  currentStringsBox = new wxListBox(dialog, (wxFunction)ChoiceStringsCallback,
                           "Values", wxSINGLE,-1, -1, 300, 100);
  wxNode *node = strings.First();
  while (node)
  {
    char *s = (char *)node->Data();
    currentStringsBox->Append(s, (char *)node);
    node = node->Next();
  }
  currentStringsBox->SetClientData((char *)this);
  dialog->NewLine();
  dialog->SetLabelPosition(wxHORIZONTAL);
  currentTextBox = new wxText(dialog, (wxFunction)NULL, "Value", "", -1, -1, 200);
//  dialog->NewLine();
  wxButton *addBut = new wxButton(dialog, (wxFunction)ChoiceAddCallback, "Add");
  wxButton *deleteBut = new wxButton(dialog, (wxFunction)ChoiceDeleteCallback, "Delete");
  addBut->SetClientData((char *)this);
  deleteBut->SetClientData((char *)this);
}

Bool BuildChoiceData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValueStringList("values", &strings);
  return TRUE;
}

Bool BuildChoiceData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::WritePrologAttributes(expr, database);
  expr->AddAttributeValueStringList("values", &strings);
  return TRUE;
}

Bool BuildChoiceData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  wxPanel *panel = (wxPanel *)buildParent->userWindow;
  if (labelPosition == wxDEFAULT_POSITION)
    panel->SetLabelPosition((((BuildPanelData *)buildParent)->horizLabelPosition ? wxHORIZONTAL : wxVERTICAL));
  else
    panel->SetLabelPosition(labelPosition);

  if (autoSize)
  {
    width = -1;
    height = -1;
  }

  windowStyle = 0;

  char **theStrings = new char*[strings.Number()];
  for (int i = 0; i < strings.Number(); i++)
    theStrings[i] = (char *)(strings.Nth(i))->Data();

  UserChoice *choice = new UserChoice((wxPanel *)buildParent->userWindow,
                 (wxFunction)NULL, title, x, y,
                 width, -1, strings.Number(), theStrings, windowStyle);

  delete[] theStrings;

  userWindow = choice;
  choice->buildWindow = this;

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildChoiceData::DestroyRealWindow(void)
{
  if (userWindow)
  {
    userWindow->GetPosition(&x, &y);
    userWindow->GetSize(&width, &height);
    delete userWindow;
    userWindow = NULL;
  }
  return TRUE;
}

void BuildChoiceData::WriteClassDeclaration(ostream& stream)
{
  stream << "void " << functionName << "(wxChoice& choice, wxCommandEvent& event);\n\n";
}

// Generation
void BuildChoiceData::WriteClassImplementation(ostream& stream)
{
  stream << "void " << functionName << "(wxChoice& choice, wxCommandEvent& event)\n";
  stream << "{\n";
  stream << "}\n\n";
}

void BuildChoiceData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << "wxChoice" << "(" << parentName;
  stream << ", " << "(wxFunction)" << (functionName ? functionName : "NULL") << ", " << SafeString(title) << ", " << x << ", " << y << ", ";
  stream << "-1" << ", " << "-1" << ", ";

  stream << strings.Number() << ", " << ((strings.Number() == 0) ? "NULL" : tmpStringArray) << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}

void BuildChoiceData::GenerateWindowInitialisationPre(ostream& stream, char *instanceName, char *indentation)
{
  BuildItemData::GenerateWindowInitialisationPre(stream, instanceName, indentation);
  if (strings.Number() != 0)
  {
    if (tmpStringArray) delete[] tmpStringArray;
    tmpStringArray = GetNewObjectName("stringArray");
    stream << indentation << "char *" << tmpStringArray << "[] = { ";
    wxNode *node = strings.First();
    while (node)
    {
      char *s = (char *)node->Data();
      stream << "\"" << s << "\"";
      node = node->Next();
      if (node) stream << ", "; else stream << " ";
    }
    stream << "};\n";
  }
}

/*
 * RadioBox
 *
 */

BuildRadioBoxData::BuildRadioBoxData(BuildPanelData *theParent):
  BuildItemData(theParent)
{
  x = -1;
  y = -1;
  width = 200;
  height = 100;
  name = copystring(GetNewObjectName("radiobox"));
  memberName = copystring(name);
  className = copystring("wxRadioBox");
  title = copystring("untitled");
  windowStyle = 0;
  windowType = wxTYPE_RADIO_BOX;
  functionName = copystring(GetNewObjectName("RadioBoxProc"));
  noRowsCols = 2;
  tmpStringArray = NULL;
}

BuildRadioBoxData::~BuildRadioBoxData(void)
{
}

Bool BuildRadioBoxData::EditAttributes(void)
{
  return BuildItemData::EditAttributes();
}

// Add class-specific items to form
void BuildRadioBoxData::AddFormItems(wxForm *form)
{
  BuildItemData::AddFormItems(form);
  form->Add(wxMakeFormShort("Rows/cols", &noRowsCols, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL,
               100));
  
  SetHelpTopic("Creating a radiobox item");
}

// Add class-specific items to dialog, since forms can't
// cope with everything.
static void RadioBoxStringsCallback(wxListBox& lbox, wxCommandEvent& event)
{
}

static void RadioBoxAddCallback(wxButton& but, wxCommandEvent& event)
{
  BuildRadioBoxData *data = (BuildRadioBoxData *)but.GetClientData();
  char *s = (data->currentTextBox->GetValue());
  if (s && (strlen(s) > 0))
  {
    wxNode *node = data->strings.Add(s);
    data->currentStringsBox->Append(s, (char *)node);
    data->currentTextBox->SetValue("");
  }
}

static void RadioBoxDeleteCallback(wxButton& but, wxCommandEvent& event)
{
  BuildRadioBoxData *data = (BuildRadioBoxData *)but.GetClientData();
  int sel = data->currentStringsBox->GetSelection();
  if (sel > -1)
  {
    wxNode *node = (wxNode *)data->currentStringsBox->GetClientData(sel);
    delete[] (char *)node->Data();
    delete node;
    data->currentStringsBox->Delete(sel);
  }
}

void BuildRadioBoxData::AddDialogItems(wxDialogBox *dialog)
{
  dialog->NewLine();
  dialog->SetLabelPosition(wxVERTICAL);
  currentStringsBox = new wxListBox(dialog, (wxFunction)RadioBoxStringsCallback,
                           "Values", wxSINGLE,-1, -1, 300, 100);
  currentStringsBox->SetClientData((char *)this);
  wxNode *node = strings.First();
  while (node)
  {
    char *s = (char *)node->Data();
    currentStringsBox->Append(s, (char *)node);
    node = node->Next();
  }
  dialog->NewLine();
  dialog->SetLabelPosition(wxHORIZONTAL);
  currentTextBox = new wxText(dialog, (wxFunction)NULL, "Value", "", -1, -1, 200);
//  dialog->NewLine();
  wxButton *addBut = new wxButton(dialog, (wxFunction)RadioBoxAddCallback, "Add");
  wxButton *deleteBut = new wxButton(dialog, (wxFunction)RadioBoxDeleteCallback, "Delete");
  addBut->SetClientData((char *)this);
  deleteBut->SetClientData((char *)this);
}

Bool BuildRadioBoxData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValueStringList("values", &strings);
  expr->AssignAttributeValue("rows_or_cols", &noRowsCols);
  return TRUE;
}

Bool BuildRadioBoxData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::WritePrologAttributes(expr, database);
  expr->AddAttributeValueStringList("values", &strings);
  expr->AddAttributeValue("rows_or_cols", (long)noRowsCols);
  return TRUE;
}

Bool BuildRadioBoxData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  wxPanel *panel = (wxPanel *)buildParent->userWindow;
  if (labelPosition == wxDEFAULT_POSITION)
    panel->SetLabelPosition((((BuildPanelData *)buildParent)->horizLabelPosition ? wxHORIZONTAL : wxVERTICAL));
  else
    panel->SetLabelPosition(labelPosition);

  if (autoSize)
  {
    width = -1;
    height = -1;
  }

  windowStyle = 0;

  char **theStrings = new char*[strings.Number()];
  for (int i = 0; i < strings.Number(); i++)
    theStrings[i] = (char *)(strings.Nth(i))->Data();

  UserRadioBox *radiobox = new UserRadioBox((wxPanel *)buildParent->userWindow,
                 (wxFunction)NULL, title, x, y,
                 -1, -1, strings.Number(), theStrings, noRowsCols, windowStyle);

  delete[] theStrings;
  
  userWindow = radiobox;
  radiobox->buildWindow = this;

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildRadioBoxData::DestroyRealWindow(void)
{
  if (userWindow)
  {
    userWindow->GetPosition(&x, &y);
    userWindow->GetSize(&width, &height);
    delete userWindow;
    userWindow = NULL;
  }
  return TRUE;
}

void BuildRadioBoxData::WriteClassDeclaration(ostream& stream)
{
  stream << "void " << functionName << "(wxRadioBox& radioBox, wxCommandEvent& event);\n\n";
}

// Generation
void BuildRadioBoxData::WriteClassImplementation(ostream& stream)
{
  stream << "void " << functionName << "(wxRadioBox& radioBox, wxCommandEvent& event)\n";
  stream << "{\n";
  stream << "}\n\n";
}

void BuildRadioBoxData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  stream << "new ";
  stream << "wxRadioBox" << "(" << parentName;
  stream << ", " << "(wxFunction)" << (functionName ? functionName : "NULL") << ", " << SafeString(title) << ", " << x << ", " << y << ", ";
  stream << "-1" << ", " << "-1" << ", ";
  stream << strings.Number() << ", " << ((strings.Number() == 0) ? "NULL" : tmpStringArray) << ", ";
  stream << noRowsCols << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}

void BuildRadioBoxData::GenerateWindowInitialisationPre(ostream& stream, char *instanceName, char *indentation)
{
  BuildItemData::GenerateWindowInitialisationPre(stream, instanceName, indentation);
  if (strings.Number() != 0)
  {
    if (tmpStringArray) delete[] tmpStringArray;
    tmpStringArray = GetNewObjectName("stringArray");
    stream << indentation << "char *" << tmpStringArray << "[] = { ";
    wxNode *node = strings.First();
    while (node)
    {
      char *s = (char *)node->Data();
      stream << "\"" << s << "\"";
      node = node->Next();
      if (node) stream << ", "; else stream << " ";
    }
    stream << "};\n";
  }
}

/*
 * slider
 *
 */

BuildSliderData::BuildSliderData(BuildPanelData *theParent):
  BuildItemData(theParent)
{
  x = -1;
  y = -1;
  width = 200;
  height = -1;
  minValue = 0;
  maxValue = 100;
  sliderValue = 0;
  
  name = copystring(GetNewObjectName("slider"));
  memberName = copystring(name);
  className = copystring("wxSlider");
  title = copystring("untitled");
  windowStyle = 0;
  windowType = wxTYPE_SLIDER;
  functionName = copystring(GetNewObjectName("SliderProc"));
}

BuildSliderData::~BuildSliderData(void)
{
}

Bool BuildSliderData::EditAttributes(void)
{
  return BuildItemData::EditAttributes();
}

// Add class-specific items to form
void BuildSliderData::AddFormItems(wxForm *form)
{
  BuildItemData::AddFormItems(form);
  form->Add(wxMakeFormShort("Min value", &minValue, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL,
               100));
  form->Add(wxMakeFormShort("Max value", &maxValue, wxFORM_DEFAULT, NULL, NULL, wxVERTICAL,
               100));
  
  SetHelpTopic("Creating a radiobox item");
}

void BuildSliderData::AddDialogItems(wxDialogBox *dialog)
{
}

Bool BuildSliderData::ReadPrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  BuildItemData::ReadPrologAttributes(expr, database);
  expr->AssignAttributeValue("min_value", &minValue);
  expr->AssignAttributeValue("max_value", &maxValue);
  expr->AssignAttributeValue("slider_value", &sliderValue);
  return TRUE;
}

Bool BuildSliderData::WritePrologAttributes(PrologExpr *expr, PrologDatabase *database)
{
  if (userWindow)
    sliderValue = ((wxSlider *)userWindow)->GetValue();

  BuildItemData::WritePrologAttributes(expr, database);
  expr->AddAttributeValue("min_value", (long)minValue);
  expr->AddAttributeValue("max_value", (long)maxValue);
  expr->AddAttributeValue("slider_value", (long)sliderValue);
  return TRUE;
}

Bool BuildSliderData::MakeRealWindow(void)
{
  if (!(buildParent && buildParent->userWindow)) return FALSE;

  wxPanel *panel = (wxPanel *)buildParent->userWindow;
  if (labelPosition == wxDEFAULT_POSITION)
    panel->SetLabelPosition((((BuildPanelData *)buildParent)->horizLabelPosition ? wxHORIZONTAL : wxVERTICAL));
  else
    panel->SetLabelPosition(labelPosition);

  if (autoSize)
  {
    width = -1;
    height = -1;
  }

  windowStyle = 0;

  UserSlider *slider = new UserSlider((wxPanel *)buildParent->userWindow,
                 (wxFunction)NULL, title, sliderValue, minValue, maxValue, width, x, y,
                 windowStyle);

  userWindow = slider;
  slider->buildWindow = this;

  userWindow->Show(TRUE);
  PositionWindowObject();
  return TRUE;
}

Bool BuildSliderData::DestroyRealWindow(void)
{
  if (userWindow)
  {
    sliderValue = ((wxSlider *)userWindow)->GetValue();
    userWindow->GetPosition(&x, &y);
    userWindow->GetSize(&width, &height);
    delete userWindow;
    userWindow = NULL;
  }
  return TRUE;
}

void BuildSliderData::WriteClassDeclaration(ostream& stream)
{
  stream << "void " << functionName << "(wxSlider& slider, wxCommandEvent& event);\n\n";
}

// Generation
void BuildSliderData::WriteClassImplementation(ostream& stream)
{
  stream << "void " << functionName << "(wxSlider& slider, wxCommandEvent& event)\n";
  stream << "{\n";
  stream << "}\n\n";
}

void BuildSliderData::GenerateConstructorCall(ostream& stream, char *parentName)
{
  if (userWindow)
  {
    sliderValue = ((wxSlider *)userWindow)->GetValue();
  }
  stream << "new ";
  stream << "wxSlider" << "(" << parentName;
  stream << ", " << "(wxFunction)" << (functionName ? functionName : "NULL") << ", " << SafeString(title) << ", ";
  stream << sliderValue << ", " << minValue << ", " << maxValue << ", " << width << ", ";
  stream << x << ", " << y << ", ";

  char buf[400];
  GenerateWindowStyleString(buf);
  stream << buf << ", \"" << name << "\")";
}

void BuildSliderData::GenerateWindowInitialisationPre(ostream& stream, char *instanceName, char *indentation)
{
  BuildItemData::GenerateWindowInitialisationPre(stream, instanceName, indentation);
}

/*
 * Callbacks
 *
 */
 
void UserButtonCallback(UserButton& but, wxCommandEvent& event)
{
  wxNode *node = but.buildWindow->buildActions.First();
  if (buildApp.TestMode())
  {
    if (node)
    {
      BuildAction *action = (BuildAction *)node->Data();
      action->DoAction();
    }
  }
  else
  {
    BuildAction *action = NULL;
    
    if (!node)
    {
      action = CreateNewAction();
      if (!action) return;
    }
    else
      action = (BuildAction *)node->Data();

    Bool ok = action->EditAction();
    if (ok)
    {
      if (!node)
        but.buildWindow->AddAction(action);
    }
    else
    {
      but.buildWindow->DeleteAction(action);
    }
  }
}
 
