# version 2.0 (15 Jun '94)

@class XfwfTabs (XfwfBoard)  @file=Tabs

@ The |XfwfTabs| widget displays a series of tabs, similar to the
alphabet tabs along the top of index cards. One tab, the front one, is
completely visible, the others are partially hidden behind it. Each of
the tabs can be clicked on with the mouse.

Although |XfwfTabs| is a descendant of |XfwfBoard|, |XfwfFrame| and
|XfwfCommon|, it does not currently use many of the resources defined
by those classes. In particular, this implementation does not take
part in keyboard traversal, and it has no 3D frame. The only resources
that it uses are the location resources defined in |XfwfBoard|.

@PUBLIC

@ |lefttabs| is the number of partially visible tabs to the left of
the main one. The main tab always occupies the same amount of space,
but the tabs to the left become narrower when |lefttabs| is increased.

	@var int lefttabs = 0

@ The number of tabs to the right of the main one. The higher the
number, the smaller the tabs will be.

	@var int righttabs = 0

@ The labels on each of the tabs must be simple strings. There is no
support for multiline labels or different fonts in this version of
|XfwfTabs|. The array |labels| should contain no less than |lefttabs +
righttabs + 1| strings. The widget makes private copies of the array
and of the strings.

The leftmost tab displays |labels[0]|. Note that the labels may be too
large to fit in the tab. In that case they are clipped. The left tabs
show only the initial part of the labels, the right tabs show the last
part of their labels and the main (front) tab shows the middle part of
its label.

	@var StringArray labels = NULL

@ The color of the text.

	@var Pixel foreground = <String> XtDefaultForeground

@ The color of the lines around the tabs.

	@var Pixel linecolor = <String> XtDefaultForeground

@ The text is drawn in the font which is give as the |font| resource.

	@var <FontStruct> XFontStruct *font = <String> XtDefaultFont

@ When the user clicks on a tab, a callback function is called. The
|call_data| argument to the callback will be a relative number: 0 if
the user clicked on the main tab, -1 if he clicked on the tab
immediately to the left, +1 if he clicked on the first right tab, and
higher numbers for tabs that are further removed from the main one.

	@var <Callback> XtCallbackList activate = NULL

@ Currently no keyboard traversal.

	@var traversalOn = False

@PRIVATE

@ A GC for drawing the labels.

	@var GC textgc

@ A GC for drawing the lines.

	@var GC linegc

@METHODS

@ The |initialize| method checks the resources and creates the local
variables.

@proc initialize
{
    String *h;
    int i;

    $linegc = NULL;
    create_linegc($);
    $textgc = NULL;
    create_textgc($);
    if ($labels) {
	h = (String*) XtMalloc(($lefttabs + $righttabs + 1) * sizeof(*h));
	for (i = $lefttabs + $righttabs; i >= 0; i--)
	    h[i] = XtNewString($labels[i]);
	$labels = h;
    }
}

@ The |set_values| method checks changed resources and recomputes some
local variables. In this case the GC's may have to be changed.

@proc set_values
{
    Boolean redraw = False;
    String *h;
    int i;

    if ($foreground != $old$foreground || $font != $old$font) {
	create_textgc($);
	redraw = True;
    }
    if ($linecolor != $old$linecolor) {
	create_linegc($);
	redraw = True;
    }
    if ($labels != $old$labels) {
	if ($labels) {
	    h = (String*) XtMalloc(($lefttabs + $righttabs + 1) * sizeof(*h));
	    for (i = $lefttabs + $righttabs; i >= 0; i--)
		h[i] = XtNewString($labels[i]);
	    $labels = h;
	}
	if ($old$labels) {
	    for (i = $old$lefttabs + $old$righttabs; i >= 0; i--)
		XtFree($old$labels[i]);
	    XtFree((XtPointer) $old$labels);
	}
	redraw = True;
    }
    if ($lefttabs != $old$lefttabs || $righttabs != $old$righttabs)
	redraw = True;

    return redraw;
}

@ When the widget is realized or resized, a new shape mask is
installed. The shape mask makes the corners transparent.

@proc realize
{
    Region region;
    Dimension delta;
    XPoint poly[7];

    #realize($, mask, attributes);
    delta = $height/2;
    /*
     *     2 o------------------o 3
     *      /                    \
     *     /                      \
     *  1 o                        o 4
     *  0 o------------------------o 5
     */
    poly[0].x = 0;		poly[0].y = $height;
    poly[1].x = 0;		poly[1].y = $height - 1;
    poly[2].x = delta;		poly[2].y = 0;
    poly[3].x = $width - delta;	poly[3].y = 0;
    poly[4].x = $width;		poly[4].y = $height - 1;
    poly[5].x = $width;		poly[5].y = $height;
    poly[6] = poly[0];

    region = XPolygonRegion(poly, 7, EvenOddRule);
    XShapeCombineRegion(XtDisplay($), XtWindow($), ShapeBounding,
			0, 0, region, ShapeSet);
    XDestroyRegion(region);
}

@proc resize
{
    Region region;
    Dimension delta;
    XPoint poly[7];

    #resize($);
    if (! XtIsRealized($)) return;

    delta = $height/2;

    poly[0].x = 0;		poly[0].y = $height;
    poly[1].x = 0;		poly[1].y = $height - 1;
    poly[2].x = delta;		poly[2].y = 0;
    poly[3].x = $width - delta;	poly[3].y = 0;
    poly[4].x = $width;		poly[4].y = $height - 1;
    poly[5].x = $width;		poly[5].y = $height;
    poly[6] = poly[0];

    region = XPolygonRegion(poly, 7, EvenOddRule);
    XShapeCombineRegion(XtDisplay($), XtWindow($), ShapeBounding,
			0, 0, region, ShapeSet);
    XDestroyRegion(region);
}

@ The |expose| method draws the tabs. First the tabs to the left, then
the tabs to the right, and finally the main one. Three private
routines do most of the work. The inherited |expose| method is called
to draw the frame, if any, although the tabs should look best without
a frame.

@proc expose
{
    int i;
    Dimension width, height;
    Position x, y;

    if (! XtIsRealized($)) return;
    #expose($, event, region);
    for (i = 0; i < $lefttabs; i++)
	draw_left_tab($, region, i);
    for (i = $lefttabs + 1; i <= $lefttabs + $righttabs; i++)
	draw_right_tab($, region, i);
    draw_main_tab($, region);
}

@UTILITIES

@ The |draw_left_tab| routine draws a narrow tab to the left of the
main one. The |poly| variable holds the corner points of the area
occupied by this tab, which looks like a lozenge, or a rectangle
bending to the right. The label is written inside this area,
left-aligned.

@def MARGIN = 2

@proc draw_left_tab($, Region region, int i)
{
    Region clip;
    XPoint poly[5];
    Dimension w, delta;
    Position x1, y1;

    w = $width/2/($lefttabs + $righttabs);
    delta = $height/2;				/* Skew */

    /*
     *    2 o--------o 3
     *     /
     *    /
     * 1 o--------o 0
     */
    poly[0].x = i * w + w - 1;          poly[0].y = $height - 1;
    poly[1].x = i * w;                  poly[1].y = $height - 1;
    poly[2].x = i * w + delta;          poly[2].y = 0;
    poly[3].x = i * w + w + delta - 1;  poly[3].y = 0;
    poly[4] = poly[0];

    clip = XPolygonRegion(poly, 5, EvenOddRule);
    if (region) XIntersectRegion(clip, region, clip);
    XSetRegion(XtDisplay($), $textgc, clip);

    if ($labels) {
	y1 = ($height - $font->ascent - $font->descent)/2;
	x1 = i * w + ($height - y1)/2 + MARGIN;
	XDrawString(XtDisplay($), XtWindow($), $textgc, x1, y1 + $font->ascent,
		    $labels[i], strlen($labels[i]));
    }
    XDrawLines(XtDisplay($), XtWindow($), $linegc, poly, 4, CoordModeOrigin);
    XDestroyRegion(clip);
}

@proc draw_right_tab($, Region region, int i)
{
    Region clip;
    XPoint poly[5];
    Dimension w, delta, textw, len;
    Position x1, y1;

    w = $width/2/($lefttabs + $righttabs);
    delta = $height/2;				/* Skew */

    /*
     * 0 o--------o 1
     *             \
     *              \
     *    3 o--------o 2
     */
    x1 = $width - w * ($lefttabs + $righttabs - i);
    poly[0].x = x1 - w - delta;  poly[0].y = 0;
    poly[1].x = x1 - delta - 1;  poly[1].y = 0;
    poly[2].x = x1 - 1;          poly[2].y = $height - 1;
    poly[3].x = x1 - w;          poly[3].y = $height - 1;
    poly[4] = poly[0];

    clip = XPolygonRegion(poly, 4, EvenOddRule);
    if (region) XIntersectRegion(region, clip, clip);
    XSetRegion(XtDisplay($), $textgc, clip);

    if ($labels) {
	len = strlen($labels[i]);
	textw = XTextWidth($font, $labels[i], len);
	y1 = ($height - $font->ascent - $font->descent)/2;
	x1 = x1 - ($height - y1)/2 - textw - MARGIN;
	XDrawString(XtDisplay($), XtWindow($), $textgc, x1, y1 + $font->ascent,
		    $labels[i], len);
    }
    XDrawLines(XtDisplay($), XtWindow($), $linegc, poly, 4, CoordModeOrigin);
    XDestroyRegion(clip);
}

@proc draw_main_tab($, Region region)
{
    Dimension textw, w1, w2, delta, len;
    Position x1, y1;
    Region clip;
    XPoint poly[6];

    delta = $height/2;
    w1 = $lefttabs == 0 ? 0 : $lefttabs * ($width/2/($lefttabs + $righttabs));
    w2 = $righttabs == 0 ? 0 : $righttabs * ($width/2/($lefttabs + $righttabs));

    /*
     *          2 o-------------o 3
     *           /               \
     *          /                 \
     * 0 ------o 1               4 o------o 5
     */
    poly[0].x = 0;                    poly[0].y = $height - 1;
    poly[1].x = w1;                   poly[1].y = $height - 1;
    poly[2].x = w1 + delta;           poly[2].y = 0;
    poly[3].x = $width - w2 - delta;  poly[3].y = 0;
    poly[4].x = $width - w2;          poly[4].y = $height - 1;

    poly[5] = poly[1];

    clip = XPolygonRegion(poly + 1, 5, EvenOddRule);

    poly[5].x = $width;               poly[5].y = $height;

    if (region) XIntersectRegion(clip, region, clip);
    XSetRegion(XtDisplay($), $textgc, clip);

    if ($labels) {
	len = strlen($labels[$lefttabs]);
	textw = XTextWidth($font, $labels[$lefttabs], len);
	x1 = (w1 + $width - w2)/2 - textw/2;
	y1 = ($height - $font->ascent - $font->descent)/2 + $font->ascent;
	XDrawString(XtDisplay($), XtWindow($), $textgc, x1, y1,
		    $labels[$lefttabs], len);
    }
    XDrawLines(XtDisplay($), XtWindow($), $linegc, poly, 6, CoordModeOrigin);
    XDestroyRegion(clip);
}

@ The GCs are not shared, because the clip masks are changed
sometimes.

@proc create_linegc($)
{
    XtGCMask mask;
    XGCValues values;

    if ($linegc != NULL) XFreeGC(XtDisplay($), $linegc);
    values.foreground = $linecolor;
    mask = GCForeground;
    $linegc = XCreateGC(XtDisplay($), RootWindowOfScreen(XtScreen($)),
		    mask, &values);
}

@proc create_textgc($)
{
    XtGCMask mask;
    XGCValues values;

    if ($textgc != NULL) XFreeGC(XtDisplay($), $textgc);
    values.foreground = $linecolor;
    values.font = $font->fid;
    mask = GCForeground | GCFont;
    $textgc = XCreateGC(XtDisplay($), RootWindowOfScreen(XtScreen($)),
		    mask, &values);
}

@TRANSLATIONS

@trans <Btn1Down>,<Btn1Up>: activate()

@ACTIONS

@proc activate
{
    Dimension w, delta, ww, w1, w2;
    int i;

    delta = ($height - event->xbutton.y)/2;
    w = $lefttabs + $righttabs == 0 ? 0 : $width/2/($lefttabs + $righttabs);
    w1 = $lefttabs == 0 ? 0 : $lefttabs * w;
    w2 = $righttabs == 0 ? 0 : $righttabs * w;

    if (event->xbutton.x < w1 + delta) {
	/* Click left of main tab */
	if ($lefttabs != 0) {
	    ww = w + delta;
	    for (i = -$lefttabs; event->xbutton.x >= ww; i++, ww += w) ;
	    XtCallCallbackList($, $activate, (XtPointer) i);
	}
    } else if (event->xbutton.x < $width - w2 - delta) {
	/* Click on main tab */
	XtCallCallbackList($, $activate, (XtPointer) 0);
    } else {
	/* Click right of main tab */
	if ($righttabs != 0) {
	    ww = $width - w - delta;
	    for (i = $righttabs; event->xbutton.x < ww; i--, ww -= w) ;
	    XtCallCallbackList($, $activate, (XtPointer) i);
	}
    }
}

@IMPORTS

@incl <stdio.h>
@incl <X11/extensions/shape.h>
@incl <Xfwf/Converters.h>
