-- ---------------------------------------------------------------------------
-- Toi
-- ---------------------------------------------------------------------------
-- 
-- Toi.cs -- command object interface.
-- 
--   Copyright (C) SieTec Systemtechnik GmbH & Co OHG 1993
--   All rights reserved
-- 
-- Author          : Lutz Hilken
-- 
-- PURPOSE
-- 	|>Description of module's purpose<|
-- 
-- DOCUMENTS
-- 	|>See Toi Reference Manual.<|
-- ---------------------------------------------------------------------------
-- 
-- Created On      : Thu Feb 17 10:02:22 1994
-- Last Modified By: Lutz Hilken
-- Last Modified On: Mon Mar 14 16:42:02 1994
-- Update Count    : 36
-- 
-- HISTORY
-- ---------------------------------------------------------------------------
-- 

SPECIFICATION Toi

IMPORT TextFile 
  FROM TextFile;
-- IMPORT Set FROM Set;

-- -----------------------------------------------------------------
-- CONSTANTS
-- -----------------------------------------------------------------

CONST MAX_CMD_STRING : INT = 500;  -- possible length of one command line

-- -----------------------------------------------------------------
-- TYPES
-- -----------------------------------------------------------------

TYPE CommandMethod = METHOD (IN ArgumentList);
  -- -----------------------------------------------------------------
  -- Type of methods for executing diffenrent commands
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE Toi = OBJECT (IN Prompt        : STRING,
		   IN Separator     : CHAR,
		   IN EndCmd        : STRING,
		   IN InputFile     : STRING,
		   IN OutputFile    : TextFile,
		   IN HistoryLength : INT)

    -- -----------------------------------------------------------------
    -- The Toi objeccttype. See documentation...
    -- -----------------------------------------------------------------
  
PUBLIC
  
  METHOD EventLoop;
    -- -----------------------------------------------------------------
    -- Execute a loop and interprete the given commands. Exit, if
    -- end_cmd is given.
    -- -----------------------------------------------------------------
  
  METHOD InterpreteCommand  (INOUT complete_cmd_str : STRING,
			     OUT   stop             : BOOL);
    -- -----------------------------------------------------------------
    -- Interprete the given command. Stop is true, if the end_cmd is
    -- interpreted or read from a file.
    -- -----------------------------------------------------------------
  
  METHOD SearchCommand      (IN    srchstr : STRING) : Command;
    -- -----------------------------------------------------------------
    -- Look for a command named srchstr. NIL is returned otherwise.
    -- -----------------------------------------------------------------
  
  METHOD PutOutput          (IN    File    : TextFile);
    -- -----------------------------------------------------------------
    -- Set the Output to File from now on.
    -- -----------------------------------------------------------------
  
  METHOD ResetOutput    ();
    -- -----------------------------------------------------------------
    -- Set the former output file.
    -- -----------------------------------------------------------------
  
PROTECTED    
  
  METHOD AppendCommand      (IN cmd_name : STRING,
			     IN handler  : CommandMethod);
    -- -----------------------------------------------------------------
    -- This method adds a new command to the Toi command set.
    -- -----------------------------------------------------------------
  
  METHOD HandleAlias        (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- This method realizes the alias command.
    -- -----------------------------------------------------------------
  
  METHOD HandleUnalias      (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- This method realizes the unalias command.
    -- -----------------------------------------------------------------
  
  METHOD HandleHistory      (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- This method realizes the history command.
    -- -----------------------------------------------------------------
  
  METHOD HandleReadfromfile (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- This method realizes the readfromfile command.
    -- -----------------------------------------------------------------
  
  METHOD HandleShell        (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- This method realizes the shell command.
    -- -----------------------------------------------------------------
  
  METHOD HandleQuit         (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- This method realizes the quit command.
    -- -----------------------------------------------------------------
  
  METHOD HandleSavehistory  (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- This method realizes the savehistory command.
    -- -----------------------------------------------------------------
  
  METHOD HandleHelp         (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- This method realizes the help command.
    -- -----------------------------------------------------------------
  
  METHOD HandleDefine       (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- This method realizes the #define command. If no parameter is
    -- specified, it initiates the listing of all macros.
    -- Otherwise it makes a syntax check for the #define command
    -- and calls the routine adding the new macro, if the syntax was correct.
    -- -----------------------------------------------------------------
  
  METHOD HandleUndefine     (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- This method realizes the unset command.
    -- The deletion of a macro must be outside the search loop, because
    -- DELETE is a write operation, that may not be performed on a
    -- for loop variable. 
    -- -----------------------------------------------------------------
  
  METHOD HandleDefines      (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- This method realizes the #define command. If no parameter is
    -- specified, it initiates the listing of all macros.
    -- Otherwise it makes a syntax check for the #define command
    -- and calls the routine adding the new macro, if the syntax was correct.
    -- -----------------------------------------------------------------
  
  METHOD HandleDummy        (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- Dummy routine.
    -- -----------------------------------------------------------------

  METHOD HandleVerbose      (IN parameter : ArgumentList);
    -- -----------------------------------------------------------------
    -- For debugging.
    -- -----------------------------------------------------------------

STATE

  -- commands          : Set[Command];     -- all commands
  commands          : CommandSET;	 -- until we have SET...
  history           : History;		 -- history
  prompt            : STRING;	 -- contains the user defined prompt
  end_cmd           : STRING;	 -- specifies the quit command
  seperator         : CHAR;		 -- specifies the command seperator
  commandFile       : STRING;
  -- user_variables    : Set[UserVariable];
  user_variables    : UserVariableSET;	 -- until we have SET...
  leave_interpreter : BOOL;
  output_file       : TextFile;
  output_file_stack : OutputStack;
  initial_output    : TextFile;
  readfromfile_active : BOOL;
  verbose_mode      : BOOL;

END OBJECT;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE ArgumentList = OBJECT (IN string : STRING,
			    IN nxt    : ArgumentList, 
			    IN prv    : ArgumentList)
  -- -----------------------------------------------------------------
  -- This type serves as a list of strings plus their count.
  -- -----------------------------------------------------------------
    
PUBLIC
  
  METHOD GetString    : STRING;
    -- -----------------------------------------------------------------
    -- Retrieve the string of this list element
    -- -----------------------------------------------------------------
    
  METHOD GetArgString : STRING;
    -- -----------------------------------------------------------------
    -- Retrieve the complete argument string of this list from this
    -- argument on
    -- -----------------------------------------------------------------
    
  METHOD GetArgCount  : INT;
    -- -----------------------------------------------------------------
    -- Return the number of arguments in this list.
    -- -----------------------------------------------------------------
    
  METHOD GetNext : ArgumentList;
  METHOD PutNext (IN nxt : ArgumentList);
  METHOD GetPrev : ArgumentList;
  METHOD PutPrev (IN prv : ArgumentList);
    -- -----------------------------------------------------------------
    -- This methods are needed to realize a double linked list.
    -- -----------------------------------------------------------------

PROTECTED
  
STATE

  str    : STRING;	     -- The string.
  next   : ArgumentList;     -- link to the next element of the double 
                             -- linked vlist
  prev   : ArgumentList;     -- link to the previous element of the double

END OBJECT;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE OutputStack = OBJECT (IN File : TextFile)
  -- -----------------------------------------------------------------
  -- Output stack for remembering output files for nested rederections
  -- -----------------------------------------------------------------

PUBLIC
  
  METHOD Push (IN File : TextFile) : OutputStack;
    -- -----------------------------------------------------------------
    -- Push. Simple.
    -- -----------------------------------------------------------------
    
  METHOD Pop : OutputStack;
    -- -----------------------------------------------------------------
    -- Pop. Simple.
    -- -----------------------------------------------------------------
    
  METHOD Top : TextFile;
    -- -----------------------------------------------------------------
    -- Top. Simple.
    -- -----------------------------------------------------------------
    
PROTECTED

  METHOD PutPrev (IN newprev : OutputStack);
    -- -----------------------------------------------------------------
    -- Needed for stack maintenance.
    -- -----------------------------------------------------------------
    
STATE
  
  elem : TextFile;			 -- element of this stack elem.
  prev : OutputStack;			 -- previous element

END OBJECT;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE UserVariable = OBJECT (IN name1 : STRING,  -- macro name
			    IN name2 : STRING)  -- macro text
    -- -----------------------------------------------------------------
    -- The type UserVariable represents a Toi macro. It consists of the 
    -- macro name (name) and the macro text (value).
    -- -----------------------------------------------------------------
  
PUBLIC
  
  METHOD GetVarName  : STRING;
    -- -----------------------------------------------------------------
    -- Return the name of this variable
    -- -----------------------------------------------------------------
  
  METHOD GetVarValue : STRING;
    -- -----------------------------------------------------------------
    -- Return the value of this variable
    -- -----------------------------------------------------------------
  
  METHOD PutVarValue (IN val    : STRING);
    -- -----------------------------------------------------------------
    -- Assign a new value to this variable
    -- -----------------------------------------------------------------
  
  METHOD Expand      (INOUT str         : STRING,
		      IN    output_file : TextFile) : BOOL;
    -- -----------------------------------------------------------------
    -- change each occurance of macro name with the macro text in the
    -- string "str". Returns "FALSE" in case of overflow (strlen >
    -- MAX_STRING) 
    -- -----------------------------------------------------------------

PROTECTED

STATE
  name  : STRING;			 -- the name of the macro
  value : STRING;			 -- the macro text
  
END OBJECT;

TYPE UserVariableSET = OBJECT
    -- -----------------------------------------------------------------
    -- Only needed until we have generecity...
    -- -----------------------------------------------------------------
  
PUBLIC
  METHOD Reset   () : UserVariable;
  METHOD GetNext () : UserVariable;
  METHOD CHOOSE  () : UserVariable;
  METHOD INCL    (IN elem : UserVariable);
  METHOD EXCL    (IN elem : UserVariable);
PROTECTED
STATE
  first  : UserVariableList;
  cursor : UserVariableList;
END OBJECT;

TYPE UserVariableList = OBJECT (IN new : UserVariable)
    -- -----------------------------------------------------------------
    -- Only needed until we have generecity...
    -- -----------------------------------------------------------------
  
PUBLIC
  METHOD GetElem : UserVariable;
  METHOD PutNext (IN new : UserVariableList);
  METHOD GetNext : UserVariableList;
PROTECTED
STATE
  elem : UserVariable;
  next : UserVariableList;
END OBJECT;

TYPE Command = OBJECT (IN string   : STRING,
		       IN method   : CommandMethod,
		       IN CmdInter : Toi)
  -- -----------------------------------------------------------------
  -- This type describes one Toi command. The instance variables of
  -- type Command describe the command name, the procedure realizing
  -- this command, and a set of aliases for this command.  
  -- -----------------------------------------------------------------

PUBLIC
  
  METHOD Execute    (IN parameter : ArgumentList) : BOOL;
    -- -----------------------------------------------------------------
    -- Execute this command
    -- -----------------------------------------------------------------
    
  METHOD IsCommand  (IN srchstr : STRING) : BOOL;
    -- -----------------------------------------------------------------
    -- Return TRUE, if this command is named like srchstr.
    -- -----------------------------------------------------------------

  METHOD ListAlias  (IN output_file : TextFile);
    -- -----------------------------------------------------------------
    -- List name and all aliases for this command.
    -- -----------------------------------------------------------------

  METHOD IsAliasFor (IN srchstr : STRING) : BOOL;
    -- -----------------------------------------------------------------
    -- Return TRUE, if this command is an alias for srchstr.
    -- -----------------------------------------------------------------

PROTECTED
  
  METHOD Redirection (INOUT parameter : ArgumentList,
		      OUT   file      : TextFile) : BOOL;
    -- -----------------------------------------------------------------
    -- Check params for pipe or file rederection and open it. Return TRUE on
    -- success. 
    -- -----------------------------------------------------------------
    
STATE

  str       : STRING;
  handle    : CommandMethod;		 -- command realizer
  cmdInter  : Toi;			 -- command interpreter
  
END OBJECT;

TYPE CommandSET = OBJECT
    -- -----------------------------------------------------------------
    -- Only needed until we have generecity...
    -- -----------------------------------------------------------------
  
PUBLIC
  METHOD Reset   () : Command;
  METHOD GetNext () : Command;
  METHOD CHOOSE  () : Command;
  METHOD INCL    (IN elem : Command);
  METHOD EXCL    (IN elem : Command);
PROTECTED
STATE
  first  : CommandList;
  cursor : CommandList;
END OBJECT;

TYPE CommandList = OBJECT (IN new : Command)
    -- -----------------------------------------------------------------
    -- Only needed until we have generecity...
    -- -----------------------------------------------------------------
  
PUBLIC
  METHOD GetElem : Command;
  METHOD PutNext (IN new : CommandList);
  METHOD GetNext : CommandList;
PROTECTED
STATE
  elem : Command;
  next : CommandList;
END OBJECT;

TYPE History = OBJECT (IN length : INT)
  -- -----------------------------------------------------------------
  -- This type realizes the history mechanisms of Toi. The instance
  -- variables are the number of history elements to print during the
  -- list history command, the history represented by the double
  -- linked list of STRING objects, and an initialized flag. 
  -- -----------------------------------------------------------------

PUBLIC
  
  METHOD ListHistory      (IN output_file : TextFile);
    -- -----------------------------------------------------------------
    -- This methods produces a list of the last 'count' history elements.
    -- -----------------------------------------------------------------
    
  METHOD DeleteCommand;
    -- -----------------------------------------------------------------
    -- This method deletes the actual history element and resets the
    -- history list 
    -- -----------------------------------------------------------------
    
  METHOD NewCommand       (IN str : STRING);
    -- -----------------------------------------------------------------
    -- This method adds a new command to the history list. The new command is
    -- always the root of the list (==> instance variable 'history').
    -- -----------------------------------------------------------------
    
  METHOD NewHistoryLength (IN length : INT,
			   IN output_file : TextFile);
    -- -----------------------------------------------------------------
    -- This method resets the number of commands to be printed in the list
    -- history command.
    -- -----------------------------------------------------------------
    
  METHOD CallCommand      (INOUT command_string : STRING,
			   IN    output_file : TextFile);
    -- -----------------------------------------------------------------
    -- This method handles the user command "!". 
    -- There are two possibilities: The user type the "!!" command or
    -- the history search command "!string_or_number".
    -- -----------------------------------------------------------------
    
  METHOD EditCommand      (INOUT command_string : STRING,
			   IN    output_file    : TextFile);
    -- -----------------------------------------------------------------
    -- This method handles the user command "^"
    -- -----------------------------------------------------------------
    
  METHOD GetInitialized   : BOOL;
    -- -----------------------------------------------------------------
    -- Is the history inintialized?
    -- -----------------------------------------------------------------
    
  METHOD GetActNumber     : INT;
    -- -----------------------------------------------------------------
    -- Ths method returns the number of the actual history element.
    -- -----------------------------------------------------------------
    
  METHOD SaveHistory      (IN filename : STRING,
			   IN output_file : TextFile);
    -- -----------------------------------------------------------------
    -- This method saves the complete history into the file specified by
    -- filename. If it is impossible to open the file by writing, an error
    -- message is produced. Otherwise teh complete history, starting from
    -- number 1, is listed into the file.
    -- -----------------------------------------------------------------
    
PROTECTED

STATE

  count       : INT;           -- number of elements to list
  history     : HistoryList;   -- the list
  initialized : BOOL;          -- the problem of the very first command:
                               -- if the user types a <CR> without any command,
			       -- the last executed command will be executed.
			       -- But what is the last executed command in case
			       -- of the first command? Before the user 
			       -- may type his first command this variable
			       -- will be checked. If it's TRUE, an init file
			       -- was executed and so a previous command exist.
			       -- Otherwise an artificial first command
			       -- will be created: "alias".

END OBJECT;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE HistoryList = OBJECT (IN string : STRING,
			   IN nxt    : HistoryList, 
			   IN prv    : HistoryList,
			   IN num    : INT)
  -- -----------------------------------------------------------------
  -- Each list element has a specific number. 
  -- -----------------------------------------------------------------
    
PUBLIC
  
  METHOD GetString : STRING;
  METHOD GetNumber : INT;
  METHOD GetNext : HistoryList;
  METHOD PutNext (IN nxt : HistoryList);
  METHOD GetPrev : HistoryList;
  METHOD PutPrev (IN prv : HistoryList);

PROTECTED
  
STATE

  str    : STRING;	     -- The string.
  next   : HistoryList;      -- link to the next element of the double 
                             -- linked vlist
  prev   : HistoryList;      -- link to the previous element of the double
  number : INT;              -- actual number of the list element

END OBJECT;

END SPECIFICATION;



