/* ---------------------------------------------------------------- */
/* string.c - string implementation for CooL V2.0 RTS               */
/* ---------------------------------------------------------------- */
/* author : Jens-Peter Fiehn Sietec Systemtechnik GmbH & Co. OHG,   */
/*          department STM 11 (CooL-SPE Development Team)           */
/* date :   October of 1993                                         */
/* description :                                                    */
/*          This module contains the definition and implementation  */
/*          of the CooL-STRING for the CooL V2.0 RTS.               */
/* ---------------------------------------------------------------- */
#include <string.h>
#include <malloc.h>
#include "C3Irts.h"

extern void C3IC3IRAISE( void );

/* ----- macros for handling strings ------------------------------ */
/* ----- STRING_of extracts a string(memory block) out of a         */
/*       memory block description                                   */
/* #define C3IC3ISTRING_of(mbd) ((mbd)->MemoryBlock)                */
#define C3IC3ISTRING_of(s) ((s).Value)


/* ----- implementation of string functionality ------------------- */
/*       internal functions                                         */

/* ----- C3IC3IFreeLengthIn --------------------------------------- */
/*       F. returns the number of characters which would fit into   */
/*       the memoryBlock.                                           */
static int C3IC3IFreeLengthIn(C3ISTRING * String)
{
  return ((int) C3IC3IMaxStringLength - C3IC3ILengthOf(String));
}


/* FOREIGN PROCEDURE C3IC3IEqualString( S1 : STRING, S2 : STRING )       */
/*					: BOOL;			         */
C3IBOOL C3IC3IEqualString(C3ISTRING S1, 
			  C3ISTRING S2)	
{
  return (strcmp(C3IC3ISTRING_of(S1), C3IC3ISTRING_of(S2)) == 0);
}


/* FOREIGN PROCEDURE C3IC3INotEqualString( S1 : STRING, S2 : STRING )    */ 
/*					   : BOOL;			 */
C3IBOOL C3IC3INotEqualString(C3ISTRING S1, 
			     C3ISTRING S2)
{
  return (strcmp(C3IC3ISTRING_of(S1), C3IC3ISTRING_of(S2)) != 0);
}


/* FOREIGN PROCEDURE C3IC3IGreaterString( S1 : STRING, S2 : STRING )     */ 
/* 					  : BOOL;			 */
C3IBOOL C3IC3IGreaterString(C3ISTRING S1, 
  			    C3ISTRING S2)
{
  return (strcmp(C3IC3ISTRING_of(S1), C3IC3ISTRING_of(S2)) > 0);
}


/* FOREIGN PROCEDURE C3IC3IGreaterEqualString( S1 : STRING, S2 : STRING )*/ 
/*				       	       : BOOL;			 */
C3IBOOL C3IC3IGreaterEqualString(C3ISTRING S1, 
				 C3ISTRING S2)	
{
  return (strcmp(C3IC3ISTRING_of(S1), C3IC3ISTRING_of(S2)) >= 0);
}


/* FOREIGN PROCEDURE C3IC3ILessString( S1 : STRING, S2 : STRING ) 	 */
/*                                     : BOOL;				 */
C3IBOOL C3IC3ILessString(C3ISTRING S1, 
			 C3ISTRING S2)	
{
  return (strcmp(C3IC3ISTRING_of(S1), C3IC3ISTRING_of(S2)) < 0);
}


/* FOREIGN PROCEDURE C3IC3ILessEqualString( S1 : STRING, S2 : STRING )   */ 
/*					    : BOOL;			 */
C3IBOOL C3IC3ILessEqualString(C3ISTRING S1, 
		       	      C3ISTRING S2)	
{
  return (strcmp(C3IC3ISTRING_of(S1), C3IC3ISTRING_of(S2)) <= 0);
}


/* [Index], [Lwb..Upb] on right side ------------------------------------ */

/* FOREIGN PROCEDURE C3IC3ICharAtIndex( S : STRING , Index : INT ) : CHAR; */
/* ----- C3IC3ICharAtIndex ---------------------------------------- */
char C3IC3ICharAtIndex (C3ISTRING S, int Index)
{
  if (Index > C3IC3ILengthOf(&S) || Index < 1)
  {
     C3IC3ICurrExcept = &CooLC3IPositionIsOutOfRange;
     C3IC3IRAISE();
  }
  return (C3IC3ISTRING_of(S)[Index-1]);
}

/* C3IC3ICheckRange ---------------------------------------------- */
/* The range given by Lwb and Upb is checked                       */
static void C3IC3ICheckRange (C3ISTRING * S, int C3IC3ILwb, int C3IC3IUpb)
{
  if (C3IC3IUpb > C3IC3ILengthOf (S) ||
      C3IC3ILwb < 1)
  {
     C3IC3ICurrExcept = &CooLC3IPositionIsOutOfRange;
     C3IC3IRAISE();
  }
}

/* FOREIGN PROCEDURE C3IC3ISubString( S : STRING, Lwb : INT, Upb : INT ) */
/* 				      : STRING;                          */
/* ----- C3IC3ISubString ------------------------------------ */
C3ISTRING C3IC3ISubString(C3ISTRING S, 
		          int Lwb, 
			  int Upb)
{
  C3ISTRING NewString;
  int Length;

  C3IC3ICheckRange (&S, Lwb, Upb);

  Length = Upb - Lwb + 1; /* Upb is inclusive */
  if (Length >= 1)
     memccpy(C3IC3ISTRING_of(NewString), 
	     C3IC3ISTRING_of(S) + Lwb - 1, '\0', Length);
  NewString.Value[Length >= 1 ? Length : 0] = '\0';

  return (NewString);
}

/* [Index], [Lwb..Upb] on left side ------------------------------------- */

/* FOREIGN PROCEDURE C3IC3IReplaceSubString(INOUT S : STRING,       */
/*                                          Lwb : INT, Upb : INT,   */
/*                                          Replacement : STRING ); */
/* ----- C3IC3IReplaceSubString ----------------------------------- */
void C3IC3IReplaceSubString(C3ISTRING *S,
                            int Lwb, int Upb,
	                    C3ISTRING Replacement)
{
  int SubstringSize = C3IC3ILengthOf(&Replacement);
  int StringSize = C3IC3ILengthOf(S);
  int for_insert = 0;

  C3IC3ICheckRange (S, Lwb, Upb);
  if (Lwb > Upb) {Upb = Lwb;for_insert = 1;}
  if (C3IC3IFreeLengthIn(S) < SubstringSize - (Upb-Lwb+1-for_insert))
  {
     C3IC3ICurrExcept = &CooLC3IMaximumStringLengthExceeded;
     C3IC3IRAISE();
  }
  memmove(C3IC3ISTRING_of(*S) + Lwb + SubstringSize - 1,
          C3IC3ISTRING_of(*S) + Upb - for_insert,
          StringSize - Upb + for_insert);
         /* string part above upb is now moved */
  (*S).Value [(Lwb-1) + SubstringSize + (StringSize-Upb) + for_insert] = '\0';
  memmove(C3IC3ISTRING_of(*S) + Lwb - 1,
          C3IC3ISTRING_of(Replacement),
          SubstringSize); /* now that upper string part is in right */
                          /* position, copy substring */
}


/* FOREIGN PROCEDURE C3IC3IReplaceChar( INOUT S : STRING,	    */ 
/*					Index : INT, C : CHAR );    */
/* ----- C3IC3IReplaceChar ---------------------------------------- */
/*       The indexed character in S is repleced by C                */
void C3IC3IReplaceChar (C3ISTRING * S, int Index, char C)
{
  if (C3IC3ILengthOf(S) < Index || Index < 1)
  {
     C3IC3ICurrExcept = &CooLC3IPositionIsOutOfRange;
     C3IC3IRAISE();
  } 
  C3IC3ISTRING_of(*S)[Index-1] = C;
}

/* <STRING> := <STRING>, <STRING> := <REF ARRAY OF CHAR>, <STRING> := <CHAR> */

/* FOREIGN PROCEDURE C3IC3ICopyString( OUT Dest : STRING, Src : STRING ); */
/* ----- C3ICopyString -------------------------------------------- */
/*       A copy of the specified string is returned.                */
void C3IC3ICopyString (C3ISTRING * Dest,
		       C3ISTRING Src)
{
  (*Dest) = Src;
}


/* FOREIGN PROCEDURE C3IC3IInitWithRefChar( CString : REF ARRAY OF CHAR )*/
/*                                        : STRING;			*/
/* ----- C3IC3IInitWithRefChar ---------------------------------------- */
C3ISTRING C3IC3IInitWithRefChar (char (*CString)[])
{
  C3ISTRING S;
  if ((int) strlen(*CString) > C3IC3IMaxStringLength)
  {
     C3IC3ICurrExcept = &CooLC3IMaximumStringLengthExceeded;
     C3IC3IRAISE();
  }
  strcpy (C3IC3ISTRING_of(S), *CString);
  return (S);
}


/* FOREIGN PROCEDURE C3IC3IInitWithChar( OUT S : STRING, Char : CHAR ); */
/* ----- C3IInitWithChar ------------------------------------------ */
void C3IC3IInitWithChar (C3ISTRING *S, char Char)
{
  C3IC3ISTRING_of(*S)[0] = Char;
  C3IC3ISTRING_of(*S)[1] = '\0';
}


/* +(S,S), +(S,C), +(C,S), +(C,C) --------------------------------------- */

/* FOREIGN PROCEDURE C3IC3IConcatStrings( S1 : STRING, S2 : STRING ) 
					: STRING; */
/* ----- C3IC3IConcatStrings --------------------------------------------- */
C3ISTRING C3IC3IConcatStrings (C3ISTRING S1, 
			       C3ISTRING S2)
{
  C3ISTRING NewString;

  if (C3IC3ILengthOf (&S1) + C3IC3ILengthOf (&S2) - 1 > C3IC3IMaxStringLength)
  {
     C3IC3ICurrExcept = &CooLC3IMaximumStringLengthExceeded;
     C3IC3IRAISE();
  }
  strcpy (C3IC3ISTRING_of(NewString), C3IC3ISTRING_of(S1));
  strcat (C3IC3ISTRING_of(NewString), C3IC3ISTRING_of(S2));

  return (NewString);
}


/* FOREIGN PROCEDURE C3IC3IAppendChar( S : STRING, C : CHAR ) : STRING ; */
/* ----- C3IAddChar ----------------------------------------------- */
C3ISTRING C3IC3IAppendChar (C3ISTRING S, char C)
{
  C3ISTRING NewString;
  int len = C3IC3ILengthOf (&S);

  if (len >= C3IC3IMaxStringLength)
  {
     C3IC3ICurrExcept = &CooLC3IMaximumStringLengthExceeded;
     C3IC3IRAISE();
  }
  strcpy (C3IC3ISTRING_of(NewString), C3IC3ISTRING_of(S));
  C3IC3ISTRING_of(NewString)[len] = C;
  C3IC3ISTRING_of(NewString)[len+1] = '\0';

  return (NewString);
}


/* FOREIGN PROCEDURE C3IC3IPrependChar( C : CHAR, S : STRING ) : STRING; */
/* ----- C3IPrependChar ------------------------------------------- */
C3ISTRING C3IC3IPrependChar (char C, C3ISTRING S)
{
  C3ISTRING NewString;
  int len = C3IC3ILengthOf(&S);
 
  if (len >= C3IC3IMaxStringLength)
  {
     C3IC3ICurrExcept = &CooLC3IMaximumStringLengthExceeded;
     C3IC3IRAISE();
  }
  memmove (C3IC3ISTRING_of(NewString) + sizeof(char), C3IC3ISTRING_of(S), len);
  C3IC3ISTRING_of(NewString)[0] = C;
  C3IC3ISTRING_of(NewString)[len+1] = '\0';

  return (NewString);
}


/* FOREIGN PROCEDURE C3IC3IConcatChars( C1 : CHAR, C2 : CHAR ) : STRING; */
/* ----- C3IC3IConcatChars ----------------------------------------- */
C3ISTRING C3IC3IConcatChars (char C1, char C2)
{
  C3ISTRING NewString;

  C3IC3ISTRING_of(NewString)[0] = C1;
  C3IC3ISTRING_of(NewString)[1] = C2;
  C3IC3ISTRING_of(NewString)[2] = '\0';

  return (NewString);
}

/* FOREIGN PROCEDURE C3IC3ILengthOf( String : REF STRING ) : INT; ----- */
/* ----- C3IC3ILengthOf ------------------------------------------- */
/*       L. returns the length of the string which isn't necesarily */
/*       the length of the MemoryBlock. The latter may be bigger to */
/*       optimize concatenations.                                   */
int C3IC3ILengthOf (C3ISTRING * String)
{
  return (strlen(C3IC3ISTRING_of(* String)));
}


/* ----- C3IC3IToRefArrayOfChar -------------------------------------- */
char (*C3IC3IToRefArrayOfChar (C3ISTRING *S ))[]
{
  return ((char (*)[]) S->Value);
}

/* ----- C3IC3IStrCpy ------------------------------------------------ */
void C3IC3IStrCpy (char * S, char * RefChar)
{
  strcpy (S, RefChar);
}
