--                             -*- Mode: Cool2 -*- 
-- break.cs -- 
-- 
--   Copyright (C) SieTec Systemtechnik GmbH & Co OHG 1993
--   All rights reserved
-- 
-- AtFSID          : $Header: break.cs[1.0] Tue Feb 22 09:59:19 1994 lutz@hawaii saved $
-- Author          : Lutz Hilken
-- 
-- Created On      : Thu Jan 27 09:42:40 1994
-- Last Modified By: Lutz Hilken
-- Last Modified On: Thu Mar 24 09:56:54 1994
-- Update Count    : 9
-- 
-- HISTORY
-- PURPOSE
-- 	|>Description of module's purpose<|
-- 

SPECIFICATION break

IMPORT Function 
  FROM function;
IMPORT Variable 
  FROM variable;
IMPORT String 
  FROM mystring;
IMPORT MySet 
  FROM Bayer;
IMPORT STACK_ADDRESS_T 
  FROM runstack;
IMPORT OBJECTType 
  FROM objecttype;
IMPORT MachineCode 
  FROM machine;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE BreakMode = ENUM 
  NONE,					 -- normal breakpoint
  BREAK, TRACE,				 -- break or trace
  BREAKOTYPE, TRACEOTYPE,		 -- break/trace for objecttype
  BREAKOBJECT, TRACEOBJECT		 -- break/trace for object
END ENUM;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE StopPoint = ABSTRACT OBJECT (IN txt_addr : INT, 
				  IN func     : Function,
				  IN filename : String)
  -- -----------------------------------------------------------------
  -- This type describes a general breakpoint. It is used as the
  -- supertype for "normal" breaklines and for entry points of
  -- functions. It's a deferred type.
  -- -----------------------------------------------------------------

PUBLIC
  
  METHOD PatchIllegalCode;
    -- -----------------------------------------------------------------
    -- This method patches this breakpoint with the code that causes
    -- the debugged process to stop.
    -- -----------------------------------------------------------------

  METHOD PatchOriginalCode;
    -- -----------------------------------------------------------------
    -- This method patches this breakpoint with the original code.
    -- -----------------------------------------------------------------

  METHOD GetTextAddress    : INT;
    -- -----------------------------------------------------------------
    -- This method returns the text address of this breakpoint.
    -- -----------------------------------------------------------------

  METHOD GetFunction       : Function; 
    -- -----------------------------------------------------------------
    -- This method returns the object describing the function this
    -- breakpoint is contained in.
    -- -----------------------------------------------------------------

  ABSTRACT METHOD GetBreakMode (OUT otypes : MySet) : BreakMode;
    -- -----------------------------------------------------------------
    -- This method returns the kind of breakpoint (BREAK, TRACE,
    -- BREAKOTYPE, TRACEOTYPE, NONE) and the objecttype for it.
    -- This method is deferred.
    -- -----------------------------------------------------------------

  ABSTRACT METHOD GetBreakModeMark () : CHAR;
    -- -----------------------------------------------------------------
    -- This method returns a sign depending on the breakmode of this
    -- breakpoint for showing in window and list.
    -- This method is deferred.
    -- -----------------------------------------------------------------

  ABSTRACT METHOD Handle : BOOL;
    -- -----------------------------------------------------------------
    -- This method performs the steps necessary for MaX, when this
    -- breakpoint is hit during execution in normal run mode
    -- -----------------------------------------------------------------

  ABSTRACT METHOD HandleSingleStep  : BOOL;
    -- -----------------------------------------------------------------
    -- This method performs the steps necessary for MaX, when this
    -- breakpoint is hit during execution in single step mode
    -- -----------------------------------------------------------------

  ABSTRACT METHOD HandleSkip (skip_fp : STACK_ADDRESS_T) : BOOL;
    -- -----------------------------------------------------------------
    -- This method performs the steps necessary for MaX, when this
    -- breakpoint is hit during execution in skip function call mode
    -- -----------------------------------------------------------------

  ABSTRACT METHOD HandleReturn (skip_fp : STACK_ADDRESS_T) : BOOL;
    -- -----------------------------------------------------------------
    -- This method performs the steps necessary for MaX, when this
    -- breakpoint is hit during execution in return from function call mode
    -- -----------------------------------------------------------------

  ABSTRACT METHOD ShowInfo           ();
    -- -----------------------------------------------------------------
    -- This method offers the kind of this breakpoint in user friendly
    -- string form.
    -- -----------------------------------------------------------------

  ABSTRACT METHOD GetSourceLineNumber : INT;
    -- -----------------------------------------------------------------
    -- This method returns the source line number of this breakpoint.
    -- -----------------------------------------------------------------
  
  METHOD GetSourceFileName : String;
    -- -----------------------------------------------------------------
    -- This method returns the source line number of this breakpoint.
    -- -----------------------------------------------------------------
  
  ABSTRACT METHOD GetOldSourceLineNumber : INT;
    -- -----------------------------------------------------------------
    -- This method returns the original C source line number of this
    -- breakpoint. 
    -- -----------------------------------------------------------------
  
  ABSTRACT METHOD SetSourceLineNumber (IN lineno : INT);
    -- -----------------------------------------------------------------
    -- This method sets the source line number of this breakpoint.
    -- This is needed in the case, that a breakline mutates from the
    -- C-Breakline to a CooL-Breakline. They differ only in this point.
    -- -----------------------------------------------------------------
  
  METHOD ChangeInitialized ();
    -- -----------------------------------------------------------------
    -- Needed for the restart command: changes the flag...
    -- -----------------------------------------------------------------
    
  ABSTRACT METHOD IsHidden () : BOOL;
    -- -----------------------------------------------------------------
    -- Is this breakpoint visible?
    -- -----------------------------------------------------------------

  METHOD PrintInfo (IN long : BOOL);
    -- -----------------------------------------------------------------
    -- Print info of breakpoint for debugging needs
    -- -----------------------------------------------------------------

  METHOD PrintTypename ();
    -- -----------------------------------------------------------------
    -- Print info of breakpoint for debugging needs
    -- -----------------------------------------------------------------

PROTECTED
    
STATE

  text_address     : INT;		 -- Address of the first
					 -- assembler statement
					 -- of this line
  original_code    : MachineCode;	 -- Contains the *original* code before
					 -- patching. Used to restore
					 -- the  original code
  illegal_code     : MachineCode;	 -- Contains the *patched*
					 -- code. Used to break this line.
  initialized      : BOOL;		 -- The original code must be
					 -- fetched only once. It
					 -- can't change.
  function         : Function;		 -- reference to the function
					 -- containing the stop point.
  file_name        : String;		 -- the name of the file
					 -- (nessesary for pragmas)
  patched          : BOOL;		 -- For debugging
  
END OBJECT;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE BreakTracePoint = StopPoint OBJECT (IN txt_addr : INT, 
                                         IN func     : Function,
					 IN filename : String,
                                         IN src_line : INT)
  -- -----------------------------------------------------------------
  -- This object type describes a normal breakpoints.
  -- It inherits all infomations and methods offered by StopPoint.
  -- It adds the following instance variables:
  --    source_line_number : INT;
  --    condition          : Condition;
  --    trace_or_break     : INT;
  -- The following methods are redefined:
  --    GetSourceLineNumber
  --    ShowInfo
  --    GetBreakMode
  --    Handle
  --    HandleSingleStep
  --    HandleSkip
  --    HandleReturn
  -- -----------------------------------------------------------------

PUBLIC
  
  METHOD GetBreakMode (OUT otypes : MySet) : BreakMode;
    -- -----------------------------------------------------------------
    -- This method returns the kind of breakpoint (BREAK, TRACE,
    -- BREAKOTYPE, TRACEOTYPE, NONE) and the objecttype for it.
    -- -----------------------------------------------------------------

  METHOD GetBreakModeMark () : CHAR;
    -- -----------------------------------------------------------------
    -- This method returns a sign depending on the breakmode of this
    -- breakpoint for showing in window and list.
    -- -----------------------------------------------------------------
  
  METHOD Handle : BOOL;
    -- -----------------------------------------------------------------
    -- This method performs the steps necessary for MaX, when this
    -- breakpoint is hit during execution in normal run mode
    -- -----------------------------------------------------------------

  METHOD HandleSingleStep  : BOOL;
    -- -----------------------------------------------------------------
    -- This method performs the steps necessary for MaX, when this
    -- breakpoint is hit during execution in single step mode
    -- -----------------------------------------------------------------

  METHOD HandleSkip (skip_fp : STACK_ADDRESS_T) : BOOL;
    -- -----------------------------------------------------------------
    -- This method performs the steps necessary for MaX, when this
    -- breakpoint is hit during execution in skip function call mode
    -- -----------------------------------------------------------------

  METHOD HandleReturn (skip_fp : STACK_ADDRESS_T) : BOOL;
    -- -----------------------------------------------------------------
    -- This method performs the steps necessary for MaX, when this
    -- breakpoint is hit during execution in return from function call mode
    -- -----------------------------------------------------------------

  METHOD ShowInfo           ();
    -- -----------------------------------------------------------------
    -- This method offers the kind of this breakpoint in user friendly
    -- string form.
    -- -----------------------------------------------------------------

  METHOD GetSourceLineNumber : INT;
    -- -----------------------------------------------------------------
    -- This method returns the source line number of this breakpoint.
    -- -----------------------------------------------------------------
  
  METHOD GetOldSourceLineNumber : INT;
    -- -----------------------------------------------------------------
    -- This method returns the original C source line number of this
    -- breakpoint. 
    -- -----------------------------------------------------------------
  
  METHOD SetSourceLineNumber (IN lineno : INT);
    -- -----------------------------------------------------------------
    -- This method sets the source line number of this breakpoint.
    -- This is needed in the case, that a breakline mutates from the
    -- C-Breakline to a CooL-Breakline. They differ only in this point.
    -- -----------------------------------------------------------------
  
  METHOD IsHidden () : BOOL;
    -- -----------------------------------------------------------------
    -- Is this breakpoint visible?
    -- -----------------------------------------------------------------

  METHOD DefineCondition;
    -- -----------------------------------------------------------------
    -- This method defines a condition for this breakpoint.
    -- It is empty in this version.
    -- -----------------------------------------------------------------

  METHOD PrintCondition;
    -- -----------------------------------------------------------------
    -- This method prints the condition of this breakpoint.
    -- It is empty in this version.
    -- -----------------------------------------------------------------

  METHOD CheckCondition;
    -- -----------------------------------------------------------------
    -- This method checks the condition for this breakpoint.
    -- It is empty in this version.
    -- -----------------------------------------------------------------

  METHOD DefineMode         (IN mode   : BreakMode,
			     IN otype  : OBJECTType,
			     IN handle : Variable);
    -- -----------------------------------------------------------------
    -- This method defines the break mode of this break point (BREAK,
    -- TRACE) and also sets the breakpoint by patching the code.
    -- -----------------------------------------------------------------

  METHOD UndefineMode       ();
    -- -----------------------------------------------------------------
    -- This method deletes the breakpoint and pathches the original
    -- code. 
    -- -----------------------------------------------------------------

  METHOD ReDefineBreak ();
    -- -----------------------------------------------------------------
    -- This method redefined the breakpoint and pathches the illegal
    -- code. 
    -- -----------------------------------------------------------------
    
  METHOD DefineSinglestepBreak;
    -- -----------------------------------------------------------------
    -- This method defines this breakpoint as being an invisible
    -- breakpoint used for snglestepping.
    -- -----------------------------------------------------------------

  METHOD UndefineSinglestepBreak;
    -- -----------------------------------------------------------------
    -- This method deletes the invisible breakpoint information for
    -- this breakpoint.
    -- -----------------------------------------------------------------

PROTECTED
  
STATE

  source_line_number : INT;		 -- source line number of this
					 -- breakpoint. 
  old_src_line       : INT;		 -- Needed for S-ODT/ATT
					 -- compiler to remember the
					 -- original C-Linenumber.
  condition          : Condition;	 -- condition. NIL for this version.
  trace_or_break     : BreakMode;	 -- break mode of this breakpoint.
  objecttypes        : MySet;		 -- needed for stopping in
					 -- given objecttypes
  objects            : MySet;		 -- needed for stopping in
					 -- given objects
  
END OBJECT;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE FirstLine = BreakTracePoint OBJECT (IN txt_addr : INT, 
					 IN func     : Function,
					 IN filename : String,
					 IN src_line : INT)
  -- -----------------------------------------------------------------
  -- This object type describes first lines of functions.
  -- It inherits all infomations and methods offered by
  -- BreakTracePoint.
  -- The following methods are redefined:
  --    UndefineMode
  --    UndefineSinglestepBreak
  --    Handle
  --    HandleSingleStep
  --    HandleSkip
  --    HandleReturn
  -- -----------------------------------------------------------------
PUBLIC
  
PROTECTED
  
STATE

END OBJECT;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE HiddenFirstLine = FirstLine OBJECT (IN txt_addr : INT, 
					 IN func     : Function,
					 IN filename : String,
					 IN src_line : INT)
  -- -----------------------------------------------------------------
  -- This object type describes first lines of INITIALLY Methods  and
  -- C3INEW routines....
  -- It inherits all infomations and methods offered by
  -- FirstLine.
  -- The following methods are redefined:
  --    Handle
  --    HandleSingleStep
  --    HandleSkip
  --    HandleReturn
  -- -----------------------------------------------------------------
PUBLIC
  
PROTECTED
  
STATE

END OBJECT;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE HiddenBreakTracePoint = BreakTracePoint OBJECT (IN txt_addr : INT, 
						     IN func     : Function,
						     IN filename : String,
						     IN src_line : INT)
  -- -----------------------------------------------------------------
  -- This object type describes first lines of INITIALLY Methods  and
  -- C3INEW routines....
  -- It inherits all infomations and methods offered by
  -- BreakTracePoint.
  -- The following methods are redefined:
  --    Handle
  --    HandleSingleStep
  --    HandleSkip
  --    HandleReturn
  -- -----------------------------------------------------------------

PUBLIC
  
PROTECTED
  
STATE

END OBJECT;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

TYPE Condition = OBJECT
  -- -----------------------------------------------------------------
  -- The following type is used in the near future. It is used for
  -- handling conditional breakpoints.
  -- -----------------------------------------------------------------
PUBLIC
  
PROTECTED

STATE

END OBJECT;

-- -----------------------------------------------------------------
-- 
-- -----------------------------------------------------------------

END SPECIFICATION;

