;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; dabbrev-ex.el -- extension of dabbrev-expand
;;  (c) sietec Systemtechnik GmbH & Co OHG 
;; AtFSID          : $__Header$
;; Author          : Martin Weber
;; 
;; Created On      : Tue Apr 27 17:09:24 1993
;; Last Modified By: Martin Weber
;; Last Modified On: Mon Jul  5 14:42:08 1993
;; Update Count    : 2
;; 
;; HISTORY
;; PURPOSE
;; 	This file defines an extension of the original dabbrev-expand,
;;      looking also for mode-specific keywords listed in a table the name
;;      of which must be assigned to abbrev-keywords-table.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; keyword specific additions for dabbrev
;;

;; !!! A T T E N T I O N !!!
;;
;; dabbrev-expand uses dabbrev-search which has changed in recent versions
;; of emacs (it has an additional parameter). If you have an older version
;; remove the parameter `nocase' in the calls of dabbrev-search. If that
;; does not work compare this version of `dabbrev-expand' with your official
;; version. This version only needs two extensions which are properly marked
;; as such. The rest should be the same.
;;

(load "dabbrev" t t)
(provide 'dabbrev-ex)

(defvar abbrev-current-keywords-table nil
  "Table of mode specific keywords not yet tried. (local).")
(make-variable-buffer-local 'abbrev-current-keywords-table)

(defvar abbrev-keywords-table nil
  "Holds the symbol of a variable containing a mode specific keyword list.")
(make-variable-buffer-local 'abbrev-keywords-table)

(defun dabbrev-expand (arg &optional search-buffer)
  "Expand previous word \"dynamically\".
Expands to the most recent, preceding word for which this is a prefix.
If no suitable preceding word is found, words following point are considered.
After that, keywords in a mode specific list are considered.

A positive prefix argument, N, says to take the Nth backward DISTINCT
possibility.  A negative argument says search forward.  The variable
dabbrev-backward-only may be used to limit the direction of search to
backward if set non-nil.

If the cursor has not moved from the end of the previous expansion and
no argument is given, replace the previously-made expansion
with the next possible expansion not yet tried."
  (interactive "*P")
  (let (old-buffer abbrev expansion old which loc n pattern
	(nocase (and case-fold-search case-replace)))
    ;; old-buffer -- buffer where the command was issued
    ;; abbrev -- the abbrev to expand
    ;; expansion -- the expansion found (eventually) or nil until then
    ;; old -- the text currently in the buffer
    ;;    (the abbrev, or the previously-made expansion)
    ;; loc -- place where expansion is found
    ;;    (to start search there for next expansion if requested later)
    ;; nocase -- non-nil if should consider case significant.
    (save-excursion
      (if (and (null arg)
	       (eq last-command this-command)
	       last-dabbrevs-abbrev-location)
	  (progn
	    (setq abbrev last-dabbrevs-abbreviation)
	    (setq old last-dabbrevs-expansion)
	    (setq which last-dabbrevs-direction))
	(setq which (if (null arg)
			(if dabbrevs-backward-only 1 0)
		        (prefix-numeric-value arg)))
	(setq loc (point))
	(forward-word -1)
	(setq last-dabbrevs-abbrev-location (point)) ; Original location.
	(setq abbrev (buffer-substring (point) loc))
	(setq old abbrev)
	(setq last-dabbrevs-expansion-location nil)
	(setq last-dabbrev-table nil))  	; Clear table of things seen.

      ;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      ;;and the minibuffer extension:
      (if search-buffer
	  (progn
	    (setq old-buffer (current-buffer))
	    (set-buffer search-buffer)))
      ;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      ;;first extension for keywords follows:
      (setq abbrev-current-keywords-table 
	    (copy-alist (eval abbrev-keywords-table)))
      ;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

      (setq pattern (concat "\\b" (regexp-quote abbrev) "\\(\\sw\\|\\s_\\)+"))
      ;; Try looking backward unless inhibited.
      (if (>= which 0)
	  (progn 
	    (setq n (max 1 which))
	    (if last-dabbrevs-expansion-location
		(goto-char last-dabbrevs-expansion-location))
	    (while (and (> n 0)
			(setq expansion (dabbrevs-search pattern t nocase)))
	      (setq loc (point-marker))
	      (setq last-dabbrev-table (cons expansion last-dabbrev-table))
	      (setq n (1- n)))
	    (or expansion
		(setq last-dabbrevs-expansion-location nil))
	    (setq last-dabbrevs-direction (min 1 which))))

      (if (and (<= which 0) (not expansion)) ; Then look forward.
	  (progn 
	    (setq n (max 1 (- which)))
	    (if last-dabbrevs-expansion-location
		(goto-char last-dabbrevs-expansion-location))
	    (while (and (> n 0)
			(setq expansion (dabbrevs-search pattern nil nocase)))
	      (setq loc (point-marker))
	      (setq last-dabbrev-table (cons expansion last-dabbrev-table))
	      (setq n (1- n)))
	    (setq last-dabbrevs-direction -1)))
    
      ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      ;;second extension for keywords follows:
      (if (not expansion) ; Then look for keywords.
	  (progn 
	    (while (and abbrev-current-keywords-table (not expansion))
	      (setq expansion 
		    (if (string= 
			 abbrev (substring 
				 (car abbrev-current-keywords-table) 0 
				 (min (length abbrev) 
				      (length 
				       (car abbrev-current-keywords-table)))))
			(car abbrev-current-keywords-table)))
	      (setq abbrev-current-keywords-table (cdr abbrev-current-keywords-table)))
	    (if expansion
		(setq last-dabbrev-table (cons expansion last-dabbrev-table))))))
    ;; and minibuffer extension:
    (if search-buffer
	(set-buffer old-buffer))
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

    (if (not expansion)
	(let ((first (string= abbrev old)))
	  (setq last-dabbrevs-abbrev-location nil)
	  (if (not first)
	      (progn (undo-boundary)
		     (delete-backward-char (length old))
		     (insert abbrev)))
	  (error (if first
		     "No dynamic expansion for \"%s\" found."
		     "No further dynamic expansions for \"%s\" found.")
		 abbrev))
      ;; Success: stick it in and return.
      (undo-boundary)
      (search-backward old)
      ;; Make case of replacement conform to case of abbreviation
      ;; provided (1) that kind of thing is enabled in this buffer
      ;; and (2) the replacement itself is all lower case
      ;; except perhaps for the first character.
      (let ((nocase (and nocase
			 (string= expansion
				  (concat (substring expansion 0 1)
					  (downcase (substring expansion 1)))))))
	(replace-match (if nocase (downcase expansion) expansion)
		       (not nocase)
		       'literal))
      ;; Save state for re-expand.
      (setq last-dabbrevs-abbreviation abbrev)
      (setq last-dabbrevs-expansion expansion)
      (setq last-dabbrevs-expansion-location loc))))

;;
;; end of keyword specific extensions for dabbrev
;;
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; dabbrev-expand for minibuffer
;;
(defun mini-dabbrev-expand (arg)
  "dynamic expansion for minibuffer"
  (interactive "*P")
  (dabbrev-expand arg (window-buffer (previous-window))))

