;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; cool2-mode.el -- Cool2 editing support package
;; 
;;   Copyright (C) sietec Systemtechnik GmbH & Co OHG 1993
;;   All rights reserved
;; 
;; AtFSID          : $__Header$
;; Author          : Martin Weber
;; 
;; Created On      : Fri Apr 23 15:35:09 1993
;; Last Modified By: Martin Weber TA1 Tel.22260
;; Last Modified On: Thu Sep 15 16:26:08 1994
;; Update Count    : 227
;; 
;; HISTORY
;; 9-Jul-1993		Lutz Hilken	
;;    Last Modified: Thu Jul  8 15:23:12 1993 #35 (Lutz Hilken)
;;    Now we provide things for lucid-emacs as well.
;; 24-May-1993		Martin Weber	
;;    Last Modified: Mon May 24 13:01:41 1993 #14 (Martin Weber)
;;    added code to make cool2-font-lock-keywords table automatically from 
;;    cool2-mode-keywords-table
;; 23-Apr-1993		Martin Weber
;;    New indentation with new landESP.
;; PURPOSE
;;    This is a mode to write CooL V2.1 programs. For a description see
;;    the description of cool2-mode below and the info on landESP.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;
;; To use the cool2 mode you should insert the following lines into
;; your .emacs file:
;;
;;(setq auto-mode-alist (cons (cons "\\.cs$" 'cool2-mode)
;;			      (cons (cons "\\.ci$" 'cool2-mode)
;;				    (cons (cons "\\.cm$" 'cool2-mode)
;;					  auto-mode-alist))))
;;
;;(autoload 'cool2-mode                    "cool2-mode" t t)
;;
;;
;; If you want your code to be fontified, i.e. keywords, strings, comments
;; will each appear in different fonts, you must add the following:
;;
;;(load "font-lock" t t)
;;(add-hook 'cool2-mode-hook 
;;	  'turn-on-font-lock)
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(require 'landESP)

(defvar completion-ignore-case nil)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; add the cool2 functions to the menubar
;;
;;

(defvar cool2-menu
  '(
    ["All method interfaces for type hierarchy"
     cool2-focus-on-method-interface t]
    ["Public method interfaces for type hierarchy"
     cool2-focus-on-public-method-interface t]
    ["Protected method interfaces for type hierarchy"
     cool2-focus-on-protectedmethod-interface t]
    ["Method body for type hierarchy"
     cool2-focus-on-method-body t]
    ["Toggle method interface/body"  cool2-focus-toggle-method t]
    "----"
    ["OBJECT"             cool2-objecttype t]
    ["METHOD"             cool2-method t]
    ["PROCEDURE"          cool2-procedure t]
    ["FINALLY"            cool2-finally t]
    ["RECORD"             cool2-record t]
    ["ARRAY"              cool2-array t]
    "----"
    ["IF"                 cool2-if t]
    ["ELSIF"              cool2-elsif t]
    ["SELECT"             cool2-select t]
    ["TYPESELECT"         cool2-typeselect t]
    ["WHILE"              cool2-while t]
    ["FOR"                cool2-for t]
    ["LOOP"               cool2-loop t]
    "----"
    "----"
    ["Indent line"        landESP-indent-line t]
    ["Indent region"      landESP-beautify-region t]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar cool2-mode-syntax-table nil
  "Syntax table in use in Cool2-mode buffers.")

(if cool2-mode-syntax-table
    ()
  (let ((table (make-syntax-table)))
    (modify-syntax-entry ?\n ">   " table)
    (modify-syntax-entry ?\\ "." table)
    (modify-syntax-entry ?\/ "." table)
    (modify-syntax-entry ?\| "." table)
    (modify-syntax-entry ?\& "." table)
    (modify-syntax-entry ?\$ "." table)
    (modify-syntax-entry ?\( "() 1" table)
    (modify-syntax-entry ?\) ")( 4" table)
    (modify-syntax-entry ?* ". 23" table)
    (modify-syntax-entry ?+ "." table)
    (modify-syntax-entry ?- ". 12" table)
    (modify-syntax-entry ?= "." table)
    (modify-syntax-entry ?% "." table)
    (modify-syntax-entry ?< "." table)
    (modify-syntax-entry ?> "." table)
    (modify-syntax-entry ?\' "\"" table)
    (modify-syntax-entry ?\_ "w" table)
    (setq cool2-mode-syntax-table table)))

(defvar cool2-mode-map nil
  "Keymap used in Cool2 mode.")

(defvar cool2-keywords-indent-table nil
  "Indentations of lines beginning with certain regular expressions.")

(if cool2-keywords-indent-table
    ()
  (setq cool2-keywords-indent-table 
	'(( ":="                                        RELATIVE 2 -2)
	  ( "\\bBODY\\b"				ABSOLUTE  0 1) 
	  ( "\\bCASE\\b"				RELATIVE -1 1) 
	  ( "\\bCONST\\b"				ABSOLUTE  0 0) 
	  ;; box comment should be indented
	  ( "---"                                       RELATIVE  0 0)
	  ;; care for comment and blank lines, do nothing
	  ( "\\ *$"                                     IGNORE    0 0)
	  ( "--"                                        IGNORE    0 0)
	  ( "\\ *(\\*"                                  IGNORE    0 0)
	  ( "\\ *\\*)"                                  IGNORE    0 0)
	  ;;    DO and END DO in one line
	  ( "\\bDO\\b.*\\bEND\\ +DO\\b"	         	RELATIVE -1 0)
	  ( "\\bDO\\b"					RELATIVE -1 1) 
	  ( "\\bELSE\\b"				RELATIVE -1 1)
	  ;;    ELSIF <expr> THEN
	  ( "\\bELSIF\\b[.\n]*\\bTHEN\\b"		RELATIVE -1 1)
	  ( "\\bELSIF\\b"				RELATIVE -1 1) 
	  ( "\\bEND\\ +DO\\b"				RELATIVE -1 0) 
	  ( "\\bEND\\b"					RELATIVE -1 0) 
	  ( "\\bENUM\\b"                                RELATIVE  0 1)
	  ( "\\bEXCEPT\\b"				RELATIVE -1 1)
	  ( "\\bEXCEPTION\\b"				ABSOLUTE  0 1)
	  ( "\\bFINALLY\\b"                             ABSOLUTE  1 1)
	  ;;    FOR ... LOOP
	  ( "\\bFOR\\b.*\\bLOOP\\b"			RELATIVE  0 1)
	  ( "\\bFOREIGN\\b"                             ABSOLUTE  0 0)
	  ( "\\bFROM\\b"				ABSOLUTE  1 1) 
	  ;;    IF <expr> THEN
	  ( "\\bIF\\b.*\\bEND IF\\ *;"			RELATIVE  0 0)
	  ( "\\bIF\\b[.\n]*\\bTHEN\\b"			RELATIVE  0 1)
	  ( "\\bIF\\b"           			RELATIVE  0 1)
	  ( "\\bIMPLEMENTATION\\b"                      ABSOLUTE  0 0)
	  ( "\\bIMPORT\\b"				ABSOLUTE  0 1) 
	  ( "\\bLOOP\\b"				RELATIVE  0 1) 
	  ( "\\bINITIALLY\\b"				ABSOLUTE  1 1)
	  ( "\\(\\b\\(REDEFINED\\|ABSTRACT\\)\\ +\\)?METHOD\\b"
	                                                ABSOLUTE  1 1) 
	  ( "\\bTHEN\\b"				RELATIVE -1 1)
	  ( "\\bOBJECT\\b"				RELATIVE  0 1) 
	  ( "\\bOTHERWISE\\b"				RELATIVE -1 1) 
	  ( "\\bPROCEDURE\\b[^;]*;"			ABSOLUTE  0 0) 
	  ( "\\bPROCEDURE\\b"				ABSOLUTE  0 1) 
	  ( "\\bPROTECTED\\b"                           ABSOLUTE  0 1)
	  ( "\\bPUBLIC\\b"                              ABSOLUTE  0 1)
	  ( "\\bRECORD\\b"				RELATIVE  0 1) 
	  ( "\\bROOT\\b"                                ABSOLUTE  0 0)
	  ( "\\bSELECT\\b"				RELATIVE  0 1)
	  ( "\\bSPECIFICATION\\b"                       ABSOLUTE  0 0)
	  ( "\\bSTATE\\b"				ABSOLUTE  0 1)
	  ( "\\bTRY\\b"					RELATIVE  0 1) 
	  ( "\\bTYPESELECT\\b"				RELATIVE  0 1) 
	  ( "\\bTYPE\\ +\\w+\\ *=\\ *RECORD\\b"		ABSOLUTE  0 1)
	  ( "\\bTYPE\\ +\\w+\\ *=\\ *\\w*\\ *OBJECT\\b" ABSOLUTE  0 1)
	  ( "\\bTYPE\\ +\\w+\\ *=\\ *UNION"             ABSOLUTE  0 1)
	  ( "\\bTYPE\\b"                                ABSOLUTE  0 0)
	  ( "\\bUNION\\b"                               RELATIVE  0 1) 
	  ;; WHILE <expr> LOOP
	  ( "\\bWHILE\\b.*LOOP\\b"                      RELATIVE  0 1) 
	  ;; any line that is not matched by one of the above expressions 
	  ;; could be defined as ( ".*" RELATIVE 0 0)
	  ))) 

(defvar cool2-mode-keywords-table nil
  "Keywords in use in Cool2-mode buffers.")

(defvar cool2-font-lock-keywords nil
  "Keywords in use in Cool2-mode buffers for fontification.")

(if cool2-mode-keywords-table
    ()
  (setq cool2-mode-keywords-table '( "ABSTRACT" "ADR" "ADDRESS" "ARRAY" "AND"
                                    "BOOL" "BODY" "CASE" "CHAR" "CONST"
				    "CURRENT" "DECR" "DELETE" "DO" "DOUBLE"
				    "ELSE" "ELSIF" "END" 
				    "END DO" "END ENUM"
				    "END IF" "END LOOP" "END OBJECT"
				    "END RECORD" "END SELECT" 
				    "END TRY" "END TYPESELECT" "END UNION"
				    "ENUM"
				    "EXCEPT" "EXCEPTION" "EXIT" "EXPORT"
				    "FALSE" "FINALLY" "FLOAT" "FOR" "FOREIGN" 
				    "FROM" 
				    "IF"  
				    "IMPLEMENTATION" "IMPORT" "INITIALLY"
				    "INCR" "INOUT" "IN" "INT" "LOOP" "LONG"
				    "METHOD" "MOD" "NEW" "NIL" "NOT"
				    "OBJECT" "OF" "OR" "OTHERWISE" "OUT"
				    "PASCAL" "PROCEDURE" "PROTECTED" "PUBLIC"
				    "RAISE" "RECORD" "REDEFINED"
				    "REF" "RETURN" "ROOT"
				    "SELECT"
				    "SHORT" "SIZEOF" "SPECIFICATION"
				    "STATE" "STRING" "SUPER" 
				    "THEN"
				    "TRUE" "TRY" "TYPE" "TYPESELECT"
				    "UNION"
				    "UNSIGNED" "VAR" "WHILE"))
  (setq cool2-font-lock-keywords
	(landESP-make-font-lock-keywords cool2-mode-keywords-table)))


(if cool2-mode-map ()
  (let ((map (copy-keymap landESP-mode-map)))
    (define-key map "\C-ca" 'cool2-array)
    (define-key map "\C-ce" 'cool2-elsif)
    (define-key map "\C-cf" 'cool2-for)
    (define-key map "\C-ci" 'cool2-if)
    (define-key map "\C-cl" 'cool2-loop)
    (define-key map "\C-cm" 'cool2-method)
    (define-key map "\C-cd" 'cool2-finally)
    (define-key map "\C-co" 'cool2-objecttype)
    (define-key map "\C-cp" 'cool2-procedure)
    (define-key map "\C-cr" 'cool2-record)
    (define-key map "\C-cs" 'cool2-select)
    (define-key map "\C-ct" 'cool2-typeselect)
    (define-key map "\C-cw" 'cool2-while)
    (define-key map "\C-c{" 'cool2-begin-comment)
    (define-key map "\C-c}" 'cool2-end-comment)
    (define-key map "\C-c\C-c" 'cool2-compile)            
    (define-key map "\C-c\C-mi" 'cool2-focus-on-method-interface)
    (define-key map "\C-c\C-mb" 'cool2-focus-on-method-body)
    (define-key map ";" 'cool2-mode-electric-semi)
    (setq cool2-mode-map map)))

(defun cool2-mode-electric-semi ()
  (interactive)
  (landESP-indent-line)
  (insert ";"))

(defun cool2-mode ()
  "This is a mode intended to support program development in Cool V2.1.
Its functionality consists of three main parts:
- generation of frames for each CooL construct
- automatic indentation
- a sophisticated `focus'-mechanism \(see info on landESP for more
  information\)

All control constructs of Cool2 can be generated by typing
Control-C followed by the first character of the construct, or by choosing
the corresponding item in the CooL2 menu bar (you can get the same menu
by pressing the right mouse button if you are using lucid emacs).
In some of those cases you will be asked for certain components of
that construct (e.g. its name,...). In some other cases you will be asked
to choose between alternatives (SPC and TAB will complete your choice or
show you all alternatives).

The dynamic-abbreviation-expansion is searching the 
cool2-mode-keywords-table after searching the current buffer for an 
expansion. So you can expand all CooL keywords.

With the TAB key you can indent a line of code accordingly in respect
to the preceding lines. With Control-C TAB you can indent a marked region.
If the variable landESP-electric-newline is set to t (default is t)
<RET> will indent the current line automatically and position the
cursor in the new line accordingly. If you do not want to have that
feature set the  variable to nil (in your cool2-mode-hook!). 
The Cool2 mode provides a simple focussing mechanism. It has four levels
of units:
   1. the file
   1a.if you are editing a file `file.cm' which contains both the 
      implementation and speciication of a module there is an extra level:
      the module part
   2. a type, procedure, constant or exception defined in a file
   3. the public or protected interface, state or body of an object type
   4. a constructor or method in the interface or the body of an objecttype.
 
If you are focussed on a unit you cannot see anything outside that unit. 
You can focus on a unit that is contained in your current unit with the key 
Control-c Control-f. You are asked to type in the name of a unit to focus on.
Names are completed as far as possible by pressing TAB in a similar way as in
the find-file emacs function. If there is more than one name to complete the
typed name, a list of all those names is printed. If you type <return> without
choosing any unit, you will be focussed at the unit your cursor points at.
To go up one level you press Control-c Control-b. To come to the next or 
previous unit on the same level press Control-c Control-n or Control-c 
Control-p, respectively. You can also focus on any other unit of the current
level by pressing Control-c Control-d. Then you are asked for the unit's
name in the same way as above.

Key bindings for functions to indent your code:

   TAB = indent line
   Control-c TAB = indent marked region

Key bindings for focusing functions:
NOTE: Some of these functions rely on two conventions:
1. Do not put comments within KEY EXPRESSIONS. 
2. We assume that files containing CooL code have the following names:
   `<module-name>.cs'    these files contain the specification part
                         of the module <module-name>
   `<module-name>.ci'    these files contain the implementation part
                         of the module <module-name>
   `<module-name>.cm'    these files contain the specification part and the
                         implementation part of the module <module-name>
   When you edit a file the corresponding CooL constructs will automatically
   be inserted into the file. Do NOT use other file names!

   Control-c Control-f = narrow focus to subunit (selected by its name)
   Control-c Control-b = widen focus to surrounding unit
   Control-c Control-n = move focus to next unit
   Control-c Control-p = move focus to previous unit
   Control-c Control-d = move focus to another unit (selected by its name)
   Control-c Control-i = move focus to unit defining the identifier where 
                         the cursor is positioned
   Control-c Control-l = move focus to the last unit visited
   Control-c BS        = select a focus from the history

   Control-c Control-m b = move focus to a method body of the current 
                           object type (including its supertype hierarchy)
   Control-c Control-m i = move focus to a method interface of the current 
                           object type (including its supertype hierarchy)
   - in the menu only -  = move focus to the interface of a public method of
                           the current object type (and its supertypes)
   - in the menu only -  = move focus to the interface of a protected method of
                           the current object type (and its supertypes)
   - in the menu only -  = move focus to another file of your project


Here are key bindings for functions to generate CooL constructs:

   Control-c o = OBJECT     = Generate a frame for an object type.
                              The definition is automatically distributed
                              between the specification and implementation
                              part of a module
   Control-c n = INITIALLY  = Generate a frame for the constructor
   Control-c d = FINALLY    = Generate a frame for the destructor
   Control-c m = METHOD     = Generate a frame for a method. Depending on
                              the object part you are focused you may choose
                              a public, protected, abstract or local method.
                              The method specification and implementation
                              are automatically generated and you are asked
                              to place it in the corresponding part of
                              the object type (just try and see what happens)
   Control-c p = PROCEDURE  = Generate a frame for a procedure. The definition
                              is automatically distributed between the
                              specification and implementation part of a module
   Control-c r = RECORD     = Generate a frame for a RECORD
   Control-c a = ARRAY      = Generate a frame for an ARRAY
   Control-c i = IF         = Generate a frame for an IF THEN ELSE
   Control-c e = ELSIF      = Insert an ELSIF part
   Control-c s = SELECT     = Generate a frame for a SELECT
   Control-c t = TYPESELECT = Generate a frame for a TYPESELECT
   Control-c w = WHILE      = Generate a frame for a WHILE loop
   Control-c l = LOOP       = Generate a frame for a LOOP
   Control-c f = FOR        = Generate a frame for a FOR statement




   landESP-indent controls the number of spaces for each indentation."
  (interactive)
;;  (kill-all-local-variables)
  (setq major-mode 'cool2-mode)
  (let ((name (file-name-sans-versions buffer-file-name))
	(case-fold-search (eq system-type 'vax-vms)))
    (cond ((string-match "\\.cs$" name)
	   (setq file-type 'cool2-spec-file)
	   (setq mode-name "Cool2 Specification"))
	  ((string-match "\\.ci$" name)
	   (setq file-type 'cool2-impl-file)
	   (setq mode-name "Cool2 Implementation"))
	  ((string-match "\\.cm$" name)
	   (setq file-type 'cool2-file)
	   (setq mode-name "Cool2")))
    (use-local-map cool2-mode-map)
    (make-local-variable 'comment-column)
    (setq comment-column 41)
    (make-local-variable 'end-comment-column)
    (setq end-comment-column 75)
    (set-syntax-table cool2-mode-syntax-table)
    (make-local-variable 'indent-line-function)
    (setq indent-line-function 'landESP-indent-line)
    (make-local-variable 'require-final-newline)
    (setq require-final-newline t)
    (make-local-variable 'comment-start)
    (setq comment-start "-- ")
    (make-local-variable 'alternate-comment-start)
    (setq alternate-comment-start "(\\*")
    (make-local-variable 'comment-end)
    (setq comment-end "")
    (make-local-variable 'comment-column)
    (setq comment-column 41)
    (make-local-variable 'comment-start-skip)
    (setq comment-start-skip "--+[ \t]*")
    (make-local-variable 'comment-indent-function)
    (setq comment-indent-function 'c-comment-indent)
    (setq case-fold-search nil)
    (setq abbrev-keywords-table 'cool2-mode-keywords-table)
    (auto-fill-mode 1)
    (landESP-initialize file-type 'cool2-keywords-indent-table ";"
			cool2-mode-map "Cool" cool2-menu)
    (if (string-match "\|" (buffer-name))
	t
      (if (re-search-forward
	   "\\bEND\\ +\\(SPECIFICATION\\|IMPLEMENTATION\\)\\b" nil t)
	  (goto-char (point-min))
	(cool2-module)))
    (run-hooks 'cool2-mode-hook)))
  
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; functions to print Cool2 constructs
;;

(defun cool2-module ()
  "Creates a frame for a module. Depending on the file type this is
a specification module, an implementation module or both."
  (interactive)
  (cond ((eq (landESP-get-focus-type) 'cool2-file)
	 (cool2-specification)
	 (forward-line 3)
	 (cool2-implementation))
	((eq (landESP-get-focus-type) 'cool2-spec-file)
	 (cool2-specification))
	((eq (landESP-get-focus-type) 'cool2-impl-file)
	 (cool2-implementation))
	(t
	 (error "a module definition is not allowed here."))))

(defun cool2-specification ()
  "Create a frame for a specification module."
  (interactive)
  (landESP-print cool2-specification-string))

(defconst cool2-specification-string
  '(SEQ
    "SPECIFICATION " (cool2-module-name) CR
    CR
    "CONST spec_vid : STRING = \"@(#) $__Header$\";" CR
    CR
    @ CR
    CR
    "END SPECIFICATION;" CR))

(defun cool2-implementation ()
  "Create a frame for a implementation module."
  (interactive)
  (landESP-print cool2-implementation-string))

(defconst cool2-implementation-string
  '(SEQ
    "IMPLEMENTATION " (cool2-module-name) CR
    CR
    "CONST impl_vid : STRING = \"@(#) $__Header$\";" CR
    CR
    @ CR
    CR
    "END IMPLEMENTATION;" CR))

(defun cool2-module-name ()
  (substring (file-name-nondirectory (buffer-file-name))
	     0 -3))

(defun cool2-try ()
  "Build skeleton TRY statement, prompting for the <expression>."
  (interactive)
  (landESP-print cool2-try-string))

(defconst cool2-try-string
  '(SEQ
    "TRY" CR
    @ CR
    "EXCEPT" CR
    CR
    "END TRY;" CR))

(defun cool2-select ()
  "Build skeleton SELECT statement, prompting for the <expression>."
  (interactive)
  (landESP-print cool2-select-string))

(defconst cool2-select-string
  '(SEQ
    "SELECT" " " "?expression" " " "OF" CR
    "CASE " @ CR
    "OTHERWISE" CR
    CR
    "END SELECT;" CR))

(defun cool2-typeselect ()
  "Build skeleton TYPESELECT statement, prompting for the <expression>."
  (interactive)
  (landESP-print cool2-typeselect-string))

(defconst cool2-typeselect-string
  '(SEQ
    "TYPESELECT " "?variable" " := " "?expression" " OF" CR
    "CASE " @ CR
    "OTHERWISE" CR
    CR
    "END TYPESELECT;" CR))

(defun cool2-for ()
  "Build skeleton FOR loop statement, prompting for the loop parameters."
  (interactive)
  (landESP-print cool2-for-string))

(defconst cool2-for-string
  '(SEQ
    "FOR " "?variable" " IN " "?range" "INCR" "?increment" CR
    "LOOP" CR
    @ CR
    "END LOOP;" CR))

(defun cool2-if ()
  "Insert skeleton IF statement, prompting for <boolean-expression>."
  (interactive)
  (landESP-print cool2-if-string))

(defconst cool2-if-string
  '(SEQ
    "IF " "?boolean-expression" CR 
    "THEN" CR
    @ CR
    "ELSE" CR
    CR
    "END IF;" CR))

(defun cool2-elsif ()
  "Insert ELSIF keyword and indent for next line."
  (interactive)
  (landESP-print cool2-elsif-string))

(defconst cool2-elsif-string
  '(SEQ
    "ELSIF " "?boolean-expression" CR 
    "THEN" CR
    @ CR))

(defun cool2-loop ()
  "Build skeleton LOOP (with END)."
  (interactive)
  (landESP-print cool2-loop-string))

(defconst cool2-loop-string
  '(SEQ
    "LOOP" CR
    @ CR
    "END LOOP;" CR))

(defun cool2-finally ()
  "Build skeleton FINALLY."
  (interactive)
  (let ((type (landESP-get-focus-type))
	(old-with-file-part landESP-with-file-part))
;;;    (setq landESP-with-file-part nil)
    (if (equal type 'cool2-file)
	(landESP-narrow-to-point-intern nil))
    (if	(or (equal type 'cool2-impl-file)
	    (equal type 'cool2-spec-file)
	    (equal type 'cool2-implementation)
	    (equal type 'cool2-specification))
	(landESP-narrow-to-point-intern nil))
    (if (or (equal (landESP-get-focus-type) 'cool2-object-type)
	    (equal (landESP-get-focus-type) 'cool2-object-type-spec))
	(landESP-narrow-to-point-intern nil))
    (cond ((equal (landESP-get-focus-type) 'cool2-body)
	   (landESP-print cool2-finally-string))
	  (t (message "METHOD is not allowed here.")))
    (while (not (equal (landESP-get-focus-type) type))
      (landESP-widen-focus-intern nil))))

(defconst cool2-finally-string
  '(SEQ
    "FINALLY" CR
    CR
    cool2-routine-body))

(defun cool2-objecttype ()
  "Build skeleton objecttype, prompting for supertype and parameters."
  (interactive)
  (let ((type (landESP-get-focus-type)))
    (if (equal type 'cool2-file)
	(landESP-narrow-to-point-intern nil))
    (cond ((or (equal type 'cool2-impl-file)
	       (equal type 'cool2-implementation))
	   (landESP-print cool2-objecttype-string))
	  ((or (equal type 'cool2-spec-file)
	       (equal type 'cool2-specification))
	   (landESP-print cool2-objecttype-spec-string))
	  (t
	   error "OBJECT type is not allowed here."))
    (while (not (equal (landESP-get-focus-type) type))
      (landESP-widen-focus-intern nil))))

(defconst cool2-objecttype-string
  '(ALT
    ("local objecttype" cool2-local-ot)
    ("exported objecttype" cool2-exported-ot)))

(defconst cool2-objecttype-spec-string
  '(SEQ
    cool2-objecttype-sig-string (cool2-define-ot) CR
    "PUBLIC" CR
    CR
    "PROTECTED" CR
    CR
    "STATE" CR
    @ CR
    "END OBJECT;" CR))

(defconst cool2-objecttype-sig-string
  '(SEQ
    cool2-type-string-for-ot
    "?Supertype name" cool2-optional-abstract
    " OBJECT " cool2-parameters (cool2-store-initially)))

(defconst cool2-type-string-for-ot
  '(SEQ "TYPE " 
	"?Name"
	cool2-generic-parameters
	" = " @))

(defconst cool2-generic-parameters
  '(OPT "?First generic argument"
	" [" <> cool2-opt-argument-rest "] " ))

(defconst cool2-ot-body
  '(SEQ
    "BODY" CR
    CR
    "INITIALLY " initially-signature CR
    cool2-routine-body
    CR
    "END OBJECT;" CR))

(defconst cool2-ot-interface
  '(SEQ
    "PUBLIC" CR
    @ CR
    "PROTECTED" CR
    CR
    "STATE" CR
    CR))

(defun cool2-define-ot ()
  (let ((ot-sig (buffer-substring (landESP-last-start)
				  (landESP-last-end)))
	(ot-head ""))
    (if (string-match "=" ot-sig)
	(setq ot-head (substring ot-sig 0 (match-end 0))))
    (setq ot-head (concat (substring ot-sig 0 (match-end 0))
			  (if (string-match "\\bABSTRACT\\b" ot-sig)
			      " ABSTRACT OBJECT"
			    " OBJECT")))
    (save-excursion
      (cool2-goto-module-part 'cool2-implementation)
      (goto-char (point-min))
      (search-forward "END IMPLEMENTATION" nil t)
      (beginning-of-line 1)
      (newline 1)
      (beginning-of-line 0)
      (insert ot-head)
      (landESP-print '(SEQ CR cool2-ot-body))))
  "")

(defun cool2-export-ot ()
  (let ((ot-sig (buffer-substring (landESP-last-start)
				  (landESP-last-end)))
	(ot-head ""))
    (delete-region (landESP-last-start) (landESP-last-end))
    (goto-char (landESP-last-start))
    (if (string-match "=" ot-sig)
      (progn
	(setq ot-head (substring ot-sig 0 (match-end 0)))
	(insert ot-head
		(if (string-match "\\bABSTRACT\\b" ot-sig)
		    " ABSTRACT OBJECT"
		  " OBJECT"))))
    (save-excursion
      (cool2-goto-module-part 'cool2-specification)
      (goto-char (point-min))
      (search-forward "END SPECIFICATION" nil t)
      (beginning-of-line 1)
      (newline 1)
      (beginning-of-line 0)
      (insert ot-sig)
      (landESP-print '(SEQ CR cool2-ot-interface "END OBJECT;" CR))))
  "")

(defconst cool2-local-ot
  '(SEQ
    cool2-objecttype-sig-string CR
    cool2-ot-interface
    cool2-ot-body))

(defconst cool2-exported-ot
  '(SEQ
    cool2-objecttype-sig-string (cool2-export-ot) CR
    cool2-ot-body))

(defun cool2-store-initially ()
  (setq initially-signature (buffer-substring (landESP-last-start)
					      (landESP-last-end)))
  "")

(defconst cool2-optional-abstract
  '(ALT 
    ("abstract" " ABSTRACT")
    ("concrete" "")))

(defconst cool2-parameters
  '(SEQ "("
	(OPT "?First argument"
	     <> cool2-opt-argument-rest) ")" ))

(defconst cool2-opt-argument-rest
  '(OPT "?Next argument"
	"," CR
	<> cool2-opt-argument-rest))

(defconst cool2-routine-body
  '(SEQ
    "DO" CR
    @ CR
    "END DO;" CR))

(defconst cool2-result
  '(OPT "?Result Type"
	" : " <> ))

(defun cool2-public-method ()
  "export the method."
  (cool2-export-routine (buffer-substring (landESP-last-start)
				    (landESP-last-end)) 'cool2-public))

(defun cool2-protected-method ()
  "export the method."
  (cool2-export-routine (buffer-substring (landESP-last-start)
				    (landESP-last-end)) 'cool2-protected))

(defconst cool2-method-sig
  '(SEQ "METHOD " "?Name" " " cool2-parameters cool2-result))

(defun cool2-define-public-method ()
  "import the method, i.e. copy the signature to the BODY and add an empty
body. Return an empty string since we don't want landESP-print to print
anything else."
  (save-excursion
    (cool2-define-routine (buffer-substring (landESP-last-start)
					    (landESP-last-end))))
  "")

(defun cool2-define-protected-method ()
  "import the method, i.e. copy the signature to the BODY and add an empty
body. Return an empty string since we don't want landESP-print to print
anything else."
  (save-excursion
    (cool2-define-routine (buffer-substring (landESP-last-start)
					    (landESP-last-end))))
  "")

(defconst cool2-method-string
  '(SEQ
    (ALT 
     ("public method"    (SEQ cool2-method-sig 
			      (cool2-public-method)))
     ("protected method" (SEQ cool2-method-sig
			      (cool2-protected-method)))
     ("redefined method" (SEQ "REDEFINED " cool2-method-sig))
     ("local method"     cool2-method-sig))
    CR cool2-routine-body))
  
(defconst cool2-public-method-string
  '(SEQ
    (ALT 
     ("abstract" (SEQ "ABSTRACT " cool2-method-sig))
     ("concrete" (SEQ cool2-method-sig (cool2-define-public-method))))
    ";" CR))

(defconst cool2-protected-method-string
  '(SEQ
    (ALT 
     ("abstract" (SEQ "ABSTRACT " cool2-method-sig))
     ("concrete" (SEQ cool2-method-sig (cool2-define-protected-method))))
    ";" CR))

(defun cool2-goto-object-part (part)
  "Goto a point of the body or interface so that you are at the same 
position in the sequence of methods. The current focus must be in the PUBLIC,
PROTECTED or BODY part of an object type."
  (let ((focus-list (save-excursion
		      (landESP-make-focus-list (landESP-get-current-focus))))
	old-focus new-focus-list new-focus)
    (setq old-focus (if focus-list
			(landESP-where-in-focus-list (point) focus-list)
		      nil))
    (setq focus-list (if old-focus
			 (nthcdr (landESP-count old-focus) focus-list)
		       nil))
    (if (cool2-focus-on-type-interface part)
	(progn
	  (cond ((equal part 'cool2-public)
		 t)
		((equal part 'cool2-protected)
		 (landESP-move-focus +1))
		((equal part 'cool2-body)
		 (landESP-nth-focus 0)))
	  (let ((ret-function (key-binding "\C-m"))
		(quit-function (key-binding "\C-g")))
	    (message "Move the cursor to the insertion position and press RET")
	    (local-set-key "\C-m" 'exit-recursive-edit)
	    (local-set-key "\C-g" 'landESP-quit-recursive-edit)
	    (recursive-edit)
	    (local-set-key "\C-m" ret-function)
	    (local-set-key "\C-g" quit-function)
	    (if landESP-quit-flag
		(signal 'quit nil)
	      'method))))))
		   
(defun cool2-export-routine (signature part)
  "Copy the signature to the interface of the OT."
  (save-excursion
    (let* ((start-entry (landESP-correct-current-focus nil))
	   (start-pos (landESP-position start-entry))
	   (target (cool2-goto-object-part part))
	   (start (point)))
      (cond ((eq target 'method)
	     (insert signature ";")
	     (newline 1))
	    ((eq target 'end)
	     (newline 1)
	     (insert signature ";")))
      (landESP-beautify-region start (point))
      (landESP-focus-on-entry start-entry nil)
      (goto-char start-pos)))
  "")

(defun cool2-define-routine (signature)
  "Copy the signature to the body of the OT and add an empty routine body"
  (save-excursion
    (let* ((start-entry (landESP-correct-current-focus nil))
	   (start-pos (landESP-position start-entry))
	   (target (cool2-goto-object-part 'cool2-body))
	   start)
      (if (eq target 'end)
	  (search-backward-regexp "END\ +OBJECT" nil t))
      (setq start (point))
      (insert signature "\n")
      (landESP-print cool2-routine-body)
      (goto-char (landESP-last-end))
      (newline 1)
      (landESP-beautify-region start (point))
      (landESP-focus-on-entry start-entry nil)
      (goto-char start-pos)))
  "")
  
(defun cool2-method ()
  "Build skeleton method. Prompt for NAME, PARAMETERs and RETURN type. 
Optionally export method (i.e. copy signature to the interface part of
the object type)."
  (interactive)
  (let ((type (landESP-get-focus-type))
	(old-with-file-part landESP-with-file-part))
;    (setq landESP-with-file-part nil)
    (if (equal type 'cool2-file)
	(landESP-narrow-to-point-intern nil))
    (if	(or (equal type 'cool2-impl-file)
	    (equal type 'cool2-spec-file)
	    (equal type 'cool2-implementation)
	    (equal type 'cool2-specification))
	(landESP-narrow-to-point-intern nil))
    (if (or (equal (landESP-get-focus-type) 'cool2-object-type)
	    (equal (landESP-get-focus-type) 'cool2-object-type-spec))
	(landESP-narrow-to-point-intern nil))
    (cond ((equal (landESP-get-focus-type) 'cool2-body)
	   (landESP-print cool2-method-string))
	  ((equal (landESP-get-focus-type) 'cool2-public)
	   (landESP-print cool2-public-method-string))
	  ((equal (landESP-get-focus-type) 'cool2-protected)
	   (landESP-print cool2-protected-method-string))
	  (t (message "METHOD is not allowed here.")))
    (while (not (equal (landESP-get-focus-type) type))
      (landESP-widen-focus-intern nil))
;    (setq landESP-with-file-part old-with-file-part)
    ))

(defun cool2-procedure ()
  "Build skeleton procedure."
  (interactive)
  (let ((type (landESP-get-focus-type)))
    (if (equal type 'cool2-file)
	(landESP-narrow-to-point-intern nil))
    (cond ((or (equal type 'cool2-impl-file)
	       (equal type 'cool2-implementation))
	   (landESP-print cool2-procedure-string))
	  ((or (equal type 'cool2-spec-file)
	       (equal type 'cool2-specification))
	   (landESP-print cool2-procedure-spec-string))
	  (t
	   error "PROCEDURE is not allowed here."))
    (while (not (equal (landESP-get-focus-type) type))
      (landESP-widen-focus-intern nil))))

(defconst cool2-procedure-string
  '(ALT
    ("local procedure" cool2-local-proc-string)
    ("exported procedure" cool2-exported-proc-string)))

(defun cool2-define-proc ()
  (let ((proc-sig (buffer-substring (landESP-last-start)
				  (landESP-last-end))))
    (save-excursion
      (cool2-goto-module-part 'cool2-implementation)
      (goto-char (point-min))
      (search-forward "END IMPLEMENTATION" nil t)
      (beginning-of-line 1)
      (newline 1)
      (beginning-of-line 0)
      (insert proc-sig)
      (landESP-print '(SEQ CR cool2-routine-body))))
  "")

(defun cool2-export-proc ()
  (let ((proc-sig (buffer-substring (landESP-last-start)
				    (landESP-last-end))))
    (save-excursion
      (cool2-goto-module-part 'cool2-specification)
      (goto-char (point-min))
      (search-forward "END SPECIFICATION" nil t)
      (beginning-of-line 1)
      (newline 1)
      (beginning-of-line 0)
      (insert proc-sig)
      (landESP-print '(SEQ ";" CR ))))
  "")

(defconst cool2-local-proc-string
  '(SEQ
    cool2-procedure-sig-string CR
    cool2-routine-body))

(defconst cool2-exported-proc-string
  '(SEQ
    cool2-procedure-sig-string (cool2-export-proc) CR
    cool2-routine-body))

(defconst cool2-procedure-sig-string
  '(SEQ
    "PROCEDURE " "?Name" " " cool2-parameters cool2-result))

(defconst cool2-procedure-spec-string
  '(SEQ
    cool2-procedure-sig-string (cool2-define-proc) ";" CR))

(defun cool2-record ()
  "Build skeleton record."
  (interactive)
  (landESP-print cool2-record-string))

(defconst cool2-record-string
  '(SEQ
    "RECORD" CR
    @ CR
    "END RECORD;" CR))

(defun cool2-array ()
  "Build skeleton record."
  (interactive)
  (landESP-print cool2-array-string))

(defconst cool2-array-dimension
  '(OPT "?dimension"
	"[" <>
	cool2-array-dimensions "]"))

(defconst cool2-array-dimensions
  '(OPT "?next dimension" 
	", " <> cool2-array-dimensions))

(defconst cool2-array-string
  '(SEQ
    "ARRAY " cool2-array-dimension " OF " "?Type" ";" CR @ ))

(defconst cool2-type-string
  '(SEQ "TYPE " "?Name" " = " @))

(defun cool2-while ()
  (interactive)
  (landESP-print cool2-while-string))

(defconst cool2-while-string
  '(SEQ
    "WHILE " "?expression" CR 
    "LOOP" CR
    @ CR
    "END LOOP;" CR))

(defun cool2-begin-comment ()
  (interactive)
  (if (not (bolp))
      (indent-to comment-column 0))
  (insert "(*  "))

(defun cool2-end-comment ()
  (interactive)
  (if (not (bolp))
      (indent-to end-comment-column))
  (insert "*)"))

(defun cool2-compile ()
  (interactive)
  (let* ((compile-command (concat "cool2 " (buffer-name))))
    (compile compile-command)))

(defun execute-monitor-command (command)
  (let* ((shell shell-file-name)
	 (csh (equal (file-name-nondirectory shell) "csh")))
    (call-process shell nil t t "-cf" (concat "exec " command))))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 
;; addition to get the whole inherited interface of an object type
;;

(defconst cool2-super-type-key
  '("\\bTYPE\\b\\ +\\w+\\ *\\(\[.*\]\\)?\\ *=[ \t\n]*\\(\\w+\\b\\)[ \t\n]*OBJECT" 1))

(defun cool2-file-filter (part-of-type file-alist)
  "This function returns an alist with all elements from FILE-ALIST
where the corresponding file may contain PART-OF-TYPE of an object type."
  (let ((file-pattern (cond ((equal part-of-type 'cool2-public)
			     "\\.cs$")
			    ((equal part-of-type 'cool2-protected)
			     "\\.cs$")
			    ((equal part-of-type 'cool2-body)
			     "\\.c[im]$")
			    (t
			     "$")))
	alist)
    (while file-alist
      (if (string-match file-pattern (car (car file-alist)))
	  (setq alist (cons (car file-alist) alist)))
      (setq file-alist (cdr file-alist)))
    alist))

(defun cool2-goto-module-part (part)
  "Find the file containing the module part PART of the module containing
the current focus and focus on PART."
  (let ((file-alist
	 (if (equal landESP-file-type 'cool2-file)
	     (list
	      (list (landESP-file (landESP-get-current-focus))
		    (landESP-file (landESP-get-current-focus))))
	   (let ((file 
		  (concat 
		   (substring (landESP-file (landESP-get-current-focus)) 0 -3)
		   (if (equal part 'cool2-implementation)
		       ".ci"
		     ".cs"))))
	     (list (list file file)))))
	new-buffer)
    (if (string= (car (car file-alist))
		 (landESP-file (landESP-get-current-focus)))
	()
      (setq new-buffer (find-file-noselect (car (car file-alist))))
      (switch-to-buffer new-buffer))
    (if (equal landESP-file-type 'cool2-file)
	(progn
	  (landESP-global-focus-on-expr
	   (if (equal part 'cool2-implementation)
	       "IMPLEMENTATION"
	     "SPECIFICATION")
	   file-alist
	   part))
      (while (not (or (equal (landESP-get-focus-type) part)
		      (and (equal part 'cool2-implementation)
			   (equal (landESP-get-focus-type) 'cool2-impl-file))
		      (and (equal part 'cool2-specification)
			   (equal (landESP-get-focus-type) 'cool2-spec-file))))
	(landESP-widen-focus-intern nil)))))
			   
(defun cool2-goto-type-in-module-part (type-entry part)
  "Find the file containing the module part PART and focus on the entry
with th same key as TYPE-ENTRY."
  (cool2-goto-module-part part)
  (if (re-search-forward (landESP-key type-entry) nil t)
      (landESP-focus-down-on-expr (match-beginning 0)
				  (landESP-key type-entry) nil t)))

(defun cool2-list-methods-of-type (part)
  "List all methods of the type we are focused on which are defined in PART.
The current focus must be cool2-object within the correct module part."
  (let (method-list)
    (cond ((equal part 'cool2-interface)
	   (setq method-list 
		 (landESP-make-focus-list(landESP-get-current-focus)))
	   (landESP-move-focus +1)
	   (setq method-list
		 (append method-list (landESP-make-focus-list
				      (landESP-get-current-focus))))
	   (landESP-move-focus -1))
	  ((equal part 'cool2-public)
	   (setq method-list 
		 (landESP-make-focus-list(landESP-get-current-focus))))
	  ((equal part 'cool2-protected)
	   (landESP-move-focus +1)
	   (setq method-list 
		 (landESP-make-focus-list (landESP-get-current-focus)))
	   (landESP-move-focus -1))
	  ((equal part 'cool2-body)
	   (landESP-nth-focus 0)
	   (setq method-list 
		 (landESP-make-focus-list (landESP-get-current-focus)))
	   (landESP-nth-focus 1)))
    method-list))


(defun cool2-list-all-methods (type-name part)
  "List all methods of type TYPE-NAME and its supertypes, which are
defined in PART ('cool2-interface, 'cool2-public, 'cool2-protected
or 'cool2-body). The current focus must be cool2-public within the
correct module part."

  (let ((start-entry (landESP-get-current-focus))
	original-entry supertype method-list)
    (setq method-list (cool2-list-methods-of-type part))
    ;; OK, got all methods of this type.

    ;; make an alist of the method list where the car of each elem is
    ;; a string consisting of the type name and the method key, and where
    ;; the cdr of each elem is the method entry.
    (setq method-list (mapcar '(lambda (elem) (cons (format "%s:%s" type-name
							    (landESP-key elem))
						    elem))
			      method-list))
    
    ;; search for a supertype name
    (goto-char (point-min))
    (if (re-search-forward (nth 0 cool2-super-type-key) nil t)
	(if (match-beginning (nth 1 cool2-super-type-key))
	    (progn
	      (setq type-name (buffer-substring 
			       (match-beginning (nth 1 cool2-super-type-key))
			       (match-end (nth 1 cool2-super-type-key))))
	      (setq supertype (concat "TYPE\\ +" type-name))
	      
	      ;; try to focus on the supertype
	      (setq original-entry
		    (landESP-global-focus-on-expr 
		     supertype      
		     (cool2-file-filter part 
					(landESP-file-list
					 (landESP-grep-command type-name)))
		     'cool2-method t nil))

	      (if original-entry
		  (progn
		    ;; OK, got a supertype. Now focus on its interface,
		    ;; and apply cool2-list-all-methods recursively.
		    (goto-char (point-min))
		    (landESP-narrow-to-point-intern nil)
		    (setq method-list 
			  (append 
			   (cool2-list-all-methods type-name part)
			   method-list))

		    ;; we still have to reset the state of the buffer
		    ;; containing the supertype.
		    (landESP-focus-on-entry original-entry nil))))))
    (landESP-focus-on-entry start-entry nil)
    method-list))

(defun cool2-focus-on-method-interface ()
  "Focus on the protected interface of a method of the type we are focused on.
The whole type hierarchy is searched."
  (interactive)
  (cool2-focus-on-method 'cool2-interface))

(defun cool2-focus-on-protected-method-interface ()
  "Focus on the protected interface of a method of the type we are focused on.
The whole type hierarchy is searched."
  (interactive)
  (cool2-focus-on-method 'cool2-protected))

(defun cool2-focus-on-public-method-interface ()
  "Focus on the public interface of a method of the type we are focused on.
The whole type hierarchy is searched."
  (interactive)
  (cool2-focus-on-method 'cool2-public))

(defun cool2-focus-on-method-body ()
  "Focus on the body of a method of the type we are focused on.
The whole type hierarchy is searched."
  (interactive)
  (cool2-focus-on-method 'cool2-body))

(defun cool2-get-type-entry ()
  "Get the entry of the type containing the current entry. If the current
entry is not within a type return nil."
  (let ((index (cond ((or (equal (landESP-get-focus-type) 'cool2-method)
			  (equal (landESP-get-focus-type) 'cool2-method-spec))
		      2)
		     ((or (equal (landESP-get-focus-type) 'cool2-public)
			  (equal (landESP-get-focus-type) 'cool2-protected)
			  (equal (landESP-get-focus-type) 'cool2-body))
		      1)
		     ((or (equal (landESP-get-focus-type) 'cool2-object-type)
			  (equal (landESP-get-focus-type)
				 'cool2-object-type-spec))
		      0)
		     (t nil))))
    (if index
	(car (nth index landESP-focus-list-stack)))))

(defun cool2-focus-on-type-interface (part)
  "Focus on the interface part of the type containing the current entry.
PART determines whether to look for the objecttype specification or for 
its implementation. Returns t if successful, nil otherwise."
  (let ((type-entry (cool2-get-type-entry)))
    (if type-entry
	(progn
	  (landESP-focus-on-entry type-entry nil)
	  (cool2-goto-type-in-module-part 
	   type-entry
	   (cond ((or (equal part 'cool2-interface)
		      (equal part 'cool2-public)
		      (equal part 'cool2-protected))
		  'cool2-specification)
		 ((equal part 'cool2-body)
		  'cool2-implementation)
		 (t
		  (error "internal error in focus-on-method."))))
	  (goto-char (point-min))
	  (landESP-narrow-to-point-intern nil)
	  t))))

(defun cool2-focus-on-method (part)
  "focus on a method of the type we are focused on. All methods are searched 
throughout the type hierarchy and the user can select one. PART defines 
whether the function will focus on the method in the type interface
\(PART = 'cool2-interface\) or in the type body."
  (fset 'turn-on-font-lock (symbol-function 'ignore))
  (let ((start-entry (landESP-correct-current-focus nil))
	list entry type-name (num 0))
    (if (cool2-focus-on-type-interface part)
	(if (re-search-forward (car (car cool2-object-key)) nil t)
	    (progn
	      (setq type-name (buffer-substring
			       (match-beginning 2)
			       (match-end 2)))
	      (setq list (mapcar '(lambda (elem)
				    (setq num (1+ num))
				    (cons (format "%02d: %s" num (car elem))
					  (cdr elem)))
				 (cool2-list-all-methods type-name part)))
	      (setq entry 
		    (cdr (assoc (car (landESP-completing-read 0 list))
				list)))
	      (if (and entry (landESP-focus-on-entry entry t))
		  ()
		(landESP-focus-on-entry start-entry nil))
	      (landESP-print-modeline)))
      (message "not focused in object type.")))
  (fset 'turn-on-font-lock (symbol-function 'old-turn-on-font-lock))
  (turn-on-font-lock))

(defun cool2-focus-toggle-method ()
  "If focused on a method specification then focus on the corresponding
method implementation, and vice versa."
  (interactive)
  (let* ((start-entry (landESP-get-current-focus))
	 (focus-list 
	  (cond ((equal (landESP-get-focus-type) 'cool2-method)
		 (cool2-focus-on-type-interface 'cool2-interface)
		 (cool2-list-methods-of-type 'cool2-interface))
		((equal (landESP-get-focus-type) 'cool2-method-spec)
		 (cool2-focus-on-type-interface 'cool2-body)
		 (cool2-list-methods-of-type 'cool2-body))
		(t
		 (error "not focused on method."))))
	 (new-focus (assoc (landESP-key start-entry) focus-list)))
    (if new-focus
	(landESP-focus-on-entry new-focus t)
      (error (format "Couldn't toggle for %s." 
		     (landESP-key start-entry)))
      (landESP-focus-on-entry start-entry nil))
    (landESP-print-modeline)))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; focus parameters used by the landESP functions
;;

(defconst cool2-import-key
  (list
   '("\\(\\bIMPORT\\b\\)[^;]*;" 1)
   '(END "\\([ \t\n]*\\bIMPORT\\b[^;]*;.*\n\\)+")))
(defconst cool2-const-key
  '(("\\bCONST\\b\\ +\\w+" 0)))
(defconst cool2-var-key
  '(("\\bVAR\\b\\ +\\w+" 0)))
(defconst cool2-exception-key
  '(("\\bEXCEPTION\\b\\ +\\w+" 0)))
(defconst cool2-type-key
  (list
   '("\\(\\bTYPE\\b\\ +\\w+\\)\\ *=" 1)
   '(END "\\bEND\\ +\\w+\\ *;.*\n")))
(defconst cool2-object-type-start
  '("\\(\\bTYPE\\b\\ +\\w+\\)\\ *\\(\\[[^]]+\\]\\)?\\ *=\\ *\\(\\w+\\ +\\)?\\(ABSTRACT\\ +\\)?OBJECT" 1))
(defconst cool2-object-type-key
  (list cool2-object-type-start
	'(END "\\bEND\\ +OBJECT\\ *;.*\n")))
(defconst cool2-procedure-start
  '("\\bPROCEDURE\\b\\ +\\w+" 0))
(defconst cool2-procedure-key
  (list cool2-procedure-start
	'(END "\\bEND\\ +DO\\b\\ *;")))
(defconst cool2-procedure-spec-key
  (list cool2-procedure-start
	'(END "[^;]*;.*\n")))
(defconst cool2-object-key
  '(("\\(\\bTYPE\\b\\)\\ +\\(\\w+\\)[^=]*=" 1)))
(defconst cool2-public-key
  '(("\\bPUBLIC\\b" 0)))
(defconst cool2-protected-key
  '(("\\bPROTECTED\\b" 0)))
(defconst cool2-state-key
  '(("\\bSTATE\\b" 0)))
(defconst cool2-body-key
  '(("\\bBODY\\b" 0)))
(defconst cool2-initially-key
  (list '("\\bINITIALLY\\b" 0)
	'(END "\\bEND\\ +DO\\b\\ *;")))
(defconst cool2-method-start
  '("\\(\\b\\(REDEFINED\\|ABSTRACT\\)\\b\\ +\\)?\\bMETHOD\\b\\ +\\w+" 0))
(defconst cool2-method-key
  (list cool2-method-start
	'(END "\\bEND\\ +DO\\b\\ *;")))
(defconst cool2-method-spec-key
  (list cool2-method-start
	'(END "[^;]*;.*\n")))

(defconst cool2-import
  (list cool2-import-key))

(defconst cool2-const
  (list cool2-const-key))

(defconst cool2-var
  (list cool2-var-key))

(defconst cool2-exception
  (list cool2-exception-key))

(defconst cool2-method
  (list cool2-method-key))

(defconst cool2-method-spec
  (list cool2-method-spec-key))

(defconst cool2-initially
  (list cool2-initially-key))

(defconst cool2-public
  (list cool2-public-key
	'cool2-method-spec))

(defconst cool2-protected
  (list cool2-protected-key
	'cool2-method-spec))

(defconst cool2-state
  (list cool2-state-key))

(defconst cool2-body
  (list cool2-body-key
	'cool2-initially 'cool2-method))

(defconst cool2-object-type
  (list cool2-object-type-key
	'cool2-public 'cool2-protected 'cool2-state 'cool2-body))

(defconst cool2-object-type-spec
  (list cool2-object-type-key
	'cool2-public 'cool2-protected 'cool2-state))

(defconst cool2-procedure
  (list cool2-procedure-key))

(defconst cool2-type
  (list cool2-type-key))

(defconst cool2-procedure-spec
  (list cool2-procedure-spec-key))

(defconst cool2-impl-file
  '((("IMPLEMENTATION\\ +\\w+\\b" 0))
    cool2-import cool2-object-type cool2-type cool2-procedure
    cool2-const cool2-var cool2-exception))

(defconst cool2-spec-file
  '((("SPECIFICATION\\ +\\w+\\b" 0))
    cool2-import cool2-object-type-spec cool2-type cool2-procedure-spec
    cool2-const cool2-var cool2-exception))

(defconst cool2-implementation cool2-impl-file)

(defconst cool2-specification cool2-spec-file)

(defconst cool2-file
  '((("\\`." 0))
    cool2-implementation cool2-specification))


(provide 'cool2-mode)
