-- ***************************************************************
-- CoLibri
-- ---------------------------------------------------------------
--
-- List.cs --
-- 
--   Copyright (C) sietec Systemtechnik GmbH & Co OHG 1994
--   All rights reserved
-- 
-- Author          : Lutz Hilken
-- 
-- PURPOSE 
--       This object type realizes the frame for the 'semi-generic' 
--       foundation class List. This class provides the functionality 
--       of unsorted lists in different ways.
--       The strings Element and <ContainerFrame> are replaced by a 
--       special tool, which creates a real container type with the
--       features of this skeleton.
--
-- REFER
--      |>Documents with client documentation<|
--
-- ---------------------------------------------------------------
--
-- Created On      : Wed Jan 12 14:57:15 1994
-- Last Modified By: Karl-Heinz Koester
-- Last Modified On: Wed Jan 12 16:12:38 1994
-- Update Count    : 6
-- 
-- ***************************************************************

SPECIFICATION List

IMPORT Node FROM Node;

TYPE List [EntryType] = 
OBJECT

-- -----------------------------------------------------------------
-- General
-- -----------------------------------------------------------------
-- An object of the type List contains a cursor on a list of
-- entries. The entries are referred by the cursor, which referes
-- to one specific entry. This entry contains the concrete
-- object and the link mechanism, say the next and previous objects.
-- Contents of the List container is also the number of entries in
-- the list and the beginning and the end of that list. 
-- -----------------------------------------------------------------

PUBLIC

-- -----------------------------------------------------------------
-- Construction of a list (done internally by INITIALLY)
-- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Copying
-- -----------------------------------------------------------------
  
  METHOD Copy () : List [EntryType];
  -- -----------------------------------------------------------------
  -- Copy all entries of the container to a new one and return it.
  -- Leave the cursor unchanged. 
  -- -----------------------------------------------------------------

  METHOD GetNumberOfEntries () : INT;
  -- -----------------------------------------------------------------
  -- Deliver the cardinality of the container. Leave the cursor
  -- unchanged. 
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Selective access
-- -----------------------------------------------------------------

  METHOD InsertEntry (IN entry : EntryType);
  -- -----------------------------------------------------------------
  -- Insert the ENTRY into this container after the entry at the
  -- actual cursor position. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

  METHOD RemoveEntry ();
  -- -----------------------------------------------------------------
  -- Delete the entry of this container at the actual cursor
  -- position. Set the cursor to the next entry.
  -- -----------------------------------------------------------------

  METHOD GetEntryAtCursor () : EntryType;
  -- -----------------------------------------------------------------
  -- Deliver the contents of this container at the actual cursor
  -- position. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

  METHOD SetEntryAtCursor (IN entry : EntryType);
  -- -----------------------------------------------------------------
  -- Write the ENTRY in this container at the actual cursor
  -- position. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Navigation through the container
-- -----------------------------------------------------------------

  METHOD SetCursorToBeginMargin ();
  METHOD SetCursorToEndMargin ();
  METHOD SetCursorToFirst ();
  METHOD SetCursorToLast ();
  METHOD SetCursorToNext ();
  METHOD SetCursorToPrevious ();
  -- -----------------------------------------------------------------
  -- Move the cursor to the beginning or at the end of this container.
  -- Move the cursor to the first or last position of this container
  -- containing an entry. 
  -- SetCursorToNext and SetCursorToPrevious move the cursor relativ
  -- to the actual position back or forth.
  -- -----------------------------------------------------------------

  METHOD GetPositionOfCursor () : INT;
  -- -----------------------------------------------------------------
  -- Deliver the number of the element at the actual cursor position.
  -- The first element has the number 1.
  -- -----------------------------------------------------------------

  METHOD SetCursorToPosition (IN position : INT);
  -- -----------------------------------------------------------------
  -- Move the cursor to the POSITIONth position
  -- -----------------------------------------------------------------

  METHOD StepToNextMatch
    (IN predicate : PROCEDURE (IN EntryType) : BOOL);
  METHOD StepToPreviousMatch
    (IN predicate : PROCEDURE (IN EntryType) : BOOL);
  -- -----------------------------------------------------------------
  -- Move the cursor back or forth to the entry which satisfies the
  -- PREDICATE. Start the search at the next or previous entry to the
  -- actual position.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Predicates
-- -----------------------------------------------------------------

  METHOD IsEmpty () : BOOL;
  -- ---------------------------------------------------------------
  -- Test this container for emptyness. Leave the cursor unchanged.
  -- ---------------------------------------------------------------

  METHOD IsEqual 
    (IN other_list  : List [EntryType],
     IN equal_entry : PROCEDURE (IN EntryType, IN EntryType) : BOOL)
    : BOOL;
  -- -----------------------------------------------------------------
  -- Compare this conatiner to OtherList. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

  METHOD IsBeginMargin () : BOOL;
  METHOD IsEndMargin   () : BOOL;
  -- -----------------------------------------------------------------
  -- Is the cursor position at the begin or end margin of this 
  -- list.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Complex operations on the container
-- -----------------------------------------------------------------

  METHOD Merge (IN list : List [EntryType]);
  -- -----------------------------------------------------------------
  -- Merge the List at the actual cursor position of this container.
  -- Leave the cursor unchanged. 
  -- -----------------------------------------------------------------

  METHOD Apply (IN routine : PROCEDURE (INOUT EntryType));
  -- -----------------------------------------------------------------
  -- Apply ROUTINE to every entry of this container. Start at the
  -- actual cursor position. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

  METHOD RemoveMatchingEntries 
    (IN predicate : PROCEDURE (IN EntryType) : BOOL);
  -- -----------------------------------------------------------------
  -- Delete all entries in this container, which satisfies the
  -- PREDICATE. Start at the actual cursor position. Set the cursor
  -- after the operation to the first entry.
  -- -----------------------------------------------------------------

  METHOD RemoveDuplicateEntries
    (IN equal_entry : PROCEDURE (IN EntryType, IN EntryType) : BOOL);
  -- -----------------------------------------------------------------
  -- Delete all entries in this container, which are equal. 
  -- Set the cursor after the operation to the first entry.
  -- -----------------------------------------------------------------

  METHOD GetNumberOfMatches
    (IN predicate : PROCEDURE (IN EntryType) : BOOL) : INT;
  -- -----------------------------------------------------------------
  -- Deliver the number of entries in this container, which satisfy
  -- the PREDICATE. Leave the cursor unchanged. 
  -- -----------------------------------------------------------------

  METHOD Split () : List [EntryType];
  -- -----------------------------------------------------------------
  -- Split this container and return the List starting at the actual
  -- cursor position. The last entry of this container will be the
  -- entry before the actual position. Set the cursor after this
  -- operation to the first entry. 
  -- -----------------------------------------------------------------

  METHOD Substitute 
    (IN predicate : PROCEDURE (IN EntryType) : BOOL,
     IN entry     : EntryType);
  -- -----------------------------------------------------------------
  -- Replace the contents of all entries of this container which
  -- satisfy the PREDICATE with ENTRY. Start at the actual cursor
  -- position. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

  METHOD Head () : EntryType;
  -- -----------------------------------------------------------------
  -- Return the first entry of this container. Leave the cursor
  -- unchanged. 
  -- -----------------------------------------------------------------

  METHOD Tail () : List [EntryType];
  -- -----------------------------------------------------------------
  -- Return this container without the first entry. Leave the cursor
  -- unchanged. The cursor of the new list is set to the first
  -- entry. 
  -- -----------------------------------------------------------------

PROTECTED
  
STATE

  cursor          : Node [EntryType]; -- actual position in this container
  cursorPosition  : INT;
  numberOfEntries : INT;	      -- number of entries in this container
  beginMargin     : Node [EntryType];   -- Left end of this container
  endMargin       : Node [EntryType];   -- Right end of this container

END OBJECT;

EXCEPTION ContainerEmpty;
EXCEPTION PositionOutOfBounds;
EXCEPTION NoMatch;

-- ***************************************************************
CONST spec_vid : STRING = 
   "@(#) $Header: List.cs[1.2] Wed Jan 12 18:10:56 1994 kalle@gianna proposed $" ;

END SPECIFICATION (* List *);
