/**************************************************************************/
/***  mkxconfig								***/
/***									***/
/***  This program attempts to create an Xconfig based on user input	***/
/***  and an X probe.  To compile, type `cc mkxconfig.c -o mkxconfig -lm' */
/***									***/
/***  STANDARD DISCLAIMER:						***/
/***	USE AT YOU OWN RISK.  NIETHER THE AUTHOR NOR THOSE WHO HELPED	***/
/***	ITS DESIGN WILL ASSUME ANY RESPONSIBILITY FOR DAMAGE OR		***/
/***	PROPERTY LOSS.							***/
/***  SPECIAL WARNING/DISCLAIMER:					***/
/***	PLAYING WITH VIDEO CLOCKS AND MONITOR RESOLUTIONS CAN		***/
/***	PERMANTLY AND IRREPARABLY DAMAGE YOUR MONITOR AND/OR CARD.	***/
/***	USE OF THIS PROGRAM DIRECTLY PLACES RESPONSIBILITY UPON THE	***/
/***	USER.								***/
/***									***/
/***  Copyright 1994 Sean Walton.  All rights reserved.  Free to copy	***/
/***  according to the GNU General Public License.			***/
/***									***/
/**************************************************************************/
/***  Revision Control							***/
/***									***/
/***  0.0.1	`mkmodes' program which made the ModeDB table		***/
/***  0.1.0	`mkxconfig' first crack at the Xconfig builder (SVGA)	***/
/***  0.1.1	Begun support for accelerator cards (didn't work)	***/
/***  0.1.2	Virtual resolutions, more command-line hooks		***/
/***  0.1.3	Additional hooks for S3 & Mach32 (still didn't work)	***/
/***  0.2.0	Debugged and got Mach32 to work.			***/
/***  0.2.1	Fixed Virtual?, >1MB mem, <24MHz clocks, min clocks	***/
/**************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#ifdef DEBUG
#define PROBESOURCE	"probe"
#else
#define PROBESOURCE	"X -probeonly 2>&1"
#endif

#define TRUE		1
#define FALSE		0
#define ONE_K		1024
#define TWO_K		2048
#define FOUR_K		4096
#define ACCELSCRATCH	256
#define VGA256		1
#define VGA16		2
#define VGAMONO		3
#define ACCEL		4

#define MAXWORD		32
#define MAXLINE		200
#define MAXFONTS	5
#define MAXVGAS		5
#define MAXCLOCKS	40
#define MAXNUM		12
#define MAXVIRTUALX	1536

typedef struct
	{
	double Low, High;
	} TRange;

typedef struct
	{
	double HTicks, HS1, HS2, HFL;
	double VTicks, VS1, VS2, VFL;
	int BeyondRange;
	double RR, HSF, Clock;
	} tModeDB;

typedef struct
	{
	int Fonts[MAXFONTS];
	int MouseType;
	int Has3Buttons;
	int VGAType;
	const char *VGAPrompt;
	int ScratchRam;
	int MemSize;
	char ChipSet[MAXWORD];
	double Clocks[MAXCLOCKS];
	int ClockCount;
	tModeDB ModeDB[MAXCLOCKS];
	} tXSettings;

const char *MiceTypes[]= 
	{"BusMouse", "Logitech", "Microsoft", "mmseries", "mouseman", 
	"mousesystems", "ps/2", "mmhittab"};
const char *FontSources[MAXFONTS]=
	{ "/usr/X386/lib/X11/fonts/misc/", "/usr/X386/lib/X11/fonts/Type1/",
	"/usr/X386/lib/X11/fonts/Speedo/", "/usr/X386/lib/X11/fonts/75dpi/",
	"/usr/X386/lib/X11/fonts/100dpi/" };
const char *VGANames[MAXVGAS]=
	{ "", "VGA256", "VGA16", "VGA2", "ACCEL" };

/*********
*** While I frown on use of global variables, these are basic system
*** system parameters which will be modified ONLY at initialization.
*** Therefore, I feel that making them global is okay (not great).
**********/
double AspectRatio=0.75;
double VerticalSyncPulse=150.0;
double HorizontalSyncPulse=3.8;
TRange VerticalRange={59.0,65.0};
TRange HorizontalRange={30.0e3,45.0e3};
double MinimumClock=24.0;
char   XConfigFile[MAXLINE];

/*********
*  GetRange--Converts a string representation of a range into a floating
*            point range which will be used to determine device boundaries.
*            This routine supports a multiplier so that ranges may be KHz
*            or Hz.
**********/
TRange GetRange(char *String, double Multiplier)
{   TRange Range;

    Range.Low = atof(String) * Multiplier;
    while (*String != '-'  &&  *String != 0)
	String++;
    Range.High = atof(String+1) * Multiplier;
    return Range;
}

/*********
*  ProcessCommandLine--Retrieve and convert information passed in through
*                      the command line.
**********/
void ProcessCommandLine(int Count, char **Args, char **Env)
{   int i;

    for ( i = 1; i < Count; i++)
	{
	if (strcmp(Args[i], "-vr") == 0)
	    VerticalRange = GetRange(Args[++i], 1);
        else if (strcmp(Args[i], "-hr") == 0)
	    HorizontalRange = GetRange(Args[++i], 1e3);
	else if (strcmp(Args[i], "-aspect") == 0)
	    {
	    AspectRatio = atof(Args[++i]);
	    if (AspectRatio < .5)
		{
		fprintf(stderr, "Bad aspect ratio: reseting to .75\n");
		AspectRatio = .75;
		}
	    }
	else if (strcmp(Args[i], "-vsync") == 0)
	    {
	    VerticalSyncPulse = atof(Args[++i]);
	    if (VerticalSyncPulse < 50  ||  VerticalSyncPulse > 300)
		{
		fprintf(stderr, "Bad vertical sync pulse width: reseting to 150\n");
		VerticalSyncPulse = 150;
		}
	    }
	else if (strcmp(Args[i], "-hsync") == 0)
	    {
	    HorizontalSyncPulse = atof(Args[++i]);
	    if (HorizontalSyncPulse < 3.5  ||  HorizontalSyncPulse > 4.0)
		{
		fprintf(stderr, "Bad horizontal sync pulse width: reseting to 3.8\n");
		HorizontalSyncPulse = 3.8;
		}
	    }
	else if (strcmp(Args[i], "-minclock") == 0)
	    {
	    MinimumClock = atof(Args[++i]);
	    if (MinimumClock < 24  ||  MinimumClock > 110)
		{
		fprintf(stderr, "Bad minimum clock: reseting to 24MHz\n");
		MinimumClock = 24;
		}
	    }
	else
	    {
	    fprintf(stderr, "Usage: mkmodes [-vr <num>-<num>] [-hr <num>-<num>] [-aspect <num>] [-vsync <num>] [-hsync <num>] [-minclock <num>]\n");
	    fprintf(stderr, "Defaults:\n");
            fprintf(stderr, "\tVertical Range=59.0 thru 65.0 Hz\n");
	    fprintf(stderr, "\tHorizontal Range=30.0 thru 45.0 KHz\n");
	    fprintf(stderr, "\tAspect Ratio=.75\n\t\t(Aspect >=.5 permitted)\n");
	    fprintf(stderr, "\tVertical Sync Pulse Width=150 microseconds\n\t\t(50 <= vsync <= 300 permitted)\n");
	    fprintf(stderr, "\tHorizontal Sync Pulse Width=3.8 microseconds\n\t\t(3.5 <= hsync <=4.0 permitted)\n");
	    fprintf(stderr, "\tMinimum dot clock=24.0MHz\n\t\t(24-110 permitted)\n");
	    fprintf(stderr, "Version: 0.2.1\n");
	    fprintf(stderr, "!!!Note: if your monitor or card can't handle `X -probeonly', DON'T USE THIS PROGRAM!\n");
	    exit(0);
	    }
	}
    for ( i = 0; Env[i] != NULL; i++)
	if (strsub(Env[i], "HOME="))
	    {
	    sprintf(XConfigFile, "%s/Xconfig", Env[i]+strlen("HOME="));
	    break;
	    }
}

/*********
*  strsub--an ancorred string compare: true when Sub is a substring of Str
**********/
int strsub(char *Str, char *Sub)
{
    while (tolower(*Str) == tolower(*Sub)  &&  *Sub != 0)
	{
	Str++;
	Sub++;
	}
    return *Sub == 0;
}

/*********
*  SortModeDB--Sort the ModeDB table using a simple bubble sort (bleah!)
**********/
void SortModeDB(tXSettings *Settings)
{   int i, j;
    tModeDB TempModeDB;

    for ( j = 0; j < Settings->ClockCount; j++)
	for ( i = 0; i < Settings->ClockCount-1; i++)
	    if (Settings->ModeDB[i].Clock > Settings->ModeDB[i+1].Clock)
		{
		TempModeDB = Settings->ModeDB[i];
		Settings->ModeDB[i] = Settings->ModeDB[i+1];
		Settings->ModeDB[i+1] = TempModeDB;
		}
}

/*********
*  InsertNumber--Inserts a new element in the clocks array.  This originally
*                was a bit more complicated.
**********/
void InsertNumber(tXSettings *Settings, double NewNum, int MaxArray)
{
    if (Settings->ClockCount < MaxArray)
	Settings->Clocks[Settings->ClockCount++] = NewNum;
}

/*********
*  FillSettings--primarily responsible for Settings initialization and 
*                user input.
**********/
void FillSettings(tXSettings *Settings)
{   int i;
    char TempS[MAXWORD];

    Settings->ClockCount = 0;
    Settings->ScratchRam = 0;
    Settings->MemSize = 0;
    Settings->ChipSet[0] = 0;
    for ( i = 0; i < MAXCLOCKS; i++)
	Settings->Clocks[i] = 0;
    for ( i = 0; i < MAXFONTS; i++)
	Settings->Fonts[i] = TRUE;
    do  {
    	printf("Do you have a 1) BusMouse, 2) Logitech, 3) Microsoft, 4) mmseries,\n"
		"5) mouseman, 6) mousesystems, 7) ps/2, 8) mmhittab: ");
	gets(TempS);
	Settings->MouseType = atoi(TempS)-1;
	}
    while (Settings->MouseType < 0  &&  Settings->MouseType > 7);
    do  {
	printf("Do you have three buttons on your mouse? ");
	gets(TempS);
	Settings->Has3Buttons = tolower(TempS[0]);
	}
    while (Settings->Has3Buttons != 'y'  &&  Settings->Has3Buttons != 'n');
    do  {
	printf("Do you have a VGA\n"
		"   1) 256 color,\n"
		"   2) 16 color,\n"
		"   3) mono adaptor,\n"
		"or 4) accelerator\n? ");
	gets(TempS);
	Settings->VGAType = atoi(TempS);
	}
    while (Settings->VGAType < VGA256  &&  Settings->VGAType >= ACCEL);
    Settings->VGAPrompt = VGANames[Settings->VGAType];
    if (Settings->VGAType == ACCEL)
	{
	do  {
	    printf("Which kind of accellerator do you have?\n"
		"   1) S3 accelerator\n"
		"   2) Mach32 accelerator\n"
		"   3) Mach8 accelerator\n"
		"or 4) 8514\n? ");
	    gets(TempS);
	    i = atoi(TempS);
	    }
	while (i < 1  ||  i > 4);
	switch (i)
	    {
	    case 1: strcpy(Settings->ChipSet, "S3"); break;
	    case 2: strcpy(Settings->ChipSet, "Mach32"); break;
	    case 3: strcpy(Settings->ChipSet, "Mach8"); break;
	    case 4: strcpy(Settings->ChipSet, "8514"); break;
	    }
#ifdef S3TESTING
	if (strcmp(Settings->ChipSet, "S3") == 0)
	    {
	    InsertNumber(Settings, 25, MAXCLOCKS);
	    InsertNumber(Settings, 36, MAXCLOCKS);
	    InsertNumber(Settings, 65, MAXCLOCKS);
	    InsertNumber(Settings, 110, MAXCLOCKS);
	    }
#endif
	do  {
	    printf("How much video RAM does your card have?\n"
		"   1) 1MB\n"
		"   2) 2MB\n"
		"   3) 4MB\n?");
	    gets(TempS);
	    i = atoi(TempS);
	    }
	while (i < 1  ||  i > 3);
	switch (i)
	    {
	    case 1: 
		Settings->MemSize=ONE_K; 
		break;
	    case 2: 
		Settings->ScratchRam=ACCELSCRATCH; 
		Settings->MemSize=TWO_K-Settings->ScratchRam; 
		break;
	    case 3: 
		Settings->ScratchRam=ACCELSCRATCH; 
		Settings->MemSize=FOUR_K-Settings->ScratchRam; 
		break;
	    }
	}
}

/*********
*  ReportModeDB--Writes the ModeDB information to OutFile.  If the values
*                are beyond the established refresh range, the data will
*                be written but commented out.
**********/
void ReportModeDB(FILE *OutFile, tModeDB *Timings)
{
    fprintf(OutFile, "# Resolution= %4gx%4g    Refresh rate=%1g Hz\n", 
		Timings->HTicks, Timings->VTicks, Timings->RR);
    fprintf(OutFile, "# Horizontal Refresh= %1g KHz   Memory required= %3gK\n", 
		Timings->HSF/1e3, rint(Timings->HTicks*Timings->VTicks/ONE_K));
    fprintf(OutFile, "#\t\tDot Clock   HTicks  HS1  HS2  HFL\tVTicks VS1  VS2  VFL\n");
    if (Timings->BeyondRange)
	fprintf(OutFile, "#");
    fprintf(OutFile, "\"%gx%g\"\t  %5g     %4g   %4g %4g %4g\t%4g  %4g %4g %4g\n",
		Timings->HTicks, Timings->VTicks, Timings->Clock/1e6, 
		Timings->HTicks, Timings->HS1, Timings->HS2, Timings->HFL,
		Timings->VTicks, Timings->VS1, Timings->VS2, Timings->VFL);
    if (Timings->BeyondRange)
	fprintf(OutFile, "#!!! Note: the horizontal refresh is beyond the established range. !!!\n");
    fprintf(OutFile, "\n");
}

static void VirtualWindow(FILE *OutFile, tXSettings *Settings)
{   int i, Max;
    int VirtualX, VirtualY;

    Max = 0;
    for ( i = 0; i < Settings->ClockCount; i++)
	if (Settings->ModeDB[i].HTicks > Max)
	    Max = Settings->ModeDB[i].HTicks;
    if (Settings->MemSize > 0)
	{
	VirtualX = sqrt(Settings->MemSize*ONE_K/AspectRatio);
	VirtualX -= (VirtualX & 7);	/* Round to multiple of 8 */
	if (Max > VirtualX)
	    VirtualX = Max;
	if (VirtualX > MAXVIRTUALX)
	    VirtualX = MAXVIRTUALX;
	VirtualY = (Settings->MemSize*ONE_K)/VirtualX;
	fprintf(OutFile, "  VideoRam\t%dk\n", Settings->MemSize+Settings->ScratchRam);
	fprintf(OutFile, "  Virtual\t%d %d\n", VirtualX, VirtualY);
	}
}

/*********
*  BuildXconfig--Using the information at hand, this routine will build
*                a temporary (for `X -probeonly' probing) and a full Xconfig.
**********/
void BuildXconfig(tXSettings *Settings)
{   FILE *OutFile;
    int i;

    if ( (OutFile = fopen(XConfigFile, "w")) == NULL)
	{
	perror(XConfigFile);
	exit(errno);
	}
    fprintf(OutFile, "RGBPath\t\t\"/usr/X386/lib/X11/rgb\"\n\n");
    for ( i = 0; i < MAXFONTS; i++)
	if (Settings->Fonts[i])
	    fprintf(OutFile, "FontPath\t\"%s\"\n", FontSources[i]);
    fprintf(OutFile, "\nKeyboard\n  AutoRepeat 500 5\n  ServerNumLock\n" "\n");
    fprintf(OutFile, "%s \"/dev/mouse\"\n", MiceTypes[Settings->MouseType]);
    if (Settings->MouseType == 1)
	fprintf(OutFile, "  SampleRate   150\n");
    if (Settings->Has3Buttons == 'n')
	fprintf(OutFile, "  Emulate3Buttons\n");
    fprintf(OutFile, "\n%s\n", VGANames[Settings->VGAType]);
    fprintf(OutFile, "  ViewPort\t0 0\n");
    if (Settings->ChipSet[0] != 0)
	fprintf(OutFile, "  ChipSet\t\"%s\"\n", Settings->ChipSet);
    if (Settings->ClockCount != 0)
	{
	VirtualWindow(OutFile, Settings);
	fprintf(OutFile, "#Do _not_ rearrange, remove or change these clocks (unless you know what you're doing;)!\n");
	fprintf(OutFile, "  Clocks\t");
	for ( i = 0; i < Settings->ClockCount; i++)
	    fprintf(OutFile, "%2g ", Settings->Clocks[i]);
	fprintf(OutFile, "\n#You may change your displays by reordering or deleting these entries.\n");
	fprintf(OutFile, "  Modes\t\t");
	for ( i = 0; i < Settings->ClockCount; i++)
	    if (Settings->ModeDB[i].Clock != 0  &&  !Settings->ModeDB[i].BeyondRange)
		fprintf(OutFile, "\"%gx%g\" ", Settings->ModeDB[i].HTicks, 
			Settings->ModeDB[i].VTicks);
	fprintf(OutFile, "\n\nModeDB\n");
	for ( i = 0; i < Settings->ClockCount; i++)
	    if (Settings->ModeDB[i].Clock != 0) 
		ReportModeDB(OutFile, &Settings->ModeDB[i]);
	}
    else
	fprintf(OutFile, "  Modes\t \"640x480\"\n\nModeDB\n");
    fclose(OutFile);
}

/*********
*  FilterProbe--Using the initial settings and a temporary Xconfig, a pipe
*               is opened to `X -probeonly' so that its autodetection can
*               used.  This routine parses through the result, looking for
*               specific tags to fill in the remaining fields.
**********/
void FilterProbe(tXSettings *Settings)
{   FILE *InFile;
    const char *TempS, *TempPrompt;
    static char Line[MAXLINE];
    static char Prompt[MAXLINE];
    int i;

#ifdef DEBUG
    if ( (InFile = fopen(PROBESOURCE, "r")) == NULL)
#else
    if ( (InFile = popen(PROBESOURCE, "r")) == NULL)
#endif
	{
	perror(PROBESOURCE);
	exit(errno);
	}
    if (Settings->VGAType < 4)
	TempPrompt = Settings->VGAPrompt;
    else
	TempPrompt = Settings->ChipSet;
    while (fgets(Line, MAXLINE, InFile) != NULL)
	{
	if (strsub(Line, "Warning: The directory \""))
	    {
	    for ( i = 0; i < MAXFONTS; i++)
		if (strstr(Line, FontSources[i]) != NULL)
		    break;
	    if (i < MAXFONTS)
		Settings->Fonts[i] = FALSE;
	    }
	sprintf(Prompt, "(--) %s: available videoram reduced by ", TempPrompt);
	if (strsub(Line, Prompt))
	    Settings->ScratchRam += atoi(Line+strlen(Prompt));
	sprintf(Prompt, "(--) %s: chipset:  ", TempPrompt);
	if (strcmp(TempPrompt, "S3") != 0  &&  strsub(Line, Prompt))
	    {
	    TempS = Line+strlen(Prompt);
	    for ( i = 0; TempS[i] > ' '  &&  TempS[i] <= 'z'; i++)
		Settings->ChipSet[i] = TempS[i];
	    Settings->ChipSet[i] = 0;
	    }
	sprintf(Prompt, "(--) %s: ", TempPrompt);
	if (strsub(Line, Prompt)  &&  strstr(Line, "kb memory available") != NULL)
	    {
	    Settings->MemSize = atoi(Line+strlen(Prompt));
	    if (Settings->VGAType == ACCEL  &&  Settings->MemSize > ONE_K)
		{
		Settings->MemSize -= ACCELSCRATCH;
		Settings->ScratchRam = ACCELSCRATCH;
		}
	    }
	sprintf(Prompt, "(--) %s: videoram: ", TempPrompt);
	if (strsub(Line, Prompt))
	    {
	    Settings->MemSize = atoi(Line+strlen(Prompt));
	    if (Settings->VGAType == ACCEL  &&  Settings->MemSize > ONE_K)
		{
		Settings->MemSize -= ACCELSCRATCH;
		Settings->ScratchRam = ACCELSCRATCH;
		}
	    }
	sprintf(Prompt, "(--) %s: clocks: ", TempPrompt);
	if (strsub(Line, Prompt))
	    {
	    TempS = Line+strlen(Prompt);
	    do  {
		InsertNumber(Settings, atof(TempS), MAXCLOCKS);
		while (*TempS != 0  &&  *TempS == ' ')
		    TempS++;
		while (*TempS != 0  &&  *TempS != ' ')
		    TempS++;
		}
	    while (*TempS != 0);
	    }
	}
#ifdef DEBUG
    fclose(InFile);
#else
    pclose(InFile);
#endif
}

/*********
*  RoundTo8--Converts Val to a number that is always divisible by 8.  This
*            is important for most SVGA cards.
**********/
int RoundTo8(double Val)
{
    return((long)(Val+4)&0xFFF8);
}

/*********
*  CalcClocks--The central algorithm to calculate the ModeDB clocks.  These
*              equations were extracted/derived from Chin Fang's document.
*              The clocks are recalculated iteratively until they within the
*              established refresh range.  The algorithm favors the lower end
*              of that range.
**********/
void CalcClocks(double Clock, tModeDB *Timings)
{   double HSP, VSP;

    Timings->Clock = Clock;
    Timings->HSF = HorizontalRange.Low;
    do 
	{
        HSP = RoundTo8(Clock * HorizontalSyncPulse*1e-6);
        Timings->HFL = RoundTo8(Clock/Timings->HSF);
        Timings->HTicks = Timings->HFL - HSP - (32*2);
        Timings->HS1 = Timings->HTicks + 32;
        Timings->HS2 = Timings->HFL - 32;
        Timings->VTicks = rint(Timings->HTicks * AspectRatio);
        Timings->VFL = rint(Timings->VTicks * 1.05);
        VSP = rint((VerticalSyncPulse*1e-6 * Clock) / Timings->HFL);
        Timings->VS1 = Timings->VTicks + 3;
        Timings->VS2 = Timings->VS1 + VSP;
	Timings->RR = Clock / (Timings->HFL * Timings->VFL);
	if ( Timings->RR > VerticalRange.High )
	    Timings->HSF -= 50;
	if ( Timings->RR < VerticalRange.Low )
	    Timings->HSF += 50;
	}
    while ( Timings->RR < VerticalRange.Low  ||  Timings->RR > VerticalRange.High );
    Timings->BeyondRange = (Timings->HSF > HorizontalRange.High);
}

/*********
*  Main--Process the commandline; fill/initialize the Settings structure;
*        make a temporary Xconfig and call the probe; sort; and make the
*        final Xconfig.
**********/
int main(int Count, char *Args[], char *Env[])
{   static tXSettings Settings;
    int i;

    ProcessCommandLine(Count, Args, Env);
    FillSettings(&Settings);
    BuildXconfig(&Settings);
    fprintf(stderr, "About to call `X -probeonly' to get your card's settings.\n"
	"If you want to stop this, you have 5 seconds to interrupt with <Ctrl>C.\n");
#ifndef DEBUG
    sleep(5);
#endif
    FilterProbe(&Settings);
    for ( i = 0; i < Settings.ClockCount; i++)
	if (Settings.Clocks[i] >= MinimumClock)
	    CalcClocks(Settings.Clocks[i]*1e6, &(Settings.ModeDB[i]));
    SortModeDB(&Settings);
    BuildXconfig(&Settings);
    fprintf(stderr, "All done!!\n\n"
	"NOTE: Sometimes, X -probeonly reports more clocks than are real.\n"
	"Also, not all monitors specify maximum dot clocks, so comment out\n"
	"resolutions which you seem \"too high\".\n"
	"Lastly, if any display does not look stable or `normal',\n"
	"SWITCH TO ANOTHER RESOLUTION or KILL X11 WITH <CTRL><ALT><BACKSPACE>!!\n"
	"Test each resolution, noting which you want to keep.  Watch out for\n"
	"consistant stray lines, zagged lines and smirred images!\n\n");
    return 0;
}
