------------------------------------------------------------------------------
--                                                                          --
--                         GNAT RUNTIME COMPONENTS                          --
--                                                                          --
--                       I N T E R F A C E S . C P P                        --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

with Interfaces.C;

package body Interfaces.CPP is

   use System.Storage_Elements;
   use type Interfaces.C.Short;

   type Vtable_Entry is record
     Delta1 : C.Short := 0;
     Index  : C.Short := 0;
     Pfn    : System.Address := System.Null_Address;
   end record;
   --  The entry in the vtable. This is the most compiler dependant part.

   type Vtable_Entry_Array is array (Positive range <>) of Vtable_Entry;

   subtype Big_Vtable_Entry_Array is Vtable_Entry_Array (Positive);
   --  Dummy type only used to declare Vtable_Ptr which must be a pointer
   --  to a constrained array

   type Vtable is record
      Idepth    : C.Short;
      Unused    : C.Short;
      Ancestors : System.Address;
      --  The first entry in the G++ VTable is unused, we take advantage of
      --  that for inserting our type specific information

      Table     : Big_Vtable_Entry_Array;
   end record;

   type Address_Array is array (C.Short range <>) of System.Address;
   subtype Big_Address_Array is Address_Array (C.Short);
   type Address_Array_Ptr is access all Big_Address_Array;

   function To_Address_Array_Ptr is
     new Unchecked_Conversion (System.Address, Address_Array_Ptr);

   function To_Address is
     new Unchecked_Conversion (Vtable_Ptr, System.Address);

   ---------------------------
   -- Set_Vfunction_Address --
   ---------------------------

   procedure Set_Vfunction_Address
     (Vptr     : Vtable_Ptr;
      Position : Positive;
      Value    : System.Address)
   is
   begin
      Vptr.Table (Position).Pfn := Value;
   end Set_Vfunction_Address;

   ---------------------------
   -- Get_Vfunction_Address --
   ---------------------------

   function Get_Vfunction_Address
     (Vptr     : Vtable_Ptr;
      Position : Positive)
      return     System.Address
   is
   begin
      return Vptr.Table (Position).Pfn;
   end Get_Vfunction_Address;

   ----------------
   -- Set_Idepth --
   ----------------

   procedure Set_Idepth (Vptr  : Vtable_Ptr; Value : Natural) is
   begin
      Vptr.Idepth := C.Short (Value);
   end Set_Idepth;

   ----------------
   -- Get_Idepth --
   ----------------

   function Get_Idepth (Vptr : Vtable_Ptr) return Natural is
   begin
      return Natural (Vptr.Idepth);
   end Get_Idepth;

   ------------------------
   -- Set_Ancestor_Vptrs --
   ------------------------

   procedure Set_Ancestor_Vptrs (Vptr : Vtable_Ptr; Value : System.Address) is
   begin
      Vptr.Ancestors := Value;
   end Set_Ancestor_Vptrs;

   ------------------------
   -- Get_Ancestor_Vptrs --
   ------------------------

   function  Get_Ancestor_Vptrs (Vptr : Vtable_Ptr) return System.Address is
   begin
      return Vptr.Ancestors;
   end Get_Ancestor_Vptrs;

   --------------------
   -- Displaced_This --
   --------------------

   function Displaced_This
    (Current_This : System.Address;
     Vptr         : Vtable_Ptr;
     Position     : Positive)
     return         System.Address
   is
   begin
      return Current_This + Storage_Offset (Vptr.Table (Position).Delta1);
   end Displaced_This;

   -----------------
   -- Vtable_Size --
   -----------------

   function Vtable_Size (Entry_Count : Natural) return Storage_Count is

      type VT is record
         Idepth    : C.Short;
         Unused    : C.Short;
         Ancestors : System.Address;
         Table     : Vtable_Entry_Array (1 .. Entry_Count);
      end record;
      --  Dummy declaration, just to get the size

   begin
      return (VT'Size + System.Storage_Unit - 1) / System.Storage_Unit;
   end Vtable_Size;

   --------------------
   -- Inherit_Vtable --
   --------------------

   procedure Inherit_Vtable
    (Old_Vptr    : Vtable_Ptr;
     New_Vptr    : Vtable_Ptr;
     Entry_Count : Natural)
   is
   begin
      --  Inherit Virtual functions

      New_Vptr.Table (1 .. Entry_Count) := Old_Vptr.Table (1 .. Entry_Count);

      --  The inheritance depth is incremented

      New_Vptr.Idepth := Old_Vptr.Idepth + 1;

      --  The Ancestor Vtable ptr Table is also inherited (with a shift)

      To_Address_Array_Ptr (New_Vptr.Ancestors) (1 .. New_Vptr.Idepth)
        := To_Address_Array_Ptr (Old_Vptr.Ancestors) (0 .. Old_Vptr.Idepth);

      To_Address_Array_Ptr (New_Vptr.Ancestors) (0) := To_Address (New_Vptr);
   end Inherit_Vtable;


   --------------------
   -- CPP_Membership --
   --------------------

   function CPP_Membership
     (Obj_Vptr : Vtable_Ptr;
      Typ_Vptr : Vtable_Ptr)
      return     Boolean
   is
      Pos : constant C.Short := Obj_Vptr.Idepth - Typ_Vptr.Idepth;

   begin
      return Pos >= 0
        and then To_Address_Array_Ptr (Obj_Vptr.Ancestors) (Pos) =
                 To_Address (Typ_Vptr);
   end CPP_Membership;

end Interfaces.CPP;
