------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               F R E E Z E                                --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.5 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Types; use Types;

package Freeze is

   --------------------------
   -- Handling of Freezing --
   --------------------------

   --  In the formal Ada semantics, freezing of entities occurs at a well
   --  defined point, described in (RM 13.14). The model in GNAT of freezing
   --  is that a Freeze_Entity node is generated at the point where an entity
   --  is frozen, and the entity contains a pointer (Freeze_Node) to this
   --  generated freeze node.

   --  The freeze node is processed in the expander to generate associated
   --  data and subprograms (e.g. an initialization procedure) which must
   --  be delayed until the type is frozen and its representation can be
   --  fully determined. Subsequently the freeze node is used by Gigi to
   --  determine the point at which it should elaborate the corresponding
   --  entity (this elaboration also requires the representation of the
   --  entity to be fully determinable). The freeze node is also used to
   --  provide additional diagnostic information (pinpointing the freeze
   --  point), when order of freezing errors are detected.

   --  If we were fully faithful to the Ada model, we would generate freeze
   --  nodes for all entities, but that is a bit heavy so we optimize (that
   --  is the nice word) or cut corners (which is a bit more honest). For
   --  many entities, we do not need to delay the freeze and instead can
   --  freeze them at the point of declaration. The conditions for this
   --  early freezing being permissible are as follows:

   --    There is no associated expander activity that needs to be delayed

   --    Gigi can fully elaborate the entity at the point of occurrence (or,
   --    equivalently, no real elaboration is required for the entity).

   --  In order for these conditions to be met (especially the second), it
   --  must be the case that all representation characteristics of the entity
   --  can be determined at declaration time.

   --  The following indicates how freezing is handled for all entity kinds:

   --    Types

   --      All declared types have freeze nodes, as well as anonymous base
   --      types created for type declarations where the defining identifier
   --      is a first subtype of the anonymous type.

   --    Subtypes

   --      All first subtypes have freeze nodes. Other subtypes need freeze
   --      nodes if the corresponding base type has not yet been frozen. If
   --      the base type has been frozen, then there is no need for a freeze
   --      node, since no rep clauses can appear for the subtype in any case.

   --    Implicit types and subtypes

   --      As noted above, implicit base types always have freeze nodes. Other
   --      implicit types and subtypes typically do not require freeze nodes,
   --      because there is no possibility of delaying any information about
   --      their representation.

   --    Subprograms
   --
   --      Are frozen at the point of declaration unless one or more of the
   --      formal types or return type themselves have delayed freezing and
   --      are not yet frozen. This includes the case of a formal access type
   --      where the designated type is not frozen. Note that we are talking
   --      about subprogram specs here (subprogram body entities have no
   --      relevance), and in any case, subprogram bodies freeze everything.

   --    Objects with dynamic address clauses
   --
   --      These have a delayed freeze. Gigi will generate code to evaluate
   --      the initialization expression if present and store it in a temp.
   --      The actual object is created at the point of the freeze, and if
   --      neccessary initialized by copying the value of this temporary.

   --    Formal Parameters
   --
   --      Are frozen when the associated subprogram is frozen, so there is
   --      never any need for them to have delayed freezing.

   --    Other Objects
   --
   --      Are always frozen at the point of declaration

   --    All Other Entities

   --      Are always frozen at the point of declaration

   --  The flag Has_Delayed_Freeze is used for to indicate that delayed
   --  freezing is required. Usually the associated freeze node is allocated
   --  at the freezing point. One special exception occurs with anonymous
   --  base types, where the freeze node is preallocated at the point of
   --  declaration, so that the First_Subtype_Link field can be set.

   -----------------
   -- Subprograms --
   -----------------

   procedure Check_Compile_Time_Size (T : Entity_Id);
   --  Check whether the size of a type is known to the back-end, and set
   --  the entity flag accordingly. Note that the size can be knonw even if
   --  the context is not static in the Ada sense. The flag serves to determine
   --  whether the secondary stack must be used to return a value of the type.

   function Freeze_Entity (E : Entity_Id; Loc : Source_Ptr) return List_Id;
   --  Freeze an entity, and return Freeze nodes, to be inserted at the
   --  point of call. Loc is a source location which corresponds to the
   --  freeze point. This is used in placing warning messages in the
   --  situation where it appears that a type has been frozen too early,
   --  e.g. when a primitive operation is declared after the freezing
   --  point of its tagged type.

   procedure Freeze_All (From : Entity_Id; After : in out Node_Id);
   --  Before a non-instance body, or at the end of a declarative part
   --  freeze all entities therein that are not yet frozen. Calls itself
   --  recursively to catch types in inner packages that were not frozen
   --  at the inner level because they were not yet completely defined.
   --  This routine also analyzes and freezes default parameter expressions
   --  in subprogram specifications (this has to be delayed until all the
   --  types are frozen). The resulting freeze nodes are inserted just
   --  after node After (which is a list node) and analyzed. On return,
   --  'After' is updated to point to the last node inserted (or is returned
   --  unchanged if no nodes were inserted). 'From' is the last entity frozen
   --  in the scope. It is used to prevent a quadratic traversal over already
   --  frozen entities.

   procedure Freeze_Before (N : Node_Id; T : Entity_Id);
   --  Freeze T then Insert the generated Freeze nodes before the node N.

   procedure Freeze_Expression (N : Node_Id);
   --  Freezes the required entities when the Expression N causes freezing.

end Freeze;
