------------------------------------------------------------------------------
--                                                                          --
--                         GNAT RUNTIME COMPONENTS                          --
--                                                                          --
--                A D A . T E X T _ I O . F L O A T _ A U X                 --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.7 $                              --
--                                                                          --
--        Copyright (c) 1992,1993,1994, 1995 NYU, All Rights Reserved       --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Text_IO.Generic_Aux; use Ada.Text_IO.Generic_Aux;

with System.Img_Real;  use System.Img_Real;
with System.Val_Real;  use System.Val_Real;

package body Ada.Text_IO.Float_Aux is

   ---------
   -- Get --
   ---------

   procedure Get
     (File  : in File_Type;
      Item  : out Long_Long_Float;
      Width : in Field)
   is
      Buf  : String (1 .. Field'Last);
      Stop : Integer := 0;
      Ptr  : aliased Integer := 1;

   begin
      if Width /= 0 then
         Load_Width (File, Width, Buf, Stop);
         String_Skip (Buf, Ptr);
      else
         Load_Real (File, Buf, Stop);
      end if;

      Item := Scan_Real (Buf, Ptr'Access, Stop);

      Check_End_Of_Field (File, Buf, Stop, Ptr, Width);
   end Get;

   ----------
   -- Gets --
   ----------

   procedure Gets
     (From : in String;
      Item : out Long_Long_Float;
      Last : out Positive)
   is
      Pos : aliased Integer;

   begin
      String_Skip (From, Pos);
      Item := Scan_Real (From, Pos'Access, From'Last);
      Last := Pos - 1;

   exception
      when Constraint_Error =>
         Last := Pos - 1;
         raise Data_Error;
   end Gets;

   ---------------
   -- Load_Real --
   ---------------

   procedure Load_Real
     (File : in File_Type;
      Buf  : out String;
      Ptr  : in out Natural)
   is
      Loaded   : Boolean;

   begin
      --  Skip initial blanks, and load possible sign

      Load_Skip (File);
      Load (File, Buf, Ptr, '+', '-');

      --  Case of .nnnn

      Load (File, Buf, Ptr, '.', Loaded);

      if Loaded then
         Load_Digits (File, Buf, Ptr, Loaded);

         --  Hopeless junk if no digits loaded

         if not Loaded then
            return;
         end if;

      --  Otherwise must have digits to start

      else
         Load_Digits (File, Buf, Ptr, Loaded);

         --  Hopeless junk if no digits loaded

         if not Loaded then
            return;
         end if;

         --  Based cases

         Load (File, Buf, Ptr, '#', ':', Loaded);

         if Loaded then

            --  Case of nnn#.xxx#

            Load (File, Buf, Ptr, '.', Loaded);

            if Loaded then
               Load_Extended_Digits (File, Buf, Ptr);

            --  Case of nnn#xxx.[xxx]# or nnn#xxx#

            else
               Load_Extended_Digits (File, Buf, Ptr);
               Load (File, Buf, Ptr, '.');
               Load_Extended_Digits (File, Buf, Ptr);

               --  As usual, it seems strange to allow mixed base characters,
               --  but that is what ACVC tests expect, see CE3804M, case (3).

               Load (File, Buf, Ptr, '#', ':');
            end if;

         --  Case of nnn.[nnn] or nnn

         else
            Load (File, Buf, Ptr, '.', Loaded);

            if Loaded then
               Load_Digits (File, Buf, Ptr);
            end if;
         end if;
      end if;

      --  Deal with exponent

      Load (File, Buf, Ptr, 'E', 'e', Loaded);

      if Loaded then
         Load (File, Buf, Ptr, '+', '-');
         Load_Digits (File, Buf, Ptr);
      end if;
   end Load_Real;

   ---------
   -- Put --
   ---------

   procedure Put
     (File : in File_Type;
      Item : in Long_Long_Float;
      Fore : in Field;
      Aft  : in Field;
      Exp  : in Field)
   is
      Buf : String (1 .. Field'Last);
      Ptr : Natural := 0;

   begin
      Set_Image_Real (Item, Buf, Ptr, Fore, Aft, Exp);
      Put_Item (File, Buf (1 .. Ptr));
   end Put;

   ----------
   -- Puts --
   ----------

   procedure Puts
     (To   : out String;
      Item : in Long_Long_Float;
      Aft  : in Field;
      Exp  : in Field)
   is
      Buf  : String (1 .. Field'Last);
      Fore : Integer;
      Ptr  : Natural := 0;

   begin
      if Exp = 0 then
         Fore := To'Length - 1 - Aft;
      else
         Fore := To'Length - 2 - Aft - Exp;
      end if;

      if Fore < 1 then
         raise Layout_Error;
      end if;

      Set_Image_Real (Item, Buf, Ptr, Fore, Aft, Exp);

      if Ptr > To'Length then
         raise Layout_Error;
      else
         To := Buf (1 .. Ptr);
      end if;
   end Puts;

end Ada.Text_IO.Float_Aux;
