------------------------------------------------------------------------------
--                                                                          --
--                         GNAT RUNTIME COMPONENTS                          --
--                                                                          --
--              A D A . T E X T _ I O . D E C I M A L _ A U X               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.7 $                              --
--                                                                          --
--        Copyright (c) 1992,1993,1994,1995 NYU, All Rights Reserved        --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Text_IO.Generic_Aux;   use Ada.Text_IO.Generic_Aux;
with Ada.Text_IO.Float_Aux;     use Ada.Text_IO.Float_Aux;

with System.Img_Decimal; use System.Img_Decimal;
with System.Img_LLD;     use System.Img_LLD;
with System.Val_Dec;     use System.Val_Dec;
with System.Val_LLD;     use System.Val_LLD;

package body Ada.Text_IO.Decimal_Aux is

   -------------
   -- Get_Dec --
   -------------

   function Get_Dec
     (File   : in File_Type;
      Width  : in Field;
      Scale  : Integer)
      return   Integer
   is
      Buf  : String (1 .. Field'Last);
      Ptr  : aliased Integer := 0;
      Stop : Integer := 0;
      Item : Integer;

   begin
      if Width /= 0 then
         Load_Width (File, Width, Buf, Stop);
         String_Skip (Buf, Ptr);
      else
         Load_Real (File, Buf, Stop);
      end if;

      Item := Scan_Decimal (Buf, Ptr'Access, Stop, Scale);
      Check_End_Of_Field (File, Buf, Stop, Ptr, Width);
      return Item;
   end Get_Dec;

   -------------
   -- Get_LLD --
   -------------

   function Get_LLD
     (File   : in File_Type;
      Width  : in Field;
      Scale  : Integer)
      return   Long_Long_Integer
   is
      Buf  : String (1 .. Field'Last);
      Ptr  : aliased Integer := 0;
      Stop : Integer := 0;
      Item : Long_Long_Integer;

   begin
      if Width /= 0 then
         Load_Width (File, Width, Buf, Stop);
         String_Skip (Buf, Ptr);
      else
         Load_Real (File, Buf, Stop);
      end if;

      Item := Scan_Long_Long_Decimal (Buf, Ptr'Access, Stop, Scale);
      Check_End_Of_Field (File, Buf, Stop, Ptr, Width);
      return Item;
   end Get_LLD;

   -------------
   -- Put_Dec --
   -------------

   procedure Put_Dec
     (File  : in File_Type;
      Item  : in Integer;
      Fore  : in Field;
      Aft   : in Field;
      Exp   : in Field;
      Scale : Integer)
   is
      Buf : String (1 .. Field'Last);
      Ptr : Natural := 0;

   begin
      Set_Image_Decimal (Item, Buf, Ptr, Fore, Aft, Exp, Scale);
      Put_Item (File, Buf (1 .. Ptr));
   end Put_Dec;

   -------------
   -- Put_LLD --
   -------------

   procedure Put_LLD
     (File  : in File_Type;
      Item  : in Long_Long_Integer;
      Fore  : in Field;
      Aft   : in Field;
      Exp   : in Field;
      Scale : Integer)
   is
      Buf : String (1 .. Field'Last);
      Ptr : Natural := 0;

   begin
      Set_Image_Long_Long_Decimal (Item, Buf, Ptr, Fore, Aft, Exp, Scale);
      Put_Item (File, Buf (1 .. Ptr));
   end Put_LLD;

   --------------
   -- Gets_Dec --
   --------------

   function Gets_Dec
     (From  : in String;
      Last  : access Positive;
      Scale : Integer)
      return  Integer
   is
      Pos  : aliased Integer;
      Item : Integer;

   begin
      String_Skip (From, Pos);
      Item := Scan_Decimal (From, Pos'Access, From'Last, Scale);
      Last.all := Pos - 1;
      return Item;

   exception
      when Constraint_Error =>
         Last.all := Pos - 1;
         raise Data_Error;
   end Gets_Dec;

   --------------
   -- Gets_LLD --
   --------------

   function Gets_LLD
     (From  : in String;
      Last  : access Positive;
      Scale : Integer)
      return  Long_Long_Integer
   is
      Pos  : aliased Integer := From'First;
      Item : Long_Long_Integer;

   begin
      String_Skip (From, Pos);
      Item := Scan_Long_Long_Decimal (From, Pos'Access, From'Last, Scale);
      Last.all := Pos - 1;
      return Item;

   exception
      when Constraint_Error =>
         Last.all := Pos - 1;
         raise Data_Error;
   end Gets_LLD;

   --------------
   -- Puts_Dec --
   --------------

   procedure Puts_Dec
     (To    : out String;
      Item  : in Integer;
      Aft   : in Field;
      Exp   : in Field;
      Scale : Integer)
   is
      Buf  : String (1 .. Field'Last);
      Fore : Integer;
      Ptr  : Natural := 0;

   begin
      if Exp = 0 then
         Fore := To'Length - 1 - Aft;
      else
         Fore := To'Length - 2 - Aft - Exp;
      end if;

      if Fore < 1 then
         raise Layout_Error;
      end if;

      Set_Image_Decimal (Item, Buf, Ptr, Fore, Aft, Exp, Scale);

      if Ptr > To'Length then
         raise Layout_Error;
      else
         To := Buf (1 .. Ptr);
      end if;
   end Puts_Dec;

   --------------
   -- Puts_Dec --
   --------------

   procedure Puts_LLD
     (To    : out String;
      Item  : in Long_Long_Integer;
      Aft   : in Field;
      Exp   : in Field;
      Scale : Integer)
   is
      Buf  : String (1 .. Field'Last);
      Fore : Integer;
      Ptr  : Natural := 0;

   begin
      if Exp = 0 then
         Fore := To'Length - 1 - Aft;
      else
         Fore := To'Length - 2 - Aft - Exp;
      end if;

      if Fore < 1 then
         raise Layout_Error;
      end if;

      Set_Image_Long_Long_Decimal (Item, Buf, Ptr, Fore, Aft, Exp, Scale);

      if Ptr > To'Length then
         raise Layout_Error;
      else
         To := Buf (1 .. Ptr);
      end if;
   end Puts_LLD;

end Ada.Text_IO.Decimal_Aux;
