/* look.c -- find lines in a sorted list
 * Created: Mon Mar  8 23:16:36 1993 by faith@cs.unc.edu
 * Revised: Wed Mar 10 14:18:27 1993 by faith@cs.unc.edu
 * Copyright 1993 Rickard E. Faith (faith@cs.unc.edu)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 1, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * $Log$
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <getopt.h>

static int dictionary_flag = 0;
static int fold_flag = 0;
static int debug_flag = 0;

static char *default_wordlist   = "/usr/dict/words";
static char *alternate_wordlist = "/usr/dict/web2";

static void usage( void )
{
   fprintf( stderr, "usage: look [-dfa] string [file]\n" );
   exit( 1 );
}

static void getword( FILE *str, char *word, int length )
{
   int count = 0;
   int c;

   while ((c = getc( str )) != EOF && c != '\n' && count < length) {
      *word++ = c;
      ++count;
   }
   *word = '\0';
}

static int compare( char *a, char *b, int length_of_b )
{
   int i;
   int A;
   int B;

   for (i = 0; i < length_of_b; i++) {
      if (!*a) return -1;	/* a is shorter than b */
      if (dictionary_flag) {
	 while (*a && !isalnum(*a) && *a != '\t' && *a != ' ') a++;
	 while (*b && !isalnum(*b) && *b != '\t' && *b != ' ') {
	    b++;
	    length_of_b--;
	 }
      }
      if (fold_flag) {
	 A = isupper( *a ) ? tolower( *a ) : *a;
	 B = isupper( *b ) ? tolower( *b ) : *b;
	 if (A < B) return -1;
	 if (A > B) return 1;
      } else {
	 if (*a < *b) return -1;
	 if (*a > *b) return 1;
      }
      ++a;
      ++b;
   }
   return 0;
}

static void search( char *filename, char *target )
{
   FILE      *str;
   long int  left = 0;
   long int  mid;
   long int  right;
   int       c;
   const int LEN = 255;
   char      word[LEN];
   int       target_length = strlen( target );


   if (!(str = fopen( filename, "r"))) {
      fprintf( stderr, "look: cannot open \"%s\" for read\n", filename );
      perror( "look" );
      exit( 1 );
   }

   fseek( str, 0, SEEK_END );
   right = ftell( str );

   if (debug_flag) fprintf( stderr, "end = %ld\n", right );

   mid = right / 2;

   for (;;) {
      if (debug_flag) fprintf( stderr, "seeking %ld\n", mid );
      fseek( str, mid, SEEK_SET );
      while ((c = getc( str )) != EOF && c != '\n')
	    ;
      getword( str, word, LEN );
      if (debug_flag) fprintf( stderr, "Found \"%s\" (%ld,%ld,%ld)\n",
			       word,
			       left,
			       mid,
			       right );
      if (compare( word, target, target_length ) < 0) {
	 left = mid;
	 mid = (left + right) / 2;
	 if (left == mid) break;
      } else {
	 right = mid;
	 mid = (left + right) / 2;
	 if (right == mid) break;
      }
   }

   if (!compare( word, target, target_length))
	 printf( "%s\n", word );
   for (;;) {
      getword( str, word, LEN );
      if (compare( word, target, target_length )) return;
      printf( "%s\n", word );
   }
}

int main( int argc, char **argv )
{
   int  c;
   char *filename = NULL;
   char *target;

   while ((c = getopt( argc, argv, "dft:aD" )) != EOF)
	 switch (c) {
	 case 'd':
	    ++dictionary_flag;
	    break;
	 case 'f':
	    ++fold_flag;
	    break;
	 case 'a':
	    filename = alternate_wordlist;
	    ++dictionary_flag;
	    ++fold_flag;
	    break;
	 case 'D':
	    ++debug_flag;
	    break;
	 default:
	    usage();
	    break;
	 }

   switch (argc - optind) {
   case 1:
      target = argv[optind];
      if (!filename) {
	 filename = default_wordlist;
	 ++dictionary_flag;
	 ++fold_flag;
      }
      break;
   case 2:
      target = argv[optind];
      filename = argv[optind + 1];
      break;
   default:
      usage();
      break;
   }

   if (debug_flag) {
      fprintf( stderr, "Searching for \"%s\" in \"%s\"\n", target, filename );
      if (dictionary_flag) fprintf( stderr, "Dictionary order\n" );
      if (fold_flag)       fprintf( stderr, "Fold\n" );
   }

   search( filename, target );
   return 0;
}
