------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                          G N A T . O S _ L I B                           --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.24 $                             --
--                                                                          --
--        Copyright (c) 1992,1993,1994,1995 NYU, All Rights Reserved        --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

--  This package types and procedures for interfacing to the underlying OS.
--  It is used by the GNAT compiler and by tools associated with the GNAT
--  compiler, and therefore works for the various OS-s to which GNAT has
--  been ported.  This package will undoubtedly grow as new services are
--  needed by various tools.

--  This package tends to use fairly low-level Ada in order to not bring
--  in large portions of the RTL.  For example, functions return access
--  to string as part of avoiding functions returning unconstrained types;
--  types related to dates are defined here instead of using the types
--  from Calendar, since use of Calendar forces linking in of tasking code.

with System; use System;

package GNAT.OS_Lib is

   type String_Access is access String;

   ---------------------
   -- Time/Date Stuff --
   ---------------------

   --  The OS's notion of time is represented by the private type OS_Time.
   --  This is the type returned by the File_Time_Stamp functions to obtain
   --  the file stam of a specified file. Functions and a procedure (modeled
   --  after the similar subprograms in package Calendar) are provided for
   --  extracting information from a value of this type. Although these are
   --  called GM, the intention is not that they provide GMT times in all
   --  cases but rather the actual (time-zone independent) time stamp of the
   --  file (of course in Unix systems, this *is* in GMT form).

   type OS_Time is private;

   subtype Year_Type   is Integer range 1900 .. 2099;
   subtype Month_Type  is Integer range    1 ..   12;
   subtype Day_Type    is Integer range    1 ..   31;
   subtype Hour_Type   is Integer range    0 ..   23;
   subtype Minute_Type is Integer range    0 ..   59;
   subtype Second_Type is Integer range    0 ..   59;

   function GM_Year    (Date : OS_Time) return Year_Type;
   function GM_Month   (Date : OS_Time) return Month_Type;
   function GM_Day     (Date : OS_Time) return Day_Type;
   function GM_Hour    (Date : OS_Time) return Hour_Type;
   function GM_Minute  (Date : OS_Time) return Minute_Type;
   function GM_Second  (Date : OS_Time) return Second_Type;

   procedure GM_Split
     (Date    : OS_Time;
      Year    : out Year_Type;
      Month   : out Month_Type;
      Day     : out Day_Type;
      Hour    : out Hour_Type;
      Minute  : out Minute_Type;
      Second  : out Second_Type);

   ----------------
   -- File Stuff --
   ----------------

   --  These routines give access to the open/creat/close/read/write level
   --  of I/O routines in the typical C library (these functions are not
   --  part of the ANSI C standard, but are typically available in all
   --  systems). See also package Interfaces.C_Streams for access to the
   --  stream level routines.

   type File_Descriptor is private;
   --  Corresponds to the int file handle values used in the C routines,

   Standin  : constant File_Descriptor;
   Standout : constant File_Descriptor;
   Standerr : constant File_Descriptor;
   --  File descriptors for standard input output files

   Invalid_FD : constant File_Descriptor;
   --  File descriptor returned when error in opening/creating file;

   type Mode is (Binary, Text);
   for Mode'Size use Integer'Size;
   for Mode use (Binary => 0, Text => 1);
   --  Used in all the Open and Create calls to specify if the file is to be
   --  opened in binary mode or text mode. In systems like Unix, this has no
   --  effect, but in systems capable of text mode translation, the use of
   --  Text as the mode parameter causes the system to do CR/LF translation
   --  and also to recognize the DOS end of file character on input. The use
   --  of Text where appropriate allows programs to take a portable Unix view
   --  of DOs-format files and process them appropriately.

   function Open_Read (Name : Address; Fmode : Mode) return File_Descriptor;
   pragma Import (C, Open_Read, "open_read");
   --  Open file Name (NUL-terminated) for reading, returning file descriptor
   --  File descriptor returned is Invalid_FD if file cannot be opened.

   function Create_File (Name : Address; Fmode : Mode) return File_Descriptor;
   pragma Import (C, Create_File, "open_create");
   --  Creates new file with given name (NUL-terminated) for writing, returning
   --  file descriptor for subsequent use in Write calls. File descriptor
   --  returned is Invalid_FD if file cannot be successfully created.

   function Create_New_File
     (Name  : Address;
      Fmode : Mode)
      return  File_Descriptor;
   pragma Import (C, Create_New_File, "open_new");
   --  Create new file with given name (NUL-terminated) for writing,
   --  returning file descriptor for subsequent use in Write calls.  This
   --  differs from Create_File in that it fails if the file already exists.
   --  File descriptor returned is Invalid_FD if the file exists or cannot
   --  be created.

   Temp_File_Len : constant Integer := 12;
   --  Length of name returned by Create_Temp_File call (GNAT-XXXXXX & NUL)

   subtype Temp_File_Name is String (1 .. Temp_File_Len);
   --  String subtype set by Create_Temp_File

   procedure Create_Temp_File
     (FD   : out File_Descriptor;
      Name : out Temp_File_Name);
   --  Create and open for writing a temporary file.  The name of the
   --  file and the File Descriptor are returned.  The File Descriptor
   --  returned is Invalid_FD in the case of failure. No mode parameter
   --  is provided. Since this is a temporary file, there is no point in
   --  doing text translation on it.

   procedure Close (FD : File_Descriptor);
   pragma Import (C, Close, "close");
   --  Close file referenced by FD

   function Read
     (FD : File_Descriptor;
      A  : Address;
      N  : Integer)
   return Integer;
   pragma Import (C, Read, "read");
   --  Read N bytes to address A from file referenced by FD. Returned value
   --  is count of bytes actually read, which can be less than N at EOF.

   procedure Write (FD : File_Descriptor; A : Address; N : Integer);
   pragma Import (C, Write, "write");
   --  Write N bytes from address A to file referenced by FD

   function File_Length (FD : File_Descriptor) return Long_Integer;
   pragma Import (C, File_Length, "file_length");
   --  Get length of file from descriptor

   function File_Time_Stamp (Name : String) return OS_Time;
   --  Get time stamp of file from name (use for unopened file)

   function File_Time_Stamp (FD : File_Descriptor) return OS_Time;
   --  Get time stamp of file from descriptor

   function Is_Regular_File (Name : String) return Boolean;
   --  The named file exists and is a regular file

   function Is_Directory (Name : String) return Boolean;
   --  The named file exists and is a directory

   function Locate_Regular_File
     (File_Name : String;
      Path      : String)
      return      String_Access;
   --  Try to locate a regular file whose name is given by File_Name in the
   --  directories listed in Path.  If a file is found, its full pathname is
   --  returned; otherwise, a null pointer is returned. If the File_Name given
   --  is an absolute pathname, then Locate_Regular_File just checks that the
   --  file exists and is a regular file.  Otherwise, the Path argument is
   --  parsed according to OS conventions, and for each directory in the Path
   --  a check is made if File_Name is a relative pathname of a regular file
   --  from that directory.
   --
   --  This is most often used for finding executables.

   ------------------
   -- Subprocesses --
   ------------------

   type Argument_List is array (Integer range <>) of String_Access;

   procedure Spawn
     (Program_Name : String;
      Args         : Argument_List;
      Success      : out Boolean);
   --  The first parameter of function Spawn is the FULL PATHNAME of
   --  the executable.  The second parameter contains the arguments
   --  to be passed to the program (in typical argc/argv form).

   -------------------
   -- Miscellaneous --
   -------------------

   function Getenv (Name : String) return String_Access;
   --  Get the value of the environment variable. Returns the empty
   --  string if the environment variable does not exist.
   --  doesn't exist

   procedure OS_Exit  (Status : Integer);
   pragma Import (C, OS_Exit, "exit");
   --  Exit to OS with given status code (program is terminated)

   procedure OS_Abort;
   pragma Import (C, OS_Abort, "abort");
   --  Exit to OS signalling an abort (traceback or other appropriate
   --  diagnostic information should be given if possible, or entry made
   --  to the debugger if that is possible).

   Directory_Separator : constant Character;
   --  The character that is used to separate parts of a pathname.

   Path_Separator      : constant Character;
   --  The character to separate paths in an environment variable value.

private
   Dirsep_Char : Character;
   pragma Import (C, Dirsep_Char, "dirsep_char");
   Directory_Separator : constant Character := Dirsep_Char;

   Pathsep_Char : Character;
   pragma Import (C, Pathsep_Char, "pathsep_char");
   Path_Separator      : constant Character := Pathsep_Char;

   type OS_Time is new Long_Integer;

   type File_Descriptor is new Integer;

   Standin    : constant File_Descriptor :=  0;
   Standout   : constant File_Descriptor :=  1;
   Standerr   : constant File_Descriptor :=  2;
   Invalid_FD : constant File_Descriptor := -1;

end GNAT.OS_Lib;
