/*
 *  Project   : tin - a Usenet reader
 *  Module    : post.c
 *  Author    : I.Lea
 *  Created   : 01-04-91
 *  Updated   : 24-09-93
 *  Notes     : mail/post/replyto/followup/crosspost & cancel articles
 *  Copyright : (c) Copyright 1991-93 by Iain Lea
 *              You may  freely  copy or  redistribute  this software,
 *              so  long as there is no profit made from its use, sale
 *              trade or  reproduction.  You may not change this copy-
 *              right notice, and it must be included in any copy made
 */

#include	"tin.h"

#define	PRINT_LF()	Raw (FALSE); fputc ('\n', stdout); fflush (stdout); Raw (TRUE);

extern char note_h_distrib[PATH_LEN];		/* Distribution: */
extern char note_h_followup[LEN];			/* Followup-To: */
extern char note_h_messageid[PATH_LEN];		/* Message-ID:	*/
extern char note_h_references[PATH_LEN];	/* References:	*/
extern char note_h_newsgroups[LEN];			/* Newsgroups:	*/
extern char note_h_subj[LEN];				/* Subject:	*/
extern char note_h_date[PATH_LEN];			/* Date:	*/
extern FILE *note_fp;						/* the body of the current article */
extern long note_mark[MAX_PAGES];			/* ftells on beginnings of pages */

int unlink_article = TRUE;
static int reread_active_for_posted_arts = FALSE;
struct t_posted *posted;


int 
user_posted_messages ()
{
	char buf[LEN];
	FILE *fp;
	int i, j, k;
	int no_of_lines = 0;

	if ((fp = fopen (postfile, "r")) == NULL) {
		clear_message ();
		return FALSE;
	} else {
		while (fgets (buf, sizeof (buf), fp) != NULL) {
			no_of_lines++;
		}
		if (! no_of_lines) {
			fclose (fp);
			info_message (txt_no_arts_posted);
			return FALSE;
		}
		rewind (fp);
		posted = (struct t_posted *) my_malloc ((unsigned) (no_of_lines+1) * sizeof (struct t_posted));
		for (i=0 ; fgets (buf, sizeof (buf), fp) != NULL ; i++) {
			for (j=0 ; buf[j] != '|' && buf[j] != '\n' ; j++) {
				posted[i].date[j] = buf[j];		/* posted date */
			}
			if (buf[j] == '\n') {	
				error_message ("Corrupted file %s", postfile);
				sleep (1);
				fclose (fp);
				clear_message ();
				return FALSE;
			}
			posted[i].date[j++] = '\0';
			posted[i].action = buf[j];
			j += 2;
			for (k=j,j=0 ; buf[k] != '|' && buf[k] != ',' ; k++, j++) {
				if (j < sizeof (posted[i].group)) {
					posted[i].group[j] = buf[k];
				}
			}
			if (buf[k] == ',') {
				while (buf[k] != '|' && buf[k] != '\n') {
					k++;
				}
				posted[i].group[j++] = ',';
				posted[i].group[j++] = '.';
				posted[i].group[j++] = '.';
				posted[i].group[j++] = '.';
			}
			posted[i].group[j++] = '\0';
			k++;
			for (j=k,k=0 ; buf[j] != '\n' ; j++, k++) {
				if (k < sizeof (posted[i].subj)) {
					posted[i].subj[k] = buf[j];
				}
			}
			posted[i].subj[k++] = '\0';
		}
		fclose (fp);

		show_info_page (POST_INFO, (char **) 0, txt_post_history_menu);
		if (posted != (struct t_posted *) 0) {
			free ((char *) posted);
			posted = (struct t_posted *) 0;
		}
		return TRUE;
	}
}


void 
update_art_posted_file (group, action, subj)
	char *group;
	int action;
	char *subj;
{
	char buf[LEN];
	char tmp_post[LEN];
	FILE *fp, *tmp_fp;
	long epoch;
	struct tm *tm;

	sprintf (tmp_post, "%s.%d", postfile, process_id);

	if ((tmp_fp = fopen (tmp_post, "w")) != NULL) {
		time (&epoch);
		tm = localtime (&epoch);
		fprintf (tmp_fp, "%02d-%02d-%02d|%c|%s|%s\n",
			tm->tm_mday, tm->tm_mon+1, tm->tm_year,
			action, group, subj);
		fclose (tmp_fp);
	}

	if ((tmp_fp = fopen (tmp_post, "a+")) != NULL) {
		if ((fp = fopen (postfile, "r")) != NULL) {
			while (fgets (buf, sizeof buf, fp) != NULL) {
				fprintf (tmp_fp, "%s", buf);
			}	
			fclose (fp);
			rename_file (tmp_post, postfile);
		}
		fclose (tmp_fp);
	}
}

/*
 * Check the article file for correct header syntax and if there 
 * is a blank between the header information and the text.
 *
 * 1. Subject header present
 * 2. Newsgroups header present
 * 3. Space after every colon in header
 * 4. Colon in every header line
 * 5. Newsgroups line has no spaces, only comma separated
 * 6. List of newsgroups is presented to user with description
 * 7. Lines in body that are to long causes a warning to be printed
 * 8. Group(s) must be listed in the active file
 * 9. No From: header allowed (limit accidental forging) and
 *    rejection by inn servers
 */

int 
check_article_to_be_posted (article)
	char *article;
{
	char *ngptrs[NGLIMIT];
	char line[LEN], *cp, *cp2;
	FILE *fp;
	int cnt = 0;
	int col, len, i, n;
	int end_of_header = FALSE;
	int errors = 0;
	int found_newsgroups_line = FALSE;
	int found_subject_line = FALSE;
	int init = TRUE;
	int ngcnt = 0;
	int nglens[NGLIMIT];
	int oldraw;	/* save previous raw state */

	if ((fp = fopen (article, "r")) == (FILE *) 0) {
		perror_message (txt_cannot_open, article);
		return FALSE;
	}

	oldraw = RawState();	/* save state */

	while (fgets (line, sizeof (line), fp) != NULL) {
		cnt++;
		len= strlen (line);
		if (len > 0) {
			if (line[len - 1] == '\n') {
				line[--len]= 0;
			}
		}
		if ((cnt == 1) && (len == 0)) {
			setup_check_article_screen (&init);
			fprintf (stderr, txt_error_header_line_blank);
			fflush (stderr);
			errors++;
			end_of_header = TRUE;
			break;
		}
		if ((len == 0) && (cnt >= 2)) {
			end_of_header = TRUE;
			break;
		}
		/* 
		 * ignore continuation lines - they start with white space 
		 */
		if ((line[0] == ' ' || line[0] == '\t') && (cnt != 1)) {
			continue;
		}
		cp = strchr (line, ':');
		if (cp == (char *) 0) {
			setup_check_article_screen (&init);
			fprintf (stderr, txt_error_header_line_colon, cnt, line);
			fflush (stderr);
			errors++;
			continue;
		}
		if (cp[1] != ' ') {
			setup_check_article_screen (&init);
			fprintf (stderr, txt_error_header_line_space, cnt, line);
			fflush (stderr);
			errors++;
		}
		if (cp - line == 7 && ! strncmp (line, "Subject", 7)) {
			found_subject_line = TRUE;
		}
		if (cp - line == 4 && ! strncmp (line, "From", 4)) {
			fprintf (stderr, txt_error_from_in_header_not_allowed, cnt);
			fflush (stderr);
			errors++;
		}
		if (cp - line == 10 && ! strncmp (line, "Newsgroups", 10)) {
			found_newsgroups_line = TRUE;
			for (cp = line + 11; *cp == ' '; cp++) {
				;
			}
			if (strchr (cp, ' ')) {
				setup_check_article_screen (&init);
				fprintf (stderr, txt_error_header_line_comma);
				fflush (stderr);
				errors++;
				continue;
			}
			while (*cp) {
				if (! (cp2 = strchr (cp, ','))) {
					cp2 = cp + strlen (cp);
				} else {
					*cp2++ = '\0';
				}
				if (ngcnt < NGLIMIT) {
					nglens[ngcnt] = strlen (cp);
					ngptrs[ngcnt] = my_malloc (nglens[ngcnt]+1);
					if (! ngptrs[ngcnt]) {
						Raw (oldraw);
						return (TRUE);
					}
					strcpy (ngptrs[ngcnt], cp);
					ngcnt++;
				}
				cp = cp2;
			}
			if (! ngcnt) {
				setup_check_article_screen (&init);
				fprintf (stderr, txt_error_header_line_empty_newsgroups);
				fflush (stderr);
				errors++;
				continue;
			}
		}
	}

	if (! found_subject_line) {
		setup_check_article_screen (&init);
		fprintf (stderr, txt_error_header_line_missing_subject);
		fflush (stderr);
		errors++;
	}

	if (! found_newsgroups_line) {
		setup_check_article_screen (&init);
		fprintf (stderr, txt_error_header_line_missing_newsgroups);
		fflush (stderr);
		errors++;
	}

	/* 
	 * Check the body of the article for long lines 
	 */
	while (fgets (line, sizeof (line), fp)) {
		cnt++;
		cp = strrchr (line, '\n');
		if (cp != (char *) 0) {
			*cp = '\0';
		}
		col = 0;
		for (cp = line; *cp; cp++) {
			if (*cp == '\t') {
				col += 8 - (col%8);
			} else {
				col++;
			}
		}
		if (col > MAX_COL) {
			setup_check_article_screen (&init);
			fprintf (stderr, txt_warn_art_line_too_long, MAX_COL, cnt, line);
			fflush (stderr);
			break;
		}
	}
	if (! end_of_header) {
		setup_check_article_screen (&init);
		fprintf (stderr, txt_error_header_and_body_not_seperate);
		fflush (stderr);
		errors++;
	}

	if (ngcnt && errors == 0) {
		/* 
		 * Print a note about each newsgroup 
		 */
		setup_check_article_screen (&init);
		fprintf (stderr, txt_art_newsgroups, ngcnt == 1? "" : "s");
		fflush (stderr);
		for (i = 0; i < ngcnt; i++) {
			n = find_group_index (ngptrs[i]);
			if (n != -1) {
				fprintf (stderr, "  %s\t%s\n", ngptrs[i],
					 (active[n].description ? 
					 active[n].description : ""));
				fflush (stderr);
			} else {
#ifdef HAVE_FACIST_NEWSADMIN
				fprintf (stderr, txt_error_not_valid_newsgroup, ngptrs[i]);
				errors++;
#else
				fprintf (stderr, txt_warn_not_valid_newsgroup, ngptrs[i]);
#endif
				fflush (stderr);
			}
			free (ngptrs[i]);
		}
	}
	fclose (fp);

	Raw (oldraw);	/* restore raw/unraw state */

	return (errors ? FALSE : TRUE);
}


void 
setup_check_article_screen (init)
	int *init;
{
	if (*init) {
		ClearScreen ();
		center_line (0, TRUE, txt_check_article);
		MoveCursor (INDEX_TOP, 0);
		Raw(FALSE);
		*init = FALSE;
	}
}

/*
 *  Quick post an article (not a followup)
 */

void 
quick_post_article ()
{
	FILE *fp;
	char ch, *ptr;
	char ch_default = 'p';
	char group[PATH_LEN];
	char subj[PATH_LEN];
	char buf[LEN], tmp[LEN];
	int i, done = FALSE;

	if (! can_post) {
		info_message (txt_cannot_post);
		return;
	}

	/*
	 * Don't allow if not active news feed 
	 */
	if (! spooldir_is_active) {
		info_message (txt_not_active_newsfeed);
		return;
	}
	
	setup_screen ();
	InitScreen();

	/*
	 * Get groupname & subject for posting article.
	 * If multiple newsgroups test all to see if any are moderated.
	 */
	sprintf (buf, txt_post_newsgroups, default_post_newsgroups);
	
	if (! prompt_string (buf, group)) {
		fprintf (stderr, "%s\n", txt_no_quick_newsgroups);
		return;
	}

	if (strlen (group)) {
		my_strncpy (default_post_newsgroups, group,
			sizeof (default_post_newsgroups));
	} else {
		if (default_post_newsgroups[0]) {
			my_strncpy (group, default_post_newsgroups, sizeof (group));
		} else {
			fprintf (stderr, "%s\n", txt_no_quick_newsgroups);
			return;
		}
	}

	/*
	 * Check if any of the newsgroups are moderated.
	 */
	strcpy (tmp, group);
	while (! done) {
		strcpy (buf, tmp);
		ptr = buf;
		ptr = strchr(buf, ',');
		if (ptr != (char *) 0) {
			strcpy (tmp, ptr+1);
			*ptr = '\0';
		} else {
			done = TRUE;	
		}
		i = find_group_index (buf);

		if (debug == 2) {
			sprintf (msg, "Group=[%s] index=[%d]", buf, i);
			wait_message (msg);
		}
		
		if (i == -1) {
			Raw(FALSE);
			fprintf (stderr, "\nGroup %s not found in active file. Exiting...\n", buf);
			return;
		}
		if (active[i].moderated == 'm') {
			sprintf (msg, txt_group_is_moderated, buf);
			if (! prompt_yn (cLINES, msg, 'y')) {
				Raw(FALSE);
				fprintf (stderr, "\nExiting...\n");
				return;
			}
		}
	}

	PRINT_LF();
	sprintf (buf, txt_post_subject, default_post_subject);
	
	if (! prompt_string (buf, subj)) {
		Raw (FALSE);
		fprintf (stderr, "%s\n", txt_no_quick_subject);
		return;
	}

	if (strlen (subj)) {
		my_strncpy (default_post_subject, subj,
			sizeof (default_post_subject));
	} else {
		if (default_post_subject[0]) {
			my_strncpy (subj, default_post_subject, sizeof (subj));
		} else {
			Raw (FALSE);
			fprintf (stderr, "%s\n", txt_no_quick_subject);
			return;
		}
	}
	
	PRINT_LF();
	start_line_offset = 6;

	if ((fp = fopen (article, "w")) == NULL) {
		Raw (FALSE);
		perror_message (txt_cannot_open, article);
		return;
	}
	chmod (article, 0600);

/* FIXME so that group only contains 1 group when finding an index number */
i = find_group_index (group);

	fprintf (fp, "Subject: %s\n", subj);
	fprintf (fp, "Newsgroups: %s\n", group);
	if (i >= 0 && active[i].attribute.organization != (char *) 0) {
		fprintf (fp, "Organization: %s\n", active[i].attribute.organization);
		start_line_offset++;
	}
	if (*reply_to) {
		fprintf (fp, "Reply-To: %s\n", reply_to);
		start_line_offset++;
	}
	if (i >= 0 && active[i].attribute.followup_to != (char *) 0) {
		fprintf (fp, "Followup-To: %s\n", active[i].attribute.followup_to);
		start_line_offset++;
	}
	if (*my_distribution) {
		fprintf (fp, "Distribution: %s\n", my_distribution);
		start_line_offset++;
	}
	fprintf (fp, "Summary: \n");
	fprintf (fp, "Keywords: \n\n\n");

	add_signature (fp, FALSE);
	fclose (fp);

	ch = 'e';
	while (1) {
		switch (ch) {
		case 'e':
			invoke_editor (article, start_line_offset);
			while (! check_article_to_be_posted (article)) {
				do {
					sprintf (msg, "%s%c", txt_bad_article, 'e');
					wait_message (msg);
					MoveCursor (cLINES, (int) strlen (txt_bad_article));
					if ((ch = (char) ReadCh ()) == CR)
						ch = 'e';
				} while (! strchr ("eq\033", ch));
				if (ch == 'e') {
					invoke_editor (article, start_line_offset);
				} else {
					break;
				}
			}
			if (ch == 'e') {
				break;
			}

		case 'q':
		case ESC:
			if (unlink_article)
				unlink (article);
			clear_message ();
			return;

		case 'i':
			invoke_ispell (article);
			break;

		case 'p':
			wait_message (txt_posting);
			if (submit_file (article)) {
				Raw (FALSE);
				info_message (txt_art_posted);
				reread_active_for_posted_arts = TRUE;		
				goto post_article_done;
			} else {
				rename_file (article, dead_article);
				Raw (FALSE);
				error_message (txt_art_rejected, dead_article);
				return;
			}
		}

		do {
			sprintf (msg, "%s%c", txt_quit_edit_post, ch_default);
			wait_message (msg);
			MoveCursor (cLINES, (int) strlen (txt_quit_edit_post));
			if ((ch = (char) ReadCh ()) == CR)
				ch = ch_default;
		} while (! strchr ("eipq\033", ch));
	}

post_article_done:
	find_mail_header (HEADER_NEWSGROUPS, article, group);
	find_mail_header (HEADER_SUBJECT, article, subj);

	if (unlink_article) {
		unlink (article);
	}
	update_art_posted_file (group, 'w', subj);

	my_strncpy (default_post_newsgroups, group, sizeof (default_post_newsgroups));
	my_strncpy (default_post_subject, subj, sizeof (default_post_subject));

	write_rcfile ();
	
	return;
}


/*
 *  Post an original article (not a followup)
 */

int 
post_article (group, posted)
	char *group;
	int *posted;
{
	FILE *fp;
	char ch;
	char ch_default = 'p';
	char subj[LEN];
	char buf[LEN];
	int i, redraw_screen = FALSE;

	if (! can_post) {
		info_message (txt_cannot_post);
		return (redraw_screen);
	}

	/*
	 * Don't allow if not active news feed 
	 */
	if (! spooldir_is_active) {
		info_message (txt_not_active_newsfeed);
		return (redraw_screen);
	}
	
	*posted = FALSE;
	start_line_offset = 6;

	if (active[my_group[cur_groupnum]].moderated == 'm') {
		sprintf (msg, txt_group_is_moderated, group);
		if (! prompt_yn (cLINES, msg, 'y')) {
			clear_message ();
			return (redraw_screen);
		}
	}

	sprintf (msg, txt_post_subject, default_post_subject);
	
	if (! prompt_string (msg, subj)) {
		clear_message ();
		return (redraw_screen);
	}

	if (strlen (subj)) {
		my_strncpy (default_post_subject, subj,
			sizeof (default_post_subject));
	} else {
		if (default_post_subject[0]) {
			my_strncpy (subj, default_post_subject, sizeof (subj));
		} else {
			info_message (txt_no_subject);
			return (redraw_screen);
		}
	}
	
	wait_message (txt_post_an_article);

	if ((fp = fopen (article, "w")) == NULL) {
		perror_message (txt_cannot_open, article);
		return (redraw_screen);
	}
	chmod (article, 0600);

	i = find_group_index (group);

	fprintf (fp, "Subject: %s\n", subj);
	fprintf (fp, "Newsgroups: %s\n", group);
	if (i >= 0 && active[i].attribute.organization != (char *) 0) {
		fprintf (fp, "Organization: %s\n", active[i].attribute.organization);
		start_line_offset++;
	}
	if (*reply_to) {
		fprintf (fp, "Reply-To: %s\n", reply_to);
		start_line_offset++;
	}
	if (i >= 0 && active[i].attribute.followup_to != (char *) 0) {
		fprintf (fp, "Followup-To: %s\n", active[i].attribute.followup_to);
		start_line_offset++;
	}
	if (*my_distribution) {
		fprintf (fp, "Distribution: %s\n", my_distribution);
		start_line_offset++;
	}
	fprintf (fp, "Summary: \n");
	fprintf (fp, "Keywords: \n\n\n");

	add_signature (fp, FALSE);
	fclose (fp);

	ch = 'e';
	while (1) {
		switch (ch) {
		case 'e':
			invoke_editor (article, start_line_offset);
			while (! check_article_to_be_posted (article)) {
				do {
					sprintf (msg, "%s%c", txt_bad_article, 'e');
					wait_message (msg);
					MoveCursor (cLINES, (int) strlen (txt_bad_article));
					if ((ch = (char) ReadCh ()) == CR)
						ch = 'e';
				} while (! strchr ("eq\033", ch));
				if (ch == 'e') {
					invoke_editor (article, start_line_offset);
				} else {
					break;
				}
			}
			redraw_screen = TRUE;
			if (ch == 'e') {
				break;
			}

		case 'q':
		case ESC:
			if (unlink_article)
				unlink (article);
			clear_message ();
			return (redraw_screen);

		case 'i':
			invoke_ispell (article);
			break;

		case 'p':
			wait_message (txt_posting);
			if (submit_file (article)) {
				info_message (txt_art_posted);
				*posted = TRUE;
				reread_active_for_posted_arts = TRUE;		
				goto post_article_done;
			} else {
				rename_file (article, dead_article);
				sprintf (buf, txt_art_rejected, dead_article);
				info_message (buf);
				sleep (3);
				return (redraw_screen);
			}
		}

		do {
			sprintf (msg, "%s%c", txt_quit_edit_post, ch_default);
			wait_message (msg);
			MoveCursor (cLINES, (int) strlen (txt_quit_edit_post));
			if ((ch = (char) ReadCh ()) == CR)
				ch = ch_default;
		} while (! strchr ("eipq\033", ch));
	}

post_article_done:
	find_mail_header (HEADER_SUBJECT, article, subj);

	if (unlink_article) {
		unlink (article);
	}
	update_art_posted_file (group, 'w', subj);

	my_strncpy (default_post_newsgroups, group, sizeof (default_post_newsgroups));
	my_strncpy (default_post_subject, subj, sizeof (default_post_subject));

	return (redraw_screen);
}


int 
post_response (group, respnum, copy_text)
	char *group;
	int respnum;
	int copy_text;
{
	FILE *fp;
	char ch, *ptr;
	char ch_default = 'p';
	char buf[LEN];
	int i;
	int ret_code = POSTED_NONE;
	
	/*
	 * Don't allow if not active news feed 
	 */
	if (! spooldir_is_active) {
		info_message (txt_not_active_newsfeed);
		return (ret_code);
	}

	start_line_offset = 4;

	wait_message (txt_post_a_followup);
	
	if (*note_h_followup && strcmp (note_h_followup, "poster") == 0) {
		clear_message ();
		if (! prompt_yn (cLINES, txt_resp_to_poster, 'y')) {
			return (ret_code);
		}
		*note_h_followup = '\0';
		find_reply_to_addr (respnum, buf);
		mail_to_someone (respnum, buf, TRUE, FALSE, &ret_code);
		return (ret_code);
	} else if (*note_h_followup && strcmp (note_h_followup, group) != 0) {
		MoveCursor (cLINES/2, 0);
		CleartoEOS ();
		center_line ((cLINES/2)+2, TRUE, txt_resp_redirect);
		MoveCursor ((cLINES/2)+4, 0);

		fputs ("    ", stdout);
		ptr = note_h_followup;
		while (*ptr) {
			if (*ptr != ',') {
				fputc (*ptr, stdout);
			} else {
				fputs ("\r\n    ", stdout);
			}
			ptr++;
		}
		fflush (stdout);

		if (! prompt_yn (cLINES, txt_continue, 'y')) {
			return (ret_code);
		}
	}

	if ((fp = fopen (article, "w")) == NULL) {
		perror_message (txt_cannot_open, article);
		return (ret_code);
	}
	chmod (article, 0600);

	i = find_group_index (group);

	fprintf (fp, "Subject: Re: %s\n", eat_re (note_h_subj));

	if (*note_h_followup && strcmp (note_h_followup, "poster") != 0) {
		fprintf (fp, "Newsgroups: %s\n", note_h_followup);
	} else {
		fprintf (fp, "Newsgroups: %s\n", note_h_newsgroups);
		if (i >= 0 && active[i].attribute.followup_to != (char *) 0) {
			fprintf (fp, "Followup-To: %s\n", 
				active[i].attribute.followup_to);
			start_line_offset++;
		} else {
			ptr = (char *) strchr (note_h_newsgroups, ',');
			if (ptr) {
				fprintf (fp, "Followup-To: %s\n", 
					note_h_newsgroups);
				start_line_offset++;
			}
		}
	}

	/*
	 * Append to References: line if its already there
	 */
	if (note_h_references[0]) {
		fprintf (fp, "References: %s %s\n", note_h_references, note_h_messageid);
	} else {
		fprintf (fp, "References: %s\n", note_h_messageid);
	}

	if (i >= 0 && active[i].attribute.organization != (char *) 0) {
		fprintf (fp, "Organization: %s\n", active[i].attribute.organization);
		start_line_offset++;
	}
	if (*reply_to) {
		fprintf (fp, "Reply-To: %s\n", reply_to);
		start_line_offset++;
	}
	if (*note_h_distrib) {
		fprintf (fp, "Distribution: %s\n", note_h_distrib);
	} else {
		fprintf (fp, "Distribution: %s\n", my_distribution);
	}
	start_line_offset++;
	fprintf (fp, "\n");

	if (copy_text) {
		if (strfquote (group, respnum, buf, sizeof (buf), news_quote_format)) {
			fprintf (fp, "%s\n", buf);
		}
		fseek (note_fp, note_mark[0], 0);
		copy_fp (note_fp, fp, quote_chars);
	}

	add_signature (fp, FALSE);
	fclose (fp);

	ch = 'e';
	while (1) {
		switch (ch) {
		case 'e':
			invoke_editor (article, start_line_offset);
			while (! check_article_to_be_posted (article)) {
				do {
					sprintf (msg, "%s%c", txt_bad_article, 'e');
					wait_message (msg);
					MoveCursor (cLINES, (int) strlen (txt_bad_article));
					if ((ch = (char) ReadCh ()) == CR)
						ch = 'e';
				} while (! strchr ("eq\033", ch));
				if (ch == 'e') {
					invoke_editor (article, start_line_offset);
				} else {
					break;
				}
			}
			ret_code = POSTED_REDRAW;
			if (ch == 'e') {
				break;
			}

		case 'q':
		case ESC:
			if (unlink_article)
				unlink (article);
			clear_message ();
			return (ret_code);

		case 'i':
			invoke_ispell (article);
			ret_code = POSTED_REDRAW;
			break;
			
		case 'p':
			wait_message (txt_posting);
			if (submit_file (article)) {
				info_message (txt_art_posted);
				ret_code = POSTED_OK;
				reread_active_for_posted_arts = TRUE;		
				goto post_response_done;
			} else {
				rename_file (article, dead_article);
				sprintf (buf, txt_art_rejected, dead_article);
				info_message (buf);
				sleep (3);
				return (ret_code);
			}
		}

		do {
			sprintf (msg, "%s%c", txt_quit_edit_post, ch_default);
			wait_message (msg);
			MoveCursor(cLINES, (int) strlen (txt_quit_edit_post));
			if ((ch = (char) ReadCh()) == CR)
				ch = ch_default;
		} while (! strchr ("eipq\033", ch));
	}

post_response_done:
	if (*note_h_followup && strcmp(note_h_followup, "poster") != 0) {
		find_mail_header (HEADER_SUBJECT, article, buf);
		update_art_posted_file (note_h_followup, 'f', buf);
	} else {
		find_mail_header (HEADER_SUBJECT, article, buf);
		update_art_posted_file (note_h_newsgroups, 'f', buf);
		my_strncpy (default_post_newsgroups, note_h_newsgroups, 
			sizeof (default_post_newsgroups));
	}

	my_strncpy (default_post_subject, buf, sizeof (default_post_subject));

	if (unlink_article) {
		unlink (article);
	}
	
	return (ret_code);
}


int 
mail_to_someone (respnum, address, mail_to_poster, confirm_to_mail, mailed_ok)
	int respnum;
	char *address;
	int mail_to_poster;
	int confirm_to_mail;
	int *mailed_ok;
{
	char nam[100];
	char ch = 's';
	char ch_default = 's';
	char buf[LEN];
	char mail_to[LEN];
	FILE *fp;
	int redraw_screen = FALSE;

	start_line_offset = 4;
	
	strcpy (mail_to, address);
	clear_message ();
	
	joinpath (nam, homedir, ".letter");
	if ((fp = fopen (nam, "w")) == NULL) {
		perror_message (txt_cannot_open, nam);
		return (redraw_screen);
	}
	chmod (nam, 0600);

	fprintf (fp, "To: %s\n", mail_to);

	if (mail_to_poster) {
		fprintf (fp, "Subject: Re: %s\n", eat_re (note_h_subj));
	} else {
		fprintf (fp, "Subject: (fwd) %s\n", note_h_subj);
	}
	
	if (*note_h_followup) {
		fprintf (fp, "Newsgroups: %s\n\n", note_h_followup);
	} else {
		fprintf (fp, "Newsgroups: %s\n", note_h_newsgroups);
	}
	if (*default_organization) {
		fprintf (fp, "Organization: %s\n", default_organization);
		start_line_offset++;
	}
	if (*reply_to) {
		fprintf (fp, "Reply-To: %s\n", reply_to);
		start_line_offset++;
	}
	fputc ('\n', fp);
	
	if (mail_to_poster) {
		ch = 'e';
		if (strfquote (active[my_group[cur_groupnum]].name, 
		    respnum, buf, sizeof (buf), mail_quote_format)) {
			fprintf (fp, "%s\n", buf);
		}
		fseek (note_fp, note_mark[0], 0);
		copy_fp (note_fp, fp, quote_chars);
	} else {
		fseek (note_fp, 0L, 0);
		copy_fp (note_fp, fp, "");
	}
	
	add_signature (fp, TRUE);
	fclose (fp);
	
	while (1) {
		if (confirm_to_mail) {
			do {
				sprintf (msg, "%s [%.*s]: %c", txt_quit_edit_ispell_send, 
					cCOLS-36, note_h_subj, ch_default);
				wait_message (msg);
				MoveCursor (cLINES, (int) (strlen (msg)-1));
				if ((ch = (char) ReadCh ()) == CR)
					ch = ch_default;
			} while (! strchr ("eiqs\033", ch));
		}
		switch (ch) {
			case 'e':
				invoke_editor (nam, start_line_offset);
				redraw_screen = TRUE;
				break;

			case 'i':
				invoke_ispell (nam);
				break;

			case 'q':
			case ESC:
				unlink (nam);
				clear_message ();
				*mailed_ok = FALSE;
				return (redraw_screen);

			case 's':
				/*
				 *  Open letter and get the To: line in case 
				 *  they changed it with the editor
				 */
				find_mail_header (HEADER_TO, nam, mail_to);
				sprintf (msg, txt_mailing_to, mail_to);
				wait_message (msg);
#ifdef M_UNIX
				sprintf (buf, "%s \"%s\" < %s", mailer, mail_to, nam);
#else				
				sprintf (buf, mailer, nam, userid, mail_to);
#endif
				if (invoke_cmd (buf)) {
					goto mail_to_someone_done;
				} else {
					error_message (txt_command_failed_s, buf);
					*mailed_ok = FALSE;
					break;
				}
		}
		if (mail_to_poster) {
			do {
				sprintf (msg, "%s [Re: %.*s]: %c", txt_quit_edit_send, 
					cCOLS-36, eat_re (note_h_subj), ch_default);
				wait_message (msg);
				MoveCursor (cLINES, (int) (strlen (msg)-1));
				if ((ch = (char) ReadCh ()) == CR)
					ch = ch_default;
			} while (! strchr ("eqs\033", ch));
		}	
	}

mail_to_someone_done:
	unlink (nam);
	*mailed_ok = TRUE;
	return (redraw_screen);
}


int 
mail_bug_report ()
{
	char buf[LEN], nam[100];
	char *gateway = (char *) 0;
	char *domain = (char *) 0;
	char ch, ch_default = 's';
	char mail_to[PATH_LEN];
	FILE *fp;
	FILE *fp_uname;
	int is_debug = FALSE;
	int is_longfiles = FALSE;
	int is_nntp = FALSE;
	int is_nntp_only = FALSE;
	int uname_ok = FALSE;

	start_line_offset = 5;
	
	wait_message (txt_mail_bug_report);
	
	joinpath (nam, homedir, ".bugreport");
	if ((fp = fopen (nam, "w")) == NULL) {
		perror_message (txt_cannot_open, nam);
		return FALSE;
	}
	chmod(nam, 0600);

	fprintf (fp, "To: %s%s\n", bug_addr, add_addr);
	fprintf (fp, "Subject: BUG REPORT tin %s PL%s %s\n", VERSION, PATCHLEVEL, OS);
	if (*default_organization) {
		fprintf (fp, "Organization: %s\n", default_organization);
		start_line_offset++;
	}
	if (*reply_to) {
		fprintf (fp, "Reply-To: %s\n", reply_to);
		start_line_offset++;
	}

#ifdef HAVE_UNAME
	if ((fp_uname = (FILE *) popen ("uname -a", "r")) != NULL) {
		while (fgets (buf, sizeof (buf), fp_uname) != NULL) {
			fprintf (fp, "\nBOX1: %s", buf);
			start_line_offset += 2;
			uname_ok = TRUE;
		}
		pclose (fp_uname);
	}
#endif	/* HAVE_UNAME */

	if (! uname_ok) {
		fprintf (fp, "\nPlease enter the following information:\n");
		fprintf (fp, "BOX1: Machine+OS:\n");
	}
#ifdef HAVE_LONG_FILENAMES
	is_longfiles = TRUE;
#endif
#ifdef NNTP_ABLE
	is_nntp = TRUE;
#endif
#ifdef NNTP_ONLY
	is_nntp_only = TRUE;
#endif
#ifdef DEBUG
	is_debug = TRUE;
#endif
#ifdef NNTP_INEWS_GATEWAY
	gateway = NNTP_INEWS_GATEWAY;
#endif
#ifdef NNTP_INEWS_DOMAIN
	domain = NNTP_INEWS_DOMAIN;
#endif
	fprintf (fp, "\nCFG1: active=%d  arts=%d  reread=%d  longfilenames=%d  setuid=%d\n",
		DEFAULT_ACTIVE_NUM, 
		DEFAULT_ARTICLE_NUM, 
		reread_active_file_secs,
		is_longfiles, 
		(tin_uid == real_uid ? 0 : 1));
	fprintf (fp, "CFG2: nntp=%d  nntp_only=%d  nntp_xindex=%d  nntp_xover=%d\n",
		is_nntp,
		is_nntp_only,  
		xindex_supported, 
		xover_supported);
	fprintf (fp, "CFG3: debug=%d gateway=[%s] domain=[%s]\n",
		is_debug,
		(gateway ? gateway : ""),
		(domain ? domain : ""));

	start_line_offset += 3;
	
	fprintf (fp, "\nPlease enter bug report/gripe/comment:\n");

	add_signature (fp, TRUE);
	fclose (fp);
	
	ch = 'e';
	while (1) {
		switch (ch) {
			case 'e':
				invoke_editor (nam, start_line_offset);
				break;

			case 'i':
				invoke_ispell (nam);
				break;

			case 'q':
			case ESC:
				unlink (nam);
				clear_message ();
				return TRUE;

			case 's':
				sprintf (msg, txt_mail_bug_report_confirm, bug_addr, add_addr);
				if (prompt_yn (cLINES, msg, 'n')) {
					strcpy (mail_to, bug_addr);
					find_mail_header (HEADER_TO, nam, mail_to);
					sprintf (msg, txt_mailing_to, mail_to);
					wait_message (msg);
#ifdef M_UNIX
					sprintf (buf, "%s \"%s\" < %s", mailer, mail_to, nam);
#else
					sprintf (buf, mailer, nam, userid, mail_to);
#endif
					if (invoke_cmd (buf)) {
						sprintf (msg, txt_mailed, 1);
						info_message (msg);
						goto mail_bug_report_done;
					} else {
						error_message (txt_command_failed_s, buf);
						break;
					}
				} else {
					goto mail_bug_report_done;
				}
		}
		do {
			sprintf (msg, "%s: %c", txt_quit_edit_ispell_send, ch_default);
			wait_message (msg);
			MoveCursor (cLINES, (int) strlen (msg)-1);
			if ((ch = (char) ReadCh ()) == CR)
				ch = ch_default;
		} while (! strchr ("eiqs\033", ch));
	}

mail_bug_report_done:
	unlink (nam);

	return TRUE;
}


int 
mail_to_author (group, respnum, copy_text)
	char *group;
	int respnum;
	int copy_text;
{
	char buf[LEN];
	char from_addr[LEN];
	char nam[100];
	char mail_to[LEN];
	char ch, ch_default = 's';
	FILE *fp;
	int redraw_screen = FALSE;

	start_line_offset = 5;
	
	wait_message (txt_reply_to_author);

	joinpath (nam, homedir, ".letter");
	if ((fp = fopen (nam, "w")) == NULL) {
		perror_message (txt_cannot_open, nam);
		return (redraw_screen);
	}
	chmod (nam, 0600);

	find_reply_to_addr (respnum, from_addr);

	fprintf (fp, "To: %s\n", from_addr);
	fprintf (fp, "Subject: Re: %s\n", eat_re(note_h_subj) );
	if (auto_cc) {
		fprintf (fp, "Cc: %s\n", userid);
	}
	fprintf (fp, "Newsgroups: %s\n", note_h_newsgroups);
	if (*default_organization) {
		fprintf (fp, "Organization: %s\n", default_organization);
		start_line_offset++;
	}
	if (*reply_to) {
		fprintf (fp, "Reply-To: %s\n", reply_to);
		start_line_offset++;
	}
	fputc ('\n', fp);

	if (copy_text) {
		if (strfquote (group, respnum, buf, sizeof (buf), mail_quote_format)) {
			fprintf (fp, "%s\n", buf);
		}
		fseek (note_fp, note_mark[0], 0);
		copy_fp (note_fp, fp, quote_chars);
	}

	add_signature (fp, TRUE);
	fclose (fp);

	ch = 'e';
	while (1) {
		switch (ch) {
		case 'e':
			invoke_editor (nam, start_line_offset);
			redraw_screen = TRUE;
			break;

		case 'q':
		case ESC:
			unlink (nam);
			clear_message ();
			return (redraw_screen);

		case 's':
			my_strncpy (mail_to, arts[respnum].from, sizeof (mail_to));
			find_mail_header (HEADER_TO, nam, mail_to);
			sprintf (msg, txt_mailing_to, mail_to);
			wait_message (msg);
			insert_x_headers (nam);
#ifdef M_UNIX
			sprintf (buf, "%s \"%s\" < %s", mailer, mail_to, nam);
#else
			sprintf (buf, mailer, nam, userid, mail_to);
#endif
			if (invoke_cmd (buf)) {
				sprintf (msg, txt_mailed, 1);
				info_message (msg);
				goto mail_to_author_done;
			} else {
				error_message (txt_command_failed_s, buf);
				break;
			}
		}

		do {
			sprintf (msg, "%s: %c", txt_quit_edit_send, ch_default);
			wait_message (msg);
			MoveCursor (cLINES, (int) strlen (msg)-1);
			if ((ch = (char) ReadCh ()) == CR)
				ch = ch_default;
		} while (! strchr ("eqs\033", ch));
	}

mail_to_author_done:
	find_mail_header (HEADER_SUBJECT, nam, buf);
	unlink (nam);
	update_art_posted_file (group, 'r', buf);

	return (redraw_screen);
}

/*
 *  Read a file grabbing the value of the specified mail header line
 */

void 
find_mail_header (header, file, value)
	int header;
	char *file;
	char *value;
{
	FILE *fp;
	char buf[LEN];
	char buf2[LEN];
	char new_value[LEN];
	char *p;
	int found = FALSE;
	int was_to = FALSE;

	*new_value = '\0';

	if ((fp = fopen (file, "r")) == NULL) {
		perror_message (txt_cannot_open, file);
		return;
	}

	while (!found && fgets (buf, sizeof (buf), fp) != NULL) {
		for (p = buf; *p && *p != '\n'; p++)
			continue;
		*p = '\0';

		if (*buf == '\0')
			break;

		switch (header) {
			case HEADER_TO:
				if (strncmp (buf, "To: ", 4) == 0 ||
				    strncmp (buf, "Cc: ", 4) == 0) {
					my_strncpy (buf2, &buf[4], sizeof (buf2));
					yank_to_addr (buf2, new_value);
					was_to = TRUE;
				} else if (*buf == ' ' || *buf == '\t' && was_to) {
					yank_to_addr (buf, new_value);
				} else {
					was_to = FALSE;
				}
				break;
			case HEADER_NEWSGROUPS:
				if (match_string (buf, "Newsgroups: ", new_value, sizeof (new_value))) {
					found = TRUE;
				}
				break;
			case HEADER_SUBJECT:
				if (strncmp (buf, "Subject: ", 9) == 0) {
					my_strncpy (new_value, &buf[9], sizeof (new_value));
					found = TRUE;
				}
				break;
		}
	}

	fclose (fp);

	if (new_value[0] == ' ') {
		strcpy (value, &new_value[1]);
	} else {
		strcpy (value, new_value);
	}
}


int 
delete_article (group, respnum)
	char *group;
	int respnum;
{
	char ch, ch_default = 'd';
	char buf[LEN];
	char delete[PATH_LEN];
	char host_name[PATH_LEN];
	char user_name[128];
	char full_name[128];
	char from[PATH_LEN];
	FILE *fp;
	int i;
	int redraw_screen = FALSE;

	/*
	 * Don't allow if not active news feed 
	 */
	if (! spooldir_is_active) {
		info_message (txt_not_active_newsfeed);
		return (redraw_screen);
	}

	/*
	 * Check if news / mail group
	 */
	i = find_group_index (group);
	if (i == -1) {
		clear_message ();
		return (redraw_screen); 
	}
	if (active[i].type == GROUP_TYPE_MAIL) {
		make_group_path (group, buf);
		sprintf (delete, "%s/%s/%d", active[i].spooldir, buf, respnum);
		sprintf (buf, "Delete article [%s]? (y/n): ", delete);
		if (prompt_yn (cLINES, buf, 'y')) {
			unlink (delete);
			return TRUE;
		} else {
			return (redraw_screen); 
		}
	}
		 
	start_line_offset = 4;

	get_host_name (host_name);
	get_user_info (user_name, full_name);
	get_from_name (user_name, host_name, full_name, delete);

	if (arts[respnum].from != arts[respnum].name) {
		sprintf (from, "%s (%s)", arts[respnum].from, arts[respnum].name);
	} else {
		my_strncpy (from, arts[respnum].from, sizeof (from));
	}

	if (debug == 2) {
		sprintf (msg, "From=[%s]  Cancel=[%s]", from, delete);		
		error_message (msg, "");
	}
	
	if (strcmp (from, delete) != 0) {
		info_message (txt_art_cannot_delete);
		return (redraw_screen);
	}
			
	clear_message ();
	
	joinpath (delete, homedir, ".cancel");
	if ((fp = fopen (delete, "w")) == NULL) {
		perror_message (txt_cannot_open, delete);
		return (redraw_screen);
	}
	chmod (delete, 0600);

	fprintf (fp, "Subject: cancel %s\n", note_h_messageid);
	if (*note_h_followup) {
		fprintf (fp, "Newsgroups: %s\n", note_h_followup);
	} else {
		fprintf (fp, "Newsgroups: %s\n", note_h_newsgroups);
	}
	fprintf (fp, "Control: cancel %s\n", note_h_messageid);
	if (*default_organization) { 
		fprintf (fp, "Organization: %s\n", default_organization);
		start_line_offset++;	
	}
	if (*reply_to) {
		fprintf (fp, "Reply-To: %s\n", reply_to);
		start_line_offset++;	
	}
	if (*note_h_distrib) {
		fprintf (fp, "Distribution: %s\n", note_h_distrib);
	} else {
		fprintf (fp, "Distribution: %s\n", my_distribution);
	}
	start_line_offset++;	
	fputc ('\n', fp);

	fprintf (fp, "Article cancelled from within tin [v%s PL%s]\n",
		VERSION, PATCHLEVEL);
	
	fclose (fp);
	
	while (1) {
		do {
			sprintf (msg, "%s [%.*s]: %c", txt_quit_edit_delete,
				cCOLS-30, note_h_subj, ch_default);
			wait_message (msg);
			MoveCursor (cLINES, (int) strlen (msg)-1);
			if ((ch = (char) ReadCh ()) == CR)
				ch = ch_default;
		} while (! strchr ("deq\033", ch));

		switch (ch) {
		case 'e':
			invoke_editor (delete, start_line_offset);
			redraw_screen = TRUE;
			break;

		case 'q':
		case ESC:
			unlink (delete);
			clear_message ();
			return (redraw_screen);

		case 'd':
			wait_message (txt_deleting_art);
			if (submit_file (delete)) {
				info_message (txt_art_deleted);
				goto delete_article_done;
			} else {
				error_message (txt_command_failed_s, delete);
				break;
			}
		}
	}

delete_article_done:
	find_mail_header (HEADER_SUBJECT, delete, buf);
	unlink (delete);
	update_art_posted_file (group, 'd', buf);

	return (redraw_screen);
}

/*
 * Crosspost an already existing article to another group (ie. local group)
 */
 
int 
crosspost_article (group, respnum)
	char *group;
	int respnum;
{
	char buf[LEN];
	char ch;
	char ch_default = 'p';
	FILE *fp;
	int i, ret_code = POSTED_NONE;
	
	start_line_offset = 4;

	if ((fp = fopen (article, "w")) == NULL) {
		perror_message (txt_cannot_open, article);
		return (ret_code);
	}
	chmod (article, 0600);

	fprintf (fp, "Subject: %s\n", eat_re (note_h_subj));
	fprintf (fp, "Newsgroups: %s\n", group);

	i = find_group_index (group);
	if (i >= 0 && active[i].attribute.organization != (char *) 0) {
		fprintf (fp, "Organization: %s\n", active[i].attribute.organization);
		start_line_offset++;
	}
	if (*reply_to) {
		fprintf (fp, "Reply-To: %s\n", reply_to);
		start_line_offset++;
	}
	if (*note_h_distrib) {
		fprintf (fp, "Distribution: %s\n", note_h_distrib);
		start_line_offset++;
	}

	fprintf (fp, "\n[ Article crossposted from %s ]", note_h_newsgroups);
	get_author (FALSE, respnum, buf);
	fprintf (fp, "\n[ Author was %s ]", buf);
	fprintf (fp, "\n[ Posted on %s ]\n\n", note_h_date);
  
	fseek (note_fp, note_mark[0], 0);
	copy_fp (note_fp, fp, "");

	add_signature (fp, FALSE);
	fclose (fp);

	while (1) {
		do {
			sprintf (msg, txt_quit_edit_xpost, 
				cCOLS-(strlen (txt_quit_edit_xpost)-1),
				note_h_subj, ch_default);
			wait_message (msg);
			MoveCursor (cLINES, (int) strlen (msg)-1);
			if ((ch = (char) ReadCh ()) == CR)
				ch = ch_default;
		} while (! strchr ("epq\033", ch));
		switch (ch) {
		case 'e':
			invoke_editor (article, start_line_offset);
			ret_code = POSTED_REDRAW;
			break;

		case 'q':
		case ESC:
			if (unlink_article)
				unlink (article);
			clear_message ();
			return (ret_code);

		case 'p':
			wait_message (txt_crosspost_an_article);
			if (submit_file (article)) {
				info_message (txt_art_posted);
				ret_code = POSTED_OK;
				reread_active_for_posted_arts = TRUE;		
				goto crosspost_done;
			} else {
				rename_file (article, dead_article);
				sprintf (buf, txt_art_rejected, dead_article);
				info_message (buf);
				sleep (3);
				return (ret_code);
			}
		}
	}

crosspost_done:
	find_mail_header (HEADER_SUBJECT, article, buf);
	update_art_posted_file (group, 'x', buf);

	if (unlink_article) {
		unlink (article);
	}
	
	return (ret_code);
}


void 
insert_x_headers (infile)
	char *infile;
{
	char line[LEN];
	char outfile[PATH_LEN];
	FILE *fp_in, *fp_out;
	int gotit = FALSE;
	
	if ((fp_in = fopen (infile, "r")) != NULL) {
		sprintf (outfile, "%s.%d", infile, process_id);
		if ((fp_out = fopen (outfile, "w")) != NULL) {
			while (fgets (line, sizeof (line), fp_in) != NULL) {
				if (! gotit && line[0] == '\n') {
					if (active[my_group[cur_groupnum]].type == GROUP_TYPE_MAIL) {
						fprintf (fp_out, "X-Mailer: TIN [version %s PL%s]\n\n",
							VERSION, PATCHLEVEL);
					} else {
						fprintf (fp_out, "X-Newsreader: TIN [version %s PL%s]\n\n",
							VERSION, PATCHLEVEL);
					}
					gotit = TRUE;
				} else {
					fputs (line, fp_out); 
				}	
			}
			fclose (fp_out);
			fclose (fp_in);
			rename_file (outfile, infile);
		}
	}
}


void 
find_reply_to_addr (respnum, from_addr)
	int respnum;
	char *from_addr;
{
	char buf[LEN];
	int found = FALSE;
	int len = 0;
	long orig_offset;

	orig_offset = ftell (note_fp);
	fseek (note_fp, 0L, 0);
	
	while (fgets (buf, sizeof (buf), note_fp) != NULL && 
		found == FALSE && buf[0] != '\n') {
		if (strncmp (buf, "Reply-To: ", 10) == 0) {
			strcpy (from_addr, &buf[10]);
			len = strlen (from_addr);
			from_addr[len-1] = '\0';
			sprintf (buf, "%s%s", from_addr, add_addr);
			strcpy (from_addr, buf);
			found = TRUE;
		}	
	}

	if (! found) {
		if (arts[respnum].name != (char *) 0 && 
		    arts[respnum].name != arts[respnum].from) { 
			sprintf (buf, "%s%s (%s)",
				 arts[respnum].from, add_addr,
				 arts[respnum].name);
			strcpy (from_addr, buf);
		} else { 
			sprintf (from_addr, "%s%s",
				 arts[respnum].from, add_addr);
		}	
	}
	
	fseek (note_fp, orig_offset, 0);
}

/*
 * If any arts have been posted by the user reread the active
 * file so that they are shown in the unread articles number
 * for each group at the group selection level.
 */
 		
void 
reread_active_after_posting ()
{
	if (reread_active_for_posted_arts) {
		yank_active_file ();		
		reread_active_for_posted_arts = FALSE;		
	}
}
